/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include "sudoku.h"

/*
 ******************************************************************************/
int fprint_sudoku(FILE *file, t_sudoku sudoku) {
	
	int i, j;
	int size = sudoku.size;
	
	for (i=0; i<size; i++) {
		for (j=0; j<size; j++) {
			fprintf(file,"%d",sudoku.grid[i][j].value);
		}
	}
	return 0;	
}

/*
 ******************************************************************************/
int print_sudoku_as_grid(t_sudoku sudoku) {

	int size = sudoku.size;
	int i, j, d=0, e=size*size;
	char format[10]; // this is enough for _really_ LARGE sudokus
	
	while ( (e /= 10) >= 1) d++;
	if (d > 1) d++;  // need spaces between numbers
	snprintf(format,10,"%%%dd",d);
	for (i=0; i<size; i++) {
		if (i%SIZE == 0) printf("\n");
		for (j=0; j<size; j++) {
			if (j>0 && j%SIZE == 0) printf("  ");
			printf(format,sudoku.grid[i][j].value);
		}
		printf("\n");
	}
  printf("\n");
	return 0;
}

/*
 ******************************************************************************/
int print_candidates_as_grid(t_sudoku sudoku) {

	int size = sudoku.size;
	int gr, gc, cr, cc, is, d=0, e=size*size;
	char format[10]; // this is enough for SIZE < 10
	
	while ( (e /= 10) >= 1) d++;
	d++; // need spaces between numbers
	snprintf(format,10,"%%%dd",d);
	for (is=0; is<size*(SIZE)*d+size+1; is++)
		if (is % (SIZE*d+1) == 0)
			printf("+");
		else
			printf("-");
	printf("\n");
	for (gr=0; gr<size; gr++) {
		for (cr=0; cr<SIZE; cr++) {
			printf("|");
			for (gc=0; gc<size; gc++) {
				if (sudoku.grid[gr][gc].value == 0) {
					for (cc=0; cc<SIZE; cc++) {
						if (sudoku.grid[gr][gc].cand[cr*SIZE+cc] > 0) {
							printf(format,sudoku.grid[gr][gc].cand[cr*SIZE+cc]);
						} else {
							for (is=0; is<d; is++) printf(" ");
						}
					}
				} else {
					for (cc=0; cc<SIZE; cc++) {
						printf(format,sudoku.grid[gr][gc].value);
					}
				}
				printf("|");
			}
			printf("\n");
		}
	for (is=0; is<size*(SIZE)*d+size+1; is++)
		if (is % (SIZE*d+1) == 0)
			printf("+");
		else
			printf("-");
		printf("\n");
	}
	return 0;
}

/*
 ******************************************************************************/
void fprint_explain(char *estr, t_sudoku sudoku, int verbosity) {
	
	char gstr[SIZE2*SIZE2 + SIZE2*SIZE2*SIZE2 + 1];
	int col=45, i;
	extern char fn_wsudognu[];
	extern FILE* explFile;
	
	if (verbosity >  9) fprintf(explFile,"%s ",estr);
	if (verbosity > 10) {
		sudoku_and_candidates_to_string(sudoku,gstr);
		for (i=strlen(estr); i<=col; i++)
			if (i%2 == 0) fprintf(explFile,"_");
			else fprintf(explFile,"_");
		fprintf(explFile," <a href=\"%s?sudoku=%s\" target=\"_blank\">sudoku</a>",fn_wsudognu,gstr);
	}
	if (verbosity >  9) fprintf(explFile,"\n");
}

/*
 ******************************************************************************/
void print_sudoku_and_candidates_as_html(const char *fn, t_sudoku sudoku) {

	int br, bc, r, c, cr, cc, ret;
	int elim[NUM_ET];
	t_sudoku dsudoku;
	FILE *out;
	char filename[512] = "sudoku.html";
	char gstr[SIZE2*SIZE2 + SIZE2*SIZE2*SIZE2 + 1];
	extern char fn_wsudognu[];
	extern FILE *explFile;
	
	if (fn != NULL) strncpy(filename,fn,511);
	filename[511] = '\0';
	out = fopen(filename,"w");
	fprintf(out,"<html>\n");
	fprintf(out,"<head><title>sudoku</title>\n");
	fprintf(out,"<style type=\"text/css\">\n");
	fprintf(out,"	body { background-color:#F8F8F8; color:black; margin: 10px 0 0 10px; }\n");
	fprintf(out,"	table.s1 { text-align: center; border-collapse: collapse; }\n");
	fprintf(out,"	table.s2 { border-collapse: collapse; }\n");
	fprintf(out,"	td.s1 { text-align: center; border: 1px solid black; padding: 1px 1px 1px 1px; margin: 0 0 0 0; }\n");
	fprintf(out,"	td.s2 { text-align: center; height: 57px; width: 57px; border: 1px solid black; padding: 1px 1px 1px 1px; margin: 0 0 0 0; font-weight: bold; font-size: 24px; }\n");
	fprintf(out,"	td.s3 { text-align: center; height: 15px; width: 15px; border: 0px solid black; padding: 0px 0px 0px 0px; margin: 0 0 0 0; font-size: 12px;}\n");
	fprintf(out,"</style>\n");
	fprintf(out,"</head>\n");
	fprintf(out,"<body>\n");
	fprintf(out,"<table class=\"s1\">\n");
	for (br=0; br<SIZE; br++) {
		fprintf(out,"<tr>\n");
		for (bc=0; bc<SIZE; bc++) {
			fprintf(out,"  <td class=\"s1\"><table class=\"s2\">\n");
			for (r=br*SIZE; r<(br+1)*SIZE; r++) {
				fprintf(out,"  <tr>\n");
				for (c=bc*SIZE; c<(bc+1)*SIZE; c++) {
					if (sudoku.grid[r][c].value != 0) {
						fprintf(out,"    <td class=\"s2\">%d</td>\n",sudoku.grid[r][c].value);
					} else {
						fprintf(out,"    <td class=\"s2\"><table class=\"s3\">\n");
						for (cr=0; cr<SIZE; cr++) {
							fprintf(out,"      <tr>");
							for (cc=0; cc<SIZE; cc++) {
								if (sudoku.grid[r][c].cand[cr*SIZE+cc] != 0) {
									fprintf(out,"<td class=\"s3\">%d</td>",sudoku.grid[r][c].cand[cr*SIZE+cc]);
								} else {
									fprintf(out,"<td class=\"s3\"> </td>");
								}
							}
							fprintf(out,"</tr>\n");
						}
						fprintf(out,"    </table></td>\n");
					}
				}
				fprintf(out,"  </tr>\n");
			}
			fprintf(out,"  </table></td>\n");
		}
		fprintf(out,"</tr>\n");
	}
	fprintf(out,"</table>\n<p>&nbsp;\n");

	copy_sudoku(sudoku,&dsudoku);
	set_stat_and_ncand_from_values_and_cands(&dsudoku);
	explFile = out;
	ret = apply_solution_technique1(&dsudoku, elim, NUM_ET-1, 10, out); 
	if (ret == 0) {
		fprintf(out,"<p><p>I have to guess or to track back now because of a\n");
		fprintf(out,"contradiction. I can't go to the next step from here.\n");
		fprintf(out,"<p>Please go back to the explanation overview and continue\n");
		fprintf(out,"from there\n");
	} else {
		sudoku_and_candidates_to_string(dsudoku,gstr);
		fprintf(out,"<p><p><a href=\"%s?sudoku=%s\" target=\"_self\">next step</a>",fn_wsudognu,gstr);
	}
	fprintf(out,"</body>\n");
	fprintf(out,"</html>\n");
	fclose(out);
	explFile = stdout;
}

/*
 ******************************************************************************/
void sudoku_and_candidates_to_string(t_sudoku sudoku, char gstr[SIZE2*SIZE2 + SIZE2*SIZE2*SIZE2 + 1]) {
	
	int i, j, k;
	int size = sudoku.size;
	
	for (i=0; i<size; i++) {
		for (j=0; j<size; j++) {
			gstr[i*size+j] = (char) (sudoku.grid[i][j].value + 48);
		}
	}
	for (i=0; i<size; i++) {
		for (j=0; j<size; j++) {
			for (k=0; k<size; k++) {
				gstr[(i+1)*size*size + j*size + k] = (char) (sudoku.grid[i][j].cand[k] + 48);
			}
		}
	}
	gstr[size*size*(size+1)] = '\0';
}
