/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#ifdef HARULIB

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <setjmp.h>
#include "sudoku.h"
#include "hpdf.h"

jmp_buf env;

/*
 ******************************************************************************/
void error_handler(HPDF_STATUS error_no, HPDF_STATUS detail_no, void *user_data){
	printf ("ERROR: error_no=%04X, detail_no=%u\n", (HPDF_UINT)error_no,
		(HPDF_UINT)detail_no);
	longjmp(env, 1);
}

/*
 ******************************************************************************/
void grid_to_pdf(t_field grid[SIZE2][SIZE2],int size, const char *title,
HPDF_Page page, HPDF_Font font, HPDF_UINT x0, HPDF_UINT y0, HPDF_UINT sqsize,
HPDF_REAL linewidth) {

	HPDF_UINT fontsize  = (2 * (sqsize + 1)) / 3;
	HPDF_UINT fontoffsy = (2 * (sqsize - fontsize + 1)) / 3;
	HPDF_UINT fontoffsx = (3 * fontoffsy) / 2;
	HPDF_UINT sudsize   = sqsize*size, x, y, origin[2]={x0, y0};
	int i, j;
	char buf[10];
	
	// print sudoku title a little above sudoku
	if (DEBUG) fprintf(stderr,"  draw title ...\n");
	HPDF_Page_SetFontAndSize(page, font, fontsize/2);
	HPDF_Page_BeginText(page);
	HPDF_Page_MoveTextPos(page, origin[0] + fontoffsx, origin[1] + fontoffsy + sudsize);
	HPDF_Page_ShowText(page, title);
	HPDF_Page_EndText(page);

	// set line property
	HPDF_Page_SetGrayStroke(page, 0.0);

	// print horizontal lines
	if (DEBUG) fprintf(stderr,"  draw horizontal lines and numbers ...\n");
	y = origin[1];
	for (i=0; i<10; i++) {
		if ((i%3) == 0) {
			HPDF_Page_SetLineWidth(page, 3*linewidth);
		} else {
			HPDF_Page_SetLineWidth(page, linewidth);
		}
		HPDF_Page_MoveTo(page, origin[0]          , y);
		HPDF_Page_LineTo(page, origin[0] + sudsize, y);
		HPDF_Page_Stroke(page);
		y += sqsize;
	}
	
	// print vertical lines
	if (DEBUG) fprintf(stderr,"  draw vertical lines ...\n");
	x = origin[0];
	for (i=0; i<10; i++) {
		if ((i%3) == 0) {
			HPDF_Page_SetLineWidth(page, 3*linewidth);
		} else {
			HPDF_Page_SetLineWidth(page, linewidth);
		}
		HPDF_Page_MoveTo(page, x, origin[1]);
		HPDF_Page_LineTo(page, x, origin[1] + sudsize);
		HPDF_Page_Stroke(page);
		x += sqsize;
	}
	
	// print numbers from top to bottom and from left to right
	if (DEBUG) fprintf(stderr,"  draw numbers ...\n");
	HPDF_Page_SetFontAndSize(page, font, fontsize);
	for (i=0; i<9; i++) {
		for (j=0; j<9; j++) {
			if (grid[i][j].value > 0) {
				HPDF_Page_BeginText(page);
				HPDF_Page_MoveTextPos(page,
					origin[0] + j*sqsize + fontoffsx,
					origin[1] - i*sqsize + fontoffsy + sudsize - sqsize);
				snprintf(buf, 12, "%d", grid[i][j].value);
				HPDF_Page_ShowText(page, buf);
				HPDF_Page_EndText(page);
			}
		}
	}
	if (DEBUG) fprintf(stderr,"  ... done printing\n");
}

/*
 ******************************************************************************/
int sudokus_to_pdf( char *filename, t_sudoku sudoku[4]) {
	
	char fname[512] = "sudoku.pdf";
	char buf[4][50], dstr[15];
	int i;
	int size = sudoku[0].size;
	HPDF_Doc  pdf;
	HPDF_Page page;
	HPDF_REAL height = 841; 	/* default resolution is 72 dpi */
	HPDF_REAL width  = 604;
	HPDF_UINT sqsize = (int) (width * 5.0 / 9.0 / 11.0), fontsize;
	HPDF_UINT x, y, dx, dy;
	HPDF_Font font;
	
	if (filename != NULL) strncpy(fname,filename,512);
	fname[19] = '\0';

	// create pdf object
	if (DEBUG) fprintf(stderr,"  create pdf object ...\n");
	pdf = HPDF_New(error_handler, NULL);
	if (!pdf) {
		fprintf(stderr,"error: cannot create PdfDoc object\n");
		return 1;
	}
	if (setjmp(env)) {
		HPDF_Free(pdf);
		fprintf(stderr,"error: setjmp\n");
		return 1;
	}
	if (DEBUG) fprintf(stderr,"  done creating pdf object\n");

	// add a new page object and get font
	if (DEBUG) fprintf(stderr,"  add page and create font ...\n");
	page = HPDF_AddPage(pdf);
	font = HPDF_GetFont(pdf, "Helvetica", NULL);
	if (DEBUG) fprintf(stderr,"  done adding page and creating font\n");

	// initialize
	if (DEBUG) fprintf(stderr,"  initialize page properties ...\n");
	HPDF_Page_SetHeight(page, height);
	HPDF_Page_SetWidth(page, width);
	HPDF_Page_SetGrayFill(page, 0.0);
	if (DEBUG) fprintf(stderr,"  done initializing page properties\n");

	// offset parameters for the sudokus
	y = (int) ((height - 2.0 * size * sqsize)/6.0);
	x = (int) ((width  - 2.0 * size * sqsize)/3.0);
	dx = size * sqsize + x;
	dy = size * sqsize + 2 * sqsize;

	// strings for rating of sudokus
	for (i=0; i<4; i++) {
		if      (sudoku[i].rating > 199) strncpy(dstr,"fiendish",8);
		else if (sudoku[i].rating >  99) strncpy(dstr,"evil",8);
		else if (sudoku[i].rating >   9) strncpy(dstr,"hard",8);
		else if (sudoku[i].rating >   1) strncpy(dstr,"moderate",8);
		else if (sudoku[i].rating ==  1) strncpy(dstr,"easy",8);
		else strncpy(dstr,"",8);
		snprintf(buf[i], 50, "Sudoku %d, difficulty %d - %s",i+1,(int)sudoku[i].rating,dstr);
	}

	// print the 4 grids	
	if (sudoku[0].rating > 0) grid_to_pdf(sudoku[0].grid, size, buf[0], page, font, x   , y+dy, sqsize, 0.5);
	if (sudoku[1].rating > 0) grid_to_pdf(sudoku[1].grid, size, buf[1], page, font, x+dx, y+dy, sqsize, 0.5);
	if (sudoku[2].rating > 0) grid_to_pdf(sudoku[2].grid, size, buf[2], page, font, x   , y   , sqsize, 0.5);
	if (sudoku[3].rating > 0) grid_to_pdf(sudoku[3].grid, size, buf[3], page, font, x+dx, y   , sqsize, 0.5);
	
	// print a nice header
	if (DEBUG) fprintf(stderr,"  print header ...\n");
	strcpy(dstr,"SUDOKU");
	fontsize = (int) (height / 12.0);
	y        = (int) (height - fontsize - fontsize / 2);
	x        = (int) (width / 2.0 - (fontsize * strlen(dstr) * 2) / 6.0);
	HPDF_Page_SetFontAndSize(page, font, fontsize);
	HPDF_Page_BeginText(page);
	HPDF_Page_MoveTextPos(page, x, y);
	HPDF_Page_ShowText(page, dstr);
	HPDF_Page_EndText(page);
	if (DEBUG) fprintf(stderr,"  done printing header\n");

	// save the document to a file
	if (DEBUG) fprintf(stderr,"  save pdf to file ...\n");
	HPDF_SaveToFile(pdf, fname);
	if (DEBUG) fprintf(stderr,"  done saving pdf to file\n");

	// clean up
	if (DEBUG) fprintf(stderr,"  clean up ...\n");
	HPDF_Free(pdf);
	if (DEBUG) fprintf(stderr,"  done cleaning up\n");
	
	return 0;
}

#endif
