/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#ifndef _SUDOKU_H
#define _SUDOKU_H

#include <stdio.h>

#define SIZE 3
#define SIZE2 SIZE*SIZE
#define DEBUG 0
#define VERSION "1.05"
#define MAX_STRLEN 1024

// number of elimination strategies, must correspond to number of elim_types, see below
#define NUM_ET 16
typedef enum {
	et_wg,   // wrong guess
	et_hs,   // hidden single
	et_s,    // single
	et_hd,   // hidden double
	et_d,    // double
	et_b,    // block-line interaction
	et_l,    // line-block interaction
	et_ht,   // hidden triple
	et_t,    // triple
	et_hq,   // hidden quadruple
	et_q,    // quadruple
	et_xw,   // x-wing
	et_sf,   // swordfish
	et_jf,   // jellyfish
	et_fc,   // forcing chain
	et_g     // guess
} elim_type;

typedef enum {
	stat_undef,
	stat_free,
	stat_given,
	stat_ggiven,
	stat_hsingle,
	stat_single,
	stat_guess,
	stat_fchain
} status;

typedef struct {
	int value;
	int cand[SIZE2];
	int ncand;
	status stat;
} t_field;

typedef struct {
	t_field grid[SIZE2][SIZE2];
	int size;
	float rating;
} t_sudoku;

/* getopt.c: parse options
 ******************************************************************************/
int getoptions(int argc, char *argv[]);
void print_version(FILE *f);
void print_usage(FILE *f);
void print_help(FILE *f);
void print_contact(FILE *f);

/* cand.c: deal with candidates
 ******************************************************************************/
void update_num_candidates(t_sudoku *sudoku);
void remove_candidates(t_sudoku *sudoku, int row, int col);
void get_candidates(t_sudoku *sudoku);
int get_random_cand(t_field cell);

/* grid.c: deal with complete grid
 ******************************************************************************/
void copy_sudoku(t_sudoku sudokuSrc ,t_sudoku *sudoku_Dest);
void reset_sudoku(t_sudoku *sudoku);
void transpose_sudoku(t_sudoku *sudoku);
void exchange_block_rc(t_sudoku *sudoku, char ent, int brcMap[SIZE]);
void exchange_rc(t_sudoku *sudoku, char ent, int rcMap[SIZE2]);
void exchange_value(t_sudoku *sudoku, int valMap[SIZE2]);
int read_sudoku_from_file(FILE *f, t_sudoku *sudoku);
int check_sudoku(t_sudoku sudoku);
int get_num_cell_with_stat1(t_sudoku sudoku, status stat);
int check_sudoku_symmetry(t_sudoku sudoku);
int read_sudoku_and_candidates_from_file(FILE *f, t_sudoku *sudoku);
void set_stat_and_ncand_from_values_and_cands(t_sudoku *sudoku);
int get_basic_rating(t_sudoku sudoku, int elim[NUM_ET]);

int get_random_cell(t_field grid[SIZE2][SIZE2], int size, status stat);
int get_random_cell2(t_field grid[SIZE2][SIZE2], int size, status stat);

/* solve.c: solve sudoku
 ******************************************************************************/
int execute_thread_solve(void *arg[]);
int solve_sudoku_and_check_uniqueness(FILE *in, int verbosity);
int solve_sudoku(t_sudoku *sudoku, int elim[NUM_ET], int depth, int verbosity, int doRate);
int check_uniqueness1(t_sudoku sudoku,t_sudoku ssudoku);
int solve_sudoku_rec(t_sudoku *sudoku, t_sudoku *dsudoku, int elim[NUM_ET], int r, int c, int cc, int depth, int verbosity, int doRate);
int eliminate(t_sudoku *sudoku, int elim[NUM_ET], int depth, int verbosity, int doRate);
int apply_solution_technique1(t_sudoku *sudoku, int elim[NUM_ET], int depth, int verbosity, FILE *out);
void find_cell_and_cand_for_guess(t_field grid[SIZE2][SIZE2], int size, int *r, int *c, int *cc);

/* create_sudoku.c
 ******************************************************************************/
int execute_thread_create(void *arg[]);
int create_sudoku(t_sudoku *sudoku);
int find_next_given_symm(t_sudoku sudoku, int *r, int *c, int *cc, int *gr, int *gc);
int find_next_given_unsymm(t_sudoku sudoku, int *r, int *c, int *cc);

/* tupel.c: deal with tupels
 ******************************************************************************/
int single(t_sudoku *sudoku, int doElim, int verbosity);
int tuple(t_sudoku *sudoku, int tuple, int doElim, int verbosity);
void find_tuple(t_sudoku *sudoku, int tuple, int itup, int c, int *cc, int colVals[SIZE2], int rowVals[SIZE2], int blkVals[SIZE2], int doElim, int verbosity, int *ntup);
void tuple_crb(t_sudoku *sudoku, int tuple, int *cc, int c, char ent, int doElim, int verbosity,	int *ntup);

/* hidden_tupel.c: deal with hidden tupels
 ******************************************************************************/
int hidden_single(t_sudoku *sudoku, int doElim, int verbosity);
int hidden_tuple(t_sudoku *sudoku, int tuple, int doElim, int verbosity);
void find_hidden_tuple(t_sudoku *sudoku, int tuple, int itup, int c, int *cc, int colVals[SIZE2], int rowVals[SIZE2], int blkVals[SIZE2], int doElim, int verbosity, int *ntup);
void hidden_tuple_crb(t_sudoku *sudoku, int tuple, int *cc, int c, char ent, int doElim, int verbosity, int *ntup);

/* block_line.c: deal with block-line interaction
 ******************************************************************************/
int block_line(t_sudoku *sudoku, int doElim, int verbosity);
int line_block(t_sudoku *sudoku, int doElim, int verbosity);

/* fish.c: deal with x-wing, swordfish, jellyfish
 ******************************************************************************/
int fish(t_sudoku *sudoku, int fish, int doElim, int verbosity);
void find_fish(t_sudoku *sudoku, int fish, int ifish, int *r, int *c, int cc, int doElim, int verbosity, int *nfish);
int fish_rc(t_sudoku *sudoku, int fish, int cc,	int *r, int *c, const char ent[12], int doElim, int verbosity);

/* forcing_chain.c
 ******************************************************************************/
int forcing_chain(t_sudoku *sudoku, int verbosity, int *fcl);
void diff_grid(t_field grid0[SIZE2][SIZE2],t_field grid1[SIZE2][SIZE2], t_field egrid[SIZE2][SIZE2], int size);
void grid_to_zero(t_field grid[SIZE2][SIZE2], int size);
//void merge_grid(t_field grid0[SIZE2][SIZE2],t_field grid1[SIZE2][SIZE2], t_field egrid[SIZE2][SIZE2], int size);
void intersect_grid_no_cands(t_field grid0[SIZE2][SIZE2],t_field grid1[SIZE2][SIZE2], t_field egrid[SIZE2][SIZE2], int size);
void intersect_grid_with_cands(t_field grid0[SIZE2][SIZE2],t_field grid1[SIZE2][SIZE2], t_field egrid[SIZE2][SIZE2], int size);

/* ascii terminal, html, pdf, svg output
 ******************************************************************************/
int fprint_sudoku(FILE *file, t_sudoku sudoku);
int print_sudoku_as_grid(t_sudoku sudoku);
int print_candidates_as_grid(t_sudoku sudoku);
void print_sudoku_and_candidates_as_html(const char *fn, t_sudoku sudoku);
void sudoku_and_candidates_to_string(t_sudoku sudoku, char gstr[SIZE2*SIZE2 + SIZE2*SIZE2*SIZE2 + 1]);
void fprint_explain(char *estr, t_sudoku sudoku, int verbosity);

/* svg output */
void sudokus_to_svg(char *fn, t_sudoku sudokus[4]);
void grid_to_svg(FILE *f, t_field grid[SIZE2][SIZE2], int size,  float x0, float y0, float squareSize, float thickLineWidth, float thinLineWidth, int sudokuFontSz, int numberFontSz, char *sudTitle);
void get_sudoku_title(int isud, int rating, char *title);

#ifdef HARULIB
#include "hpdf.h"
int sudokus_to_pdf(char *fn, t_sudoku sudokus[4]);
void grid_to_pdf(t_field grid[SIZE2][SIZE2], int size, const char *title, HPDF_Page page, HPDF_Font font, HPDF_UINT x0, HPDF_UINT y0, HPDF_UINT sqsize, HPDF_REAL linewidth);
#endif

#endif
