/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * vncviewer.c - the Xt-based VNC viewer.
 */

#include "vncviewer.h"
#include <X11/Xaw/Toggle.h>

char *programName;
XtAppContext appContext;
Display* dpy;

Widget toplevel;

void set_sbwidth(int sbw) {
	char *q, *p, t[5];
	int i, k, N = 4;
	int db = 0;

	if (sbw < 1) {
		sbw = 2;
	} else if (sbw > 100) {
		sbw = 100;
	}
	if (db) fprintf(stderr, "sbw: %d\n", sbw);

	sprintf(t, "%4d", sbw);
	k = 0;
	while (fallback_resources[k] != NULL) {
		q = strstr(fallback_resources[k], "horizontal.height: ");
		if (!q) {
			q = strstr(fallback_resources[k], "vertical.width: ");
		}
		if (q) {
			p = strdup(fallback_resources[k]);
			q = strstr(p, ":   ");
			if (q) {
				q++;
				q++;
				for (i=0; i < N; i++) {
					*(q+i) = t[i];
				}
				fallback_resources[k] = p;
				if (db) fprintf(stderr, "res: %s\n", p);
			}
		}
		k++;
	}
}

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

void unixpw(char *instr) {
	char *str, *q, *infile = NULL;
	FILE *in;
	int i, rmfile = 0;
	struct stat sb;
	int N = 99;
	char username[100], passwd[100];

	for (i=0; i<100; i++) {
		username[i] = '\0';
		passwd[i] = '\0';
	}

	if (instr == NULL) {
		return;
	} else if (!strcmp(instr, "")) {
		return;
	}

	str = strdup(instr);
	
	if (strstr(str, "rm:") == str) {
		rmfile = 1;
		infile = str + strlen("rm:");
	} else if (stat(str, &sb) == 0) {
		infile = str;
	}
	if (!strcmp(str, ".")) {
		char *p;
		fprintf(stderr, "\nUnix Username: ");
		if (fgets(username, N, stdin) == NULL) {
			exit(1);
		}
		p = getpass("Unix Password: ");
		if (! p) {
			exit(1);
		}
		strncpy(passwd, p, N);
		fprintf(stderr, "\n");
		
	} else if (!strcmp(str, "-")) {
		char *p, *q;
		p = getpass("unixuser@unixpasswd: ");
		if (! p) {
			exit(1);
		}
		q = strchr(p, '@');
		if (! q) {
			exit(1);
		}
		*q = '\0';
		strncpy(username, p, N);
		strncpy(passwd, q+1, N);
		
	} else if (infile) {
		in = fopen(infile, "r");	
		if (in == NULL) {
			fprintf(stderr, "failed to open -unixpw file.\n");
			exit(1);
		}
		if (fgets(username, N, in) == NULL) {
			exit(1);
		}
		if (fgets(passwd, N, in) == NULL) {
			exit(1);
		}
		fclose(in);
		fprintf(stderr, "read username@passwd from file: %s\n", infile);
		if (rmfile) {
			fprintf(stderr, "deleting username@passwd file:  %s\n", infile);
			unlink(infile);
		}
	} else if (strchr(str, '@'))  {
		char *q = strchr(str, '@');
		*q = '\0';
		strncpy(username, str, N);
		strncpy(passwd, q+1, N);
	} else {
		exit(1);
	}

	free(str);

	if (! getenv("SSVNC_UNIXPW_NOESC")) {
		SendKeyEvent(XK_Escape, 1);
		SendKeyEvent(XK_Escape, 0);
	}

	q = username;
	while (*q != '\0' && *q != '\n') {
		char c = *q;
		if (c >= 0x20 && c <= 0x07e) {
			KeySym ks = (KeySym) c;
			SendKeyEvent(ks, 1);
			SendKeyEvent(ks, 0);
		}
		q++;
	}

	SendKeyEvent(XK_Return, 1);
	SendKeyEvent(XK_Return, 0);
	
	q = passwd;
	while (*q != '\0' && *q != '\n') {
		char c = *q;
		if (c >= 0x20 && c <= 0x07e) {
			KeySym ks = (KeySym) c;
			SendKeyEvent(ks, 1);
			SendKeyEvent(ks, 0);
		}
		q++;
	}

	SendKeyEvent(XK_Return, 1);
	SendKeyEvent(XK_Return, 0);
}

int
main(int argc, char **argv)
{
	int i, save_sbw;
	programName = argv[0];

  /* The -listen option is used to make us a daemon process which listens for
     incoming connections from servers, rather than actively connecting to a
     given server. The -tunnel and -via options are useful to create
     connections tunneled via SSH port forwarding. We must test for the
     -listen option before invoking any Xt functions - this is because we use
     forking, and Xt doesn't seem to cope with forking very well. For -listen
     option, when a successful incoming connection has been accepted,
     listenForIncomingConnections() returns, setting the listenSpecified
     flag. */

	for (i = 1; i < argc; i++) {
		if (strcmp(argv[i], "-listen") == 0) {
			listenForIncomingConnections(&argc, argv, i);
			break;
		}
		if (strcmp(argv[i], "-tunnel") == 0 || strcmp(argv[i], "-via") == 0) {
			if (!createTunnel(&argc, argv, i)) {
				exit(1);
			}
			break;
		}
	}

	if (argc > 1 && strstr(argv[1], "-h") == argv[1]) {
		usage();
		return 0;
	}

  /* Call the main Xt initialisation function.  It parses command-line options,
     generating appropriate resource specs, and makes a connection to the X
     display. */

	appData.sbWidth = 0;
	if (getenv("VNCVIEWER_SBWIDTH")) {
		int sbw = atoi(getenv("VNCVIEWER_SBWIDTH"));
		if (sbw > 0) {
			appData.sbWidth = sbw;
		}
	}
	if (appData.sbWidth == 0) {
		int i, sbw = 0;
		for (i = 1; i < argc - 1; i++) {
			if (!strcmp(argv[i], "-sbwidth")) {
				sbw = atoi(argv[i+1]);
			}
		}
		if (sbw > 0) {
			appData.sbWidth = sbw;
		}
	}
	save_sbw = appData.sbWidth;
	if (save_sbw > 0) {
		set_sbwidth(save_sbw);
	} else {
		set_sbwidth(6);
	}

	toplevel = XtVaAppInitialize(&appContext, "Vncviewer", cmdLineOptions,
	    numCmdLineOptions, &argc, argv, fallback_resources,
	    XtNborderWidth, 0, NULL);

	dpy = XtDisplay(toplevel);

  /* Interpret resource specs and process any remaining command-line arguments
     (i.e. the VNC server name).  If the server name isn't specified on the
     command line, getArgsAndResources() will pop up a dialog box and wait
     for one to be entered. */

	GetArgsAndResources(argc, argv);

	if (save_sbw) {
		appData.sbWidth = save_sbw;
	}

  /* Unless we accepted an incoming connection, make a TCP connection to the
     given VNC server */

	if (!listenSpecified) {
		if (!ConnectToRFBServer(vncServerHost, vncServerPort)) {
			exit(1);
		}
		if (appData.repeaterUltra != NULL) {
			char tmp[256];
			if (ReadFromRFBServer(tmp, 12)) {
				if (strstr(tmp, "RFB 000.000") == tmp) {
					int i;
					for (i=0; i<256; i++) {
						tmp[i] = '\0';
					}
					for (i=0; i<250; i++) {
						if (i >= strlen(appData.repeaterUltra)) {
							break;
						}
						tmp[i] = appData.repeaterUltra[i];
					}
					WriteExact(rfbsock, tmp,250); 
				}
			}
		}
	}

  /* Initialise the VNC connection, including reading the password */

	if (!InitialiseRFBConnection()) {
		exit(1);
	}
	if (appData.unixPW != NULL) {
		unixpw(appData.unixPW);
	} else if (getenv("SSVNC_UNIXPW")) {
		unixpw(getenv("SSVNC_UNIXPW"));
	}

  /* Create the "popup" widget - this won't actually appear on the screen until
     some user-defined event causes the "ShowPopup" action to be invoked */

	CreatePopup();
	CreateScaleN();
	CreateQuality();
	CreateCompress();
	CreateChat();

  /* Find the best pixel format and X visual/colormap to use */

	SetVisualAndCmap();

  /* Create the "desktop" widget, and perform initialisation which needs doing
     before the widgets are realized */

	ToplevelInitBeforeRealization();

	DesktopInitBeforeRealization();

  /* "Realize" all the widgets, i.e. actually create and map their X windows */

	XtRealizeWidget(toplevel);

  /* Perform initialisation that needs doing after realization, now that the X
     windows exist */

	InitialiseSelection();

	ToplevelInitAfterRealization();

	DesktopInitAfterRealization();

  /* Tell the VNC server which pixel format and encodings we want to use */

	SetFormatAndEncodings();

  /* Now enter the main loop, processing VNC messages.  X events will
     automatically be processed whenever the VNC connection is idle. */

	while (1) {
		if (!HandleRFBServerMessage()) {
			break;
		}
	}

	Cleanup();

	return 0;
}

/*
 * Toggle8bpp
 */

static int last_ncolors = 0;
static int save_useBGR233 = 0;
static Bool save_useBGR565 = False;

static Widget b8    = NULL;
static Widget b16   = NULL;
static Widget bfull = NULL;

int do_format_change = 0;
int do_cursor_change = 0;
int do_fb_update = 0.0;
static void schedule_format_change(void) {
	do_format_change = 1;
	do_cursor_change = 0;
}
extern double dnow(void);
static void schedule_fb_update(void) {
	do_fb_update = dnow();
}
static void init_format_change(void) {
	appDataNew.useBGR233       = appData.useBGR233;
	appDataNew.useBGR565       = appData.useBGR565;
	appDataNew.useGreyScale    = appData.useGreyScale;
	appDataNew.enableJPEG      = appData.enableJPEG;
	appDataNew.encodingsString = appData.encodingsString;
	appDataNew.useRemoteCursor = appData.useRemoteCursor;
	appDataNew.useX11Cursor    = appData.useX11Cursor;
	appDataNew.useRawLocal     = appData.useRawLocal;
	appDataNew.qualityLevel    = appData.qualityLevel;
	appDataNew.compressLevel   = appData.compressLevel;
}
void cutover_format_change(void) {
	appData.useBGR233       = appDataNew.useBGR233;
	appData.useBGR565       = appDataNew.useBGR565;
	appData.useGreyScale    = appDataNew.useGreyScale;
	appData.enableJPEG      = appDataNew.enableJPEG;
	appData.encodingsString = appDataNew.encodingsString;
	appData.useRemoteCursor = appDataNew.useRemoteCursor;
	appData.useX11Cursor    = appDataNew.useX11Cursor;
	appData.useRawLocal     = appDataNew.useRawLocal;
	appData.qualityLevel    = appDataNew.qualityLevel;
	appData.compressLevel   = appDataNew.compressLevel;
}

void
Toggle8bpp(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	fprintf(stderr, "Toggle8bpp: %d\n", appData.useBGR233);
	b8 = w;
	init_format_change();
	if (appData.useBGR233) {
		last_ncolors = appData.useBGR233;
		appDataNew.useBGR233 = 0;
		appDataNew.useBGR565 = save_useBGR565;
		fprintf(stderr, "8bpp: off\n");
	} else {
		if (!last_ncolors) last_ncolors = 256;
		appDataNew.useBGR233 = last_ncolors;
		save_useBGR565 = appData.useBGR565;
		appDataNew.useBGR565 = False;
		fprintf(stderr, "8bpp: on (%d colors)\n", appDataNew.useBGR233);
	}
	schedule_format_change();
}


void
Toggle16bpp(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	fprintf(stderr, "Toggle16bpp: %d\n", appData.useBGR565);
	b16 = w;
	init_format_change();
	if (appData.useBGR565) {
		appDataNew.useBGR565 = False;
		appDataNew.useBGR233 = save_useBGR233;
		fprintf(stderr, "16bpp: off\n");
	} else {
		appDataNew.useBGR565 = True;
		save_useBGR233 = appData.useBGR233;
		appDataNew.useBGR233 = 0;
		fprintf(stderr, "16bpp: on\n");
	}
	schedule_format_change();
}

void
ToggleFullColor(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	fprintf(stderr, "ToggleFullColor\n");
	bfull = w;
	init_format_change();
	if (appData.useBGR565 || appData.useBGR233) {
		save_useBGR565 = appData.useBGR565;
		appDataNew.useBGR565 = False;
		save_useBGR233 = appData.useBGR233;
		appDataNew.useBGR233 = 0;
		fprintf(stderr, "FullColor: on\n");
	} else {
		if (save_useBGR565) {
			appDataNew.useBGR565 = True;
			appDataNew.useBGR233 = 0;
			fprintf(stderr, "FullColor off -> 16bpp.\n");
		} else {
			appDataNew.useBGR565 = False;
			if (!save_useBGR233) save_useBGR233 = 256;
			appDataNew.useBGR233 = save_useBGR233;
			fprintf(stderr, "FullColor off -> 8bpp.\n");
		}
	}
	schedule_format_change();
}

/*
 * ToggleNColors
 */

static Widget w256 = NULL;
static Widget w64  = NULL;
static Widget w8   = NULL;

void
Toggle256Colors(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	w256 = w;
	if (appData.useBGR233 != 256) {
		fprintf(stderr, "256 colors: on\n");
		init_format_change();
		last_ncolors = appDataNew.useBGR233 = 256;
		save_useBGR565 = appData.useBGR565;
		appDataNew.useBGR565 = False;
		schedule_format_change();
	}
}

void
Toggle64Colors(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	w64 = w;
	if (appData.useBGR233 != 64) {
		fprintf(stderr, "64 colors: on\n");
		init_format_change();
		last_ncolors = appDataNew.useBGR233 = 64;
		save_useBGR565 = appData.useBGR565;
		appDataNew.useBGR565 = False;
		schedule_format_change();
	}
}

void
Toggle8Colors(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	w8 = w;
	if (appData.useBGR233 != 8) {
		fprintf(stderr, "8 colors: on\n");
		init_format_change();
		last_ncolors = appDataNew.useBGR233 = 8;
		save_useBGR565 = appData.useBGR565;
		appDataNew.useBGR565 = False;
		schedule_format_change();
	}
}

void
ToggleGreyScale(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	fprintf(stderr, "ToggleGreyScale\n");
	init_format_change();
	if (appData.useGreyScale) {
		appDataNew.useGreyScale = False;
		fprintf(stderr, "greyscale: off\n");
	} else {
		appDataNew.useGreyScale = True;
		fprintf(stderr, "greyscale: on\n");
	}
	schedule_format_change();
}

/*
 * ToggleJPEG
 */

void
ToggleJPEG(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	init_format_change();
	if (appData.enableJPEG) {
		appDataNew.enableJPEG = False;
		fprintf(stderr, "JPEG: off\n");
	} else {
		appDataNew.enableJPEG = True;
		fprintf(stderr, "JPEG: on\n");
	}
	schedule_format_change();
}

/*
 * ToggleTightZRLE
 */

static Bool usingZRLE = False;
static Bool usingZYWRLE = False;
extern int skip_maybe_sync;

void
ToggleTightZRLE(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char prefTight[] = "copyrect tight zrle zywrle zlib hextile corre rre raw";
	char prefZRLE[]  = "copyrect zrle zywrle tight zlib hextile corre rre raw";
	init_format_change();
	if (! appData.encodingsString) {
		appDataNew.encodingsString = strdup(prefZRLE);
		usingZRLE = True;
		fprintf(stderr, "prefer: ZRLE\n");
	} else {
		char *t, *z;
		static int first = 1;
		t = strstr(appData.encodingsString, "tight");
		z = strstr(appData.encodingsString, "zrle");
		if (first && usingZRLE) {
			appDataNew.encodingsString = strdup(prefTight);
			usingZRLE = False;
			usingZYWRLE = False;
		} else if (! t) {
			appDataNew.encodingsString = strdup(prefZRLE);
			usingZRLE = True;
			fprintf(stderr, "prefer: ZRLE\n");
		} else if (! z) {
			appDataNew.encodingsString = strdup(prefTight);
			usingZRLE = False;
			usingZYWRLE = False;
			skip_maybe_sync = 0;
			fprintf(stderr, "prefer: Tight\n");
		} else {
			if (t < z) {
				appDataNew.encodingsString = strdup(prefZRLE);
				usingZRLE = True;
				fprintf(stderr, "prefer: ZRLE\n");
			} else {
				appDataNew.encodingsString = strdup(prefTight);
				usingZRLE = False;
				usingZYWRLE = False;
				skip_maybe_sync = 0;
				fprintf(stderr, "prefer: Tight\n");
			}
		}
		first = 0;
	}
	schedule_format_change();
}

void
ToggleZRLEZYWRLE(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char prefZYWRLE[] = "copyrect zywrle zrle tight zlib hextile corre rre raw";
	char prefZRLE[]   = "copyrect zrle zywrle tight zlib hextile corre rre raw";
	init_format_change();
	usingZRLE = True;
	if (! appData.encodingsString) {
		appDataNew.encodingsString = strdup(prefZYWRLE);
		usingZYWRLE = True;
		fprintf(stderr, "prefer: ZYWRLE\n");
	} else {
		char *z, *w;
		w = strstr(appData.encodingsString, "zywrle");
		z = strstr(appData.encodingsString, "zrle");
		if (usingZYWRLE) {
			appDataNew.encodingsString = strdup(prefZRLE);
			fprintf(stderr, "prefer: ZRLE\n");
			usingZYWRLE = False;
			skip_maybe_sync = 0;
		} else {
			appDataNew.encodingsString = strdup(prefZYWRLE);
			fprintf(stderr, "prefer: ZYWRLE\n");
			usingZYWRLE = True;
		}
	}
	schedule_format_change();
}

/*
 * ToggleViewOnly
 */

void
ToggleViewOnly(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.viewOnly) {
		appData.viewOnly = False;
		fprintf(stderr, "viewonly: off\n");
	} else {
		appData.viewOnly = True;
		fprintf(stderr, "viewonly: on\n");
	}
	Xcursors(1);
}

void
ToggleCursorShape(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	init_format_change();
	if (appData.useRemoteCursor) {
		appDataNew.useRemoteCursor = False;
		fprintf(stderr, "useRemoteCursor: off\n");
	} else {
		appDataNew.useRemoteCursor = True;
		fprintf(stderr, "useRemoteCursor: on\n");
	}
	schedule_format_change();
	if (!appDataNew.useRemoteCursor) {
		do_cursor_change = 1;
	} else {
		do_cursor_change = -1;
	}
}

void
ToggleCursorAlpha(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useCursorAlpha) {
		appData.useCursorAlpha = False;
		fprintf(stderr, "useCursorAlpha: off\n");
	} else {
		appData.useCursorAlpha = True;
		fprintf(stderr, "useCursorAlpha: on\n");
	}
}

void
ToggleX11Cursor(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	init_format_change();
	if (appData.useX11Cursor) {
		appDataNew.useX11Cursor = False;
		fprintf(stderr, "useX11Cursor: off\n");
	} else {
		appDataNew.useX11Cursor = True;
		fprintf(stderr, "useX11Cursor: on\n");
	}
	schedule_format_change();
	do_cursor_change = 1;
}

void
ToggleBell(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBell) {
		appData.useBell = False;
		fprintf(stderr, "useBell: off\n");
	} else {
		appData.useBell = True;
		fprintf(stderr, "useBell: on\n");
	}
}

void
ToggleRawLocal(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	init_format_change();
	if (appData.useRawLocal) {
		appDataNew.useRawLocal = False;
		fprintf(stderr, "useRawLocal: off\n");
	} else {
		appDataNew.useRawLocal = True;
		fprintf(stderr, "useRawLocal: on\n");
	}
	schedule_format_change();
}

void
ToggleServerInput(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.serverInput) {
		appData.serverInput= False;
		fprintf(stderr, "serverInput: off\n");
		SendServerInput(True);
	} else {
		appData.serverInput = True;
		fprintf(stderr, "serverInput: on\n");
		SendServerInput(False);
	}
}

Bool _sw1_ = False;	/* XXX this is a weird bug... */
Bool _sw2_ = False;
Bool _sw3_ = False;
Bool selectingSingleWindow = False;

extern Cursor bogoCursor;

void
ToggleSingleWindow(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.singleWindow) {
		appData.singleWindow= False;
		fprintf(stderr, "singleWindow: off\n");
		SendSingleWindow(-1, -1);
	} else {
		appData.singleWindow = True;
		selectingSingleWindow = True;
		fprintf(stderr, "singleWindow: on\n");
		if (bogoCursor != None) {
			XDefineCursor(dpy, desktopWin, bogoCursor);
		}
	}
}

void raiseme(int force);
void AppendChatInput(char *);

void printChat(char *str, Bool raise) {
	if (appData.termChat) {
		if (raise) {
			raiseme(0);
		}
		fprintf(stderr, str);
	} else {
		if (raise) {
			ShowChat();
		}
		AppendChatInput(str);
	}
}

void
ToggleTextChat(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.chatActive) {
		printChat("\n*SentClose*\n\n", False);
		SendTextChatClose();
		HideChat();
		appData.chatActive= False;
	} else {
		ShowChat();
		SendTextChatOpen();
		if (appData.termChat) {
			printChat("\n*SentOpen*\n\nSend: ", True);
		} else {
			printChat("\n*SentOpen*\n", True);
		}
		appData.chatActive = True;
	}
}

void
ToggleFileXfer(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.fileActive) {
		HideFile(w, ev, params, num_params);
		appData.fileActive= False;
	} else {
		ShowFile(w, ev, params, num_params);
		appData.fileActive = True;
	}
}

static int fooHandler(Display *dpy, XErrorEvent *error) {
	return 0;
}

void raiseme(int force) {
	if ((force || appData.termChat) && getenv("WINDOWID")) {
		unsigned long w;
		if (sscanf(getenv("WINDOWID"), "%lu", &w) == 1)  {
			;
		} else if (sscanf(getenv("WINDOWID"), "0x%lx", &w) == 1)  {
			;
		} else {
			w = 0;
		}
		if (w != 0) {
			XErrorHandler old = XSetErrorHandler(fooHandler);
			XMapRaised(dpy, (Window) w);
			XSync(dpy, False);
			XSetErrorHandler(old);
		}
	}
}

void set_server_scale(int n) {
	if (n >= 1 && n < 100) {
		int w = si.framebufferWidth;
		int h = si.framebufferHeight;
		appData.serverScale = n;
		SendServerScale(n);
		SendFramebufferUpdateRequest(0, 0, w, h, False);
		schedule_fb_update();
	}
}

void
DoServerScale(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char str[100], *s, *q;
	int n;
	if (1) {
		s = DoScaleDialog();
	} else {
		raiseme(1);
		fprintf(stderr, "\n\n\a\nEnter integer n for 1/n server scaling: ");
		str[0] = '\0';
		fgets(str, 100, stdin); 
		s = str;
		q = strstr(str, "\n");
		if (q) *q = '\0';
	}
	if (s[0] != '\0') {
		n = atoi(s);
		set_server_scale(n);
	}
}

void set_server_quality(int n) {
	fprintf(stderr, "set_quality: %d\n", n);
	if (n >= 0 && n <= 9) {
		int w = si.framebufferWidth;
		int h = si.framebufferHeight;
		init_format_change();
		appDataNew.qualityLevel = n;
		SendFramebufferUpdateRequest(0, 0, w, h, False);
		schedule_format_change();
	}
}

void
DoServerQuality(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char str[100], *s, *q;
	int n;
	if (1) {
		s = DoQualityDialog();
	} else {
		raiseme(1);
		fprintf(stderr, "\n\n\a\nEnter integer 1 <= n <= 9 for quality setting: ");
		str[0] = '\0';
		fgets(str, 100, stdin); 
		s = str;
		q = strstr(str, "\n");
		if (q) *q = '\0';
	}
	if (s[0] != '\0') {
		n = atoi(s);
		set_server_quality(n);
	}
}

void set_server_compress(int n) {
	fprintf(stderr, "set_compress: %d\n", n);
	if (n >= 0 && n <= 9) {
		int w = si.framebufferWidth;
		int h = si.framebufferHeight;
		init_format_change();
		appDataNew.compressLevel = n;
		SendFramebufferUpdateRequest(0, 0, w, h, False);
		schedule_format_change();
	}
}

void
DoServerCompress(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char str[100], *s, *q;
	int n;
	if (1) {
		s = DoCompressDialog();
	} else {
		raiseme(1);
		fprintf(stderr, "\n\n\a\nEnter integer 1 <= n <= 9 for compress level setting: ");
		str[0] = '\0';
		fgets(str, 100, stdin); 
		s = str;
		q = strstr(str, "\n");
		if (q) *q = '\0';
	}
	if (s[0] != '\0') {
		n = atoi(s);
		set_server_compress(n);
	}
}


void set_ycrop(int n) {
	if (n >= 1) {
		int w = si.framebufferWidth;
		int h = si.framebufferHeight;
		appData.yCrop = n;
		ReDoDesktop();
		SendFramebufferUpdateRequest(0, 0, w, h, False);
		schedule_fb_update();
	}
}

void
SetYCrop(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char str[100], *q, *s;
	int n;
	if (1) {
		s = DoYCropDialog();
	} else {
		raiseme(1);
		fprintf(stderr, "\n\n\a\nEnter pixel size n -ycrop maximum y-height: ");
		str[0] = '\0';
		fgets(str, 100, stdin); 
		s = str;
		q = strstr(str, "\n");
		if (q) *q = '\0';
	}
	if (s[0] != '\0') {
		n = atoi(s);
		set_ycrop(n);
	}
}

void set_scbar(int n) {
	if (n >= 1) {
		int w = si.framebufferWidth;
		int h = si.framebufferHeight;
fprintf(stderr, "set_scbat: %d\n", n);
		appData.sbWidth = n;
		ReDoDesktop();
		SendFramebufferUpdateRequest(0, 0, w, h, False);
		schedule_fb_update();
	}
}

void
SetScbar(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	char str[100], *q, *s;
	int n;
	if (1) {
		s = DoScbarDialog();
	} else {
		raiseme(1);
		fprintf(stderr, "\n\n\a\nEnter pixel size n scrollbar width: ");
		str[0] = '\0';
		fgets(str, 100, stdin); 
		s = str;
		q = strstr(str, "\n");
		if (q) *q = '\0';
	}
	if (s[0] != '\0') {
		n = atoi(s);
		set_scbar(n);
	}
}

void
SetScaleN(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (*num_params != 0) {
		int n = atoi(params[0]);
		set_server_scale(n);
	}
}

void
SetQuality(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (*num_params != 0) {
		int n = atoi(params[0]);
		set_server_quality(n);
	}
}

void
SetCompress(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (*num_params != 0) {
		int n = atoi(params[0]);
		set_server_compress(n);
	}
}

void
GotChatText(char *str, int len)
{
	static char *b = NULL;
	static int blen = -1;
	int i;
	if (appData.termChat) {
		printChat("\nChat: ", True);
	} else {
		printChat("Chat: ", True);
	}

	if (len < 0) len = 0;

	if (blen < len+1) {
		if (b) free(b);
		blen = 2 * (len + 10);
		b = (char *) malloc(blen);
	}

	for (i=0; i < len; i++) {
		b[i] = str[i];
	}
	b[len] = '\0';
	printChat(b, True);
	
	if (appData.termChat) {
		if (strstr(str, "\n")) {
			printChat("Send: ", True);
		} else {
			printChat("\nSend: ", True);
		}
	}
}

void
SetViewOnlyState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.viewOnly)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetNOJPEGState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.enableJPEG)
    XtVaSetValues(w, XtNstate, False, NULL);
  else
    XtVaSetValues(w, XtNstate, True, NULL);
}

void
SetQualityState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (*num_params != 0) {
		int n = atoi(params[0]);
		if (appData.qualityLevel == n) {
			XtVaSetValues(w, XtNstate, True, NULL);
		} else {
			XtVaSetValues(w, XtNstate, False, NULL);
		}
	}
}

void
SetCompressState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (*num_params != 0) {
		int n = atoi(params[0]);
		if (appData.compressLevel == n) {
			XtVaSetValues(w, XtNstate, True, NULL);
		} else {
			XtVaSetValues(w, XtNstate, False, NULL);
		}
	}
}

void
SetScaleNState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (*num_params != 0) {
		int n = atoi(params[0]);
		if (appData.serverScale == n || (appData.serverScale >= 6 && n >= 6)) {
			XtVaSetValues(w, XtNstate, True, NULL);
		} else {
			XtVaSetValues(w, XtNstate, False, NULL);
		}
	}
}

void
Set8bppState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBGR233) {
		XtVaSetValues(w, XtNstate, True, NULL);
		if (b16   != NULL) XtVaSetValues(b16,   XtNstate, False, NULL);
		if (bfull != NULL) XtVaSetValues(bfull, XtNstate, False, NULL);
	} else {
		XtVaSetValues(w, XtNstate, False, NULL);
	}
}

void
Set16bppState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBGR565) {
		XtVaSetValues(w, XtNstate, True, NULL);
		if (b8    != NULL) XtVaSetValues(b8,    XtNstate, False, NULL);
		if (bfull != NULL) XtVaSetValues(bfull, XtNstate, False, NULL);
	} else {
		XtVaSetValues(w, XtNstate, False, NULL);
	}
}

void
SetFullColorState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBGR565 || appData.useBGR233) {
		XtVaSetValues(w, XtNstate, False, NULL);
	} else {
		XtVaSetValues(w, XtNstate, True, NULL);
		if (b8  != NULL) XtVaSetValues(b8,  XtNstate, False, NULL);
		if (b16 != NULL) XtVaSetValues(b16, XtNstate, False, NULL);
	}
}

void
Set256ColorsState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBGR233 == 256) {
		XtVaSetValues(w, XtNstate, True, NULL);
		if (w64  != NULL) XtVaSetValues(w64 , XtNstate, False, NULL);
		if (w8   != NULL) XtVaSetValues(w8  , XtNstate, False, NULL);
	} else {
		XtVaSetValues(w, XtNstate, False, NULL);
	}
}

void
Set64ColorsState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBGR233 == 64) {
		XtVaSetValues(w, XtNstate, True, NULL);
		if (w256 != NULL) XtVaSetValues(w256, XtNstate, False, NULL);
		if (w8   != NULL) XtVaSetValues(w8  , XtNstate, False, NULL);
	} else {
		XtVaSetValues(w, XtNstate, False, NULL);
	}
}

void
Set8ColorsState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useBGR233 == 8) {
		XtVaSetValues(w, XtNstate, True, NULL);
		if (w256 != NULL) XtVaSetValues(w256, XtNstate, False, NULL);
		if (w64  != NULL) XtVaSetValues(w64 , XtNstate, False, NULL);
	} else {
		XtVaSetValues(w, XtNstate, False, NULL);
	}
}

void
SetGreyScaleState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	if (appData.useGreyScale) {
		XtVaSetValues(w, XtNstate, True, NULL);
	} else {
		XtVaSetValues(w, XtNstate, False, NULL);
	}
}

void
SetZRLEState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
	static int first = 1;
	if (first && appData.encodingsString) {
		char *t, *z, *w;
		t = strstr(appData.encodingsString, "tight");
		z = strstr(appData.encodingsString, "zrle");
		w = strstr(appData.encodingsString, "zywrle");
		if (t) {
			if (z) {
				if (w) {
					if (t < z && t < w) {
						usingZRLE = False;
					} else {
						usingZRLE = True;
					}
					if (z < w) {
						usingZYWRLE = False;
					} else {
						usingZYWRLE = True;
					}
				} else {
					if (t < z) {
						usingZRLE = False;
					} else {
						usingZRLE = True;
					}
					usingZYWRLE = False;
				}
			} else {
				if (w) {
					if (t < w) {
						usingZRLE = False;
					} else {
						usingZRLE = True;
					}
					usingZYWRLE = True;
				} else {
					usingZRLE = False;
					usingZYWRLE = False;
				}
			}
		} else {
			if (z) {
				if (w) {
					usingZRLE = True;
					if (z < w) {
						usingZYWRLE = False;
					} else {
						usingZYWRLE = True;
					}
				} else {
					usingZRLE = True;
					usingZYWRLE = False;
				}
			} else {
				if (w) {
					usingZRLE = True;
					usingZYWRLE = True;
				} else {
					usingZRLE = False;
					usingZYWRLE = False;
				}
			}
		}
	}
	first = 0;

	if (usingZRLE)
		XtVaSetValues(w, XtNstate, True, NULL);
	else
		XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetZYWRLEState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (usingZYWRLE)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetCursorShapeState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.useRemoteCursor)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetCursorAlphaState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.useCursorAlpha)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetX11CursorState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.useX11Cursor)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetBellState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.useBell)
    XtVaSetValues(w, XtNstate, False, NULL);
  else
    XtVaSetValues(w, XtNstate, True, NULL);
}

void
SetRawLocalState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.useRawLocal)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetServerInputState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (!appData.serverInput)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetSingleWindowState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.singleWindow)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetTextChatState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.chatActive)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}

void
SetFileXferState(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  if (appData.fileActive)
    XtVaSetValues(w, XtNstate, True, NULL);
  else
    XtVaSetValues(w, XtNstate, False, NULL);
}
