/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * sockets.c - functions to deal with sockets.
 */

#include <unistd.h>
#include <sys/socket.h>
#include <errno.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <assert.h>
#include <vncviewer.h>

void PrintInHex(char *buf, int len);

Bool errorMessageOnReadFailure = True;

#define BUF_SIZE 8192
static char buf[BUF_SIZE];
static char *bufoutptr = buf;
static int buffered = 0;

/*
 * ReadFromRFBServer is called whenever we want to read some data from the RFB
 * server.  It is non-trivial for two reasons:
 *
 * 1. For efficiency it performs some intelligent buffering, avoiding invoking
 *    the read() system call too often.  For small chunks of data, it simply
 *    copies the data out of an internal buffer.  For large amounts of data it
 *    reads directly into the buffer provided by the caller.
 *
 * 2. Whenever read() would block, it invokes the Xt event dispatching
 *    mechanism to process X events.  In fact, this is the only place these
 *    events are processed, as there is no XtAppMainLoop in the program.
 */

static Bool rfbsockReady = False;
static void
rfbsockReadyCallback(XtPointer clientData, int *fd, XtInputId *id)
{
  rfbsockReady = True;
  XtRemoveInput(*id);
}

extern int skip_XtUpdate;
extern int skip_XtUpdateAll;
extern int filexfer_sock, filexfer_listen;
extern time_t start_listen;
extern void CheckTextInput(void);
extern time_t last_filexfer;

static char fxfer[65536];
int fxfer_size = 65536;

void CheckFileXfer() {
	fd_set fds;
	struct timeval tv;
	int i, icnt = 0, igot = 0, bytes0 = 0, bytes = 0, grace = 0, n, list = 0;
	int db = 0;

	if (!appData.fileActive || (filexfer_sock < 0 && filexfer_listen < 0)) {
		return;
	}

	if (filexfer_listen >= 0 && time(NULL) > start_listen + 10) {
		fprintf(stderr, "filexfer closing aging listen socket.\n");
		close(filexfer_listen);
		filexfer_listen = -1;
		return;
	}
//fprintf(stderr, "In CheckFileXfer\n");

	if (filexfer_listen >=0) {
		n = filexfer_listen;
		list = 1;
	} else {
		n = filexfer_sock;
	}

	while (1) {
		icnt++;
		FD_ZERO(&fds);
		FD_SET(n,&fds);
		tv.tv_sec = 0;
		tv.tv_usec = 0;
		if (select(n+1, &fds, NULL, NULL, &tv) > 0) {
			if (FD_ISSET(n, &fds)) {
				if (list) {
					if (filexfer_sock >= 0) {
						fprintf(stderr, "filexfer close stale(?) filexfer_sock.\n");
						close(filexfer_sock);
						filexfer_sock = -1;
					}
					filexfer_sock = AcceptTcpConnection(filexfer_listen);
					if (filexfer_listen >= 0) {
						fprintf(stderr, "filexfer accept OK.\n");
						close(filexfer_listen);
						filexfer_listen = -1;
					} else {
						fprintf(stderr, "filexfer accept failed.\n");
					}
					break;
				} else {
					ssize_t rn;
					unsigned char hdr[12];
					unsigned int len;
					if (db) fprintf(stderr, "try read filexfer...\n");
#if 1
					rn = read(n, fxfer, 1*8192);
					if (0 || db) fprintf(stderr, "filexfer read[%d] %d.\n", icnt, rn);
					if (rn < 0) {
						fprintf(stderr, "filexfer bad read: %d\n", errno);
						break;
					} else if (rn == 0) {
						fprintf(stderr, "filexfer gone.\n");
						close(n);
						filexfer_sock = -1;
						last_filexfer = time(NULL);
						//fprintf(stderr, "last_filexfer-2a: %d\n", last_filexfer);
						return;
					} else if (rn > 0) {
						if (db) write(2, fxfer, rn);
						if (db) fprintf(stderr, "\n");
						bytes += rn;
						last_filexfer = time(NULL);
						//fprintf(stderr, "last_filexfer-2b: %d\n", last_filexfer);
						if (!WriteExact(rfbsock, fxfer, rn)) {
							return;
						}
						igot = 1;
					}
#else
					// not working, not always 7 msg type.	
					rn = read(n, hdr, 12);
					if (db) fprintf(stderr, "filexfer read %d.\n", rn);
					if (rn == 0) {
						fprintf(stderr, "filexfer gone.\n");
						close(n);
						filexfer_sock = -1;
						last_filexfer = time(NULL);
						return;
					}
					if (rn == 12) {
						len = (hdr[8] << 24) | (hdr[9] << 16) | (hdr[10] << 8) | hdr[11];
						if (db) fprintf(stderr, "n=%d len=%d\n", rn, len);
						if (db) write(2, hdr, rn);
						if (db) fprintf(stderr, "\n");
						WriteExact(rfbsock, hdr, rn);
						if (len > 0) {
							rn = read(len, fxfer, len);
							if (!WriteExact(rfbsock, fxfer, len)) {
								last_filexfer = time(NULL);
								return;
							}
							if (db) write(2, fxfer, len);
						}
						if (db) fprintf(stderr, "\n");
					} else {
						if (db) fprintf(stderr, "bad rn: %d\n", rn);
					}
					igot = 1;
#endif
				}
			}
		} else {
			if (bytes >= 8192) {
				int ok = 0;
				if (bytes0 == 0) {
					ok = 1;
				} else if (bytes >= bytes0 + 12) {
					ok = 1;
				} else if (grace < 20) {
					ok = 1;
				}
				if (ok) {
					grace++;
					bytes0 = bytes;
					//fprintf(stderr, "grace: %d\n", grace);
					usleep(10 * 1000);
					continue;
				}
			}
			break;
		}
	}
	if (igot) {
		last_filexfer = time(NULL);
		//fprintf(stderr, "last_filexfer-2c: %d\n", last_filexfer);
	}
	return;
}

static void
ProcessXtEvents()
{
	int y;
	if (skip_XtUpdateAll) {
		return;
	}
	if (appData.chatActive) {
		fd_set fds;
		struct timeval tv;
		int i, igot = -1, n = fileno(stdin);
		char strs[100][512];
		char buf[rfbTextMaxSize];

		if (appData.termChat) {
			for (i=0; i < 100; i++) {
				FD_ZERO(&fds);
				FD_SET(n,&fds);
				tv.tv_sec = 0;
				tv.tv_usec = 0;
				if (select(n+1, &fds, NULL, NULL, &tv) > 0) {
					if (FD_ISSET(n, &fds)) {
						fgets(strs[i], 512, stdin);
						igot = i;
					} else {
						break;
					}
				} else {
					break;
				}
			}
			buf[0] = '\0';
			for (i=0; i <= igot; i++) {
				if (strlen(buf) + strlen(strs[i]) < rfbTextMaxSize) {
					strcat(buf, strs[i]);
				} else {
					SendTextChat(buf);
					buf[0] = '0';
				}
			}
			if (buf[0] != '\0') {
				SendTextChat(buf);
			}
			if (igot >= 0) printChat("Send: ");
		} else {
			CheckTextInput();
		}
	}
	if (skip_XtUpdate) {
		return;
	}

//fprintf(stderr, "XtAppAddInput: ");
	rfbsockReady = False;
	XtAppAddInput(appContext, rfbsock, (XtPointer)XtInputReadMask,
	    rfbsockReadyCallback, NULL);

	while (!rfbsockReady) {
//fprintf(stderr, ".");
		XtAppProcessEvent(appContext, XtIMAll);
	}
//fprintf(stderr, " done.\n");
}

Bool
ReadFromRFBServer(char *out, unsigned int n)
{
  if (n <= buffered) {
    memcpy(out, bufoutptr, n);
    bufoutptr += n;
    buffered -= n;
    return True;
  }

  memcpy(out, bufoutptr, buffered);

  out += buffered;
  n -= buffered;

  bufoutptr = buf;
  buffered = 0;

  if (n <= BUF_SIZE) {

    while (buffered < n) {
      int i = read(rfbsock, buf + buffered, BUF_SIZE - buffered);
      if (i <= 0) {
	if (i < 0) {
	  if (errno == EWOULDBLOCK || errno == EAGAIN) {
	    ProcessXtEvents();
	    i = 0;
	  } else {
	    fprintf(stderr,programName);
	    perror(": read");
	    return False;
	  }
	} else {
	  if (errorMessageOnReadFailure) {
	    fprintf(stderr,"%s: VNC server closed connection\n",programName);
	  }
	  return False;
	}
      }
      buffered += i;
    }

    memcpy(out, bufoutptr, n);
    bufoutptr += n;
    buffered -= n;
    return True;

  } else {

    while (n > 0) {
      int i = read(rfbsock, out, n);
      if (i <= 0) {
	if (i < 0) {
	  if (errno == EWOULDBLOCK || errno == EAGAIN) {
	    ProcessXtEvents();
	    i = 0;
	  } else {
	    fprintf(stderr,programName);
	    perror(": read");
	    return False;
	  }
	} else {
	  if (errorMessageOnReadFailure) {
	    fprintf(stderr,"%s: VNC server closed connection\n",programName);
	  }
	  return False;
	}
      }
      out += i;
      n -= i;
    }

    return True;
  }
}


/*
 * Write an exact number of bytes, and don't return until you've sent them.
 */

Bool
WriteExact(int sock, char *buf, int n)
{
  fd_set fds;
  int i = 0;
  int j;

  while (i < n) {
    j = write(sock, buf + i, (n - i));
    if (j <= 0) {
      if (j < 0) {
	if (errno == EWOULDBLOCK || errno == EAGAIN) {
	  FD_ZERO(&fds);
	  FD_SET(rfbsock,&fds);

	  if (select(rfbsock+1, NULL, &fds, NULL, NULL) <= 0) {
	    fprintf(stderr,programName);
	    perror(": select");
	    return False;
	  }
	  j = 0;
	} else {
	  fprintf(stderr,programName);
	  perror(": write");
	  return False;
	}
      } else {
	fprintf(stderr,"%s: write failed\n",programName);
	return False;
      }
    }
    i += j;
  }
  return True;
}


/*
 * ConnectToTcpAddr connects to the given TCP port.
 */

int
ConnectToTcpAddr(unsigned int host, int port)
{
  int sock;
  struct sockaddr_in addr;
  int one = 1;

  memset(&addr, 0, sizeof(struct sockaddr_in));

  addr.sin_family = AF_INET;
  addr.sin_port = htons(port);
  addr.sin_addr.s_addr = host;

  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock < 0) {
    fprintf(stderr,programName);
    perror(": ConnectToTcpAddr: socket");
    return -1;
  }

  if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
    fprintf(stderr,programName);
    perror(": ConnectToTcpAddr: connect");
    close(sock);
    return -1;
  }

  if (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY,
		 (char *)&one, sizeof(one)) < 0) {
    fprintf(stderr,programName);
    perror(": ConnectToTcpAddr: setsockopt");
    close(sock);
    return -1;
  }

  return sock;
}



/*
 * FindFreeTcpPort tries to find unused TCP port in the range
 * (TUNNEL_PORT_OFFSET, TUNNEL_PORT_OFFSET + 99]. Returns 0 on failure.
 */

int
FindFreeTcpPort(void)
{
  int sock, port;
  struct sockaddr_in addr;

  memset(&addr, 0, sizeof(struct sockaddr_in));

  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;

  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock < 0) {
    fprintf(stderr,programName);
    perror(": FindFreeTcpPort: socket");
    return 0;
  }

  for (port = TUNNEL_PORT_OFFSET + 99; port > TUNNEL_PORT_OFFSET; port--) {
    addr.sin_port = htons((unsigned short)port);
    if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) == 0) {
      close(sock);
      return port;
    }
  }

  close(sock);
  return 0;
}


/*
 * ListenAtTcpPort starts listening at the given TCP port.
 */

int use_loopback = 0;

int
ListenAtTcpPort(int port)
{
  int sock;
  struct sockaddr_in addr;
  int one = 1;

  memset(&addr, 0, sizeof(struct sockaddr_in));

  addr.sin_family = AF_INET;
  addr.sin_port = htons(port);
  addr.sin_addr.s_addr = INADDR_ANY;

	if (getenv("VNCVIEWER_LISTEN_LOCALHOST") || use_loopback) {
		addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	}

  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock < 0) {
    fprintf(stderr,programName);
    perror(": ListenAtTcpPort: socket");
    return -1;
  }

  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
		 (const char *)&one, sizeof(one)) < 0) {
    fprintf(stderr,programName);
    perror(": ListenAtTcpPort: setsockopt");
    close(sock);
    return -1;
  }

  if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
    fprintf(stderr,programName);
    perror(": ListenAtTcpPort: bind");
    close(sock);
    return -1;
  }

  if (listen(sock, 5) < 0) {
    fprintf(stderr,programName);
    perror(": ListenAtTcpPort: listen");
    close(sock);
    return -1;
  }

  return sock;
}


/*
 * AcceptTcpConnection accepts a TCP connection.
 */

int
AcceptTcpConnection(int listenSock)
{
  int sock;
  struct sockaddr_in addr;
  int addrlen = sizeof(addr);
  int one = 1;

  sock = accept(listenSock, (struct sockaddr *) &addr, &addrlen);
  if (sock < 0) {
    fprintf(stderr,programName);
    perror(": AcceptTcpConnection: accept");
    return -1;
  }

  if (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY,
		 (char *)&one, sizeof(one)) < 0) {
    fprintf(stderr,programName);
    perror(": AcceptTcpConnection: setsockopt");
    close(sock);
    return -1;
  }

  return sock;
}


/*
 * SetNonBlocking sets a socket into non-blocking mode.
 */

Bool
SetNonBlocking(int sock)
{
  if (fcntl(sock, F_SETFL, O_NONBLOCK) < 0) {
    fprintf(stderr,programName);
    perror(": AcceptTcpConnection: fcntl");
    return False;
  }
  return True;
}


/*
 * StringToIPAddr - convert a host string to an IP address.
 */

Bool
StringToIPAddr(const char *str, unsigned int *addr)
{
  struct hostent *hp;

  if (strcmp(str,"") == 0) {
    *addr = 0; /* local */
    return True;
  }

  *addr = inet_addr(str);

  if (*addr != -1)
    return True;

  hp = gethostbyname(str);

  if (hp) {
    *addr = *(unsigned int *)hp->h_addr;
    return True;
  }

  return False;
}


/*
 * Test if the other end of a socket is on the same machine.
 */

Bool
SameMachine(int sock)
{
  struct sockaddr_in peeraddr, myaddr;
  int addrlen = sizeof(struct sockaddr_in);

  getpeername(sock, (struct sockaddr *)&peeraddr, &addrlen);
  getsockname(sock, (struct sockaddr *)&myaddr, &addrlen);

  return (peeraddr.sin_addr.s_addr == myaddr.sin_addr.s_addr);
}


/*
 * Print out the contents of a packet for debugging.
 */

void
PrintInHex(char *buf, int len)
{
  int i, j;
  char c, str[17];

  str[16] = 0;

  fprintf(stderr,"ReadExact: ");

  for (i = 0; i < len; i++)
    {
      if ((i % 16 == 0) && (i != 0)) {
	fprintf(stderr,"           ");
      }
      c = buf[i];
      str[i % 16] = (((c > 31) && (c < 127)) ? c : '.');
      fprintf(stderr,"%02x ",(unsigned char)c);
      if ((i % 4) == 3)
	fprintf(stderr," ");
      if ((i % 16) == 15)
	{
	  fprintf(stderr,"%s\n",str);
	}
    }
  if ((i % 16) != 0)
    {
      for (j = i % 16; j < 16; j++)
	{
	  fprintf(stderr,"   ");
	  if ((j % 4) == 3) fprintf(stderr," ");
	}
      str[i % 16] = 0;
      fprintf(stderr,"%s\n",str);
    }

  fflush(stderr);
}
