/* Snownews - A lightweight console RSS newsreader
 * 
 * Copyright 2003-2004 Oliver Feiler <kiza@kcore.de>
 * http://kiza.kcore.de/software/snownews/
 *
 * main.c
 *
 * Please read the file README.patching before changing any code in this file!
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
#include <ncurses.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <signal.h>

#include "config.h"
#include "interface.h"
#include "ui-support.h"
#include "main.h"
#include "updatecheck.h"
#include "setup.h"
#include "io-internal.h"

extern struct feed *first_bak;		/* For use with the signal handler. */

/* Deinit ncurses and quit. */
void MainQuit (char * func, char * error) {
	if (error == NULL) {
		/* Only save settings if we didn't exit on error. */
		WriteCache();
	}
	clear();
	refresh();
	endwin();		/* Make sure no ncurses function is called after this point! */

	if (error == NULL) {		
		printf (_("Bye.\n\n"));
		
		/* Do this after everything else. If it doesn't work or hang
		   user can ctrl-c without interfering with the program operation
		   (like writing caches). */
		AutoVersionCheck();
		
		exit(0);
	} else {
		printf (_("Aborting program execution!\nAn internal error occured. Snownews has quit, no changes has been saved!\n"));
		printf (_("This shouldn't happen, please submit a bugreport to kiza@kcore.de, tell me what you where doing and include the output below in your mail.\n"));
		printf ("----\n");
		/* Please don't localize! I don't want to receive Japanese error messages. Thanks. :) */
		printf ("While executing: %s\n", func);
		printf ("Error as reported by the system: %s\n\n", error);
		exit(1);
	}
}

/* Signal handler function. */
void MainSignalHandler (int sig) {
	if (sig == SIGINT) {
		/* If there is a first_bak!=NULL a filter is set. Reset first_ptr
		   so the correct list gets written on the disk when exisiting via SIGINT. */
		if (first_bak != NULL)
			first_ptr = first_bak;
		MainQuit(NULL, NULL);
	}
}

/* http://foo.bar/address.rdf -> http:__foo.bar_address.rdf */
void Hashify (char * url) {
	int i;
	for (i = 0; i <= strlen(url); i++) {
		if (url[i] == '/') {
			url[i] = '_';
		}
		/* Cygwin doesn't seem to like anything besides a-z0-9 in filenames.
		   Zap'em! */
#ifdef __CYGWIN__
		if (((i < 48) && (i > 57)) ||
			((i < 65) && (i > 90)) ||
			((i < 97) && (i > 122)) ||
			(i != 0)) {
			url[i] = '_';
		}
#endif
	}
}

int main (int argc, char *argv[]) {
	int autoupdate = 0;		/* Automatically update feeds on app start... or not if set to 0. */
	int numfeeds;			/* Number of feeds loaded from Config(). */
	
#ifdef LOCALEPATH
  setlocale (LC_ALL, "");
  bindtextdomain ("snownews", LOCALEPATH);
  textdomain ("snownews");
#endif

	/* Kiza's really sucky argument checker.
	   Hey, it worked for Tornado. */
	if (argc > 1) {
		if (strcmp(argv[1], "--version") == 0 ||
		    strcmp(argv[1], "-V") == 0) {
			printf (_("Snownews version %s\n\n"), VERSION);
			return 0;
		} else if (strcmp(argv[1], "--update") == 0 ||
			strcmp(argv[1], "-u") == 0) {
			autoupdate = 1;
		} else if (strcmp(argv[1], "--disable-versioncheck") == 0) {
			printf ("The option \"--disable-versioncheck\" has been obsoleted.\n");
			printf ("Please see the manpage for details.\n");
			printf ("(To deactivate check, put -1 into ~/.snownews/updatecheck)\n");
			exit(1);
		} else {
			printf (_("Snownews version %s\n\n"), VERSION);
			printf (_("usage: snownews [-huV] [--help|--update|--version]\n\n"));
			printf (_("\t--update|-u\tAutomatically update every feed.\n"));
			printf (_("\t--help|-h\tPrint this help message.\n"));
			printf (_("\t--version|-V\tPrint version number and exit.\n"));
			printf (_("\t--disable-versioncheck (obsoleted)\n"));
			return 0;
		}
	}
	
	
	/* Install SIGINT signal handler. */
	signal (SIGINT, MainSignalHandler);
	
	InitCurses();
	
	/* Check if configfiles exist and create/read them. */
	numfeeds = Config();
		
	LoadAllFeeds (numfeeds);

	if (autoupdate)
		UpdateAllFeeds();
	
	/* Init the pRNG. See about.c for usages of rand() ;) */
	srand(time(0));
	
	/* Give control to main program loop. */
	UIMainInterface();

	/* We really shouldn't be here at all... ah well. */
	MainQuit(NULL, NULL);
	return 0;
}
