/* Snownews - A lightweight console RSS newsreader
 * 
 * Copyright 2003 Oliver Feiler <kiza@kcore.de>
 * http://kiza.kcore.de/software/snownews/
 *
 * main.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
#include <ncurses.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <signal.h>

#include "config.h"
#include "interface.h"
#include "ui-support.h"
#include "netio.h"
#include "main.h"
#include "xmlparse.h"
#include "updatecheck.h"
#include "categories.h"

#ifdef SUN
#	include "os-support.h"
#endif

struct feed *first_ptr = NULL;
struct keybindings keybindings;
struct color color;

extern struct feed *first_bak;		/* For use with the signal handler. */

char *browser;						/* Browser command. lynx is standard. */
int versioncheck = 1;				/* Autoupdate check. 1=yes, 0=no. */
char *proxyname;					/* Hostname of proxyserver. */
char *useragent;					/* Snownews User-Agent string. */
unsigned short proxyport = 0;		/* Port on proxyserver to use. */
int use_colors = 0;					/* Default to disabled. */


/* Deinit ncurses and quit. */
void MainQuit (char * func, char * error) {
	if (error == NULL) {
		/* Only save settings if we didn't exit on error. */
		WriteCache();
	}
	clear();
	refresh();
	endwin();		/* Make sure no ncurses function is called after here! */

	if (error == NULL) {		
		printf (_("Bye.\n\n"));
		
		/* Do this after everything else. If it doesn't work or hang
		   user can ctrl-c without interfering with the program operation
		   (like writing caches). */
		if (versioncheck)
			AutoVersionCheck();
		
		exit(0);
	} else {
		printf (_("Aborting program execution!\nAn internal error occured. Snownews has quit, no changes has been saved!\n"));
		printf (_("This shouldn't happen, please submit a bugreport to kiza@kcore.de, tell me what you where doing and include the output below in your mail.\n"));
		printf ("----\n");
		/* Please don't localize! I don't want to receive Japanese error messages. Thanks. :) */
		printf ("While executing: %s\n", func);
		printf ("Error as reported by the system: %s\n\n", error);
		exit(1);
	}
}

/* Signal handler function. */
void MainSignalHandler (int sig) {
	if (sig == SIGINT) {
		/* If there is a first_bak!=NULL a filter is set. Reset first_ptr
		   so the correct list gets written on the disk when exisiting via SIGINT. */
		if (first_bak != NULL)
			first_ptr = first_bak;
		MainQuit(NULL, NULL);
	}
}

/* http://foo.bar/address.rdf -> http:__foo.bar_address.rdf */
void Hashify (char * url) {
	int i;

	for (i = 0; i <= strlen(url); i++) {
		if (url[i] == '/') {
			url[i] = '_';
		}
	}
}


/* Load config and populate caches. */
void Config (void) {
	char file[512];					/* File locations. */
	char filebuf[4096];
	char *freeme, *tmpbuf, *tmppos, *tmpstr;
	char *categories = NULL;		/* Holds comma seperated category string. */
	FILE *configfile;
	struct feed *new_ptr;
	struct stat dirtest;
	
	/* stat configfile dirs and create them. */
	snprintf (file, sizeof(file), "%s/.snownews", getenv("HOME"));
	if ((stat (file, &dirtest)) == -1 ) {
		/* Create directory. */
		if (mkdir (file, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) != 0)
			MainQuit ("Creating config directory ~/.snownews/", strerror(errno));
	} else {
		/* Something with the name .snownews exists, let's see what it is. */
		if ((dirtest.st_mode & S_IFDIR) != S_IFDIR) {
			MainQuit ("Creating config directory ~/.snownews/",
				"A file with the name \"~/.snownews/\" exists!");
		}
	}
	
	snprintf (file, sizeof(file), "%s/.snownews/cache", getenv("HOME"));
	if ((stat (file, &dirtest)) == -1) {
		/* Create directory. */
		if (mkdir (file, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) != 0)
			MainQuit (_("Creating config directory ~/.snownews/cache/"), strerror(errno));
	} else {
		if ((dirtest.st_mode & S_IFDIR) != S_IFDIR) {
			MainQuit ("Creating config directory ~/.snownews/cache",
				"A file with the name \"~/.snownews/cache/\" exists!");
		}
	}
	
	UIStatus (_("Reading configuration settings..."), 0);
	

	/*******************
	 * Default browser *
	 *******************/
	 
	snprintf (file, sizeof(file), "%s/.snownews/browser", getenv("HOME"));
	configfile = fopen (file, "r");
	if (configfile == NULL) {
		UIStatus (_("Creating new config \"browser\"..."), 0);
		configfile = fopen (file, "w+");
		if (configfile == NULL)
			MainQuit (_("Create initial configfile \"config\""), strerror(errno)); /* Still didn't work? */
		browser = malloc (strlen("lynx %s")+1);
		strcpy (browser, "lynx %s");
		fputs (browser, configfile);
		fclose (configfile);
	} else {
		/* Careful not to overflow char browser! */
		fgets (filebuf, sizeof(filebuf), configfile);
		browser = malloc (strlen(filebuf)+1);
		strncpy (browser, filebuf, strlen(filebuf)+1);
		fclose (configfile);
		/* Die newline, die! */
		if (browser[strlen(browser)-1] == '\n')
			browser[strlen(browser)-1] = '\0';
	}

	/*************
	 * Feed list *
	 *************/
	
	snprintf (file, sizeof(file), "%s/.snownews/urls", getenv("HOME"));
	configfile = fopen (file, "r");
	if (configfile == NULL) {
		UIStatus (_("Creating new configfile."), 0);
		configfile = fopen (file, "w+");
		if (configfile == NULL)
			MainQuit (_("Create initial configfile \"urls\""), strerror(errno));	/* Still didn't work? */
		fclose (configfile);
	} else {
		while (!feof(configfile)) {
			if ((fgets (filebuf, sizeof(filebuf), configfile)) == NULL)
				break;
			if (filebuf[0] == '\n')
				break;
			new_ptr = malloc (sizeof(struct feed));
			
			new_ptr->feed = NULL;
			/* Set to NULL so xmlparse.c is happy when we try to free there. */
			new_ptr->title = NULL;
			new_ptr->link = NULL;
			new_ptr->description = NULL;
			new_ptr->items = NULL;
			new_ptr->lastmodified = NULL;
			new_ptr->lasthttpstatus = 0;
			new_ptr->override = NULL;
			new_ptr->original = NULL;
			new_ptr->problem = 0;
			new_ptr->feedcategories = NULL;
			/* Reset to NULL on every loop run! */
			categories = NULL;
			
			/* File format is url|custom name|comma seperated categories */
			tmpbuf = strdup (filebuf);
			
			/* Munch trailing newline. */
			if (tmpbuf[strlen(tmpbuf)-1] == '\n')
				tmpbuf[strlen(tmpbuf)-1] = '\0';
				
			freeme = tmpbuf;
			tmpstr = strsep (&tmpbuf, "|");
			
			/* The first | was squished with \0 so we can strdup the first part. */
			new_ptr->feedurl = strdup(freeme);
			
			/* Save start position of override string */
			tmpstr = strsep (&tmpbuf, "|");
			if (tmpstr != NULL) {
				if (*tmpstr != '\0') {
					new_ptr->override = strdup(tmpstr);
				}
			}
			tmpstr = strsep (&tmpbuf, "|");
			if (tmpstr != NULL) {
				if (*tmpstr != '\0') {
					categories = strdup(tmpstr);
				}
			}
			
			/* We don't need freeme anymore. */
			free (freeme);
			
			/* Put categories into cat struct. */
			if (categories != NULL) {
				freeme = categories;
				
				while (1) {
					tmppos = strsep (&categories, ",");
					if (tmppos == NULL)
						break;
					
					FeedCategoryAdd (new_ptr, tmppos);
				}
				free (freeme);
			} else
				new_ptr->feedcategories = NULL;
			
			/* Add to bottom of pointer chain. */
			new_ptr->next_ptr = NULL;
			if (first_ptr == NULL) {
				new_ptr->prev_ptr = NULL;
				first_ptr = new_ptr;
			} else {
				new_ptr->prev_ptr = first_ptr;
				while (new_ptr->prev_ptr->next_ptr != NULL)
					new_ptr->prev_ptr = new_ptr->prev_ptr->next_ptr;
				new_ptr->prev_ptr->next_ptr = new_ptr;
			}
				
				
		}
		fclose (configfile);
	}
	
	/***********************
	 * Global keybindings. *
	 ***********************/
	
	/* Define default values for keybindings. If some are defined differently
	   in the keybindings file they will be overwritten. If some are missing or broken/wrong
	   these are sane defaults. */
	keybindings.next = 'n';
	keybindings.prev = 'p';
	keybindings.quit = 'q';
	keybindings.addfeed = 'a';
	keybindings.deletefeed = 'D';
	keybindings.markread = 'm';
	keybindings.dfltbrowser = 'B';
	keybindings.moveup = 'P';
	keybindings.movedown = 'N';
	keybindings.feedinfo = 'i';
	keybindings.reload = 'r';
	keybindings.reloadall = 'R';
	keybindings.urljump = 'o';
	keybindings.urljump2 = 'O';
	keybindings.changefeedname = 'c';
	keybindings.sortfeeds = 's';
	keybindings.pup = 'b';
	keybindings.pdown = ' ';
	keybindings.categorize = 'C';
	keybindings.filter = 'f';
	keybindings.filtercurrent = 'g';
	keybindings.nofilter = 'F';
	keybindings.help = 'h';
	keybindings.about = 'A';
		
	snprintf (file, sizeof(file), "%s/.snownews/keybindings", getenv("HOME"));
	configfile = fopen (file, "r");
	if (configfile == NULL) {
		/* Create new file. */
		configfile = fopen (file, "w+");
		fputs ("# Snownews keybindings configfile\n", configfile);
		fputs ("# Main menu bindings\n", configfile);
		fputs ("add feed:a\n", configfile);
		fputs ("delete feed:D\n", configfile);
		fputs ("reload all feeds:R\n", configfile);
		fputs ("change default browser:B\n", configfile);
		fputs ("move item up:P\n", configfile);
		fputs ("move item down:N\n", configfile);
		fputs ("change feedname:c\n", configfile);
		fputs ("sort feeds:s\n", configfile);
		fputs ("categorize feed:C\n", configfile);
		fputs ("apply filter:f\n", configfile);
		fputs ("only current category:g\n", configfile);
		fputs ("remove filter:F\n", configfile);
		fputs ("# Feed display menu bindings\n", configfile);
		fputs ("show feedinfo:i\n", configfile);
		fputs ("# General keybindungs\n", configfile);
		fputs ("next item:n\n", configfile);
		fputs ("previous item:p\n", configfile);
		fputs ("quit/return to previous menu:q\n", configfile);
		fputs ("reload feed:r\n", configfile);
		fputs ("mark (all) as read:m\n", configfile);
		fputs ("open url:o\n", configfile);
		fputs ("open item url in overview:O\n", configfile);
		fputs ("page up:b\n", configfile);
		fputs ("page down: \n", configfile);
		fputs ("help menu:h\n", configfile);
		fputs ("about:A\n", configfile);
		fclose (configfile);
	} else {
		/* Read keybindings and populate keybindings struct. */
		while (!feof(configfile)) {
			if ((fgets (filebuf, sizeof(filebuf), configfile)) == NULL)
				break;
			if (strstr(filebuf, "#") != NULL)
				continue;
			tmpbuf = strdup(filebuf);	/* strsep only works on *ptr */
			freeme = tmpbuf;			/* Save start pos. This is also the string we need to compare. strsep will \0 the delimiter. */
			strsep (&tmpbuf, ":");

			/* Munch trailing newline and avoid \n being bound to a function if no key is defined. */
			if (tmpbuf != NULL)
				tmpbuf[strlen(tmpbuf)-1] = 0;

			if (strcmp (freeme, "next item") == 0)
				keybindings.next = tmpbuf[0];
			if (strcmp (freeme, "previous item") == 0)
				keybindings.prev = tmpbuf[0];
			if (strcmp (freeme, "quit/return to previous menu") == 0)
				keybindings.quit = tmpbuf[0];
			if (strcmp (freeme, "add feed") == 0)
				keybindings.addfeed = tmpbuf[0];
			if (strcmp (freeme, "delete feed") == 0)
				keybindings.deletefeed = tmpbuf[0];
			if (strcmp (freeme, "mark (all) as read") == 0)
				keybindings.markread = tmpbuf[0];
			if (strcmp (freeme, "change default browser") == 0)
				keybindings.dfltbrowser = tmpbuf[0];
			if (strcmp (freeme, "sort feeds") == 0)
				keybindings.sortfeeds = tmpbuf[0];
			if (strcmp (freeme, "move item up") == 0)
				keybindings.moveup = tmpbuf[0];
			if (strcmp (freeme, "move item down") == 0)
				keybindings.movedown = tmpbuf[0];
			if (strcmp (freeme, "show feedinfo") == 0)
				keybindings.feedinfo = tmpbuf[0];
			if (strcmp (freeme, "reload feed") == 0)
				keybindings.reload = tmpbuf[0];
			if (strcmp (freeme, "reload all feeds") == 0)
				keybindings.reloadall = tmpbuf[0];
			if (strcmp (freeme, "open url") == 0)
				keybindings.urljump = tmpbuf[0];
			if (strcmp (freeme, "open item url in overview") == 0)
				keybindings.urljump2 = tmpbuf[0];
			if (strcmp (freeme, "change feedname") == 0)
				keybindings.changefeedname = tmpbuf[0];
			if (strcmp (freeme, "page up") == 0)
				keybindings.pup = tmpbuf[0];
			if (strcmp (freeme, "page down") == 0)
				keybindings.pdown = tmpbuf[0];
			if (strcmp (freeme, "categorize feed") == 0)
				keybindings.categorize = tmpbuf[0];
			if (strcmp (freeme, "apply filter") == 0)
				keybindings.filter = tmpbuf[0];
			if (strcmp (freeme, "only current category") == 0)
				keybindings.filtercurrent = tmpbuf[0];
			if (strcmp (freeme, "remove filter") == 0)
				keybindings.nofilter = tmpbuf[0];
			if (strcmp (freeme, "help menu") == 0)
				keybindings.help = tmpbuf[0];
			if (strcmp (freeme, "about") == 0)
				keybindings.about = tmpbuf[0];
			
			free (freeme);
		}
		fclose (configfile);
		
		/* Override old default settings and make sure there is no clash. */
		/* Default browser is now B; b moved to page up. */
		if (keybindings.dfltbrowser == 'b') {
			keybindings.dfltbrowser = 'B';
		}
	}
	
	/***********************
	 * COLOR suppport code *
	 ***********************/
	
	color.changed = 0;
	color.newitems = 5;
	color.newitemsbold = 0;
	color.urljump = 4;
	color.urljumpbold = 0;
	
	snprintf (file, sizeof(file), "%s/.snownews/colors", getenv("HOME"));
	configfile = fopen (file, "r");
	if (configfile == NULL) {
		/* Set color configfile update flag. */
		color.changed = 1;
	} else {
		while (!feof(configfile)) {
			if ((fgets (filebuf, sizeof(filebuf), configfile)) == NULL)
				break;
			if (strstr(filebuf, "#") != NULL)
				continue;
			tmpbuf = strdup(filebuf);	/* strsep only works on *ptr */
			freeme = tmpbuf;			/* Save start pos. This is also the string we need to compare. strsep will \0 the delimiter. */
			strsep (&tmpbuf, ":");

			/* Munch trailing newline. */
			if (tmpbuf != NULL)
				tmpbuf[strlen(tmpbuf)-1] = 0;
				
			if (strcmp (freeme, "enabled") == 0) 
				use_colors = atoi(tmpbuf);
			if (strcmp (freeme, "new item") == 0) {
				color.newitems = atoi(tmpbuf);
				if (color.newitems > 7) {
					color.newitemsbold = 1;
				} else
					color.newitemsbold = 0;
			}
			if (strcmp (freeme, "goto url") == 0) {
				color.urljump = atoi(tmpbuf);
				if (color.urljump > 7) {
					color.urljumpbold = 1;
				} else
					color.urljumpbold = 0;
			}
			
			free (freeme);
		}
	}
	
	if (use_colors) {
		start_color();
		
		/* The following call will automagically implement -1 as the terminal's
		   default color for fg/bg in init_pair. */
		use_default_colors();
		
		/* Internally used color pairs.
		   Use with WA_BOLD to get bright color (orange->yellow, gray->white, etc). */
		init_pair (10, 1, -1);		/* red */
		init_pair (11, 2, -1);		/* green */
		init_pair (12, 3, -1);		/* orange */
		init_pair (13, 4, -1);		/* blue */
		init_pair (14, 5, -1);		/* magenta */
		init_pair (15, 6, -1);		/* cyan */
		init_pair (16, 7, -1);		/* gray */
		
		/* Default terminal color color pair */
		init_pair (63, -1, -1);
		
		/* Initialize all color pairs we're gonna use. */
		/* New item. */
		if (color.newitemsbold == 1)
			init_pair (2, color.newitems-8, -1);
		else
			init_pair (2, color.newitems, -1);
			
		/* Goto url line */
		if (color.urljumpbold == 1)
			init_pair (3, color.urljump-8, -1);
		else
			init_pair (3, color.urljump, -1);
	
	}
}


/* Update given feed from server.
 * Reload XML document and replace in memory cur_ptr->feed with it.
 */
int UpdateFeed (struct feed * cur_ptr) {
	char *tmpname;
	char *freeme;

	if (cur_ptr == NULL)
		return 1;
	
	/* Need to work on a copy of ->feedurl, because DownloadFeed()
	   changes the pointer. */
	tmpname = strdup (cur_ptr->feedurl);
	freeme = tmpname;		/* Need to make a copy, otherwise we cannot free
							   all RAM. */
	
	if (cur_ptr->feed != NULL)
		free (cur_ptr->feed);
	
	cur_ptr->feed = DownloadFeed (tmpname, cur_ptr);
	
	free (freeme);
	
	/* Set title and link structure to something.
	   To the feedurl in this case so the program show something
	   as placeholder instead of crash. */
	if (cur_ptr->title == NULL)
		cur_ptr->title = strdup (cur_ptr->feedurl);
	if (cur_ptr->link == NULL)
		cur_ptr->link = strdup (cur_ptr->feedurl);
	
	/* If download function returns a NULL pointer return from here. */
	if (cur_ptr->feed == NULL) {
		return 1;
	}
	
	if ((DeXML (cur_ptr)) != 0) {
		UIStatus (_("Invalid XML! Cannot parse this feed!"), 2);
		/* Activate feed problem flag. */
		cur_ptr->problem = 1;
		return 1;
	}
	
	/* We don't need these anymore. Free the raw XML to save some memory. */
	if (cur_ptr->feed != NULL) {
		free (cur_ptr->feed);
		cur_ptr->feed = NULL;
	}
		
	return 0;
}
int UpdateAllFeeds (void) {
	struct feed *cur_ptr;
	
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		if ((UpdateFeed (cur_ptr)) != 0)
			continue;
	}
	return 0;
}


/* Load feed from disk. And call UpdateFeed if neccessary. */
int LoadFeed (struct feed * cur_ptr) {
	int feedlength = 0;			/* Internal usage for realloc. */
	char filebuf[4096];			/* File I/O block buffer. */
	char file[512];
 	char hashme[512];			/* Hashed filename. */
 	char tmp[1024];
	FILE *cache;
	
	strncpy (hashme, cur_ptr->feedurl, sizeof(hashme));
	Hashify (hashme);
	snprintf (file, sizeof(file), "%s/.snownews/cache/%s", getenv("HOME"), hashme);
	cache = fopen (file, "r");
	
	if (cache == NULL) {
		snprintf (tmp, sizeof(tmp), _("Cache for %s is toast. Reloading from server..."), cur_ptr->feedurl);
		UIStatus (tmp, 0);
		
		if ((UpdateFeed (cur_ptr)) != 0)
			return 1;
		return 0;
	}
	
	/* Workaround hack for strncat. */
	cur_ptr->feed = malloc (1);
	cur_ptr->feed[0] = '\0';
	
	/* Read complete cachefile. */
	while (!feof(cache)) {
		if ((fgets (filebuf, sizeof(filebuf), cache)) == NULL) {
			break;			
		}
		feedlength += strlen (filebuf);
		cur_ptr->feed = realloc (cur_ptr->feed, feedlength+1);
		strncat (cur_ptr->feed, filebuf, strlen(filebuf));
	}
	fclose (cache);
	
	/* After loading DeXMLize the mess. */
	if ((DeXML (cur_ptr)) != 0) {
		snprintf (tmp, sizeof(tmp), _("Cache for %s is invalid, please delete the file %s and try again."), cur_ptr->feedurl, file);
		MainQuit (_("Parsing cache"), tmp);
	}	
	if (cur_ptr->feed != NULL) {
		free (cur_ptr->feed);
		cur_ptr->feed = NULL;
	}
	
	return 0;
}
int LoadAllFeeds (void) {
	char tmp[512];
	struct feed *cur_ptr;
	
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		snprintf (tmp, sizeof(tmp), _("Loading cache for %s..."), cur_ptr->feedurl);
		UIStatus (tmp, 0);
		if ((LoadFeed (cur_ptr)) != 0) {
			continue;
		}
	}
	return 0;
}

/* Write in memory structures to disk cache.
 * Usually called before program exit.
 */
void WriteCache (void) {
	char file[512];				/* File locations. */
	char hashme[512];			/* Cache file name. */
	char readstatus[2];
	char syscall[512];
	FILE *configfile;
	FILE *cache;
	struct feed *cur_ptr;
	struct newsitem *item;
	struct stat filetest;
	struct feedcategories *category;
	char *encoded;
	
	UIStatus (_("Saving settings..."), 0);
	
	snprintf (file, sizeof(file), "%s/.snownews/browser", getenv("HOME"));
	configfile = fopen (file, "w+");
	if (configfile == NULL) {
		MainQuit (_("Save settings (browser)"), strerror(errno));
	}
	fputs (browser, configfile);
	fclose (configfile);
	
	snprintf (file, sizeof(file), "%s/.snownews/urls", getenv("HOME"));
	
	if ((stat (file, &filetest)) != -1) {
		if ((filetest.st_mode & S_IFREG) == S_IFREG) {
			snprintf (syscall, sizeof(file), "cp -f %s/.snownews/urls %s/.snownews/urls.bak", getenv("HOME"), getenv("HOME"));
			system (syscall);
		}
	}
	
	
	configfile = fopen (file, "w+");
	if (configfile == NULL) {
		MainQuit (_("Save settings (urls)"), strerror(errno));
	}	
	
	for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		fputs (cur_ptr->feedurl, configfile);
		fputc ('|', configfile);
		if (cur_ptr->override != NULL) {
			fputs (cur_ptr->title, configfile);
		}
		fputc ('|', configfile);
		if (cur_ptr->feedcategories != NULL) {
			for (category = cur_ptr->feedcategories; category != NULL; category = category->next_ptr) {
				fputs (category->name, configfile);
				/* Only add a colon of we run the loop again! */
				if (category->next_ptr != NULL)
					fputc (',', configfile);
			}
		}
		fputc ('\n', configfile);		/* Add newline character. */
		
		/* 
		 * Write cache.
		 */
		strncpy (hashme, cur_ptr->feedurl, sizeof(hashme));
		Hashify (hashme);
		snprintf (file, sizeof(file), "%s/.snownews/cache/%s", getenv("HOME"), hashme);
		cache = fopen (file, "w+");
		
		fputs ("<?xml version=\"1.0\" ?>\n\n<rdf:RDF\n  xmlns:rdf=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\"\n  xmlns=\"http://purl.org/rss/1.0/\">\n\n", cache);
		
		if (cur_ptr->lastmodified != NULL) {
			fputs ("<lastmodified>", cache);
			fputs (cur_ptr->lastmodified, cache);
			fputs ("</lastmodified>\n", cache);
		}
				
		fputs ("<channel rdf:about=\"", cache);
		
		encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->feedurl);
		fputs (encoded, cache);
		free (encoded);
		
		fputs ("\">\n<title>", cache);
		if (cur_ptr->original != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->original);
			fputs (encoded, cache);
			free (encoded);
		} else if (cur_ptr->title != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->title);
			fputs (encoded, cache);
			free (encoded);
		}
		fputs ("</title>\n<link>", cache);
		if (cur_ptr->link != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->link);
			fputs (encoded, cache);
			free (encoded);
		}
		fputs ("</link>\n<description>", cache);
		if (cur_ptr->description != NULL) {
			encoded = xmlEncodeEntitiesReentrant (NULL, cur_ptr->description);
			fputs (encoded, cache);
			free (encoded);
		}
		fputs ("</description>\n</channel>\n\n", cache);
		
		for (item = cur_ptr->items; item != NULL; item = item->next_ptr) {
			fputs ("<item rdf:about=\"", cache);
			
			if (item->link != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->link);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("\">\n<title>", cache);
			if (item->title != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->title);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("</title>\n<link>", cache);
			if (item->link != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->link);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("</link>\n<description>", cache);
			if (item->description != NULL) {
				encoded = xmlEncodeEntitiesReentrant (NULL, item->description);
				fputs (encoded, cache);
				free (encoded);
			}
			fputs ("</description>\n<readstatus>", cache);
			snprintf (readstatus, sizeof(readstatus), "%d", item->readstatus);
			fputs (readstatus, cache);			
			fputs ("</readstatus>\n</item>\n\n", cache);
		}
		
		fputs ("</rdf:RDF>", cache);
		fclose (cache);
	}
	fclose (configfile);
	
	/* Write color configfile if needed, */
	if (color.changed) {
		snprintf (file, sizeof(file), "%s/.snownews/colors", getenv("HOME"));
		configfile = fopen (file, "w+");
		
		fputs ("# Snownews color definitons\n", configfile);
		fputs ("# black:0\n", configfile);
		fputs ("# red:1\n", configfile);
		fputs ("# green:2\n", configfile);
		fputs ("# orange:3\n", configfile);
		fputs ("# blue:4\n", configfile);
		fputs ("# magenta(tm):5\n", configfile);
		fputs ("# cyan:6\n", configfile);
		fputs ("# gray:7\n", configfile);
		fputs ("# brightred:9\n", configfile);
		fputs ("# brightgreen:10\n", configfile);
		fputs ("# yellow:11\n", configfile);
		fputs ("# brightblue:12\n", configfile);
		fputs ("# brightmagenta:13\n", configfile);
		fputs ("# brightcyan:14\n", configfile);
		fputs ("# white:15\n", configfile);
		fprintf (configfile, "enabled:%d\n", use_colors);
		fprintf (configfile, "new item:%d\n", color.newitems);
		fprintf (configfile, "goto url:%d\n", color.urljump);
				
		fclose (configfile);
	}
	
	return;
}


int main (int argc, char *argv[]) {
	int autoupdate = 0;		/* Automatically update feeds on app start... or not if set to 0. */
	char *freeme;
	char *proxystring;
	char *tmp;
	char *lang;
	int ualength;
		
#ifdef LOCALEPATH
  setlocale (LC_ALL, "");
  bindtextdomain ("snownews", LOCALEPATH);
  textdomain ("snownews");
#endif

	/* Kiza's really sucky argument checker.
	   Hey, it worked for Tornado.
	   Should be replaced with getopt on a boring Friday night. */
	if (argc > 1) {
		if (strcmp(argv[1], "--version") == 0 ||
		    strcmp(argv[1], "-V") == 0) {
			printf (_("Snownews version %s\n\n"), VERSION);
			return 0;
		} else if (strcmp(argv[1], "--update") == 0 ||
			strcmp(argv[1], "-u") == 0) {
			autoupdate = 1;
			if (argv[2] != NULL) {
				if (strcmp(argv[2], "--disable-versioncheck") == 0)
					versioncheck = 0;
			}
		} else if (strcmp(argv[1], "--disable-versioncheck") == 0) {
			versioncheck = 0;
			if (argv[2] != NULL) {
				if (strcmp(argv[2], "--update") == 0 ||
					strcmp(argv[2], "-u") == 0) {
					autoupdate = 1;
				}
			}
		} else {
			printf (_("Snownews version %s\n\n"), VERSION);
			printf (_("usage: snownews [-huV] [--help|--update|--version]\n\n"));
			printf (_("\t--update|-u\tAutomatically update every feed.\n"));
			printf (_("\t--help|-h\tPrint this help message.\n"));
			printf (_("\t--version|-V\tPrint version number and exit.\n"));
			printf (_("\t--disable-versioncheck\n"));
			return 0;
		}
	}
	
	/* Check for proxy environment variable. */
	if (getenv("http_proxy") != NULL) {
		/* The pointer returned by getenv must not be altered.
		   What about mentioning this in the manpage of getenv? */
		proxystring = strdup(getenv("http_proxy"));
		freeme = proxystring;
		strsep (&proxystring, "/");
		if (proxystring != NULL) {
			strsep (&proxystring, "/");
		} else {
			free (freeme);
			goto goon;
		}
		if (proxystring != NULL) {
			tmp = strsep (&proxystring, ":");
			proxyname = strdup(tmp);
		} else {
			free (freeme);
			goto goon;
		}
		if (proxystring != NULL) {
			proxyport = atoi (strsep (&proxystring, "/"));
		} else {
			free (freeme);
			goto goon;
		}
		free (freeme);
	}
	goon:
	
	/* Constuct the User-Agent string of snownews. This is done here in program init,
	   because we need to do it exactly once and it will never change while the program
	   is running. */
	if (getenv("LANG") != NULL) {
		lang = getenv("LANG");
		/* Snonews/VERSION (Linux; de_DE; (http://kiza.kcore.de/software/snownews/) */
		ualength = strlen("Snownews/") + strlen(VERSION) + 2 + strlen(lang) + 2 + strlen(OS)+2 + strlen("http://kiza.kcore.de/software/snownews/") + 2;
		useragent = malloc(ualength);
		snprintf (useragent, ualength, "Snownews/%s (%s; %s; http://kiza.kcore.de/software/snownews/)", VERSION, OS, lang);
	} else {
		/* "Snownews/" + VERSION + "(http://kiza.kcore.de/software/snownews/)" */
		ualength = strlen("Snownews/") + strlen(VERSION) + 2 + strlen(OS) + 2 + strlen("http://kiza.kcore.de/software/snownews/") + 2;
		useragent = malloc(ualength);
		snprintf (useragent, ualength, "Snownews/%s (%s; http://kiza.kcore.de/software/snownews/)", VERSION, OS);
	}
	
	/* Install SIGINT signal handler. */
	signal (SIGINT, MainSignalHandler);
	
	InitCurses();
	
	/* Check if configfiles exist and create/read them. */
	Config();
	
	LoadAllFeeds();

	if (autoupdate)
		UpdateAllFeeds();
	
	/* Init the pRNG. */
	srand(time(0));
	
	/* Give control to main program loop. */
	UIMainInterface();

	/* We really shouldn't be here at all... ah well. */
	MainQuit(NULL, NULL);
	return 0;
}
