/* Snownews - A lightweight console RSS newsreader
 * 
 * Copyright 2003 Oliver Feiler <kiza@kcore.de>
 * http://kiza.kcore.de/software/snownews/
 *
 * interface.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <ncurses.h>

#include <time.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "interface.h"
#include "main.h"
#include "categories.h"
#include "ui-support.h"

extern char *browser;
extern struct keybindings keybindings;
extern struct categories *first_category;
extern struct color color;
 
/* Snowflake for the xmas about box. */
struct snowflake {
	int x;
	int y;
	int oldx;
	int oldy;
	char oldchar;
	char oldchar2;
	int visible;		/* Don't draw flakes over text. */
	int vspeed;			/* Vertical speed of the flake */
	int hspeed;			/* Horizontal speed */
	struct snowflake * next;
	struct snowflake * prev;
};
 
char * UIOneLineEntryField (int x, int y) {
        char *text;

        text = malloc(512);
        
        /* UIStatus switches off attron! */
        attron (WA_REVERSE);
        echo();
        curs_set(1);

        move (y, x);
        /* Beware of hardcoded textlength size! */
        getnstr (text, 511);
        
        noecho();
        curs_set(0);
        attroff (WA_REVERSE);
        
        /* This memory needs to be freed in the calling function! */
        return text;
}

int UIChangeBrowser (void) {
	char *browserstring;
	int len;
	
	/* malloc = 17 (strlen("Current setting: ") + browser
	   We will malloc a bigger junk, because other languages
	   might need longer strings and crash! */
	len = strlen(_("Current setting: %s")) + strlen(browser) + 1;
	browserstring = malloc (len);
	snprintf (browserstring, len, _("Current setting: %s"), browser);

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	UISupportDrawBox (3, 5, COLS-4, 7);
		
	UIStatus (browserstring, 0);
	free (browserstring);
	
	browserstring = UIOneLineEntryField (5, 6);
	
	if (strlen(browserstring) == 0) {
		free (browserstring);
		return 1;
	}
	
	browser = realloc (browser, strlen(browserstring)+1);
	strncpy (browser, browserstring, strlen(browserstring)+1);
	
	free (browserstring);
	
	return 0;
}

/* Dialog to change feedname.
   Return: 0	on success
           1	on user abort
		   2	original title restored
*/
int UIChangeFeedName (struct feed *cur_ptr) {
	char *newname;

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	UISupportDrawBox (3, 5, COLS-4, 7);

	UIStatus (_("Enter new name. Blank line to abort. '-' to reset."), 0);
	
	newname = UIOneLineEntryField (5, 6);
	
	/* If strlen is zero, return 1. */
	if (strlen(newname) == 0) {
		free (newname);
		return 1;
	}
	
	/* If newname contains "|", abort since this is used as a delimiter for the config file. */
	if (strstr (newname, "|") != NULL) {
		free (newname);
		return 3;
	}
	
	/* Restor original title. */
	if ((newname != NULL) && (cur_ptr->override != NULL)) {
		if (strcmp(newname, "-") == 0) {
			if (cur_ptr->title != NULL)
				free (cur_ptr->title);
			cur_ptr->title = strdup(cur_ptr->original);
			free (cur_ptr->original);
			/* Set back original to NULL pointer. */
			cur_ptr->original = NULL;
			free (cur_ptr->override);
			cur_ptr->override = NULL;
			free (newname);
			return 2;
		}
	}
	
	/* Copy new name into ->override. */
	if (cur_ptr->override != NULL)
		free (cur_ptr->override);
	cur_ptr->override = strdup (newname);
	
	/* Save original. */
	if (cur_ptr->original != NULL)
		free (cur_ptr->original);
	cur_ptr->original = strdup (cur_ptr->title);
	
	/* Set new title. */
	if (cur_ptr->title != NULL)
		free (cur_ptr->title);
	cur_ptr->title = strdup (newname);
		
	free (newname);
	return 0;
}

/* Popup window to add new RSS feed. */
int UIAddFeed (void) {
	char tmp[512];
	char *url;
	struct feed *new_ptr;
	struct feed *tmp_ptr;

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	UISupportDrawBox (3, 5, COLS-4, 7);
	
	UIStatus (_("Enter URL of the feed you want to add. Blank line to abort."), 0);

	url = UIOneLineEntryField (5, 6);
	
	/* If read stringlength is ZARO (abort of action requested) return 1
	   and confuse the calling function. */
	if (strlen(url) == 0) {
		free (url);
		return 1;
	}
	
	/* Check of entered url is valid at all. */
	if (strstr(url, "http://") == NULL) {
		free (url);
		return 2;
	}

	new_ptr = malloc (sizeof(struct feed));
	/* getnstr does not return newline... says the docs. */
	new_ptr->feedurl = malloc (strlen(url)+1);
	new_ptr->feed = NULL;
	
	/* Set to NULL so xmlparse.c is happy when we try to feed there. */
	new_ptr->title = NULL;
	new_ptr->link = NULL;
	new_ptr->description = NULL;
	new_ptr->items = NULL;
	new_ptr->lastmodified = NULL;
	new_ptr->override = NULL;
	new_ptr->original = NULL;
	new_ptr->problem = 0;
	new_ptr->feedcategories = NULL;
	
	/* Attach to feed pointer chain. */
	strncpy (new_ptr->feedurl, url, strlen(url)+1);
	new_ptr->next_ptr = first_ptr;
	if (first_ptr != NULL)
		first_ptr->prev_ptr = new_ptr;
	new_ptr->prev_ptr = NULL;
	first_ptr = new_ptr;
	
	/* Don't need url text anymore. */
	free (url);

	/* Download new feed and DeXMLize it. */	
	if ((UpdateFeed (new_ptr)) != 0) {
		if ((new_ptr->lasthttpstatus != 200) && (new_ptr->lasthttpstatus != 0)) {
			snprintf (tmp, sizeof(tmp), _("HTTP error %d - Could not update feed."), new_ptr->lasthttpstatus);
			UIStatus (tmp, 2);
		}
		
		/* Free the feeds data structure again. Otherwise we'll leak them. */
		if (new_ptr->feedurl != NULL)
			free (new_ptr->feedurl);
		if (new_ptr->link != NULL)
			free (new_ptr->link);
		if (new_ptr->title != NULL)
			free (new_ptr->title);
		if (new_ptr->feed != NULL)
			free (new_ptr->feed);
		
		/* Remove from feed pointer chain -> feed has problems -> we don't add it. */
		tmp_ptr = first_ptr;
		if (first_ptr->next_ptr != NULL)
			 first_ptr->next_ptr->prev_ptr = NULL;
		first_ptr = first_ptr->next_ptr;
		free (tmp_ptr);
		
		return -1;
	}

	return 0;
}

void FeedInfo (struct feed * current_feed) {
	int centerx, len;
	char *hashme;			/* Hashed filename. */
	char *file;
	char *categories = NULL;
	struct stat filetest;
	struct feedcategories *category;
	
	centerx = COLS / 2;
	
	hashme = malloc (strlen(current_feed->feedurl)+1);
	strncpy (hashme, current_feed->feedurl, strlen(current_feed->feedurl)+1);
	Hashify (hashme);
	
	len = (strlen(getenv("HOME")) + strlen(hashme) + 18);
	file = malloc (len);
	snprintf (file, len, "%s/.snownews/cache/%s", getenv("HOME"), hashme);

	UISupportDrawBox (5, 4, COLS-6, 12);
	
	attron (WA_REVERSE);
	mvaddstr (5, centerx-((strlen(current_feed->title))/2), current_feed->title);
	mvaddstr (7, centerx-(COLS/2-7), current_feed->feedurl);
	if (current_feed->lastmodified != NULL)
		mvprintw (8, centerx-(COLS/2-7), _("Last updated: %s"), current_feed->lastmodified);	
	else
		mvaddstr (8, centerx-(COLS/2-7), _("No modification date."));
	
	if ((stat (file, &filetest)) != -1) {
		mvprintw (9, centerx-(COLS/2-7), _("In disk cache: %lld bytes"), (long long int) filetest.st_size);
	} else
		mvaddstr (9, centerx-(COLS/2-7), _("Not in disk cache."));
		
	/* Print category info */
	mvaddstr (10, centerx-(COLS/2-7), _("Categories:"));
	if (current_feed->feedcategories == NULL)
		mvaddstr (10, centerx-(COLS/2-7)+strlen(_("Categories:"))+1, "none");
	else {
		categories = malloc(1);
		memset (categories, 0, 1);
		len = 0;
		for (category = current_feed->feedcategories; category != NULL; category = category->next_ptr) {
			len += strlen(category->name)+4;
			categories = realloc (categories, len);
			strncat (categories, category->name, len);
			if (category->next_ptr != NULL)
				strcat (categories, ", ");
		}
		mvprintw (10, centerx-(COLS/2-7)+strlen(_("Categories:"))+1, categories);
		if (categories != NULL)
			free (categories);
	}
	
	/* Add a smiley indicator to the http status telling the overall status
	   so you don't have to know what the HTTP return codes mean.
	   Yes I think I got the idea from cdparanoia. :) */
	if (current_feed->lasthttpstatus != 0) {
		mvprintw (11, centerx-(COLS/2-7), _("Last webserver status: %d"), current_feed->lasthttpstatus);
		
		switch (current_feed->lasthttpstatus) {
			case 200:
			case 304:
				mvaddstr (11, centerx-(COLS/2-7)+strlen(_("Last webserver status: %d"))+2, ":-)");
				break;
			case 403:
			case 500:
			case 503:
				mvaddstr (11, centerx-(COLS/2-7)+strlen(_("Last webserver status: %d"))+2, ":-P");
				break;
			case 404:
			case 410:
				mvaddstr (11, centerx-(COLS/2-7)+strlen(_("Last webserver status: %d"))+2, ":-(");
				break;
			default:
				mvaddstr (11, centerx-(COLS/2-7)+strlen(_("Last webserver status: %d"))+2, "8-/");
				break;
		}
	}
	
	UIStatus (_("Displaying feed information."), 0);
	
	free (file);
	free (hashme);

	getch();
}

int UIDeleteFeed (char * feedname) {
	
	UISupportDrawBox (3, 5, COLS-3, 8);
	
	attron (WA_REVERSE);
	mvaddstr (6, COLS/2-21, _("Are you sure you want to delete this feed?"));
	mvprintw (7, 5, "%s", feedname);
	
	UIStatus (_("Type 'y' to delete, any other key to abort."), 0);
	
	if (getch() == 'y')
		return 1;
	else
		return 0;
}

void UIHelpScreen (void) {
	int centerx, centery;				/* Screen center x/y coordinate. */
	int userinput;
	int offset = 18;
	int offsetstr = 12;
	
	centerx = COLS / 2;
	centery = LINES / 2;
	
	UISupportDrawBox ((COLS/2)-20, (LINES/2)-9, (COLS/2)+24, (LINES/2)+8);
	
	attron (WA_REVERSE);
	/* Keys */
	mvprintw (centery-8, centerx-offset, "%c:", keybindings.addfeed);	
	mvprintw (centery-7, centerx-offset, "%c:", keybindings.deletefeed);
	mvprintw (centery-6, centerx-offset, "%c:", keybindings.changefeedname);
	mvprintw (centery-5, centerx-offset, "%c:", keybindings.reloadall);
	mvprintw (centery-4, centerx-offset, "%c:", keybindings.reload);
	mvprintw (centery-3, centerx-offset, "%c:", keybindings.markread);
	mvprintw (centery-2, centerx-offset, "%c:", keybindings.dfltbrowser);
	mvprintw (centery-1, centerx-offset, "%c, %c:", keybindings.moveup, keybindings.movedown);
	mvprintw (centery,   centerx-offset, "%c:", keybindings.sortfeeds);
	mvprintw (centery+1, centerx-offset, "%c:", keybindings.categorize);
	mvprintw (centery+2, centerx-offset, "%c:", keybindings.filter);
	mvprintw (centery+3, centerx-offset, "%c:", keybindings.filtercurrent);
	mvprintw (centery+4, centerx-offset, "%c:", keybindings.nofilter);
	mvaddstr (centery+5, centerx-offset, _("tab:"));
	mvprintw (centery+6, centerx-offset, "%c:", keybindings.about);
	mvprintw (centery+7, centerx-offset, "%c:", keybindings.quit);
	/* Descriptions */
	mvaddstr (centery-8, centerx-offsetstr, _("Add RSS feed..."));
	mvaddstr (centery-7, centerx-offsetstr, _("Delete highlighted RSS feed..."));
	mvaddstr (centery-6, centerx-offsetstr, _("Rename feed..."));
	mvaddstr (centery-5, centerx-offsetstr, _("Reload all feeds"));
	mvaddstr (centery-4, centerx-offsetstr, _("Reload this feed"));
	mvaddstr (centery-3, centerx-offsetstr, _("Mark all read"));
	mvaddstr (centery-2, centerx-offsetstr, _("Change default browser..."));
	mvaddstr (centery-1, centerx-offsetstr, _("Move item up, down"));
	mvaddstr (centery,   centerx-offsetstr, _("Sort feed list alphabetically"));
	mvaddstr (centery+1, centerx-offsetstr, _("Categorize feed..."));
	mvaddstr (centery+2, centerx-offsetstr, _("Apply filter..."));
	mvaddstr (centery+3, centerx-offsetstr, _("Only current category"));
	mvaddstr (centery+4, centerx-offsetstr, _("Remove filter"));
	mvaddstr (centery+5, centerx-offsetstr, _("Type Ahead Find"));
	mvaddstr (centery+6, centerx-offsetstr, _("About"));
	mvaddstr (centery+7, centerx-offsetstr, _("Quit program"));
	attroff (WA_REVERSE);

	UIStatus (_("Press the any(tm) key to exit help screen."), 0);
	userinput = getch();
	
	/* Return input back into input queue so it gets automatically
	   executed. */
	if ((userinput != '\n') && (userinput != 'h') && (userinput != 'q'))
		ungetch(userinput);
}

void UIDisplayFeedHelp (void) {
	int centerx, centery;				/* Screen center x/y coordinate. */
	int userinput;
	int offset = 18;
	int offsetstr = 7;
	
	centerx = COLS / 2;
	centery = LINES / 2;
	
	UISupportDrawBox ((COLS/2)-20, (LINES/2)-5, (COLS/2)+24, (LINES/2)+6);
	
	attron (WA_REVERSE);
	/* Keys */
	mvprintw (centery-4, centerx-offset, _("%c, up:"), keybindings.prev);	
	mvprintw (centery-3, centerx-offset, _("%c, down:"), keybindings.next);
	mvaddstr (centery-2, centerx-offset, _("enter:"));
	mvprintw (centery-1, centerx-offset, "%c:", keybindings.reload);
	mvprintw (centery,   centerx-offset, "%c:", keybindings.urljump);
	mvprintw (centery+1, centerx-offset, "%c:", keybindings.urljump2);
	mvprintw (centery+2, centerx-offset, "%c:", keybindings.markread);
	mvprintw (centery+3, centerx-offset, "%c:", keybindings.feedinfo);
	mvaddstr (centery+4, centerx-offset, _("tab:"));
	mvprintw (centery+5, centerx-offset, "%c:", keybindings.quit);
	/* Descriptions */
	mvprintw (centery-4, centerx-offsetstr, _("Previous item"));	
	mvprintw (centery-3, centerx-offsetstr, _("Next item"));
	mvaddstr (centery-2, centerx-offsetstr, _("View item"));
	mvprintw (centery-1, centerx-offsetstr, _("Reload this feed"));
	mvprintw (centery,   centerx-offsetstr, _("Open homepage"));
	mvprintw (centery+1, centerx-offsetstr, _("Open link"));
	mvprintw (centery+2, centerx-offsetstr, _("Mark all read"));
	mvprintw (centery+3, centerx-offsetstr, _("Show feed info..."));
	mvaddstr (centery+4, centerx-offsetstr, _("Type Ahead Find"));
	mvprintw (centery+5, centerx-offsetstr, _("Return to main menu"));
	attroff (WA_REVERSE);

	UIStatus (_("Press the any(tm) key to exit help screen."), 0);
	userinput = getch();
	if ((userinput != '\n') && (userinput != 'h'))
		ungetch(userinput);
}

void UIDisplayItemHelp (void) {
	int centerx, centery;				/* Screen center x/y coordinate. */
	int userinput;
	int offset = 16;
	int offsetstr = 6;
	
	centerx = COLS / 2;
	centery = LINES / 2;
	
	UISupportDrawBox ((COLS/2)-18, (LINES/2)-3, (COLS/2)+18, (LINES/2)+3);
	
	attron (WA_REVERSE);
	/* Keys */
	mvaddstr (centery-2, centerx-offset, _("cursor:"));
	mvprintw (centery-1, centerx-offset, "%c, <-:", keybindings.prev);	
	mvprintw (centery,   centerx-offset, "%c, ->:", keybindings.next);
	mvprintw (centery+1, centerx-offset, "%c:", keybindings.urljump);
	mvprintw (centery+2, centerx-offset, _("%c, enter:"), keybindings.quit);
	/* Descriptions */
	mvaddstr (centery-2, centerx-offsetstr, _("Scroll text"));
	mvaddstr (centery-1, centerx-offsetstr, _("Previous item"));	
	mvaddstr (centery,   centerx-offsetstr, _("Next item"));
	mvaddstr (centery+1, centerx-offsetstr, _("Open link"));
	mvaddstr (centery+2, centerx-offsetstr, _("Return to overview"));
	attroff (WA_REVERSE);

	UIStatus (_("Press the any(tm) key to exit help screen."), 0);
	userinput = getch();
	if ((userinput != '\n') && (userinput != 'h'))
		ungetch(userinput);
}

/* This draws the colored parts of the christmas tree.
   Called everytime during SnowFall to avoid to much duplicate code.
   Use sparsly as snowflakes will vanish behind this elements. */
void ChristmasTree (void) {
	attron (COLOR_PAIR(13));
	mvaddch (6, 13, '&');
	mvaddch (16, 5, '&');
	attroff (COLOR_PAIR(13));
	attron (COLOR_PAIR(10));
	mvaddch (9, 10, '&');
	mvaddch (19, 12, '&');
	attroff (COLOR_PAIR(10));
	attron (COLOR_PAIR(15));
	mvaddch (12, 14, '&');
	mvaddch (18, 6, '&');
	attroff (COLOR_PAIR(15));
	attron (COLOR_PAIR(14));
	mvaddch (16, 15, '&');
	mvaddch (13, 10, '&');
	attroff (COLOR_PAIR(14));
}

/* This function sometimes deletes characters. Maybe oldchar doesn't
   get saved correctly everytime. */
void Snowfall (void) {
	struct snowflake *cur;
	struct snowflake *first = NULL;
	struct snowflake *new;
	struct snowflake *curnext;
	/* The idea behind wind:
	   Determine windtimer and wait that many rounds before we initially
	   select any windspeed. If we've waited enough rounds (windtimer==0)
	   select a windspeed and windtimer (which should be much less than
	   between the storms). Use (slightly randomly offset) windspeed to
	   blow snowflakes around (to left or right). If windtimer drops to 0
	   again select new windtimer, wait, repeat. */
	int windspeed = 0;
	int windtimer = 0;
	int wind = 1;			/* 1: wind active, 0: inactive 
							   set to 1 here so the first loop run clears it. */
	int newflake = 0;		/* Time until a new flake appears. */
	
	/* Set ncurses halfdelay mode. */
		halfdelay (3);
	
	/* White snowflakes! */
	/* Doesn't work on white terminal background... obviously.
	attron (COLOR_PAIR(16));
	attron (WA_BOLD); */
	
	while (1) {
		/* Set up the storm. */
		if (windtimer == 0) {
			if (wind) {
				/* Entering silence */
				windtimer = 10 + ((float)rand() / (float)RAND_MAX * 50);
				wind = 0;
				windspeed = 0;
			} else {
				/* Entering storm. */
				windtimer = 10 + ((float)rand() / (float)RAND_MAX * 20);
				wind = 1;
				windspeed = (1+(float)rand() / (float)RAND_MAX * 11)-6;
			}
		}
		//mvaddstr(2,1,     "                              ");
		//if (wind)
		//	mvprintw (2,1,"Windspeed: %d; rounds left: %d",windspeed,windtimer);
		//else
		//	mvprintw (2,1,"No wind; rounds left: %d",windtimer);

		/* Add new snowflakes. */
		if (newflake == 0) {
			/* Add new flake to pointer chain with random x offset. */
			new = malloc (sizeof (struct snowflake));
			new->y = 0;
			new->x = (float)rand() / (float)RAND_MAX * COLS;
			new->oldx = new->x;
			new->oldy = new->y;
			new->visible = 1;
			new->oldchar = new->oldchar2 = ' ';
			new->vspeed = 1+(float)rand() / (float)RAND_MAX * 2;
			new->hspeed = (1+(float)rand() / (float)RAND_MAX * 7)-4;
			
			/* Add our new snowflake to the pointer chain. */
			new->next = NULL;
			if (first == NULL) {
				new->prev = NULL;
				first = new;
			} else {
				new->prev = first;
				while (new->prev->next != NULL)
					new->prev = new->prev->next;
				new->prev->next = new;
			}
			
			/* Set new counter until next snowflake. */
			newflake = 1+(float)rand() / (float)RAND_MAX * 2;
			//mvaddstr (1,1,"      ");
			//mvprintw (1,1,"New flake in %d rounds.", newflake);
		}
		
		for (cur = first; cur != NULL; cur = curnext) {
			curnext = cur->next;
			/* Draw every snowflake at its coordinates to the screen. */
			if (cur->visible) {
				/* Only draw if y<=LINES. This makes some snow lie on bottom of screen. */
				if (cur->y <= LINES) {
					/* Restore old char with proper color.
					   See also ChristmasTree() above. */
					if (((cur->oldx < 25) && (cur->oldy > 5)) ||
						(cur->oldx < 20))
						attron (COLOR_PAIR(11));
					if ((cur->oldx < 14) && (cur->oldx > 9) && (cur->oldy < 24) && (cur->oldy > 20)) {
						attroff (COLOR_PAIR(11));
						attron (COLOR_PAIR(12));
					}
					mvaddch (cur->oldy, cur->oldx, cur->oldchar2);
					if (((cur->oldx < 25) && (cur->oldy > 5)) ||
						(cur->oldx < 20))
						attroff (COLOR_PAIR(11));
					if ((cur->oldx < 14) && (cur->oldx > 9) && (cur->oldy < 24) && (cur->oldy > 20)) {
						attroff (COLOR_PAIR(12));
					}
					ChristmasTree();
				}
				mvaddch (cur->y, cur->x, '*');
				cur->oldx = cur->x;
				cur->oldy = cur->y;
			}
			/* Set new hspeed for flake */
			cur->hspeed = (1+(float)rand() / (float)RAND_MAX * 7)-4;
			
			/* Advance every flake downwards by a random amount and to
			   the left or right.
			   Check if the next position would obscure a character on the screen
			   and set visible to 0 in this case. Clear visible flag as needed. */
			cur->y += cur->vspeed;
			if (wind)
				cur->hspeed += windspeed;
			cur->x += cur->hspeed;
			
			if (cur->y > LINES) {
				if (cur == first) {
					first = first->next;
					first->prev = NULL;
				} else if (cur->next == NULL) {
					cur->prev->next = NULL;
				} else {
					cur->prev->next = cur->next;
					cur->next->prev = cur->prev;
				}
				free (cur);
				continue;
			}
			
			/* Only draw if we're still inside the window. */
			if (cur->y <= LINES) {
				//mvaddstr (3,1,"                ");
				//mvprintw (3,1,"Flake hspeed: %d",cur->hspeed);
				cur->oldchar2 = cur->oldchar;
				/* Reset to ' ' if we accidently set it to *. */
				if (cur->oldchar2 == '*')
					cur->oldchar2 = ' ';
				if ((cur->x <= COLS) && (cur->y <= LINES))
					cur->oldchar = mvinch (cur->y, cur->x);
				else
					cur->oldchar = ' ';
			}
		}
	
		windtimer--;
		newflake--;
		
		refresh();
		
		/* Leave loop if anykey(tm) was pressed. */
		if (getch() != ERR)
			break;
	}
	/* Leave halfdelay mode. I have no idea how this is supposed to work.
	   nodelay (stdscr, FALSE) has no effect. The manpage says nocbreak(),
	   but this buffers chars. Calling cbreak() seems to do the job.
	   Someone tell me how this works please. */
	cbreak();
}

void UIAbout (void) {
	struct tm *t;
	time_t tunix;
	int xpos;
	
	clear();				/* Get the crap off the screen to make room
							   for our wonderful ASCII logo. :) */

	xpos = COLS/2 - 40;
	
	if (COLS < 80) {
		mvprintw (0, 0, _("Need at least 80 COLS terminal, sorry!"));
		refresh();
		getch();
		return;
	}
	
	/* Save unix time and pass pointer to localtime for conversion into struct tm.*/
	tunix = time(0);
	t = localtime(&tunix);
	
	/* Use Merry Christmas about screen on Dec 24th-26th. :) */
	
	/* Careful with tm_mon==(0-11) vs. tm_mday==(1-31)  STUPID! */
	if ((t->tm_mon == 11) && (t->tm_mday >= 24) && (t->tm_mday <= 26)) {
		/* Logo */
		mvaddstr (1,  0, "                      _____ _____    ____ _______ _____   ____ _______   _____");
		mvaddstr (2,  0, "                     /  __/ \\    \\  / __ \\\\ |  | \\\\    \\ /  _/ \\ |  | \\ /  __/");
		mvaddstr (3,  0, "                     \\___ \\  \\ |  \\ \\ \\/ / \\     / \\ |  \\\\ __\\  \\     / \\___ \\");
		mvaddstr (4,  0, "                     /____/  /_|__/  \\__/  /__|_/  /_|__/ \\___\\ /__|_/  /____/");
		/* Christmas tree. */
		attron (COLOR_PAIR(11));
		mvaddstr (1,  0, "");
		mvaddstr (2,  0, "");
		mvaddstr (3,  0, "          _\\/_");
		mvaddstr (4,  0, "           /\\");
		mvaddstr (5,  0, "          /  \\");
		mvaddstr (6,  0, "         /   &\\");
		mvaddstr (7,  0, "        /      \\");
		mvaddstr (8,  0, "       /        \\");
		mvaddstr (9,  0, "      /   &      \\");
		mvaddstr (10, 0, "     /            \\");
		mvaddstr (11, 0, "    /              \\");
		mvaddstr (12, 0, "   /   /      &     \\");
		mvaddstr (13, 0, "  /___/   &      \\___\\");
		mvaddstr (14, 0, "     /            \\");
		mvaddstr (15, 0, "    /              \\");
		mvaddstr (16, 0, "   / &         &    \\");
		mvaddstr (17, 0, "  /                  \\");
		mvaddstr (18, 0, " /    &               \\");
		mvaddstr (19, 0, "/____/      &      \\___\\");
		mvaddstr (20, 0, "    /         \\     \\");
		mvaddstr (21, 0, "   /______####_\\_____\\");
		mvaddstr (22, 0, "         |####|");
		mvaddstr (23, 0, "         |####|");
		attroff (COLOR_PAIR(11));
		attron (COLOR_PAIR(12));
		mvaddstr (21, 10, "####");
		mvaddstr (22, 10, "####");
		mvaddstr (23, 10, "####");
		attroff (COLOR_PAIR(12));
		ChristmasTree();
		/* Credits. */
		mvprintw (5, 21, "Version %s", VERSION);
		mvprintw (8, 29, _("Merry Christmas from the Snownews developers."));
		mvaddstr (10, 29, _("Main code"));
		mvaddstr (11, 29, "Oliver Feiler");
		mvaddstr (13, 29, _("Additional Code"));
		mvaddstr (14, 29, "Rene Puls");
		mvaddstr (16, 29, _("Translation team"));
		mvaddstr (17, 29, "Oliver Feiler, Frank van der Loo,");
		mvaddstr (18, 29, "Pascal Varet, Simon Isakovic");
		mvaddstr (19, 29, "Fernando J. Pereda, Marco Cova");
		
		Snowfall();
	} else {
		/* 80 COLS logo */
		mvaddstr (2,   xpos, "  ________ _______     ____ __________ _______     ______ __________   ________");
		mvaddstr (3, xpos, " /  _____/ \\      \\   /    \\\\  |    | \\\\      \\   /  ___/ \\  |    | \\ / ______/");
		mvaddstr (4, xpos, " \\____  \\   \\   |  \\ /  /\\  \\\\   |    / \\   |  \\ /     /   \\   |    / \\____  \\");
		mvaddstr (5, xpos, " /       \\  /   |   \\\\  \\/  / \\  |   /  /   |   \\\\  ___\\    \\  |   /  /       \\");
		mvaddstr (6, xpos, "/______  / / ___|___/ \\____/  /__|__/  / ___|___/ \\____ \\   /__|__/  /______  /");
		mvaddstr (7, xpos, "       \\/  \\/                          \\/              \\/                   \\/");
		mvprintw (9, COLS/2-(strlen("Version")+strlen(VERSION)+1)/2, "Version %s", VERSION);
	
		mvaddstr (12, COLS/2-(strlen(_("Brought to you by:")))/2, _("Brought to you by:"));
		mvaddstr (14, COLS/2-(strlen(_("Main code")))/2, _("Main code"));
		mvaddstr (15, COLS/2-6, "Oliver Feiler");
		mvaddstr (17, COLS/2-(strlen(_("Additional code")))/2, _("Additional code"));
		mvaddstr (18, COLS/2-4, "Rene Puls");
		mvaddstr (20, COLS/2-(strlen(_("Translation team")))/2, _("Translation team"));
		mvaddstr (21, COLS/2-31, "Oliver Feiler, Frank van der Loo, Pascal Varet, Simon Isakovic,");
		mvaddstr (22, COLS/2-15, "Fernando J. Pereda, Marco Cova");

		refresh();
		getch();
	}
}

/* Add/remove categories for given feed. This takes over the main interface while running. */
void CategorizeFeed (struct feed * current_feed) {
	int i, n, nglobal, y;
	int uiinput;
	char tmp[512];
	char *newcategory;
	struct feedcategories *category;
	struct categories *cur_ptr;
	
	/* Return if we got passed a NULL pointer (no feeds added to main program). */
	if (current_feed == NULL)
		return;
		
	/* Determine number of global categories. */
	nglobal = 0;
	for (cur_ptr = first_category; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		nglobal++;
	}

	/* We're taking over the program! */
	while (1) {
		/* Determine number of categories for current_feed. */
		n = 0;
		for (category = current_feed->feedcategories; category != NULL; category = category->next_ptr) {
			n++;
		}

		UISupportDrawBox ((COLS/2)-37, 2, (COLS/2)+37, LINES-3);
		
		attron (WA_REVERSE);
		mvprintw (3, (COLS/2)-((strlen(_("Category configuration for %s"))+strlen(current_feed->title))/2),
			_("Category configuration for %s"), current_feed->title);
		

		/* No category defined yet */
		if (current_feed->feedcategories == NULL) {
			i = 49;
			y = 5;
			mvaddstr (y, (COLS/2)-33, _("No category defined. Select one or add a new category:"));
			for (cur_ptr = first_category; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
				mvprintw (y+1, (COLS/2)-33, "%c. %s", i, cur_ptr->name);
				y++;
				i++;
				/* Fast forward to 'a' if we hit ASCII 58 ':' */
				if (i == 58)
					i += 39;
			}
			snprintf (tmp, sizeof(tmp), _("Select category number to add, press 'A' to add a new category or '%c' to quit."), keybindings.quit);
			UIStatus (tmp, 0);
		} else {
			i = 49;
			y = 5;
			mvaddstr (y, (COLS/2)-33, _("Categories defined for this feed:"));
			for (category = current_feed->feedcategories; category != NULL; category = category->next_ptr) {
				mvprintw (y+1, (COLS/2)-33, "%c. %s", i, category->name);
				y++;
				i++;
				/* Fast forward to 'a' if we hit ASCII 58 ':' */
				if (i == 58)
					i += 39;
			}
			snprintf (tmp, sizeof(tmp), _("Select a category number to delete, press 'A' to add a new one or '%c' to quit."), keybindings.quit);
			UIStatus (tmp, 0);
		}
		
		refresh();
		
		uiinput = getch();
		if (uiinput == keybindings.quit)
			return;
		if (uiinput == 'A') {
			/* Clear screen area we want to "draw" to. */
			UISupportDrawBox ((COLS/2)-37, 5, (COLS/2)+37, LINES-3);
			
			attron (WA_REVERSE);
			UIStatus (_("Enter new category name."), 0);
			newcategory = UIOneLineEntryField ((COLS/2)-33, 5);
			if (strlen(newcategory) == 0)
				UIStatus (_("Aborted."), 1);
			else {
				if (!FeedCategoryExists (current_feed, newcategory))
					FeedCategoryAdd (current_feed, newcategory);
				else
					UIStatus (_("Category already defined for this feed!"), 1);
			}
			free (newcategory);
		}
		/* If uiinput is 1-9,a-z
		   If uiinput < i (ASCII of max selectable element) ignore event. */
		if ((((uiinput >= 49) && (uiinput <= 57)) ||
			((uiinput >= 97) && (uiinput <= 123))) &&
			(uiinput < i)) {
			
			if (uiinput <= 57)
				i = uiinput - 48;
			else
				i = uiinput - 87;
			
			if (n > 0) {
				/* Delete category code for n (number of categories defined for current feed) > 0 */

				/* Decrease i by one while walking the category linked list.
				   If we hit 0, break. category->name will now contain the category we want to remove. */
				for (category = current_feed->feedcategories; category != NULL; category = category->next_ptr) {
					if (i == 1)
						break;
					i--;
				}
			
				/* Delete entry from feed categories. */
				FeedCategoryDelete (current_feed, category->name);
			} else {
				/* If n == 0 add category with number i to feed. */
				for (cur_ptr = first_category; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
					if (i == 1)
						break;
					i--;
				}
				
				/* Add category to current feed */
				FeedCategoryAdd (current_feed, cur_ptr->name);
			}
		}
	}

}

/* This function lets the user assign color labels to a category. */
void CatConf (void) {
	UIStatus ("Please write me! -- CatConf() 2003 A.D.", 0);
	getch();
}

/* Allocates and returns a filter string the user has chosen from the list of
   available categories.
   If no filter is chosen a NULL pointer is returned. */
char * DialogGetCategoryFilter (void) {
	int i, nglobal, y;
	char *filterstring = NULL;
	int uiinput;
	struct categories *cur_ptr;
	
	/* Determine number of global categories. */
	nglobal = 0;
	for (cur_ptr = first_category; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		nglobal++;
	}
	
	UISupportDrawBox ((COLS/2)-35, 2, (COLS/2)+35, nglobal+4);
	
	attron (WA_REVERSE);
	mvaddstr (3, (COLS/2)-(strlen(_("Select filter to apply"))/2), _("Select filter to apply"));
	
	i = 49;
	y = 3;
	for (cur_ptr = first_category; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
		mvprintw (y+1, (COLS/2)-33, "%c. %s", i, cur_ptr->name);
		y++;
		i++;
		/* Fast forward to 'a' if we hit ASCII 58 ':' */
		if (i == 58)
			i += 39;
	}
			
	UIStatus (_("Select a category number or any other key to reset filters."), 0);
	
	refresh();
		
	uiinput = getch();

	/* If uiinput is 1-9,a-z
	   If uiinput < i (ASCII of max selectable element) ignore event. */
	if ((((uiinput >= 49) && (uiinput <= 57)) ||
		((uiinput >= 97) && (uiinput <= 123))) &&
		(uiinput < i)) {
		
		if (uiinput <= 57)
			i = uiinput - 48;
		else
			i = uiinput - 87;
		
		for (cur_ptr = first_category; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
			if (i == 1)
				break;
			i--;
		}
		filterstring = strdup (cur_ptr->name);
	}
	
	return filterstring;
}
