/* Snownews - A lightweight console RSS newsreader
 * 
 * Copyright 2003 Oliver Feiler <kiza@kcore.de>
 * http://kiza.kcore.de/software/snownews/
 *
 * interface.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <ncurses.h>

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include "config.h"
#include "main.h"
#include "netio.h"
#include "interface.h"
#include "xmlparse.h"
#include "conversions.h"
#include "dialog.h"
#include "ui-support.h"

#ifdef SUN
#	include "os-support.h"
#endif


extern char *browser;
extern struct keybindings keybindings;

/* Scrollable feed->description. */
struct scrolltext {
	char * line;
	struct scrolltext * next_ptr;
	struct scrolltext * prev_ptr;
};

/* Init the ncurses library. */
void InitCurses (void) {
	initscr();

	keypad (stdscr, TRUE);	/* Activate keypad 'cuz some howto told me to do so. */
	
	cbreak();				/* No buffering. */
	noecho();				/* Do not echo typed chars onto the screen. */
	clear();
	curs_set(0);			/* Hide cursor. */
	refresh();
}


/* Print text in statusbar. */
void UIStatus (char * text, int delay) {
	int i;
			
	attron (WA_REVERSE);
	/* Clear statusbar. */
	for (i = 0; i < COLS-1; i++) {
		mvaddch (LINES-1, i, ' ');
	}
	
	mvaddnstr (LINES-1, 1, text, COLS-2);
	
	/* attroff is called here. If the calling function had it switched on,
	   switch it on again there! */
	attroff (WA_REVERSE);
	refresh();
	
	if (delay)
		sleep (delay);
}


/* View newsitem in scrollable window.
 * Speed of this code has been greatly increased in 1.2.1
 */
void UIDisplayItem (struct newsitem * current_item, struct feed * current_feed) {
	int i, j, len;
	int uiinput;
	char *syscall = NULL;			/* Needs to be set to NULL. */
	char *syscallnull = NULL;		/* gcc will issue "might be used uninitialized" otherwise. */
	char *escapetext = NULL;
	char *tmp;
	char *feeddescscreenlength;		/* Used to print a max of COLS-something
									   for feed->desc onto the screen. */
	char *newtext;
	char *newtextwrapped;
	char *textslice;
	int columns;
	struct scrolltext *textlist;
	struct scrolltext *cur_ptr;
	struct scrolltext *first_ptr = NULL;
	struct scrolltext *tmp_ptr;
	struct scrolltext *next_ptr;
	int ypos;
	int linenumber = 0;				/* First line on screen (scrolling). Ugly hack. */
	int maxlines = 0;
	char *freeme;					/* str pos pointer. */
	char *converted;				/* Converted from UTF-8 to local charset. */
	int updatelink;
	int rewrap;						/* Set to 1 if text needs to be rewrapped. */
	
	/* This flag tells whether we need to update with a new url string.
	   If it's 0 the code below will not be executed. */
	updatelink = 1;
	
	/* Activate rewrap. */
	rewrap = 1;
	
	while (1) {
		if (updatelink) {
			/* Need to do this inside while loop so it gets
			   updated if user uses <-,->,n,p keys. */
		   
			syscall = NULL;
			syscallnull = NULL;
			escapetext = NULL;
		   
			/* NULL pointer check! */
			if (current_item->link != NULL) {
				if ((strstr(current_item->link, "'")) == NULL) {
					len = strlen(current_item->link) + 3;
					escapetext = malloc (len);
					snprintf (escapetext, len, "\'%s\'", current_item->link);
					len = strlen(browser) + len + 1;
					syscall = malloc (len);
					snprintf (syscall, len, browser, escapetext);
				} else {
					len = strlen(browser) + 1;
					syscall = malloc (len);
					snprintf (syscall, len, browser, "");
				}
			} else {
				len = strlen(browser) + 1;
				syscall = malloc (len);
				snprintf (syscall, len, browser, "");
			}
			updatelink = 0;
		}
		
		ypos = 6;
		
		move (0, 0);
		clrtobot();

		attron (WA_REVERSE);
		for (i = 0; i < COLS; i++) {
			mvaddch (0, i, ' ');
		}
		mvprintw (0, 1, "* Snownews %s", VERSION);
		attroff (WA_REVERSE);

		/* If feed doesn't have a description print title instead. */
		attron (WA_BOLD);
		if (current_feed->description != NULL) {
			/* Define columns, because malloc three lines down might fail
			   and corrupt mem while resizing the window. */
			columns = COLS-10;
			/* This function only prints COLS-10 number of letters for each feed->desc. */
			if (strlen(current_feed->description) > columns) {
				feeddescscreenlength = malloc(columns+1);
				strncpy (feeddescscreenlength, current_feed->description, columns);
				feeddescscreenlength[columns] = '\0';
				converted = iconvert (feeddescscreenlength, "UTF-8", TARGET_CHARSET);
				if (converted == NULL)
					converted = strdup (feeddescscreenlength);
				mvprintw (2, 1, "%s...", converted);
				free (feeddescscreenlength);
				free (converted);
			} else {
				converted = iconvert (current_feed->description, "UTF-8", TARGET_CHARSET);
				if (converted == NULL)
					converted = strdup (current_feed->description);
				mvprintw (2, 1, converted);
				free (converted);
			}
		}
		else
			mvprintw (2, 1, current_feed->title);
		attroff (WA_BOLD);
		
		/* Don't print feed titles longer than screenwidth here. */
		columns = COLS-6;
		/* And don't crash if title is empty. */
		if (current_item->title != NULL) {
			if (strlen(current_item->title) > columns) {
				feeddescscreenlength = malloc(columns+1);
				strncpy (feeddescscreenlength, current_item->title, columns);
				feeddescscreenlength[columns] = '\0';
				converted = iconvert (feeddescscreenlength, "UTF-8", TARGET_CHARSET);
				if (converted == NULL)
					converted = strdup (feeddescscreenlength);
				mvprintw (4, 1, "%s...", converted);
				free (converted);
				free (feeddescscreenlength);
			} else {
				converted = iconvert (current_item->title, "UTF-8", TARGET_CHARSET);
				if (converted == NULL)
					converted = strdup (current_item->title);
				mvprintw (4, (COLS/2)-(strlen(converted)/2), "%s", converted);
				free (converted);
			}
		} else
			mvaddstr (4, 1, _("No title"));
		
		if (current_item->description == NULL)
			mvprintw (6, 1, _("No description available."));
		else {
			if (strlen(current_item->description) == 0)
				mvprintw (6, 1, _("No description available."));
			else {
				/* Only generate a new scroll list if we need to rewrap everything.
				   Otherwise just skip this block. */
				if (rewrap) {
					converted = iconvert (current_item->description, "UTF-8", TARGET_CHARSET);
					if (converted == NULL)
						converted = strdup (current_item->description);
					newtext = UIDejunk (converted);
					free (converted);
					newtextwrapped = WrapText (newtext, COLS-4);
					free (newtext);
					freeme = newtextwrapped;	/* Set ptr to str start so we can free later. */
					
					/* 
					 * Split newtextwrapped at \n and put into double linked list.
					 */
					while (1) {
						textslice = strsep (&newtextwrapped, "\n");
						
						/* Find out max number of lines text has. */
						maxlines++;
						
						if (textslice != NULL) {
							textlist = malloc (sizeof(struct scrolltext));
							
							textlist->line = malloc (strlen(textslice)+1);
							strcpy (textlist->line, textslice);
							
							/* Gen double list with new items at bottom. */
							textlist->next_ptr = NULL;
							if (first_ptr == NULL) {
								textlist->prev_ptr = NULL;
								first_ptr = textlist;
							} else {
								textlist->prev_ptr = first_ptr;
								while (textlist->prev_ptr->next_ptr != NULL)
									textlist->prev_ptr = textlist->prev_ptr->next_ptr;
								textlist->prev_ptr->next_ptr = textlist;
							}
						} else
							break;
					}
				
					free (freeme);
					rewrap = 0;
				}
				
				j = linenumber;
				/* We sould now have the linked list setup'ed... hopefully. */
				for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
					/* Skip lines we don't need to display. */
					if (j > 0) {
						j--;
						continue;
					}
					if (ypos > LINES-4)
						continue;
					mvaddstr (ypos, 2, cur_ptr->line);
					ypos++;
				}
			}
		}
		

		attron (WA_BOLD);
		mvprintw (LINES-2, 1, "-> %s", syscall);
		attroff (WA_BOLD);
		
		/* Set current item to read. */
		current_item->readstatus = 1;
		
		UIStatus (_("Press 'q' or Enter to return to previous screen. Hit 'h' for help screen."), 0);
		uiinput = getch();
		
		if (uiinput == 'h')
			UIDisplayItemHelp();
		if ((uiinput == '\n') ||
			(uiinput == keybindings.quit)) {
			free (escapetext);
			free (syscall);
			return;
		}
		if (uiinput == keybindings.urljump) {
			len = strlen(syscall) + strlen(_("Executing %s")) + 1;
			tmp = malloc (len);
			snprintf (tmp, len, _("Executing %s"), syscall);
			UIStatus (tmp, 0);
			len = len + 13;
			syscallnull = malloc (len);
			snprintf (syscallnull, len, "%s 2>/dev/null", syscall);
			
			/* Switch on the cursor. */
			curs_set(1);
			
			system(syscallnull);
			
			InitCurses();
			
			/* Hide cursor again. */
			curs_set(0);
			
			free (tmp);
			free (syscallnull);
		}
		if ((uiinput == keybindings.next) ||
			(uiinput == KEY_RIGHT)) {
			if (current_item->next_ptr != NULL) {
				current_item = current_item->next_ptr;
				linenumber = 0;
				/* Set flag to update with new url string. */
				updatelink = 1;
				rewrap = 1;
				maxlines = 0;
			} else {
				/* Setting rewrap to 1 to get the free block below executed. */
				rewrap = 1;
				uiinput = ungetch(keybindings.quit);
			}
		}
		if ((uiinput == keybindings.prev) ||
			(uiinput == KEY_LEFT)) {
			if (current_item->prev_ptr != NULL) {
				current_item = current_item->prev_ptr;
				linenumber = 0;
				/* Set flag to update with new url string. */
				updatelink = 1;
				rewrap = 1;
				maxlines = 0;
			} else {
				/* Setting rewrap to 1 to get the free block below executed. */
				rewrap = 1;
				uiinput = ungetch(keybindings.quit);
			}
		}
		/* Scroll by one page. */
		if ((uiinput == KEY_NPAGE) || (uiinput == 32) ||
			(uiinput == keybindings.pdown)) {
			for (i = 0; i < LINES-9; i++) {
				if (linenumber+(LINES-7) < maxlines)
					linenumber++;
			}
		}
		if ((uiinput == KEY_PPAGE) || (uiinput == keybindings.pup)) {
			for (i = 0; i < LINES-9; i++) {
				if (linenumber >= 1)
					linenumber--;
			}
		}
		if (uiinput == 'A')
			UIAbout();
		if (uiinput == KEY_UP) {
			if (linenumber >= 1)
				linenumber--;
		}
		if (uiinput == KEY_DOWN) {
			if (linenumber+(LINES-7) < maxlines)
				linenumber++;
		}
		if (uiinput == KEY_RESIZE) {
			rewrap = 1;
			/* Reset maxlines, otherwise the program will scroll to far down. */
			maxlines = 0;
		}
		
		/* Free the linked list structure if we need to rewrap the text block. */
		if (rewrap) {
			/* Free textlist list before we start loop again. */
			for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = next_ptr) {
				tmp_ptr = cur_ptr;
				if (cur_ptr->next_ptr != NULL)
					first_ptr = cur_ptr->next_ptr;
				free (tmp_ptr->line);
				next_ptr = cur_ptr->next_ptr;
				free (tmp_ptr);
			}
			/* first_ptr must be set to NULL again!
			   = back to init value. */
			first_ptr = NULL;
		}
		/* Only execute free if updatelink == 1 */
		if (updatelink) {
			/* Only free escapetext if we used it above! */
			if (escapetext != NULL)
				free (escapetext);
			free (syscall);
		}
	}
}


void UIDisplayFeed (struct feed * current_feed) {
	char *tmp;
	char *syscall, *syscallnull, *escapetext;
	int uiinput = 0;
	int typeahead = 0;				/* Typeahead enabled? */
	int skip = 0;					/* # of lines to skip (for typeahead) */
	int skipper = 0;				/* # of lines already skipped */
	char *search = NULL;			/* Typeahead search string */
	int searchlen = 0;
	int count, found;
	char tmpstr[512];
	struct newsitem *savestart;			/* Typeahead internal usage (tmp position save) */
	struct newsitem *savestart_first = NULL;
	struct newsitem *cur_ptr;
	int i, ypos, len;
	struct newsitem *curitem_ptr;
	struct newsitem *highlighted;
	struct newsitem *tmp_highlighted;
	struct newsitem *tmp_first;
	struct newsitem *first_scr_ptr;
	struct newsitem *markasread;
	int itemnum;
	int highlightnum = 1;
	char *feeddescscreenlength;		/* Used to print a max of COLS-something
									   for feed->desc onto the screen. */
	char *itemcut;					/* Don't print items longer than COLS. */
	int columns;
	char *converted;				/* Converted from UTF-8 to local charset. */
	int forceredraw = 0;
	
	syscall = NULL;
	syscallnull = NULL;
	escapetext = NULL;
	
	/* We don't need to check for NULL pointer since
	   current_feed->link will never be NULL. */
	if ((strstr(current_feed->link, "'")) == NULL) {
		len = strlen(current_feed->link) + 3;
		escapetext = malloc (len);
		snprintf (escapetext, len, "\'%s\'", current_feed->link);
		len = strlen(browser) + len + 1;
		syscall = malloc (len);
		snprintf (syscall, len, browser, escapetext);
	} else {
		len = strlen(browser) + 1;
		syscall = malloc (len);
		snprintf (syscall, len, browser, "");
	}
	
	/* Set highlighted to first_ptr at the beginning.
	   Otherwise bad things (tm) will happen. */
	first_scr_ptr = current_feed->items;
	highlighted = current_feed->items;
	
	/* Select first unread item if we enter feed view or
	   leave first item active if there is no unread. */
	tmp_highlighted = highlighted;
	tmp_first = first_scr_ptr;
	highlighted = first_scr_ptr;
	/* Moves highlight to next unread item. */
	
	/* Check if we have no items at all! */
	if (highlighted != NULL) {
		while (highlighted->next_ptr != NULL) {
			if (highlighted->readstatus == 1) {
				highlighted = highlighted->next_ptr;
				
				highlightnum++;
				if (highlightnum+1 > LINES-7) {
					/* Fell off screen. */	
					if (first_scr_ptr->next_ptr != NULL)
						first_scr_ptr = first_scr_ptr->next_ptr;
				}	
				
			} else
				break;				
		}
	}
	/* If *highlighted points to a read entry now we have hit the last
	   entry and there are no unread items anymore. Restore the original
	   location in this case. */
	/* Check if we have no items at all! */
	if (highlighted != NULL) {
		if (highlighted->readstatus == 1) {
			highlighted = tmp_highlighted;
			first_scr_ptr = tmp_first;
		}
	}
	
	/* Save first starting position. For typeahead. */
	savestart = first_scr_ptr;
	
	while (1) {
		if (forceredraw) {
			clear();
			forceredraw = 0;
		} else {
			move (0, 0);
			clrtobot();
		}
		
		/* This clears/inverts the first line and prints the app name/version. */
		attron (WA_REVERSE);
		for (i = 0; i < COLS; i++) {
			mvaddch (0, i, ' ');
		}
		mvprintw (0, 1, "* Snownews %s", VERSION);
		attroff (WA_REVERSE);
		
		if (typeahead) {
			/* This resets the offset for every typeahead loop. */
			first_scr_ptr = current_feed->items;
			
			count = 0;
			skipper = 0;
			found = 0;
			for (cur_ptr = current_feed->items; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
				/* count+1 > LINES-7:
				   If the _next_ line would go over the boundary. */
				if (count+1 > LINES-7) {
						if (first_scr_ptr->next_ptr != NULL)
							first_scr_ptr = first_scr_ptr->next_ptr;
				}
				if (searchlen > 0) {
					if (strncmp (search, cur_ptr->title, searchlen) == 0) {
						found = 1;
						highlighted = cur_ptr;
						if (skip >= 1) {
							if (skipper == skip) {
								break;
							} else {
								skipper++;
								/* We need to increase counter also, because we
								   don't reach the end of the loop after continue!. */
								count++;
								continue;
							}
						}
						break;
					}
					count++;
				}
			}
			if (!found) {
				/* Restore original position on no match. */
				highlighted = savestart;
				first_scr_ptr = savestart_first;
			}
		}
		
		attron (WA_BOLD);
		if (current_feed->description != NULL) {
			/* Print a max of COLS-something chars.
			   This function might cause overflow if user resizes screen
			   during malloc and strncpy. To prevent this we use a private
			   columns variable. */
			columns = COLS-10;
			if (strlen(current_feed->description) > columns) {
				feeddescscreenlength = malloc(columns+1);
				strncpy (feeddescscreenlength, current_feed->description, columns);
				feeddescscreenlength[columns] = '\0';
				converted = iconvert (feeddescscreenlength, "UTF-8", TARGET_CHARSET);
				/* Fallback: duplicate string into converted so free won't crash.
				   View the unconverted item instead if iconvert fails. */
				if (converted == NULL)
					converted = strdup (feeddescscreenlength);
				mvprintw (2, 1, "%s...", converted);
				free (feeddescscreenlength);
				free (converted);
			} else {
				converted = iconvert (current_feed->description, "UTF-8", TARGET_CHARSET);
				if (converted == NULL)
					converted = strdup (current_feed->description);
				mvprintw (2, 1, "%s", converted);
				free (converted);
			}
		} else {
			mvprintw (2, 1,	"%s", current_feed->title);
		}
		attroff (WA_BOLD);
		
		/* We start the item list at line 5. */
		ypos = 4;
		itemnum = 1;
		
		columns = COLS-6;			/* Cut max item length. */
		/* Print unread entries in bold. */
		for (curitem_ptr = first_scr_ptr; curitem_ptr != NULL; curitem_ptr = curitem_ptr->next_ptr) {
			/* Set cursor to start of current line and clear it. */
			move (ypos, 0);
			clrtoeol();
			
			if (curitem_ptr->readstatus != 1)
				attron (WA_BOLD);
			if (curitem_ptr == highlighted) {
				highlightnum = itemnum;
				attron (WA_REVERSE);
				for (i = 0; i < COLS-1; i++)
					mvaddch (ypos, i, ' ');
			}
			
			/* Check for empty <title> */
			if (curitem_ptr->title != NULL) {
				if (strlen(curitem_ptr->title) > columns) {
					itemcut = malloc(columns+1);
					strncpy (itemcut, curitem_ptr->title, columns);
					itemcut[columns] = '\0';
					converted = iconvert (itemcut, "UTF-8", TARGET_CHARSET);
					if (converted == NULL)
						converted = strdup (itemcut);
					mvprintw (ypos, 1, "%s...", converted);
					free (converted);
					free (itemcut);
				} else {
					converted = iconvert (curitem_ptr->title, "UTF-8", TARGET_CHARSET);
					if (converted == NULL)
						converted = strdup (curitem_ptr->title);
					mvaddstr (ypos, 1, converted);
					free (converted);
				}
			} else
				mvaddstr (ypos, 1, _("No title"));
			ypos++;
			if (curitem_ptr == highlighted)
				attroff (WA_REVERSE);
			if (curitem_ptr->readstatus != 1)
				attroff (WA_BOLD);
				
			if (itemnum >= LINES-7)
				break;
			itemnum++;
		}
		/* refresh(); */ /* Not needed since UIStatus does this for us. */
		
		attron (WA_BOLD);
		mvprintw (LINES-2, 1, "-> %s", syscall);
		attroff (WA_BOLD);
		
		if (typeahead) {
			snprintf (tmpstr, sizeof(tmpstr), "-> %s", search);
			UIStatus (tmpstr, 0);
		} else
			UIStatus (_("Press 'q' to return to main menu, 'h' to show help."), 0);
		
		uiinput = getch();
		
		if (typeahead) {
			/* Only match real characters. */
			if ((uiinput >= 32) && (uiinput <= 126)) {
				search[searchlen] = uiinput;
				search[searchlen+1] = 0;
				searchlen++;
				search = realloc (search, searchlen+2);
			/* ASCII 127 is DEL, 263 is... actually I have
			   no idea, but it's what the text console returns. */
			} else if ((uiinput == 127) || (uiinput == 263)) {
				/* Don't let searchlen go below 0! */
				if (searchlen > 0) {
					searchlen--;
					search[searchlen] = 0;
					search = realloc (search, searchlen+2);
				} else {
					typeahead = 0;
					free (search);
				}
			}
		} else {
			if (uiinput == 'h')
				UIDisplayFeedHelp();
			if (uiinput == keybindings.quit) {
				/* Only free these when we return!
				   Otherwise this would introduce multifrees. 8) */
				free (syscall);
				free (escapetext);
				return;
			}
			if ((uiinput == KEY_UP) ||
				(uiinput == keybindings.prev)) {
				/* Check if we have no items at all! */
				if (highlighted != NULL) {
					if (highlighted->prev_ptr != NULL) {
						highlighted = highlighted->prev_ptr;
						if (highlightnum-1 < 1) {
							/* First scr entry. */
							if (first_scr_ptr->prev_ptr != NULL)
								first_scr_ptr = first_scr_ptr->prev_ptr;
							}
					}
				}
			}
			if ((uiinput == KEY_DOWN) ||
				(uiinput == keybindings.next)) {
				/* Check if we have no items at all! */
				if (highlighted != NULL) {
					if (highlighted->next_ptr != NULL) {
						highlighted = highlighted->next_ptr;
						if (highlightnum+1 > LINES-7) {
							/* Fell off screen. */	
							if (first_scr_ptr->next_ptr != NULL)
								first_scr_ptr = first_scr_ptr->next_ptr;
						}	
					}
				}
			}
			/* Move highlight one page up/down == LINES-9 */
			if ((uiinput == KEY_NPAGE) || (uiinput == 32) ||
				(uiinput == keybindings.pdown)) {
				if (highlighted != NULL) {
					for (i = highlightnum; i <= LINES-8; i++) {
						if (highlighted->next_ptr != NULL) {
							highlighted = highlighted->next_ptr;
						} else
							break;
					}
				}
				first_scr_ptr = highlighted;
			}
			if ((uiinput == KEY_PPAGE) || (uiinput == keybindings.pup)) {
				if (highlighted != NULL) {
					for (i = highlightnum; i <= LINES-8; i++) {
						if (highlighted->prev_ptr != NULL) {
							highlighted = highlighted->prev_ptr;
						} else
							break;
					}
				}
				first_scr_ptr = highlighted;
			}
			if (uiinput == keybindings.reload) {
				UpdateFeed (current_feed);
				highlighted = current_feed->items;
				/* Reset first_scr_ptr if reloading. */
				first_scr_ptr = current_feed->items;
				forceredraw = 1;
			}
			if (uiinput == keybindings.urljump) {
				/* Copy "Executing lynx URL" into tmp and show it,
				   copy "lynx URL 2>/dev/null" into syscallnull and execute via system. */
				len = strlen(syscall) + strlen(_("Executing %s")) + 1;
				tmp = malloc (len);
				snprintf (tmp, len, _("Executing %s"), syscall);
				UIStatus (tmp, 0);
				len = len + 13;
				syscallnull = malloc (len);
				snprintf (syscallnull, len, "%s 2>/dev/null", syscall);
				
				/* Switch on the cursor. */
				curs_set(1);
				
				system(syscallnull);
				
				InitCurses();
				
				/* Hide cursor again. */
				curs_set(0);
				
				free (tmp);
				free (syscallnull);
			}
			if (uiinput == keybindings.markread) {
				/* Mark everything read. */
				for (markasread = current_feed->items; markasread != NULL; markasread = markasread->next_ptr) {
					markasread->readstatus = 1;
				}
			}		
			if (uiinput == 'A')
				UIAbout();
			if (uiinput == keybindings.feedinfo)
				FeedInfo(current_feed);
			
			/* DON NOT free escapetext and syscall here. It would result in multiple frees! */
		}
		
		if (uiinput == '\n') {
			/* If typeahead is active clear it's state and free the structure. */
			if (typeahead) {
				free (search);
				typeahead = 0;
				curs_set(0);
			}
			/* Check if we have no items at all!
			   Don't even try to view a non existant item. */
			if (highlighted != NULL) {
				UIDisplayItem (highlighted, current_feed);
				tmp_highlighted = highlighted;
				tmp_first = first_scr_ptr;
				
				/* highlighted = first_scr_ptr; */
					
				/* Moves highlight to next unread item. */
				while (highlighted->next_ptr != NULL) {
					if (highlighted->readstatus == 1) {
						highlighted = highlighted->next_ptr;
						
						highlightnum++;
						if (highlightnum+1 > LINES-7) {
							/* Fell off screen. */	
							if (first_scr_ptr->next_ptr != NULL)
								first_scr_ptr = first_scr_ptr->next_ptr;
						}	
						
					} else
						break;				
				}
				
				/* If *highlighted points to a read entry now we have hit the last
				   entry and there are no unread items anymore. Restore the original
				   location in this case. */
				if (highlighted->readstatus == 1) {
					highlighted = tmp_highlighted;
					first_scr_ptr = tmp_first;
				}
			}
		}
		
		/* TAB key is decimal 9. */
		if (uiinput == 9) {
			if (typeahead) {
				if (searchlen == 0) {
					typeahead = 0;
					/* Typeahead now off. */
					curs_set(0);
					free (search);
					skip = 0;
					highlighted = savestart;
					first_scr_ptr = savestart_first;
				} else {
					found = 0;
					for (cur_ptr = current_feed->items; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
						if (strncmp (search, cur_ptr->title, searchlen) == 0) {
							if (strncmp (search, cur_ptr->title, searchlen) == 0) {
								found++;
							}
						}
					}
					if (skip == found-1) {
						skip = 0;
					} else if (found > 1) {
						/* If more than one match was found and user presses tab we will skip matches. */
						skip++;
					}
				}
			} else {
				typeahead = 1;
				/* Typeahead now on. */
				curs_set(1);
				search = malloc (2);
				memset (search, 0, 2);
				searchlen = 0;
				/* Save all start positions. */
				savestart = highlighted;
				savestart_first = first_scr_ptr;
			}
		}
		/* ctrl+g clears typeahead. */
		if (uiinput == 7) {
			/* But only if it was switched on previously. */
			if (typeahead) {
				typeahead = 0;
				free (search);
				skip = 0;
				highlighted = savestart;
				first_scr_ptr = savestart_first;
			}
		}
		/* ctrl+u clears line. */
		if (uiinput == 21) {
			searchlen = 0;
			search = realloc (search, 2);
			memset (search, 0, 2);
		}

		
	}	
}

void UIMainInterface (void) {
	int i, ypos, len;
	int newcount;					/* Number of new articles. */
	int uiinput = 0;
	int typeahead = 0;
	int skip = 0;					/* Number of typeahead matches to skip. */
	int skipper = 0;				/* " skipped */
	int found = 0;					/* Number of typeahead matches. */
	char *search = NULL;			/* Typeahead search string. */
	int searchlen = 0;				/* " length. */
	char tmp[1024];
	char *file, *hashme;
	struct feed *cur_ptr;
	struct newsitem *curitem_ptr;
	struct newsitem *markasread;
	struct feed *highlighted;
	struct feed *first_scr_ptr;		/* First pointer on top of screen. Used for scrolling. */
	struct feed *removed;			/* Removed struct tmp pointer. */
	struct feed *savestart;
	struct feed *savestart_first = NULL;
	int itemnum;					/* Num of items on screen. Max is LINES-4 */
	int highlightnum = 1;
	int columns;
	int count;
	char *maxlinesize;
	int forceredraw = 0;

	first_scr_ptr = first_ptr;
	highlighted = first_ptr;
	savestart = first_scr_ptr;
	
	/* Clear produces flickering when used inside the loop!
	   Only call it once and use move+clrtobot inside! */
	clear();
	
	while (1) {
		if (forceredraw) {
			/* Call clear() to reinit everything.
			   Usually called after downloading/parsing feed to
			   avoid Ramsch on the screen.  :) */
			clear();
			forceredraw = 0;
		} else {
			/* Clear screen with clrtobot() */
			move (0,0);
			clrtobot();
		}
		
		attron (WA_REVERSE);	
		for (i = 0; i < COLS; i++) {
			mvaddch (0, i, ' ');
		}
		//mvaddnstr (0, 0, " ", COLS-1);
		mvprintw (0, 1, "* Snownews %s", VERSION);
		attroff (WA_REVERSE);
	
		ypos = 2;
		itemnum = 1;
	
		if (typeahead) {
			/* This resets the offset for every typeahead loop. */
			first_scr_ptr = first_ptr;
			
			count = 0;
			skipper = 0;
			found = 0;
			for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
				/* count+1 >= Lines-4:
				   If the _next_ line would go over the boundary. */
				if (count+1 > LINES-4) {
						if (first_scr_ptr->next_ptr != NULL)
							first_scr_ptr = first_scr_ptr->next_ptr;
				}
				if (searchlen > 0) {
					if (strncmp (search, cur_ptr->title, searchlen) == 0) {
						found = 1;
						highlighted = cur_ptr;
						if (skip >= 1) {
							if (skipper == skip) {
								break;
							} else {
								skipper++;
								/* We need to increase counter also, because we
								   don't reach the end of the loop after continue!. */
								count++;
								continue;
							}
						}
						break;
					}
				}
				count++;
			}
			if (!found) {
				/* Restore original position on no match. */
				highlighted = savestart;
				first_scr_ptr = savestart_first;
			}
		}
		
		for (cur_ptr = first_scr_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
			/* Set cursor to start of current line and clear it. */
			move (ypos, 0);
			clrtoeol();
			
			if (cur_ptr == highlighted) {
				highlightnum = itemnum;
				attron (WA_REVERSE);
				for (i = 0; i < COLS-1; i++)
					mvaddch (ypos, i, ' ');
			}
			
			columns = COLS-9-strlen(_("new"));
			if (strlen(cur_ptr->title) > columns) {
				maxlinesize = malloc(columns+1);
				strncpy (maxlinesize, cur_ptr->title, columns);
				maxlinesize[columns] = '\0';
				mvprintw (ypos, 1, "%s...", maxlinesize);
				free (maxlinesize);
				if (cur_ptr->problem)
					mvaddch (ypos, 0, '!');
			} else {
				mvaddstr (ypos, 1, cur_ptr->title);
				if (cur_ptr->problem)
					mvaddch (ypos, 0, '!');
			}
			
			newcount = 0;
			for (curitem_ptr = cur_ptr->items; curitem_ptr != NULL; curitem_ptr = curitem_ptr->next_ptr) {
				if (curitem_ptr->readstatus == 0)
					newcount++;
			}
			/* Execute this _after_ the for loop. Otherwise it'll suck CPU like crazy! */
			if (newcount != 0) {
				snprintf (tmp, sizeof(tmp), _("%3d new"), newcount);
				mvaddstr (ypos, COLS-5-strlen(_("new")), tmp);
			}
			ypos++;
			if (cur_ptr == highlighted)
				attroff (WA_REVERSE);
			
			if (itemnum >= LINES-4)
				break;
			itemnum++;
		}
		/* refresh(); */ /* Not needed since UIStatus does this for us. */

		if (typeahead) {
			snprintf (tmp, sizeof(tmp), "-> %s", search);
			UIStatus (tmp, 0);
		} else
			UIStatus (_("Press 'h' for help window."), 0);
		
		uiinput = getch();
				
		if (typeahead) {
			/* Only match real characters. */
			if ((uiinput >= 32) && (uiinput <= 126)) {
				search[searchlen] = uiinput;
				search[searchlen+1] = 0;
				searchlen++;
				search = realloc (search, searchlen+2);
			} else if ((uiinput == 127) || (uiinput == 263)) {
				/* Don't let searchlen go below 0! */
				if (searchlen > 0) {
					searchlen--;
					search[searchlen] = 0;
					search = realloc (search, searchlen+2);
				} else {
					typeahead = 0;
					free (search);
				}
			}
		} else {
			if (uiinput == keybindings.quit) {
#ifdef QUIT_CONFIRM
				snprintf (tmp, sizeof(tmp), _("Exit Snownews? Press '%c' again to quit."), keybindings.quit);
				UIStatus (tmp, 0);
				if (getch() == keybindings.quit)
#endif
					MainQuit (NULL, NULL);
			}
			if (uiinput == keybindings.reload) {
				/* Refresh highlighted feed. */
				if (UpdateFeed (highlighted) != 0) {
					if (lasthttpstatus == 0) {
						UIStatus (_("Aborted."), 1);
						continue;
					}
					if (lasthttpstatus == 304)
						continue;
					if (lasthttpstatus != 200) {
						snprintf (tmp, sizeof(tmp), _("HTTP error %d - Could not update feed."), lasthttpstatus);
						UIStatus (tmp, 2);
						continue;
					}
					if (lasthttpstatus != 0) {
						snprintf (tmp, sizeof(tmp), _("Could not update feed %s - %s"), highlighted->title, strerror(errno));
						UIStatus (tmp, 2);
					}
				}
				forceredraw = 1;
			}
			if (uiinput == keybindings.reloadall) {
				UpdateAllFeeds();
				forceredraw = 1;
			}
			if (uiinput == keybindings.addfeed) {
				switch (UIAddFeed()) {
					case 0:
						UIStatus (_("Successfully added new item..."), 1);
						
						/* If this was the first element added set highlighted to
				   		this one. */
				   		/* Why was this if == NULL added? */
						/* if (highlighted == NULL) { */
						highlighted = first_ptr;
						/* } */
						first_scr_ptr = first_ptr;
					
						break;
					case 1:
						UIStatus (_("Aborted."), 1);
						break;
					case 2:
						UIStatus (_("Invalid URL! Please add http:// if you forgot this."), 2);
						break;
					case -1:
					default:
						break;
				}
				forceredraw = 1;
			}
			if (uiinput == 'h')
				UIHelpScreen();
			if (uiinput == keybindings.deletefeed) {
				if (UIDeleteFeed(highlighted->title) == 1) {
					/* Do it! */
					removed = highlighted;
					
					/* Remove cachefile from filesystem. */
				
					hashme = malloc (strlen(highlighted->feedurl)+1);
					strncpy (hashme, highlighted->feedurl, strlen(highlighted->feedurl)+1);
					Hashify (hashme);
		
					len = (strlen(getenv("HOME")) + strlen(hashme) + 18);
					file = malloc (len);
					snprintf (file, len, "%s/.snownews/cache/%s", getenv("HOME"), hashme);
					
					/* Errors from unlink can be ignored. Worst thing that happens is that
					   we delete a file that doesn't exist. */
					unlink (file);
					
					free (file);
					free (hashme);
									
					/* Unlink pointer from chain. */
					if (highlighted == first_ptr) {
						/* first element */
						if (first_ptr->next_ptr != NULL) {
							first_ptr = first_ptr->next_ptr;
							first_scr_ptr = first_ptr;
							highlighted->next_ptr->prev_ptr = NULL;
						} else {
							first_ptr = NULL;
							first_scr_ptr = NULL;
						}
					} else if (highlighted->next_ptr == NULL) {
						/* last element */
						highlighted->prev_ptr->next_ptr = NULL;
					} else {
						/* element inside list */
						highlighted->prev_ptr->next_ptr = highlighted->next_ptr;
						highlighted->next_ptr->prev_ptr = highlighted->prev_ptr;
					}
					highlighted = first_ptr;
				
					/* free (removed) pointer */
					if (removed->items != NULL) {
						while (removed->items->next_ptr != NULL) {
							removed->items = removed->items->next_ptr;
							if (removed->items->prev_ptr->title != NULL)
								free (removed->items->prev_ptr->title);
							if (removed->items->prev_ptr->link != NULL)
								free (removed->items->prev_ptr->link);
							if (removed->items->prev_ptr->description != NULL)
								free (removed->items->prev_ptr->description);
							free (removed->items->prev_ptr);
						}
						if (removed->items->title != NULL)
							free (removed->items->title);
						if (removed->items->link != NULL)
							free (removed->items->link);
						if (removed->items->description != NULL)
							free (removed->items->description);
						free (removed->items);
					}
					/* This is ugly. :(
					   But it prevents crash if deleting a feed that is broken. */
					if (removed->feedurl != NULL)
						free (removed->feedurl);
					if (removed->feed != NULL)
						free (removed->feed);
					if (removed->title != NULL)
						free (removed->title);
					if (removed->link != NULL)
						free (removed->link);
					if (removed->description != NULL)
						free (removed->description);
					if (removed->lastmodified != NULL)
						free (removed->lastmodified);
					if (removed->override != NULL)
						free (removed->override);
					if (removed->original != NULL)
						free (removed->original);
					free (removed);			
				} else {
					UIStatus (_("Aborted."), 1);
				}
			}
			if ((uiinput == KEY_UP) ||
				(uiinput == keybindings.prev)) {
				if (highlighted != NULL) {
					if (highlighted->prev_ptr != NULL) {
						highlighted = highlighted->prev_ptr;
						if (highlightnum-1 < 1) {
							/* Reached first onscreen entry. */
							if (first_scr_ptr->prev_ptr != NULL)
								first_scr_ptr = first_scr_ptr->prev_ptr;
						}
					}
				}
			}
			if ((uiinput == KEY_DOWN) ||
				(uiinput == keybindings.next)) {
				if (highlighted != NULL) {
					if (highlighted->next_ptr != NULL) {
						highlighted = highlighted->next_ptr;
						if (highlightnum+1 > LINES-4) {
							/* If we fall off the screen, advance first_scr_ptr
							   to next entry. */
							if (first_scr_ptr->next_ptr != NULL)
								first_scr_ptr = first_scr_ptr->next_ptr;
						}
					}
				}
			}
			/* Move highlight one page up/down == LINES-6 */
			if ((uiinput == KEY_NPAGE) || (uiinput == 32) ||
				(uiinput == keybindings.pdown)) {
				if (highlighted != NULL) {
					for (i = highlightnum; i <= LINES-5; i++) {
						if (highlighted->next_ptr != NULL) {
							highlighted = highlighted->next_ptr;
						} else 
							break;
					}
				}
				first_scr_ptr = highlighted;
			}
			if ((uiinput == KEY_PPAGE) || (uiinput == keybindings.pup)) {
				if (highlighted != NULL) {
					for (i = highlightnum; i <= LINES-5; i++) {
						if (highlighted->prev_ptr != NULL) {
							highlighted = highlighted->prev_ptr;
						} else
							break;
					}
				}
				first_scr_ptr = highlighted;
			}
			if (uiinput == keybindings.moveup) {
				/* Move item up. */
				if (highlighted->prev_ptr != NULL) {
					SwapPointers (highlighted, highlighted->prev_ptr);
					highlighted = highlighted->prev_ptr;
						if (highlightnum-1 < 1) {
							if (first_scr_ptr->prev_ptr != NULL)
								first_scr_ptr = first_scr_ptr->prev_ptr;
						}
				}
			}
			if (uiinput == keybindings.movedown) {
				/* Move item down. */
				if (highlighted->next_ptr != NULL) {
					SwapPointers (highlighted, highlighted->next_ptr);
					highlighted = highlighted->next_ptr;
					if (highlightnum+1 > LINES-4) {
						if (first_scr_ptr->next_ptr != NULL)
								first_scr_ptr = first_scr_ptr->next_ptr;
					}
				}
			}
			if (uiinput == keybindings.dfltbrowser) {
				if (UIChangeBrowser()) {
					UIStatus (_("Aborted."), 1);
				}
			}
			if (uiinput == keybindings.markread) {
				for (cur_ptr = first_scr_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
					for (markasread = cur_ptr->items; markasread != NULL; markasread = markasread->next_ptr) {
						markasread->readstatus = 1;
					}
				}
			}
			if (uiinput == 'A')
				UIAbout();
			if (uiinput == keybindings.changefeedname) {
				switch (UIChangeFeedName(highlighted)) {
					case 0:
						/* Changed */
						UIStatus (_("Title changed successfully."), 2);
						break;
					case 1:
						/* Aborted */
						UIStatus (_("Aborted."), 1);
						break;
					case 2:
						UIStatus (_("Original title restored."), 2);
						break;
					default:
						break;
				}
			}
			if (uiinput == keybindings.sortfeeds)
				SnowSort();
		}
		
		if (uiinput == '\n') {
			/* If typeahead is active clear it's state and free the structure. */
			if (typeahead) {
				free (search);
				typeahead = 0;
			}
			/* Select this feed, open and view entries. */
			if (highlighted != NULL)
				UIDisplayFeed (highlighted);
				
			/* Clear screen after we return from here. */
			move(0,0);
			clrtobot();
		}
		
		/* TAB key is decimal 9. */
		if (uiinput == 9) {
			if (typeahead) {
				if (searchlen == 0) {
					typeahead = 0;
					/* Typeahead now off. */
					free (search);
					skip = 0;
					highlighted = savestart;
					first_scr_ptr = savestart_first;
				} else {
					found = 0;
					for (cur_ptr = first_ptr; cur_ptr != NULL; cur_ptr = cur_ptr->next_ptr) {
						if (strncmp (search, cur_ptr->title, searchlen) == 0) {
							if (strncmp (search, cur_ptr->title, searchlen) == 0) {
								found++;
							}
						}
					}
					if (skip == found-1) { /* found-1 to avoid empty tab cycle. */
						skip = 0;
					} else if (found > 1) {
						/* If more than one match was found and user presses tab we will skip matches. */
						skip++;
					}
				}
			} else {
				typeahead = 1;
				/* Typeahead now on. */
				search = malloc (2);
				memset (search, 0, 2);
				searchlen = 0;
				/* Save all start positions. */
				savestart = highlighted;
				savestart_first = first_scr_ptr;
			}
		}
		/* ctrl+g clears typeahead. */
		if (uiinput == 7) {
			/* But only if it was switched on previously. */
			if (typeahead) {
				typeahead = 0;
				free (search);
				skip = 0;
				highlighted = savestart;
				first_scr_ptr = savestart_first;
			}
		}
		/* ctrl+u clears line. */
		if (uiinput == 21) {
			searchlen = 0;
			search = realloc (search, 2);
			memset (search, 0, 2);
		}

	}
}

