/* Snownews - A lightweight console RSS newsreader
 * 
 * Copyright 2003 Oliver Feiler <kiza@kcore.de>
 * http://kiza.kcore.de/software/snownews/
 *
 * interface.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <ncurses.h>

#include <time.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "interface.h"
#include "main.h"

extern char *browser;
extern struct keybindings keybindings;
 
/* Snowflake for the xmas about box. */
struct snowflake {
	int x;
	int y;
	int oldx;
	int oldy;
	char oldchar;
	char oldchar2;
	int visible;		/* Don't draw flakes over text. */
	int vspeed;			/* Vertical speed of the flake */
	int hspeed;			/* Horizontal speed */
	struct snowflake * next;
	struct snowflake * prev;
};
 
char * UIOneLineEntryField (int x, int y) {
        char *text;

        text = malloc(512);
        
        /* UIStatus switches off attron! */
        attron (WA_REVERSE);
        echo();
        curs_set(1);

        move (y, x);
        /* Beware of hardcoded textlength size! */
        getnstr (text, 511);
        
        noecho();
        curs_set(0);
        attroff (WA_REVERSE);
        
        /* This memory needs to be freed in the calling function! */
        return text;
}

int UIChangeBrowser (void) {
	int centerx, centery, i, j, len;
	char *browserstring;
	
	/* malloc = 17 (strlen("Current setting: ") + browser
	   We will malloc a bigger junk, because other languages
	   might need longer strings and crash! */
	len = strlen(_("Current setting: %s")) + strlen(browser) + 1;
	browserstring = malloc (len);
	snprintf (browserstring, len, _("Current setting: %s"), browser);
	
	centerx = COLS / 2;
	centery = LINES / 2;

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	for (i = 5; i <= 7; i++) {
		for (j = -(COLS/2-3); j <= (COLS/2-3); j++) {
				mvaddch (i, centerx+j, ' ');
		}	
	}
	
	UIStatus (browserstring, 0);
	free (browserstring);
	
	browserstring = UIOneLineEntryField (5, 6);
	
	if (strlen(browserstring) == 0) {
		free (browserstring);
		return 1;
	}
	
	browser = realloc (browser, strlen(browserstring)+1);
	strncpy (browser, browserstring, strlen(browserstring)+1);
	
	free (browserstring);
	
	return 0;
}

/* Dialog to change feedname.
   Return: 0	on success
           1	on user abort
		   2	original title restored
*/
int UIChangeFeedName (struct feed *cur_ptr) {
	char *newname;
	int centerx, centery, i, j;

	centerx = COLS / 2;
	centery = LINES / 2;

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	for (i = 5; i <= 7; i++) {
		for (j = -(COLS/2-3); j <= (COLS/2-3); j++) {
				mvaddch (i, centerx+j, ' ');
		}	
	}
	UIStatus (_("Enter new name. Blank line to abort. '-' to reset."), 0);
	
	newname = UIOneLineEntryField (5, 6);
	
	/* If strlen is zero, return 1. */
	if (strlen(newname) == 0) {
		free (newname);
		return 1;
	}
	
	if ((newname != NULL) && (cur_ptr->override != NULL)) {
		if (strcmp(newname, "-") == 0) {
			if (cur_ptr->title != NULL)
				free (cur_ptr->title);
			cur_ptr->title = strdup(cur_ptr->original);
			free (cur_ptr->original);
			/* Set back original to NULL pointer. */
			cur_ptr->original = NULL;
			free (cur_ptr->override);
			cur_ptr->override = NULL;
			free (newname);
			return 2;
		}
	}
	
	/* Copy new name into ->override. */
	if (cur_ptr->override != NULL)
		free (cur_ptr->override);
	cur_ptr->override = strdup (newname);
	
	/* Save original. */
	if (cur_ptr->original != NULL)
		free (cur_ptr->original);
	cur_ptr->original = strdup (cur_ptr->title);
	
	/* Set new title. */
	if (cur_ptr->title != NULL)
		free (cur_ptr->title);
	cur_ptr->title = strdup (newname);
		
	free (newname);
	return 0;
}

/* Popup window to add new RSS feed. */
int UIAddFeed (void) {
	char tmp[512];
	char *url;
	int centerx, centery;
	int i, j;
	struct feed *new_ptr;
	struct feed *tmp_ptr;

	centerx = COLS / 2;
	centery = LINES / 2;

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	for (i = 5; i <= 7; i++) {
		for (j = -(COLS/2-3); j <= (COLS/2-3); j++) {
				mvaddch (i, centerx+j, ' ');
		}	
	}
	UIStatus (_("Enter URL of the feed you want to add. Blank line to abort."), 0);

	url = UIOneLineEntryField (5, 6);
	
	/* If read stringlength is ZARO (abort of action requested) return 1
	   and confuse the calling function. */
	if (strlen(url) == 0) {
		free (url);
		return 1;
	}
	
	/* Check of entered url is valid at all. */
	if (strstr(url, "http://") == NULL) {
		free (url);
		return 2;
	}

	new_ptr = malloc (sizeof(struct feed));
	/* getnstr does not return newline... says the docs. */
	new_ptr->feedurl = malloc (strlen(url)+1);
	new_ptr->feed = NULL;
	
	/* Set to NULL so xmlparse.c is happy when we try to feed there. */
	new_ptr->title = NULL;
	new_ptr->link = NULL;
	new_ptr->description = NULL;
	new_ptr->items = NULL;
	new_ptr->lastmodified = NULL;
	new_ptr->override = NULL;
	new_ptr->original = NULL;
	new_ptr->problem = 0;
			
	/* Attach to feed pointer chain. */
	strncpy (new_ptr->feedurl, url, strlen(url)+1);
	new_ptr->next_ptr = first_ptr;
	if (first_ptr != NULL)
		first_ptr->prev_ptr = new_ptr;
	new_ptr->prev_ptr = NULL;
	first_ptr = new_ptr;
	
	/* Don't need url text anymore. */
	free (url);

	/* Download new feed and DeXMLize it. */	
	if ((UpdateFeed (new_ptr)) != 0) {
		if ((lasthttpstatus != 200) && (lasthttpstatus != 0)) {
			snprintf (tmp, sizeof(tmp), _("HTTP error %d - Could not update feed."), lasthttpstatus);
			UIStatus (tmp, 2);
		}
		
		/* Free the feeds data structure again. Otherwise we'll leak them. */
		if (new_ptr->feedurl != NULL)
			free (new_ptr->feedurl);
		if (new_ptr->link != NULL)
			free (new_ptr->link);
		if (new_ptr->title != NULL)
			free (new_ptr->title);
		if (new_ptr->feed != NULL)
			free (new_ptr->feed);
		
		/* Remove from feed pointer chain -> feed has problems -> we don't add it. */
		tmp_ptr = first_ptr;
		if (first_ptr->next_ptr != NULL)
			 first_ptr->next_ptr->prev_ptr = NULL;
		first_ptr = first_ptr->next_ptr;
		free (tmp_ptr);
		
		return -1;
	}

	return 0;
}

void FeedInfo (struct feed * current_feed) {
	int centerx, i, j, len;
	char *hashme;			/* Hashed filename. */
	char *file;
	struct stat filetest;
	
	centerx = COLS / 2;
	
	hashme = malloc (strlen(current_feed->feedurl)+1);
	strncpy (hashme, current_feed->feedurl, strlen(current_feed->feedurl)+1);
	Hashify (hashme);
	
	len = (strlen(getenv("HOME")) + strlen(hashme) + 18);
	file = malloc (len);
	snprintf (file, len, "%s/.snownews/cache/%s", getenv("HOME"), hashme);

	/* Clear the help screen area and print the text on reverse background. */
	/*attron (WA_REVERSE);
	for (i = -4; i <= 5; i++) {
		for (j = -20; j <= 20; j++) {
				mvaddch (centery+i, centerx+j, ' ');
		}	
	}*/
	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	for (i = 0; i <= 7; i++) {
		for (j = -(COLS/2-5); j <= (COLS/2-5); j++) {
				mvaddch (4+i, centerx+j, ' ');
		}	
	}
	mvaddstr (5, centerx-((strlen(current_feed->title))/2), current_feed->title);
	mvaddstr (7, centerx-(COLS/2-7), current_feed->feedurl);
	if (current_feed->lastmodified != NULL)
		mvprintw (8, centerx-(COLS/2-7), _("Last updated: %s"), current_feed->lastmodified);	
	else
		mvaddstr (8, centerx-(COLS/2-7), _("No modification date."));
	
	if ((stat (file, &filetest)) != -1) {
		mvprintw (9, centerx-(COLS/2-7), _("In disk cache: %lld bytes"), (long long int) filetest.st_size);
	} else
		mvaddstr (9, centerx-(COLS/2-7), _("Not in disk cache."));
	
	UIStatus (_("Displaying feed information."), 0);
	
	free (file);
	free (hashme);

	getch();
}

int UIDeleteFeed (char * feedname) {
	int i, j;

	/* Clear screen area we want to "draw" to. */
	attron (WA_REVERSE);
	for (i = 5; i <= 8; i++) {
		for (j = -(COLS/2-3); j <= (COLS/2-3); j++) {
				mvaddch (i, (COLS/2)+j, ' ');
		}	
	}
	mvaddstr (6, COLS/2-21, _("Are you sure you want to delete this feed?"));
	mvprintw (7, 5, "%s", feedname);
	
	UIStatus (_("Type 'y' to delete, any other key to abort."), 0);
	
	if (getch() == 'y')
		return 1;
	else
		return 0;
}

void UIHelpScreen (void) {
	int centerx, centery;				/* Screen center x/y coordinate. */
	int i, j;
	int userinput;
	
	centerx = COLS / 2;
	centery = LINES / 2;

	/* Clear the help screen area and print the text on reverse background. */
	attron (WA_REVERSE);
	for (i = -6; i <= 7; i++) {
		for (j = -20; j <= 20; j++) {
				mvaddch (centery+i, centerx+j, ' ');
		}	
	}
	mvprintw (centery-5, centerx-18, _("%c:    Add RSS feed..."), keybindings.addfeed);	
	mvprintw (centery-4, centerx-18, _("%c:    Delete highlighted RSS feed..."), keybindings.deletefeed);
	mvprintw (centery-3, centerx-18, _("%c:    Rename feed..."), keybindings.changefeedname);
	mvprintw (centery-2, centerx-18, _("%c:    Reload all feeds"), keybindings.reloadall);
	mvprintw (centery-1, centerx-18, _("%c:    Reload this feed"), keybindings.reload);
	mvprintw (centery,   centerx-18, _("%c:    Mark all read"), keybindings.markread);
	mvprintw (centery+1, centerx-18, _("%c:    Change default browser..."), keybindings.dfltbrowser);
	mvprintw (centery+2, centerx-18, _("%c, %c: Move item up, down"), keybindings.moveup, keybindings.movedown);
	mvprintw (centery+3, centerx-18, _("%c:    Sort feed list alphabetically"), keybindings.sortfeeds);
	mvaddstr (centery+4, centerx-18, _("tab:  Type Ahead Find"));
	mvaddstr (centery+5, centerx-18, _("A:    About"));
	mvprintw (centery+6, centerx-18, _("%c:    Quit program"), keybindings.quit);
	attroff (WA_REVERSE);

	UIStatus (_("Press the any(tm) key to exit help screen."), 0);
	userinput = getch();
	
	/* Return input back into input queue so it gets automatically
	   executed. */
	if ((userinput != '\n') && (userinput != 'h') && (userinput != 'q'))
		ungetch(userinput);
}

void UIDisplayFeedHelp (void) {
	int centerx, centery;				/* Screen center x/y coordinate. */
	int i, j;
	int userinput;
	
	centerx = COLS / 2;
	centery = LINES / 2;

	/* Clear the help screen area and print the text on reverse background. */
	attron (WA_REVERSE);
	for (i = -5; i <= 6; i++) {
		for (j = -19; j <= 19; j++) {
				mvaddch (centery+i, centerx+j, ' ');
		}	
	}
	mvprintw (centery-4, centerx-17, _("%c, up:    Previous item"), keybindings.prev);	
	mvprintw (centery-3, centerx-17, _("%c, down:  Next item"), keybindings.next);
	mvaddstr (centery-2, centerx-17, _("enter:    View item"));
	mvprintw (centery-1, centerx-17, _("%c:        Reload this feed"), keybindings.reload);
	mvprintw (centery,   centerx-17, _("%c:        Open homepage"), keybindings.urljump);
	mvprintw (centery+1, centerx-17, _("%c:        Mark all read"), keybindings.markread);
	mvprintw (centery+2, centerx-17, _("%c:        Show feed info..."), keybindings.feedinfo);
	mvaddstr (centery+3, centerx-17, _("tab:      Type Ahead Find"));
	mvaddstr (centery+4, centerx-17, _("A:        About"));
	mvprintw (centery+5, centerx-17, _("%c:        Return to main menu"), keybindings.quit);
	attroff (WA_REVERSE);

	UIStatus (_("Press the any(tm) key to exit help screen."), 0);
	userinput = getch();
	if ((userinput != '\n') && (userinput != 'h'))
		ungetch(userinput);
}

void UIDisplayItemHelp (void) {
	int centerx, centery;				/* Screen center x/y coordinate. */
	int i, j;
	int userinput;
	
	centerx = COLS / 2;
	centery = LINES / 2;

	/* Clear the help screen area and print the text on reverse background. */
	attron (WA_REVERSE);
	for (i = -3; i <= 4; i++) {
		for (j = -18; j <= 18; j++) {
				mvaddch (centery+i, centerx+j, ' ');
		}	
	}
	mvaddstr (centery-2, centerx-16, _("up, down: Scroll text"));
	mvprintw (centery-1, centerx-16, _("%c, left:  Previous item"), keybindings.prev);	
	mvprintw (centery,   centerx-16, _("%c, right: Next item"), keybindings.next);
	mvprintw (centery+1, centerx-16, _("%c:        Open link..."), keybindings.urljump);
	mvaddstr (centery+2, centerx-16, _("A:        About"));
	mvprintw (centery+3, centerx-16, _("%c, enter: Return to overview"), keybindings.quit);
	attroff (WA_REVERSE);

	UIStatus (_("Press the any(tm) key to exit help screen."), 0);
	userinput = getch();
	if ((userinput != '\n') && (userinput != 'h'))
		ungetch(userinput);
}

void Snowfall (void) {
	struct snowflake *cur;
	struct snowflake *first = NULL;
	struct snowflake *new;
	struct snowflake *curnext;
	/* The idea behind wind:
	   Determine windtimer and wait that many rounds before we initially
	   select any windspeed. If we've waited enough rounds (windtimer==0)
	   select a windspeed and windtimer (which should be much less than
	   between the storms). Use (slightly randomly offset) windspeed to
	   blow snowflakes around (to left or right). If windtimer drops to 0
	   again select new windtimer, wait, repeat. */
	int windspeed = 0;
	int windtimer = 0;
	int wind = 1;			/* 1: wind active, 0: inactive 
							   set to 1 here so the first loop run clears it. */
	int newflake = 0;		/* Time until a new flake appears. */
	
	/* Set ncurses halfdelay mode. */
		halfdelay (3);
	
	while (1) {
		/* Set up the storm. */
		if (windtimer == 0) {
			if (wind) {
				/* Entering silence */
				windtimer = 10 + ((float)rand() / (float)RAND_MAX * 50);
				wind = 0;
				windspeed = 0;
			} else {
				/* Entering storm. */
				windtimer = 10 + ((float)rand() / (float)RAND_MAX * 20);
				wind = 1;
				windspeed = (1+(float)rand() / (float)RAND_MAX * 11)-6;
			}
		}
		//mvaddstr(2,1,     "                              ");
		//if (wind)
		//	mvprintw (2,1,"Windspeed: %d; rounds left: %d",windspeed,windtimer);
		//else
		//	mvprintw (2,1,"No wind; rounds left: %d",windtimer);

		/* Add new snowflakes. */
		if (newflake == 0) {
			/* Add new flake to pointer chain with random x offset. */
			new = malloc (sizeof (struct snowflake));
			new->y = 0;
			new->x = (float)rand() / (float)RAND_MAX * COLS;
			new->oldx = new->x;
			new->oldy = new->y;
			new->visible = 1;
			new->oldchar = new->oldchar2 = ' ';
			new->vspeed = 1+(float)rand() / (float)RAND_MAX * 2;
			new->hspeed = (1+(float)rand() / (float)RAND_MAX * 7)-4;
			
			/* Add our new snowflake to the pointer chain. */
			new->next = NULL;
			if (first == NULL) {
				new->prev = NULL;
				first = new;
			} else {
				new->prev = first;
				while (new->prev->next != NULL)
					new->prev = new->prev->next;
				new->prev->next = new;
			}
			
			/* Set new counter until next snowflake. */
			newflake = 1+(float)rand() / (float)RAND_MAX * 2;
			//mvaddstr (1,1,"      ");
			//mvprintw (1,1,"New flake in %d rounds.", newflake);
		}
		
		for (cur = first; cur != NULL; cur = curnext) {
			curnext = cur->next;
			/* Draw every snowflake at its coordinates to the screen. */
			if (cur->visible) {
				/* Only draw if y<=LINES. This makes some snow lie on bottom of screen. */
				if (cur->y <= LINES)
					mvaddch (cur->oldy, cur->oldx, cur->oldchar2);
				mvaddch (cur->y, cur->x, '*');
				cur->oldx = cur->x;
				cur->oldy = cur->y;
			}
			/* Set new hspeed for flake */
			cur->hspeed = (1+(float)rand() / (float)RAND_MAX * 7)-4;
			
			/* Advance every flake downwards by a random amount and to
			   the left or right.
			   Check if the next position would obscure a character on the screen
			   and set visible to 0 in this case. Clear visible flag as needed. */
			cur->y += cur->vspeed;
			if (wind)
				cur->hspeed += windspeed;
			cur->x += cur->hspeed;
			
			if (cur->y > LINES) {
				if (cur == first) {
					first = first->next;
					first->prev = NULL;
				} else if (cur->next == NULL) {
					cur->prev->next = NULL;
				} else {
					cur->prev->next = cur->next;
					cur->next->prev = cur->prev;
				}
				free (cur);
				continue;
			}
			
			/* Only draw if we're still inside the window. */
			if (cur->y <= LINES) {
				//mvaddstr (3,1,"                ");
				//mvprintw (3,1,"Flake hspeed: %d",cur->hspeed);
				cur->oldchar2 = cur->oldchar;
				/* Reset to ' ' if we accidently set it to *. */
				if (cur->oldchar2 == '*')
					cur->oldchar2 = ' ';
				if ((cur->x <= COLS) && (cur->y <= LINES))
					cur->oldchar = mvinch (cur->y, cur->x);
				else
					cur->oldchar = ' ';
			}
		}
	
		windtimer--;
		newflake--;
		
		refresh();
		
		/* Leave loop if anykey(tm) was pressed. */
		if (getch() != ERR)
			break;
	}
	/* Leave halfdelay mode. I have no idea how this is supposed to work.
	   nodelay (stdscr, FALSE) has no effect. The manpage says nocbreak(),
	   but this buffers chars. Calling cbreak() seems to do the job.
	   Someone tell me how this works please. */
	cbreak();
}

void UIAbout (void) {
	struct tm *t;
	time_t tunix;
	int ypos, xpos;
	
	clear();				/* Get the crap off the screen to make room
							   for our wonderful ASCII logo. :) */

	ypos = 2;
	xpos = COLS/2 - 40;
	
	if (COLS < 80) {
		mvprintw (0, 0, _("Need at least 80 COLS terminal, sorry!"));
		refresh();
		getch();
		return;
	}
	
	/* Save unix time and pass pointer to localtime for conversion into struct tm.*/
	tunix = time(0);
	t = localtime(&tunix);
	
	/* Use Merry Christmas about screen on Dec 24th-26th. :) */
	
	/* Careful with tm_mon==(0-11) vs. tm_mday==(1-31)  STUPID! */
	if ((t->tm_mon == 11) && (t->tm_mday >= 24) && (t->tm_mday <= 26)) {
		/* Christmas tree. */
		mvaddstr (1,  0, "                      _____ _____    ____ _______ _____   ____ _______   _____");
		mvaddstr (2,  0, "                     /  __/ \\    \\  / __ \\\\ |  | \\\\    \\ /  _/ \\ |  | \\ /  __/");
		mvaddstr (3,  0, "          _\\/_       \\___ \\  \\ |  \\ \\ \\/ / \\     / \\ |  \\\\ __\\  \\     / \\___ \\");
		mvaddstr (4,  0, "           /\\        /____/  /_|__/  \\__/  /__|_/  /_|__/ \\___\\ /__|_/  /____/");
		mvaddstr (5,  0, "          /  \\");
		mvaddstr (6,  0, "         /   &\\");
		mvaddstr (7,  0, "        /      \\");
		mvaddstr (8,  0, "       /        \\");
		mvaddstr (9,  0, "      /   &      \\");
		mvaddstr (10, 0, "     /            \\");
		mvaddstr (11, 0, "    /              \\");
		mvaddstr (12, 0, "   /   /      &     \\");
		mvaddstr (13, 0, "  /___/   &      \\___\\");
		mvaddstr (14, 0, "     /            \\");
		mvaddstr (15, 0, "    /              \\");
		mvaddstr (16, 0, "   / &         &    \\");
		mvaddstr (17, 0, "  /                  \\");
		mvaddstr (18, 0, " /    &               \\");
		mvaddstr (19, 0, "/____/      &      \\___\\");
		mvaddstr (20, 0, "    /         \\     \\");
		mvaddstr (21, 0, "   /______####_\\_____\\");
		mvaddstr (22, 0, "         |####|");
		mvaddstr (23, 0, "         |####|");
		/* Credits. */
		mvprintw (5, 21, "Version %s", VERSION);
		mvprintw (8, 35, _("Merry Christmas from the Snownews developers."));
		mvaddstr (10, 35, _("Main code"));
		mvaddstr (11, 35, "Oliver Feiler");
		mvaddstr (13, 35, _("Additional Code"));
		mvaddstr (14, 35, "Rene Puls");
		mvaddstr (16, 35, _("Translation team"));
		mvaddstr (17, 35, "Oliver Feiler, Frank van der Loo,");
		mvaddstr (18, 35, "Pascal Varet, Simon Isakovic");
		mvaddstr (19, 35, "Fernando J. Pereda");
		
		Snowfall();
	} else {
		/* 80 COLS logo */
		mvaddstr (ypos,   xpos, "  ________ _______     ____ __________ _______     ______ __________   ________");
		mvaddstr (ypos+1, xpos, " /  _____/ \\      \\   /    \\\\  |    | \\\\      \\   /  ___/ \\  |    | \\ / ______/");
		mvaddstr (ypos+2, xpos, " \\____  \\   \\   |  \\ /  /\\  \\\\   |    / \\   |  \\ /     /   \\   |    / \\____  \\");
		mvaddstr (ypos+3, xpos, " /       \\  /   |   \\\\  \\/  / \\  |   /  /   |   \\\\  ___\\    \\  |   /  /       \\");
		mvaddstr (ypos+4, xpos, "/______  / / ___|___/ \\____/  /__|__/  / ___|___/ \\____ \\   /__|__/  /______  /");
		mvaddstr (ypos+5, xpos, "       \\/  \\/                          \\/              \\/                   \\/");
		mvprintw (ypos+7, COLS/2-(strlen("Version")+strlen(VERSION)+1)/2, "Version %s", VERSION);
	
		mvaddstr (ypos+10, COLS/2-(strlen(_("Brought to you by:")))/2, _("Brought to you by:"));
		mvaddstr (ypos+12, COLS/2-(strlen(_("Main code")))/2, _("Main code"));
		mvaddstr (ypos+13, COLS/2-6, "Oliver Feiler");
		mvaddstr (ypos+15, COLS/2-(strlen(_("Additional code")))/2, _("Additional code"));
		mvaddstr (ypos+16, COLS/2-4, "Rene Puls");
		mvaddstr (ypos+18, COLS/2-(strlen(_("Translation team")))/2, _("Translation team"));
		mvaddstr (ypos+19, COLS/2-31, "Oliver Feiler, Frank van der Loo, Pascal Varet, Simon Isakovic,");
		mvaddstr (ypos+20, COLS/2-9, "Fernando J. Pereda");

		refresh();
		getch();
	}
}

 
