/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
** Copyright (C) 2001 Brian Caswell <bmc@mitre.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spo_csv.c,v 1.4.4.1 2002/03/15 14:42:32 chrisgreen Exp $ */

/* spo_csv
 * 
 * Purpose:  output plugin for full alerting
 *
 * Arguments:  alert file (eventually)
 *   
 * Effect:
 *
 * Alerts are written to a file in the snort full alert format
 *
 * Comments:   Allows use of full alerts with other output plugin types
 *
 */

/* output plugin header file */
#include "spo_csv.h"

/* external globals from rules.c */
extern char *file_name;
extern int file_line;

/*
 * Function: SetupCSV()
 *
 * Purpose: Registers the output plugin keyword and initialization 
 *          function into the output plugin list.  This is the function that
 *          gets called from InitOutputPlugins() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupCSV()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterOutputPlugin("CSV", NT_OUTPUT_ALERT, CSVInit);

#ifdef DEBUG
    printf("Output plugin: CSV is setup...\n");
#endif
}


/*
 * Function: CSVInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void CSVInit(u_char *args)
{
    SpoCSVData *data;
#ifdef DEBUG
    printf("Output: CSV Initialized\n");
#endif

    pv.alert_plugin_active = 1;

    /* parse the argument list from the rules file */
    data = ParseCSVArgs(args);

#ifdef DEBUG
    printf("Linking CSV functions to call lists...\n");
#endif

    /* Set the preprocessor function into the function list */
    AddFuncToOutputList(SpoCSV, NT_OUTPUT_ALERT, data);
    AddFuncToCleanExitList(CSVCleanExitFunc, data);
    AddFuncToRestartList(CSVRestartFunc, data);
}

void SpoCSV(Packet *p, char *msg, void *arg, Event *event)
{
    SpoCSVData *data = (SpoCSVData *)arg;
    AlertCSV(p, msg, data->file, data->args, data->numargs); 
    return;
}


/*
 * Function: ParseCSVArgs(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.  This function doesn't
 *          have to exist if it makes sense to parse the args in the init 
 *          function.
 *
 * Arguments: args => argument list
 *
 * Returns: void function
 *
 */
SpoCSVData *ParseCSVArgs(char *args)
{
    char **toks;
    int num_toks; 
    char *filename;
    SpoCSVData *data;
/*    SpoCSVConfig *config; */
    int num;

    data = (SpoCSVData *)malloc(sizeof(SpoCSVData));
#ifdef DEBUG
    printf("ParseCSVArgs: %s\n", args);
#endif


    toks = mSplit(args, " ", 2, &num_toks, 0);

    if(num_toks <= 1)
    {
        ErrorMessage("ERROR => You must supply at least TWO arguments for the CSV plugin...\n");
        FatalError("\t ... arguements of \"/path/to/output/file default\" as a minimum.\n");
    }
 
    filename = ProcessFileOption(toks[0]);
    data->file = OpenAlertFile(filename);
    free(filename);
#ifdef DEBUG
    printf("CSV Got filename\n");
#endif

    if(!strncasecmp("default", toks[1], 7))
    {
        data->csvargs = DEFAULT_CSV;
    }
    else
    {
        data->csvargs = toks[1]; 
    } 

#ifdef DEBUG
    printf("CSV Got Config ARGS\n");
#endif
  
    toks = mSplit(data->csvargs, ",", 128, &num_toks, 0);

    num = num_toks;
    data->args = toks;
    data->numargs = num;

    return data;
}

void CSVCleanExitFunc(int signal, void *arg)
{
    SpoCSVData *data = (SpoCSVData *)arg;
    /* close alert file */
#ifdef DEBUG
    printf("CSVCleanExitFunc\n");
#endif
    fclose(data->file);
    /* free memory from SpoCSVData */
    free(data);
}

void CSVRestartFunc(int signal, void *arg)
{
    SpoCSVData *data = (SpoCSVData *)arg;
    /* close alert file */
#ifdef DEBUG
    printf("CSVRestartFunc\n");
#endif
    fclose(data->file);
    /* free memory from SpoCSVData */
    free(data);
}



/*
 * Function: CSVAlert(Packet *, char *, void *, char *, const int )
 *
 * Purpose: Stub function for compatability
 *
 * Arguments:    p => ptr to packet data
 *             msg => message to send to alert facility
 *             arg => arguments to the alert facility
 *            args => CSV arguements 
 *         numargs => number of arguements
 * Returns: void function
 */
void CSVAlert(Packet * p, char *msg, void *arg, char **args, int numargs)
{
    AlertCSV(p, msg, alert, args, numargs);
    return;
}

/*
 *
 * Function: AlertCSV(Packet *, char *, FILE *, char *, numargs const int)
 *
 * Purpose: Write a user defined CSV message
 *
 * Arguments:     p => packet. (could be NULL)
 *              msg => the message to send
 *             file => file pointer to print data to
 *             args => CSV output arguements 
 *          numargs => number of arguements
 * Returns: void function
 *
 */
void AlertCSV(Packet * p, char *msg, FILE * file, char **args, int numargs)
{
    char timestamp[TIMEBUF_SIZE];
    int num; 
    char *type;
    char tcpFlags[9];

    if(p == NULL)
        return;

    bzero((char *) timestamp, TIMEBUF_SIZE);
    ts_print(p == NULL ? NULL : (struct timeval *) & p->pkth->ts, timestamp);

#ifdef DEBUG
        printf("Logging CSV Alert data\n"); 
#endif

    for (num = 0; num < numargs; num++)
    {
        type = args[num];

#ifdef DEBUG
        printf("CSV Got type %s %d\n", type, num); 
#endif
       if(!strncasecmp("timestamp", type, 9))
       {
            fwrite(timestamp, strlen(timestamp), 1, file);
       }
       else if(!strncasecmp("msg", type, 3))
       {
            fwrite(msg, strlen(msg),1,file);
       }
       else if(!strncasecmp("proto", type, 9))
       {
            switch (p->iph->ip_proto)
            {
                case IPPROTO_UDP:
                    fwrite("UDP", 3,1,file);
                    break;
                case IPPROTO_TCP:
                    fwrite("TCP", 3,1,file);
                    break;
                case IPPROTO_ICMP:
                    fwrite("ICMP", 4,1,file);
                    break;
             }
       }
       else if(!strncasecmp("ethsrc", type, 9))
       {
            if(p && p->eh)
            {
               fprintf(file, "%X:%X:%X:%X:%X:%X", p->eh->ether_src[0],
                  p->eh->ether_src[1], p->eh->ether_src[2], p->eh->ether_src[3],
                  p->eh->ether_src[4], p->eh->ether_src[5]);
            }
       } 
       else if(!strncasecmp("ethdst", type, 9))
       {
            if(p && p->eh)
            {
               fprintf(file, "%X:%X:%X:%X:%X:%X", p->eh->ether_dst[0],
                  p->eh->ether_dst[1], p->eh->ether_dst[2], p->eh->ether_dst[3],
                  p->eh->ether_dst[4], p->eh->ether_dst[5]);
            }
       }
       else if(!strncasecmp("ethtype", type, 9))
       {
            if(p && p->eh)
            {
                fprintf(file,"0x%X",ntohs(p->eh->ether_type));
            }
       }
       else if(!strncasecmp("udplenght", type, 3))
       {
            if(p->udph)
                fprintf(file,"%d",ntohs(p->udph->uh_len));
       }
       else if(!strncasecmp("ethlen", type, 9))
       {
            if(p && p->eh)
                fprintf(file,"0x%X",p->pkth->len);
       }
       else if(!strncasecmp("trheader", type, 9))
       {
            if(p && p->trh)
                PrintTrHeader(file, p);
       }
       else if(!strncasecmp("src", type, 8))
       {
            fputs(inet_ntoa(p->iph->ip_src), file);
       }
       else if(!strncasecmp("dst", type, 8))
       {
            fputs(inet_ntoa(p->iph->ip_dst), file); 
       }
       else if(!strncasecmp("srcport", type, 7))
       {
            switch(p->iph->ip_proto)
            {
                case IPPROTO_UDP:
                case IPPROTO_TCP:
                        fprintf(file, "%d", p->sp);
                        break;
            }    
        }
       else if(!strncasecmp("dstport", type, 7))
       {
            switch(p->iph->ip_proto)
            {
                case IPPROTO_UDP:
                case IPPROTO_TCP:
                        fprintf(file, "%d", p->dp);
                        break;
            }    
        }
       else if(!strncasecmp("icmptype",type,8))
       {
            if(p->icmph)
            {
                fprintf(file,"%d",p->icmph->type);
            }
       }
       else if(!strncasecmp("icmpcode",type,8))
       {
            if(p->icmph)
            {
                fprintf(file,"%d",p->icmph->code);
            }
       }
      else if(!strncasecmp("icmpid",type,6))
       {
            if(p->ext)
            {
                fprintf(file,"%d",ntohs(p->ext->id));
            }
       }
       else if(!strncasecmp("icmpseq",type,7))
       {
            if(p->ext)
                fprintf(file,"%d",ntohs(p->ext->seqno));
       }
       else if(!strncasecmp("ttl",type,3))
       {
            if(p->iph)
                fprintf(file,"%d",p->iph->ip_ttl);
       }
       else if(!strncasecmp("tos",type,3))
       {
            if(p->iph)
                fprintf(file,"%d",p->iph->ip_tos);
       }
       else if(!strncasecmp("id",type,3))
       {
            if(p->iph)
                fprintf(file,"%d",ntohs(p->iph->ip_id));
       }
       else if(!strncasecmp("iplen",type,5))
       {
            if(p->iph)
                fprintf(file,"%d",p->iph->ip_hlen << 2);
       }
       else if(!strncasecmp("dgmlen",type,6))
       {
            if(p->iph)
                fprintf(file,"%d",ntohs(p->iph->ip_len));
       }
       else if(!strncasecmp("tcpseq",type,6))
       {
            if(p->tcph)
                fprintf(file,"0x%lX",(u_long) ntohl(p->tcph->th_seq));
       }
       else if(!strncasecmp("tcpack",type,6))
       {
            if(p->tcph)
                fprintf(file,"0x%lX",(u_long) ntohl(p->tcph->th_ack));
       }
       else if(!strncasecmp("tcplen",type,6))
       {
            if(p->tcph)
                fprintf(file,"%d",p->tcph->th_off << 2);
       }
       else if(!strncasecmp("tcpwindow",type,9))
       {
            if(p->tcph)
                fprintf(file,"0x%X",ntohs(p->tcph->th_win));
       }
       else if(!strncasecmp("tcpflags",type,8))
       {
            if(p->tcph)
            {   
                CreateTCPFlagString(p, tcpFlags);
                fprintf(file,"%s", tcpFlags);
            }
       }

       if (num < numargs - 1) 
          fwrite(",",1,1,file);
    }
    fputc('\n', file);
   

    return;
}

