/*
** Copyright (C) 2000 Jed Pickel <jed@pickel.net> <jpickel@cert.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* Snort postgresql Output Plugin
 * by Jed Pickel <jed@pickel.net>, <jpickel@cert.org>
 *
 * spo_log_postgresql
 * 
 * Purpose:
 *
 * This plugin logs to a postgresql database
 *
 *    To get this working follow these steps.
 *    1) install postgresql (go to www.postgresql.org for source)
 *    2) add a user in postgresql that has permission to connect 
 *       to and create databases
 *    3) build snort (./configure; make; make install)
 *    4) add the appropriate line to your configuration file and
 *       start snort... (see Arguments section below for details)
 *    5) Try using psql to view results as a test. 
 *
 *       % psql snort
 *       .....
 *       snort=> SELECT event.signature, iphdr.sip, iphdr.dip, 
 *       snort=> event.timestamp WHERE event.id = iphdr.id 
 *       snort=> ORDER BY iphdr.sip;
 *
 * Arguments:
 * 
 * output log_postgresql: <name of database>, <parameter list>
 *
 *    The following excerpt is ripped from postgresql documentation
 *
 *    Each parameter setting is in the form keyword = value.
 *    (To write a null value or a value containing spaces, surround it
 *    with single quotes, eg, keyword = 'a value'. Single quotes within
 *    the value must be written as \'. Spaces around the equal sign are
 *    optional.) The currently recognized parameter keywords are:
 *       + host -- host to connect to. If a non-zero-length string is
 *         specified, TCP/IP communication is used. Without a host name,
 *         libpq will connect using a local Unix domain socket.
 *       + port -- port number to connect to at the server host, or
 *         socket filename extension for Unix-domain connections.
 *       + dbname -- database name.
 *       + user -- user name for authentication.
 *       + password -- password used if the backend demands password
 *         authentication.
 *       + options -- trace/debug options to send to backend.
 *       + tty -- file or tty for optional debug output from backend.
 *   
 * Effect:
 *
 *    Logs are written to a database for later analysis by other applications. 
 *
 * Environment Variables: (partially ripped from postgresql documentation)
 *
 *    The following environment variables can be used to select
 *    default connection parameter values if you don't want to have
 *    this info in your configuration file.
 *          
 *       * PGHOST sets the default server name. If a non-zero-length string is
 *         specified, TCP/IP communication is used. Without a host name, libpq
 *         will connect using a local Unix domain socket.
 *       * PGPORT sets the default port or local Unix domain socket file 
 *         extension for communicating with the Postgres backend.
 *       * PGDATABASE sets the default Postgres database name.
 *       * PGUSER sets the username used to connect to the database and for
 *         authentication.
 *       * PGPASSWORD sets the password used if the backend demands password
 *         authentication.
 *       * PGREALM sets the Kerberos realm to use with Postgres, if it is 
 *         different from the local realm. If PGREALM is set, Postgres 
 *         applications will attempt authentication with servers for this 
 *         realm and use separate ticket files to avoid conflicts with local 
 *         ticket files. This environment variable is only used if Kerberos 
 *         authentication is selected by the backend.
 *       * PGOPTIONS sets additional runtime options for the Postgres backend.
 *       * PGTTY sets the file or tty on which debugging messages from the 
 *         backend server are displayed.
 *         
 *    The following environment variables can be used to specify user-level 
 *    default behavior for every Postgres session:
 *             
 *       * PGDATESTYLE sets the default style of date/time representation.
 *       * PGTZ sets the default time zone.
 *
 * Comments:
 *
 *    Using this module could have an impact on performance if you 
 *    generate a large number of alerts in a very short time period.
 *    Once I get threads working there will be little performance  
 *    impact. Currently, there should be little performance impact 
 *    if you generate less than a few alerts per second.
 * 
 *    To see database structure go into psql and type "\d *" 
 *
 *    More documentation is available at http://www.incident.org/snortdb  
 *
 * Change Log:
 *
 *   2000-03-08: Added new table "sensor" and a new field to event table
 *               to represent the sensor
 *   2000-03-08: Added locking on inserts to eliminate concurrency problem 
 *   2000-03-08: Changed "type" and "code" in icmphdr to int2 instead of char
 *   2000-03-01: Added extra argument to RegisterOutputPlugin
 *   2000-02-28: First release
 * 
 * TODO: 
 *  
 *    - Figure out the best way to handle fragments 
 *    - More detailed logging of ICMP fields
 *    - Log the data portion of packets in a text field
 *    - Figure out the best way to log raw packets considering 
 *        the postgresql large object interface is not sufficient
 *        for this purpose */

#ifdef ENABLE_POSTGRESQL
    #include "spo_log_postgresql.h"
extern PV pv;

/*
 * Function: SetupLogPostgresql()
 *
 * Purpose: Registers the output plugin keyword and initialization 
 *          function into the output plugin list.  This is the function that
 *          gets called from InitOutputPlugins() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupLogPostgresql()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterOutputPlugin("log_postgresql", NT_OUTPUT_LOG, LogPostgresqlInit);

    /* check to see if postmaster is running */


#ifdef DEBUG
    printf("Output plugin: Log-Postgresql is setup...\n");
#endif
}

/*
 * Function: LogPostgesqlInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void LogPostgresqlInit(u_char *args)
{
#ifdef DEBUG
    printf("Output: Log-Postgresql Initialized\n");
#endif

    /* tell command line loggers to go away */
    pv.log_plugin_active = 1;

    /* parse the argument list from the rules file */
    ParsePostgresqlArgs(args);

    InitPostgresqlDB();

    /* make a connection to the database */
    conn = PQconnectdb(dbString);

    /*
     * check to see that the backend connection was successfully made
     */
    if (PQstatus(conn) == CONNECTION_BAD)
    {
        printf("Connection to database '%s' failed.\n", dbString);
        printf("%s\n", PQerrorMessage(conn));
        PQfinish(conn);
        exit(-1);
    }

    /* Set the ireprocessor function into the function list */
    AddFuncToOutputList(LogPostgresql, NT_OUTPUT_LOG);
}


/*
 * Function: ParsePostgresqlArgs(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.  This function doesn't
 *          have to exist if it makes sense to parse the args in the init 
 *          function.
 *
 * Arguments: args => argument list
 *
 * Returns: void function
 *
 */
void ParsePostgresqlArgs(char *args)
{
    char **toks;
    int numToks;

    if (args != NULL)
    {
        toks = mSplit(args,",",2,&numToks,'\\');
        dbString = (char *) calloc(strlen(args) + strlen("dbname=") + 2, sizeof(char));
        dsn = (char *) calloc(strlen(args) + 1, sizeof(char));
        if (numToks == 1)
        {
            snprintf(dbString, strlen(args) + strlen("dbname=") + 1, "dbname=%s", toks[0]);
        }
        else
        {
            snprintf(dbString, strlen(args) + strlen("dbname=") + 1, "dbname=%s%s", toks[0], toks[1]);
            dsn = toks[1];
        }
    }
    else
    {
        printf("ERROR: must supply arguments for log_postgresql plugin\n");
        exit(-1);
    }
}


/*
 * Function: PreprocFunction(Packet *)
 *
 * Purpose: Perform the preprocessor's intended function.  This can be
 *          simple (statistics collection) or complex (IP defragmentation)
 *          as you like.  Try not to destroy the performance of the whole
 *          system by trying to do too much....
 *
 * Arguments: p   => pointer to the current packet data struct 
 *            msg => pointer to the signature message
 *
 * Returns: void function
 *
 */
void LogPostgresql(Packet *p, char *msg)
{
    char *i0;
    char *i1;
    char *i2;
    /* char *i3;  Reserved for data field query */
    char sip[16];
    char dip[16];

    if (p == NULL)
    {
#ifdef DEBUG
        printf("NULL packet in postgresql alert!\n");
#endif
        return;
    }

    /* 
       Still need code to handle fragments! For now we will not log them.
       Fixing this is the first thing on the ToDo list 
    */
    if (!p->frag_flag)
    {
        i0 = (char *)malloc(MAX_QUERY_LENGTH);
        i1 = (char *)malloc(MAX_QUERY_LENGTH);
        i2 = (char *)malloc(MAX_QUERY_LENGTH);

        /* have to do this since inet_ntoa is f^@%&d up and writes to
           a static memory location */
        strncpy(sip, inet_ntoa(p->iph->ip_src), 16);
        strncpy(dip, inet_ntoa(p->iph->ip_dst), 16);

        if (p->iph->ip_proto == IPPROTO_ICMP)
        {
            /* THIS core dumps because of p->ext..... I still need to figure out why..  
              snprintf(i2, MAX_QUERY_LENGTH, "INSERT INTO icmphdr (id, type, code, icmp_id, seq) VALUES (max(event.id), '%i','%i','%i','%i');", p->icmph->type, p->icmph->code, p->ext->id, p->ext->seqno); 
             
            So for now we will go with a stripped down version */
            snprintf(i2, MAX_QUERY_LENGTH, "INSERT INTO icmphdr (id, type, code) VALUES (max(event.id), '%i','%i');", p->icmph->type, p->icmph->code);
        }
        else if (p->iph->ip_proto == IPPROTO_TCP)
        {
            snprintf(i2, MAX_QUERY_LENGTH, "INSERT INTO tcphdr (id, source, dest, flags, window, urg_ptr) VALUES (max(event.id), '%i','%i','%i','%i','%i');", ntohs(p->tcph->th_sport), ntohs(p->tcph->th_dport), p->tcph->th_flags, p->tcph->th_win, p->tcph->th_urp);
        }
        else if (p->iph->ip_proto == IPPROTO_UDP)
        {
            snprintf(i2, MAX_QUERY_LENGTH, "INSERT INTO udphdr (id, source, dest, len) VALUES (max(event.id), '%i','%i','%i');", ntohs(p->udph->uh_sport), ntohs(p->udph->uh_dport), p->udph->uh_len);
        }

        if (msg == NULL)
        {
            msg = "NULL MESSAGE";
        }

        snprintf(i0, MAX_QUERY_LENGTH, "INSERT INTO event (signature, sensor, timestamp) VALUES ('%s', '%i', now());", msg, sensor_id);
        snprintf(i1, MAX_QUERY_LENGTH, "INSERT INTO iphdr (id, protocol, sip, dip, tos, ttl, ip_id, frag_off, frag, tot_len) VALUES (max(event.id),'%i','%s','%s','%i','%i','%i','%i','%i','%i');",p->iph->ip_proto, sip, dip, p->iph->ip_tos, p->iph->ip_ttl, p->iph->ip_id, p->frag_offset, p->frag_flag, p->iph->ip_len);

        /* Reserved for data field insert 
        if(pv.data_flag)
        {
          if(pv.char_data_flag)
      {
            i3 = (char *)malloc(MAX_QUERY_LENGTH + p->dsize);
            snprintf(i3, MAX_QUERY_LENGTH + p->dsize, "INSERT INTO data (id, data) VALUES (max(event.id),'%s');", PrintCharData(fp, p->data, p->dsize));      
      }
        }   
        */

        /* Execute the qureies */
        CleanExec("BEGIN;",QUERY_FAILED,1,conn);
        CleanExec("LOCK event IN SHARE MODE;",QUERY_FAILED,1,conn);
        CleanExec(i0,QUERY_FAILED,1,conn);
        CleanExec(i1,QUERY_FAILED,1,conn);
        CleanExec(i2,QUERY_FAILED,1,conn);

        /* Reserved for data field insert
        if(pv.data_flag)
        {
          if(pv.char_data_flag)
      {
            CleanExec(i3,QUERY_FAILED,1,conn);
            free(i3);
      }
        }*/
        CleanExec("COMMIT;",QUERY_FAILED,1,conn);

        free(i0); free(i1); free(i2);
    }

    /* 
      Since the large object handling in postgresql sucks I am reluctant
      to add this functionality to store raw packets.... Perhaps I could
      insert one day worth of packets to a single large object instead of
      every packet in a single large object or something.... To be 
      determined based on how useful this plugin turns out to be...  
 
      If you uncomment this is will not work in its current state.
  
      lobjId = lo_creat(conn, INV_READ|INV_WRITE);
      if (lobjId == 0)
      {
         printf("spo_log_postgresql: Can't create object for packet\n");
      }
      else
      {
         od = lo_open(conn, lobjId, INV_WRITE);
     
         if (od >= 0)
         {
            if (lo_write(conn, od, p->pkt, p->pkth->len) >= 0)
            {
               printf("spo_log_postgresql: Can't write to object %d (od=%d)\n", lobjId, od);
               printf("spo_log_postgresql: p->pkth->len = %d\n", p->pkth->len);
            } 
            if (lo_close(conn, od) < 0)
            {
               printf("spo_log_postgresql: Can't close object %d (od=%d)\n", lobjId, od);
            } 
         }
         else
         {
            printf("spo_log_postgresql: Can't open object %d (rval = %d)\n", lobjId, od);
         } 
      }
    */
}


/****************************************************************************
 *
 * Function: InitPostgresqlDB()
 *
 * Purpose: Initialize the database if it does not exist
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void InitPostgresqlDB()
{
    PGconn * dbadmin;
    PGresult * result;
    char * dsn_str;
    char * select0;
    char * insert0;

    /* make a connection to the database */
    dsn_str = (char *) calloc(strlen(dsn) + strlen("dbname=template1 ") + 1, sizeof(char));
    snprintf(dsn_str, strlen(dsn) + strlen("dbname=template1 ") + 1, "dbname=template1 %s", dsn);
    dbadmin = PQconnectdb(dsn_str);
    free(dsn_str);

    /*
     * check to see that the backend connection was successfully made
     */
    if (PQstatus(dbadmin) == CONNECTION_BAD)
    {
        printf("Connection to database '%s' failed.\n", dsn_str);
        printf("%s\n", PQerrorMessage(dbadmin));
        PQfinish(dbadmin);
        exit(-1);
    }

    if (!CleanExec("CREATE DATABASE snort;","Database already exists\n",0,dbadmin))
    { 
        /* End connection to template1 */
        PQfinish(dbadmin);
        /* Connect to newly created database */
        dbadmin = PQconnectdb(dbString); 

        /* was connection successfull? */
        if (PQstatus(dbadmin) == CONNECTION_BAD)
        {
            printf("Connection to database '%s' failed.\n", dbString);
            printf("%s\n", PQerrorMessage(dbadmin));
            PQfinish(dbadmin);
            exit(-1);
        }

        CleanExec("CREATE TABLE event (id SERIAL, signature text not null, sensor int4 not null, timestamp datetime not null);",QUERY_FAILED,1,dbadmin);
        CleanExec("CREATE TABLE sensor (id SERIAL, hostname text, interface text, filter text);",QUERY_FAILED,1,dbadmin); 
        CleanExec("CREATE TABLE iphdr (id int4 not null, protocol int2, sip inet not null, dip inet not null, tos int2, ttl int2, ip_id int4, frag_off int4, frag int2, tot_len int4);",QUERY_FAILED,1,dbadmin);
        CleanExec("CREATE TABLE tcphdr (id int4 not null, source int4 not null, dest int4 not null, flags int4, window int4, urg_ptr int4);",QUERY_FAILED,1,dbadmin);
        CleanExec("CREATE TABLE udphdr (id int4 not null, source int4 not null, dest int4 not null, len int4);",QUERY_FAILED,1,dbadmin);
        CleanExec("CREATE TABLE icmphdr (id int4 not null, type int2 not null, code int2 not null, icmp_id int4, seq int4);",QUERY_FAILED,1,dbadmin);
        /* FUTURE FUNCTIONALITY ---- NOT FOR CURRENT VERSION
        CleanExec("CREATE TABLE data (id int4 not null, data text);",QUERY_FAILED,1,dbadmin);
        CleanExec("CREATE TABLE dns (start datetime not null, finish datetime, ip inet, fqdn text);",QUERY_FAILED,1,dbadmin);
        CleanExec("CREATE TABLE packet (id int4 not null, data text);",QUERY_FAILED,1,dbadmin);
        */

#ifdef DEBUG
        printf("Created database and tables\n");
#endif DEBUG
    }

    /* End connection to template1 or snort database */
    PQfinish(dbadmin);

    /* Connect to snort database */
    dbadmin = PQconnectdb(dbString); 

    /* was connection successfull? */
    if (PQstatus(dbadmin) == CONNECTION_BAD)
    {
        printf("Connection to database '%s' failed.\n", dbString);
        printf("%s\n", PQerrorMessage(dbadmin));
        PQfinish(dbadmin);
        exit(-1);
    }

    /* Get sensor id */
    select0 = (char *)malloc(MAX_QUERY_LENGTH);
    snprintf(select0, MAX_QUERY_LENGTH, "SELECT id FROM sensor WHERE hostname = '%s' and interface = '%s' and filter ='%s';", getenv("HOSTNAME"), pv.interface, pv.pcap_cmd);
    result = PQexec(dbadmin,select0);
    if (PQntuples(result) > 0)
    {
        sensor_id = atoi(PQgetvalue(result,0,0));
    }
    else
    {
        /* set the sensor id */
        insert0 = (char *)malloc(MAX_QUERY_LENGTH);
        snprintf(insert0, MAX_QUERY_LENGTH, "INSERT INTO sensor (hostname, interface, filter) VALUES ('%s','%s','%s');", getenv("HOSTNAME"), pv.interface, pv.pcap_cmd); 
        CleanExec(insert0,QUERY_FAILED,1,dbadmin);
        free(insert0);

        /* get the sensor id */
        result = PQexec(dbadmin,select0);
        if (PQresultStatus(result) == PGRES_TUPLES_OK)
        {
            sensor_id = atoi(PQgetvalue(result,0,0));
        }
        else
        {
            printf("%s", PQresultErrorMessage(result));
            PQfinish(dbadmin);
            exit(-1);
        }   
    } 

    free(select0);

    PQfinish(dbadmin);
}


/****************************************************************************
 *
 * Function: CleanExec()
 *
 * Purpose: Executes a postgres query and ensures it works
 *          Program will optionally die if the query does not work
 *
 * Arguments: char * query         (The query string)
 *            char * error_message (What to print out if it fails) 
 *            int fail             (Pass in anything but 0 to die on fail)  
 *            PGconn * db          (the connection to execute query on)
 *
 * Returns: 0 if successful, 1 if fail
 *
 ***************************************************************************/
int CleanExec(char * query, char * error_message, int fail, PGconn * db)
{
    PGresult * result;
#ifdef DEBUG
    printf("Executing: %s\n", query);
#endif DEBUG
    result = PQexec(db,query);
    if (PQresultStatus(result) != PGRES_COMMAND_OK)
    {
        if (fail)
        {
            printf("Query...: %s\n", query);
            printf("Result..: %s\n", PQresStatus(PQresultStatus(result)));
            printf("%s", PQresultErrorMessage(result));
            PQfinish(db);
            exit(-1);
        }
        return 1;
    }
    return 0;
}

#endif
