%{
/*	$Id: parse.y,v 1.4 2006/06/06 13:13:22 mbalmer Exp $	*/

/*
 * Copyright (c) 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/queue.h>

#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <regex.h>

#include "smtp-vilter.h"
#include "pathnames.h"
#include "vilter-regex.h"

#if defined(__linux__)
#include "strtonum.h"
#endif

extern FILE	*regexin;
extern int	 regexlineno;
extern char	*regextext;

extern int	verbose;

extern struct pat_head hdr_pats;
extern struct pat_head bdy_pats;

static struct pattern *p;
static int	cflags;

static int	 regexerrcnt;
static char	*regexcfgfile;

int	regexerror(const char *, ...);
int	regexparse(void);
int	regexlex(void);

static struct pattern *
compile_pattern(char *value, int cflags)
{
	struct pattern *p;
	int error;
	char errstr[256];

	p = malloc(sizeof(struct pattern));
	if (p == NULL)
		errx(1, "regex: out of memory");
	bzero(p, sizeof(struct pattern));
	p->pat = strdup(value);
	if (p->pat == NULL)
		errx(1, "regex: out of memory");
	if ((error = regcomp(&p->preg, p->pat, cflags)) != 0) {
		regerror(error, &p->preg, errstr, sizeof(errstr));
		errx(1, "regex:  regexp %s: %s", p->pat, errstr);
	}
	return p;
}

%}

%union {
	long number;
	char *string;
}

%token	CASE_SENSITIVE UNWANTED_HEADER UNWANTED_BODY NOTIFICATION
%token	TRUE FALSE
%token	<string>	NUMBER
%token	<string>	TEXT

%%
statement	: /* empty */
		| statement '\n'
		| statement case_sensitive '\n'
		| statement unwanted_header '\n'
		| statement unwanted_body '\n'
		;

case_sensitive	: CASE_SENSITIVE '=' TRUE		{
			cflags &= ~REG_ICASE;
		}
		| CASE_SENSITIVE '=' FALSE		{
			cflags |= REG_ICASE;
		}
		;

unwanted_header	: UNWANTED_HEADER '=' TEXT		{
			p = compile_pattern($3, cflags);
			SLIST_INSERT_HEAD(&hdr_pats, p, pats);
			if (verbose)
				warnx("regex: adding unwanted header pattern "
				    "%s", $3);

			free($3);
		}
		;

unwanted_body	: UNWANTED_BODY '=' TEXT		{
			p = compile_pattern($3, cflags);
			SLIST_INSERT_HEAD(&bdy_pats, p, pats);
			if (verbose)
				warnx("regex: adding unwanted body pattern %s",
				    $3);

			free($3);
		}
		;

%%

int
vilter_init(char *cfgfile)
{
	cflags = REG_EXTENDED | REG_ICASE | REG_NOSUB;

	if (verbose)
		warnx("regex: vilter_init()");

	SLIST_INIT(&hdr_pats);
	SLIST_INIT(&bdy_pats);

	if (cfgfile == NULL)
		cfgfile = REGEX_CONF;

	regexlineno = 1;
	regexcfgfile = cfgfile;
	regexerrcnt = 0;
	if ((regexin = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("regex: using configuration from file %s",
			    cfgfile);

		while (!feof(regexin))
			regexparse();
		fclose(regexin);

		if (regexerrcnt)
			errx(1, "configuration file contains errors, "
			    "terminating");
	} else if (verbose)
		warnx("regex: configuration file %s for regex backend not "
		    "found, using default values", cfgfile);

	if (verbose)
		warnx("regex: vilter_init() return");
	return 0;
}

int
regexerror(const char *fmt, ...)
{
	va_list		 ap;
	char		*nfmt;

	++regexerrcnt;
	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s, line %d: %s near '%s'", regexcfgfile,
	    regexlineno, fmt, regextext) == -1)
		errx(1, "regex: asprintf failed");
	va_end(ap);
	fprintf(stderr, "%s\n", nfmt);
	free(nfmt);
	return (0);
}
