/*	$Id: smtp-vilter.c,v 1.131 2006/02/06 10:33:49 mbalmer Exp $	*/

/*
 * Copyright (c) 2003-2006 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/queue.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <dlfcn.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#ifdef ENABLE_LDAP
#include <ldap.h>
#endif
#include <math.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "libmilter/mfapi.h"


#include "pathnames.h"
#include "smtp-vilter.h"

#include "imsg.h"

#ifdef __linux__
#include <sys/param.h>
#include "strlfunc.h"
#include "strtonum.h"
#else
#include <sys/syslimits.h>
#endif

#ifdef __FreeBSD__
#include "strtonum.h"
#endif

#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256
#define POLL_MAX	1

#define LOG_UNSET	-1

char	*port;
int	 virus_strategy;
int	 spam_strategy;
int	 unwanted_strategy;
int	 error_strategy;

int	 virus_reaction;
int	 spam_reaction;
int	 unwanted_reaction;
int	 clean_reaction;
long	 virus_duration;
long	 spam_duration;
long	 unwanted_duration;
long	 clean_duration;

/* there is only one reaction type: add to/clr from table */

char	*virus_table;
char	*spam_table;
char	*unwanted_table;
char	*clean_table;

int	 markall;
int	 verbose;
char	*pidfile;
char	*cfgfile;
char	*tmpdir;
int	 tmpfiles_gr;
int	 tmpfiles_setgrp;
char	*backend;
char	*recipient_notification;
char	*notify_only;
int	 log_facility;
char	*logfile;
FILE	*logfp;
char	*user;
char	*group;
char	*chrootdir;
int	 rlimit_nofile;
int	 nbe;
char	*spamprefix;
int	 logvirus, logspam, logunwanted;
int	 keep_tmpfiles;

struct imsgbuf	*ibuf_e;

extern char *__progname;

#define DEFLT_LDAPHOST	"localhost"
#define DEFLT_LDAPPORT	389

#ifdef ENABLE_LDAP
LDAP	*ld;
#endif
char	*ldaphost;
int	 ldapport;
char	*ldapurl;
char	*searchbase;
char	*binddn;
char	*bindpasswd;
int	 ldap_use_tls;

volatile sig_atomic_t quit = 0;

struct backend *be;
int	 dev = -1;

struct privdata {
	struct connection	*conn;
	struct message		*msg;
	struct be_data		*backend_data;
};

int dispatch_imsg(struct imsgbuf *, int);

extern void smtp_vilter_init(void);
extern int e_main(uid_t, gid_t, int[2]);

static void
sighdlr(int signum)
{
	switch (signum) {
	case SIGINT:
	case SIGTERM:
	case SIGCHLD:
		quit = 1;
		break;
	}
}

#if __OpenBSD__
__dead static void
#else
static void
#endif
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-V] [-m] [-k] [-p port] [-d dir] "
	    "[-T tmpfile-opt] [-e logile] [-b backend] "
	    "[-n recipient-notification] [-o notify-only] [-C configfile] "
	    "[-P pidfile] [-t chroot-dir] [-u user] [-g group] [-f maxfiles] "
	    "[-a spam-subject-prefix]", __progname);
#ifdef ENABLE_LDAP
	fprintf(stderr, " [-D binddn] [-h ldaphost] [-L ldapport] "
	    "[-U ldapurl] [-w bindpasswd] [-B searchbase]");
#endif
	fprintf(stderr, "\n");
	exit(1);
}

void
decode_backend(char *s)
{
	char	*p;
	int	 n;
	char	*q, *r;

	if ((q = strdup(s)) == NULL)
		err(1, "memory allocation error");

	nbe = 0;
	r = q;
	do {
		p = strsep(&r, ",");
		if (*p != '\0')
			++nbe;
	} while (r != NULL);

	free(q);

	if ((be = (struct backend *)malloc(nbe * sizeof(struct backend))) == NULL)
		err(1, "error allocating memory for backends");
	bzero(be, nbe * sizeof(struct backend));

	n = 0;
	do {
		p = strsep(&s, ",");
		if (*p != '\0') {
			if ((be[n].name = strdup(p)) == NULL)
				err(1, "memory allocation error");
			++n;
		}
	} while (s != NULL);
}

#ifdef ENABLE_LDAP
static int
get_rebind_credentials(LDAP *ld, char **dnp, char **pwp, int *authmethodp, int freeit)
{
	warnx("get_rebind_credentials() called");
	return LDAP_SUCCESS;
}
#endif

int
main(int argc, char *argv[])
{
	int		 ch;
	struct stat	 statbuf;
	char		 libname[MAX_FILE];
	struct passwd	*passwd;
	struct group	*grp;
	uid_t		 uid;
	gid_t		 gid;
	struct rlimit	 rl;
	struct pollfd	 pfd[1];
	int		 n;
	pid_t		 child_pid, pid;
#ifdef ENABLE_LDAP
	int		 version;
#endif
	int		 xmode;
	const char	*errstr;
	long long	 rlimit_max;
	int		 pipe_m2e[2];
	int		 nfds;
	int		 flags;

	/* Set initial values */

	port = NULL;
	tmpdir = NULL;
	tmpfiles_gr = 0;
	tmpfiles_setgrp = 0;
	pidfile = NULL;
	user = NULL;
	group = NULL;
	chrootdir = NULL;
	cfgfile = _PATH_CFGFILE;
	backend = NULL;
	virus_strategy = -1;
	spam_strategy = -1;
	unwanted_strategy = -1;
	error_strategy = -1;

	virus_reaction = REACTION_UNSET;
	spam_reaction = REACTION_UNSET;
	unwanted_reaction = REACTION_UNSET;
	clean_reaction = REACTION_UNSET;

	virus_duration = spam_duration = unwanted_duration = clean_duration = 0L;
	virus_table = spam_table = unwanted_table = clean_table = NULL;

	log_facility = LOG_UNSET;
	logfile = NULL;
	recipient_notification = NULL;
	notify_only = NULL;
	markall = 0;
	rlimit_nofile = -1;
	spamprefix = NULL;
	logvirus = logspam = logunwanted = 0;
	keep_tmpfiles = 0;
#ifdef ENABLE_LDAP
	ldaphost = NULL;
	ldapport = -1;
	ldapurl = NULL;
	searchbase = NULL;
	binddn = NULL;
	bindpasswd = NULL;
	ldap_use_tls = -1;
#endif
	xmode = 0;

	/* Process the commandline */

#ifdef ENABLE_LDAP	
	while ((ch = getopt(argc, argv, "vmp:d:T:b:e:n:o:C:P:t:u:U:g:Vf:ka:xD:h:L:w:B:?")) != -1) {
#else
	while ((ch = getopt(argc, argv, "vmp:d:T:b:e:n:o:C:P:t:u:g:Vf:ka:x?")) != -1) {
#endif
		switch (ch) {
		case 'v':
			++verbose;
			break;
		case 'm':
			markall = 1;
			break;
		case 'p':
			port = optarg;
			break;
		case 'd':
			tmpdir = optarg;
			break;
		case 'T':
			if (!strcmp(optarg, "g+r+"))
				tmpfiles_gr = 1;
			else if (!strcmp(optarg, "setgrp"))
				tmpfiles_setgrp = 1;
			else
				errx(1, "unknown temp file option %s", optarg);
			break;
		case 'b':
			backend = optarg;
			decode_backend(backend);
			break;
		case 'e':
			logfile = optarg;
			break;
		case 'n':
			recipient_notification = optarg;
			break;
		case 'o':
			notify_only = optarg;
			break;
		case 'C':
			cfgfile = optarg;
			break;
		case 'P':
			pidfile = optarg;
			break;
		case 't':
			chrootdir = optarg;
			break;
		case 'u':
			user = optarg;
			break;
#ifdef ENABLE_LDAP
		case 'U':
			ldapurl = optarg;
			break;
#endif
		case 'g':
			group = optarg;
			break;
		case 'V':
			printf("%s %s\n", __progname, VERSION);
#ifdef ENABLE_LDAP
			printf("LDAP support enabled\n");
#endif
#ifndef __OpenBSD__
#ifdef ENABLE_PF
			printf("PF support enabled\n");
#endif
#endif
			exit(0);
			/* NOTREACHED */
		case 'f':
			if (getuid()) {
				getrlimit(RLIMIT_NOFILE, &rl);
				rlimit_max = rl.rlim_max;
			} else
				rlimit_max = RLIM_INFINITY;
			rlimit_nofile = (int)strtonum(optarg, 1LL, rlimit_max, &errstr);
			if (errstr)
				errx(1, "number of files is %s: %s", errstr, optarg);
			break; 
		case 'a':
			spamprefix = optarg;
			break;
		case 'k':
			keep_tmpfiles = 1;
			break;
#ifdef ENABLE_LDAP
		case 'D':
			binddn = optarg;
			break;
		case 'h':
			ldaphost = optarg;
			break;
		case 'L':
			ldapport = strtonum(optarg, 1, 65535, &errstr);
			if (errstr)
				errx(1, "ldap port number is %s: %s", errstr, optarg);
			break;
		case 'w':
			bindpasswd = optarg;
			break;
		case 'B':
			searchbase = optarg;
			break;
#endif
		case 'x':
			xmode = 1;
			break;
		default:
			usage();
		}
	}

	/*
	argc -= optind;
	argv += optind;
	*/

	/* Read config file */

	smtp_vilter_init();

	if (!nbe)
		errx(1, "no backends defined");

	if (xmode) {
		printf("backends: ");
		for (n = 0; n < nbe; n++)
			printf("%s ", be[n].name);
		printf("\n");
		printf("user: %s\n", user != NULL ? user : "not set");
		printf("group: %s\n", group != NULL ? group : "not set");
		printf("chroot: %s\n", chrootdir != NULL ? chrootdir : "not set");
		printf("recipient-notification: %s\n", recipient_notification != NULL ? recipient_notification : "not set");
		printf("spam-subject-prefix: %s\n", spamprefix != NULL ? spamprefix : "not set");
		printf("port: %s\n", port != NULL ? port : "not set");
		printf("tmpdir: %s\n", tmpdir != NULL ? tmpdir : "not set");
		printf("pidfile: %s\n", pidfile != NULL ? pidfile : "not set");
		printf("logfile: %s\n", logfile != NULL ? logfile : "not set");
#ifdef ENABLE_LDAP
		printf("ldapurl: %s\n", ldapurl != NULL ? ldapurl : "not set");
		printf("ldaphost: %s\n", ldaphost != NULL ? ldaphost : "not set");
		printf("searchbase: %s\n", searchbase != NULL ? searchbase : "not set");
		printf("binddn: %s\n", binddn != NULL ? binddn : "not set");
		printf("bindpasswd: %s\n", bindpasswd != NULL ? bindpasswd : "not set");
#endif
		printf("virus reaction: ");
		switch (virus_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", virus_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", virus_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		printf("spam reaction: ");
		switch (spam_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", spam_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", spam_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		printf("unwanted-content reaction: ");
		switch (unwanted_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", unwanted_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", unwanted_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		printf("clean reaction: ");
		switch (clean_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", clean_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", clean_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		return 0;
	}

	/* Set default values if some variables are not set */

	if (port == NULL)
		port = _PATH_PORT;
	if (tmpdir == NULL)
		tmpdir = _PATH_TMPDIR;
	if (pidfile == NULL)
		pidfile = _PATH_PIDFILE;
	if (backend == NULL)
		backend = "clamd";
	if (virus_strategy == -1)
		virus_strategy = STRATEGY_DISCARD;
	if (spam_strategy == -1)
		spam_strategy = STRATEGY_MARK;
	if (unwanted_strategy == -1)
		unwanted_strategy = STRATEGY_MARK;
	if (error_strategy == -1)
		error_strategy = STRATEGY_TEMPFAIL;
	if (log_facility == LOG_UNSET)
		log_facility = LOG_MAIL;

#ifdef ENABLE_LDAP
	if (ldaphost == NULL)
		ldaphost = DEFLT_LDAPHOST;
	if (ldapport == -1)
		ldapport = DEFLT_LDAPPORT;
	if (ldap_use_tls == -1)
		ldap_use_tls = 0;

	if (ldapurl == NULL) {
		if ((ld = ldap_init(ldaphost, ldapport)) == NULL)
			errx(1, "No directory server at %s, %m", ldaphost);
		else if (verbose)
			warnx("using LDAP server %s:%d", ldaphost, ldapport);
	} else {
		if (ldap_initialize(&ld, ldapurl) != LDAP_SUCCESS)
			errx(1, "No directory server at %s, %m", ldapurl);
		else if (verbose)
			warnx("using LDAP server %s", ldapurl);
	}
	version = LDAP_VERSION3;
	if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version) != LDAP_OPT_SUCCESS)
		errx(1, "Failed to set LDAP version 3 protocol");

	if (ldap_use_tls)
		if (ldap_start_tls_s(ld, NULL, NULL) != LDAP_SUCCESS)
			errx(1, "Failed to start TLS for LDAP");

	/*
	 * Setup LDAP rebind, eventually continue with system defaults if
	 * the initial bind fails?
 	 */

	if (ldap_simple_bind_s(ld, binddn, bindpasswd) != LDAP_SUCCESS)
		errx(1, "Failed to bind to directory server as '%s'", binddn);

	ldap_set_rebind_proc(ld, get_rebind_credentials, NULL);

	if (verbose)
		warnx("Directory server at %s server is up, %susing TLS", ldaphost,
			ldap_use_tls ? "" : "not ");
#endif
	openlog(__progname, verbose ? LOG_CONS | LOG_NDELAY | LOG_PID | LOG_PERROR : LOG_CONS | LOG_NDELAY | LOG_PID, log_facility);

	/* Load the backends */

	for (n = 0; n < nbe; n++) {
		snprintf(libname, sizeof(libname), "libvilter-%s.so.%d", be[n].name, MAJOR);

		if (verbose)
			warnx("loading backend %s from file %s", be[n].name, libname);
			
		if ((be[n].dlhandle = dlopen(libname, RTLD_LAZY)) == NULL)
			err(1, "error loading backend %s (%s), %s", be[n].name, libname, dlerror());
		
		be[n].be_init = dlsym(be[n].dlhandle, "vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "vilter_name")) == NULL)
			errx(1, "no function vilter_name in backend %s", be[n].name);

		be[n].be_new = dlsym(be[n].dlhandle, "vilter_new");	/* Optional */
		be[n].be_end = dlsym(be[n].dlhandle, "vilter_end");	/* Optional */
		
		if ((be[n].be_type = dlsym(be[n].dlhandle, "vilter_type")) == NULL)
			errx(1, "no function vilter_type in backend %s", be[n].name);

		be[n].type = be[n].be_type();

		if (be[n].type & BE_SCAN_VIRUS)	{
			if ((be[n].be_scan.be_scan_virus = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s", be[n].name);
			if (be[n].type & (BE_SCAN_SPAM | BE_SCAN_UNWANTED))
				errx(1, "illegal backend type combination %s", be[n].name);
		} else if (be[n].type & BE_SCAN_SPAM) {
			if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s", be[n].name);
			if (be[n].type & BE_SCAN_UNWANTED)
				errx(1, "illegal backend type combination %s", be[n].name);
		} else if (be[n].type & BE_SCAN_UNWANTED) {
			if ((be[n].be_scan.be_scan_unwanted = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s", be[n].name);
		} else
			errx(1, "backend %s has an unknown vilter type", be[n].name);

		be[n].be_exit = dlsym(be[n].dlhandle, "vilter_exit");
		if (be[n].be_init != NULL) {
			if (be[n].be_init(be[n].config_file))
				errx(1, "error initializing backend %s", be[n].name);
		}
	}

	uid = getuid();
	gid = getgid();

	if (user != NULL) {
		if ((passwd = getpwnam(user)) != NULL) {
			uid = passwd->pw_uid;
			gid = passwd->pw_gid;
		} else
			err(1, "no such user '%s'", user);
		
		if (group != NULL) {
			if ((grp = getgrnam(group)) != NULL)
				gid = grp->gr_gid;
			else
				err(1, "no such group '%s'", group);
		}
	}

	if (!stat(pidfile, &statbuf))
		errx(1, "pid file %s exists, another copy running?", pidfile);

	if (verbose == 0 &&  daemon(0, 0))
		err(1, "can't run as daemon");

	/* Setup imsg stuff, this is a one-way com only */

	if (socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, pipe_m2e) == -1)
		errx(1, "imsg setup failed");
	if ((flags = fcntl(pipe_m2e[0], F_GETFL, 0)) == -1)
		errx(1, "fcntl failed");
	flags |= O_NONBLOCK;
	if (fcntl(pipe_m2e[0], F_SETFL, flags) == -1)
		errx(1, "fcntl can't set flags");
	if ((flags = fcntl(pipe_m2e[1], F_GETFL, 0)) == -1)
		errx(1, "fcntl failed");
	flags |= O_NONBLOCK;
	if (fcntl(pipe_m2e[1], F_SETFL, flags) == -1)
		errx(1, "fcntl can't set flags");

	/*
	 * Fork into two processes, one to run privileged, one to do the
	 * work.
	 */

	child_pid = e_main(uid, gid, pipe_m2e);

	/* We are the privileged process */

#ifndef __linux
	setproctitle("parent");
#endif

	signal(SIGCHLD, sighdlr);
	signal(SIGINT, sighdlr);
	signal(SIGTERM, sighdlr);

	close(pipe_m2e[1]);

	if ((ibuf_e = malloc(sizeof(struct imsgbuf))) == NULL)
		errx(1, "memory error");

	imsg_init(ibuf_e, pipe_m2e[0]);

	while (!quit) {
		pfd[0].fd = ibuf_e->fd;
		pfd[0].events = POLLIN;
		if ((nfds = poll(pfd, 1, 1000)) == -1) {
			if (errno != EINTR) {
				syslog(LOG_WARNING, "main: poll error");
				/* quit = 1; */
			}
		} else if (nfds > 0 && pfd[0].revents & POLLIN) {
			/* nfds --; */
			if (dispatch_imsg(ibuf_e, 0) == -1)
				quit = 1;
		}

#ifdef ENABLE_PF
		if (pftable_timeout())
			syslog(LOG_WARNING, "can not timeout pf tables");
#endif
	}

	kill(child_pid, SIGTERM);

#ifdef ENABLE_LDAP
	ldap_unbind(ld);
#endif	

	do {
		if ((pid = wait(NULL)) == -1 &&
		    errno != EINTR && errno != ECHILD)
			errx(1, "wait");
	} while (pid != child_pid || (pid == -1 && errno == EINTR));

	msgbuf_clear(&ibuf_e->w);
	free(ibuf_e);

	if (!verbose) {
		if (unlink(pidfile))
			syslog(LOG_ERR, "can not unlink pidfile %s", pidfile);
		closelog();
	}
	return 0;
}

int
dispatch_imsg(struct imsgbuf *ibuf, int idx)
{
	struct imsg	imsg;
	int		n;
	int		rv;

	if ((n = imsg_read(ibuf)) == -1)
		return -1;

	if (n == 0) { /* connection closed */
		syslog(LOG_WARNING, "dispatch_imsg in main: pipe closed");
		return -1;
	}

	rv = 0;
	for (;;) {
		if ((n = imsg_get(ibuf, &imsg)) == -1)
			return -1;

		if (n == 0)
			break;

		switch (imsg.hdr.type) {
#ifdef ENABLE_PF 
		case IMSG_PFTABLE_ADD:
			if (imsg.hdr.len != IMSG_HEADER_SIZE +
			    sizeof(struct pftable_msg))
				syslog(LOG_WARNING, "wrong imsg size");
			else if (pftable_addr_add(imsg.data) != 0) {
				rv = 1;
			}
			break;
		case IMSG_PFTABLE_DEL:
			if (imsg.hdr.len != IMSG_HEADER_SIZE +
			    sizeof(struct pftable_msg))
				syslog(LOG_WARNING, "wrong imsg size");
			else if (pftable_addr_del(imsg.data) != 0) {
				rv = 1;
			}
			break;
		break;
#endif
		default:
			break;
		}
		imsg_free(&imsg);
		if (rv != 0)
			return rv;
	}
	return 0;
}
