/*
 * Copyright (c) 2003, 2004 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <err.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <err.h>
#include <dlfcn.h>
#include <ctype.h>
#include <syslog.h>
#include <math.h>
#include <time.h>

#ifdef WITH_LDAP
#ifdef LINUX
#include "queue.h"
#else
#include <sys/queue.h>
#endif
#include <ldap.h>
#endif

#include "smtp-vilter.h"

#ifdef LINUX
#include <sys/param.h>
#include "strlfunc.h"
#else
#include <sys/syslimits.h>
#endif

#include "libmilter/mfapi.h"

#define PIDFILE		"/var/run/smtp-vilter.pid"
#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256
#define	CFGFILE		"/etc/smtp-vilter/smtp-vilter.conf"

enum strategy_tok {
	STRATEGY_UNSET	= -1,
	STRATEGY_DISCARD,
	STRATEGY_MARK,
	STRATEGY_NOTIFY_RECIPIENT,
	STRATEGY_TEMPFAIL,
	STRATEGY_REJECT
};

#define LOG_UNSET					-1

char *port;
int virus_strategy;
int spam_strategy;
int unwanted_strategy;
int error_strategy;
int markall;
int verbose;
char *pidfile;
char *cfgfile;
char *tmpdir;
int tmpfiles_gr;
int tmpfiles_setgrp;
char *backend;
char *recipient_notification;
char *notify_only;
int log_facility;
char *logfile;
FILE *logfp;
char *user;
char *group;
char *chrootdir;
int rlimit_nofile;
int nbe;
char *spamprefix;
int logvirus, logspam, logunwanted;
int keep_tmpfiles;

extern char *__progname;

#ifdef WITH_LDAP
#define LDAPHOST	"localhost"
#define LDAPPORT	389

LDAP *ld;
char *ldaphost;
int ldapport;
char *searchbase;
char *binddn;
char *bindpasswd;

SLIST_HEAD(dlistheadm, domain) dhead;


struct recipient {
	char *recipient;
	char *mailLocalAddress;	/* canonicalized recipient */
	int msgid;
	int virus_strategy;
	int spam_strategy;
	char *spamprefix;
	struct domain *domain;
	SLIST_ENTRY(recipient) entries;
};

struct domain {
	char *mailLocalDomain;
	int msgid;
	int virus_strategy;
	int spam_strategy;
	char *spamprefix;
	SLIST_ENTRY(domain) entries;
};

#endif

struct backend {
	char *name;
	char *config_file;
	void *dlhandle;
	int (*be_init)(char *cfgfile);	/* One time initialization */
	char *(*be_name)(void);
	int (*be_type)(void);
	int (*be_new)(struct be_data *priv, char *from, FILE *tmpfp);	/* Per thread initializer */
	union {
		int (*be_scan_virus)(struct be_data *priv, char *path, char *chroot, char *virus, int viruslen);
		int (*be_scan_spam)(struct be_data *priv, char *path, char *chroot, double *score, double *threshold);
		int (*be_scan_unwanted)(struct be_data *priv, char *path, char *chroot, char *reason, int rlen);
	} be_scan;
	int (*be_end)(struct be_data *priv);	/* Per thread finalizer */
	void (*be_exit)(void);
};

struct backend *be;

struct connection {
	char *hostname;
	_SOCK_ADDR *hostaddr;
	
	/* The following are for statistics, not used at the moment */
	
	int num_msgs;
	int num_virus;
	int num_spam;
	int num_unwanted;
};

struct message {
	char tfn[MAX_FILE];	/* Temporary file name */
	char *msgid;		/* Message ID */
	char *envfrom;		/* Envelope sender */
	char *envto;		/* Envelope recipient */
#ifdef WITH_LDAP
	SLIST_HEAD(rlisthead, recipient) rhead;	/* List of recipients */
	SLIST_HEAD(dlisthead, domain) dhead;	/* List of recp. domains */
#endif
	char *hdrfrom;		/* From: address in message header */
	char *hdrto;		/* To: address in message header */
	char *subject;		/* The message subject */
	int  notify;		/* if != 0, notify recipient */
	FILE *fp;
};


struct privdata {
	struct connection *conn;
	struct message *msg;

	struct syslog_data sdata;
	struct be_data *backend_data;
};

enum cmd_tok {
	CMD_BACKEND = 0,
	CMD_VIRUS_STRATEGY,
	CMD_ERROR_STRATEGY,
	CMD_SPAM_STRATEGY,
	CMD_UNWANTED_STRATEGY,
	CMD_PORT,
	CMD_TMPDIR,
	CMD_PIDFILE,
	CMD_LOG_FACILITY,
	CMD_RECIPIENT_NOTIFICATION,
	CMD_NOTIFY_ONLY,
	CMD_OPTION,
	CMD_USER,
	CMD_GROUP,
	CMD_CHROOT,
	CMD_TMPFILES,
	CMD_MAXFILES,
	CMD_SPAM_SUBJECT_PREFIX,
	CMD_LOGFILE,
	CMD_CFGFILE,
#ifdef WITH_LDAP
	CMD_LDAPHOST,
	CMD_LDAPPORT,
	CMD_BINDDN,
	CMD_BINDPASSWD,
	CMD_SEARCHBASE
#endif
};
	
static struct token cmd_tab[] = {
	{ "backend",	CMD_BACKEND },
	{ "virus-strategy",	CMD_VIRUS_STRATEGY },
	{ "error-strategy",	CMD_ERROR_STRATEGY },
	{ "spam-strategy",	CMD_SPAM_STRATEGY },
	{ "unwanted-strategy", CMD_UNWANTED_STRATEGY },
	{ "port",	CMD_PORT },
	{ "tmpdir",	CMD_TMPDIR },
	{ "pidfile",	CMD_PIDFILE },
	{ "log-facility",	CMD_LOG_FACILITY },
	{ "recipient-notification",	CMD_RECIPIENT_NOTIFICATION },
	{ "notify-only",	CMD_NOTIFY_ONLY },
	{ "pidfile",	CMD_PIDFILE },
	{ "option",	CMD_OPTION },
	{ "user",	CMD_USER },
	{ "group",	CMD_GROUP },
	{ "chroot",	CMD_CHROOT },
	{ "tmpfiles",	CMD_TMPFILES },
	{ "maxfiles",	CMD_MAXFILES },
	{ "spam-subject-prefix",	CMD_SPAM_SUBJECT_PREFIX },
	{ "logfile",	CMD_LOGFILE },
	{ "config-file",	CMD_CFGFILE },
#ifdef WITH_LDAP
	{ "ldaphost",	CMD_LDAPHOST },
	{ "ldapport",	CMD_LDAPPORT },
	{ "binddn",	CMD_BINDDN },
	{ "bindpasswd",	CMD_BINDPASSWD },
	{ "searchbase",	CMD_SEARCHBASE },
#endif
	{ NULL, 0 }
};

static struct token lfacilities_tab[] = {
	{ "daemon",	LOG_DAEMON, },
	{ "mail",	LOG_MAIL },
	{ "user",	LOG_USER },
	{ "local0",	LOG_LOCAL0 },
	{ "local1",	LOG_LOCAL1 },
	{ "local2",	LOG_LOCAL2 },
	{ "local3",	LOG_LOCAL3 },
	{ "local4",	LOG_LOCAL4 },
	{ "local5",	LOG_LOCAL5 },
	{ "local6",	LOG_LOCAL6 },
	{ "local7",	LOG_LOCAL7 },
	{ NULL,		0 }
};

static struct token virus_strategy_tab[] = {
	{ "discard",			STRATEGY_DISCARD },
	{ "mark",				STRATEGY_MARK },
	{ "notify-recipient",	STRATEGY_NOTIFY_RECIPIENT },
	{ "reject",				STRATEGY_REJECT },
	{ NULL,	0 }
};

static struct token error_strategy_tab[] = {
	{ "discard",	STRATEGY_DISCARD },
	{ "mark",		STRATEGY_MARK },
	{ "tempfail",	STRATEGY_TEMPFAIL },
	{ NULL,	0 }
};

static struct token spam_strategy_tab[] = {
	{ "discard",	STRATEGY_DISCARD },
	{ "mark",		STRATEGY_MARK },
	{ "reject",		STRATEGY_REJECT },
	{ NULL,	0 }
};

static struct token unwanted_strategy_tab[] = {
	{ "discard",			STRATEGY_DISCARD },
	{ "mark",				STRATEGY_MARK },
	{ "reject",				STRATEGY_REJECT },
	{ NULL,	0 }
};

enum option_tok {
	OPT_MARKALL = 0,
	OPT_LOGVIRUS,
	OPT_LOGSPAM,
	OPT_LOGUNWANTED
};

static struct token option_tab[] = {
	{ "markall",		OPT_MARKALL },
	{ "logvirus",		OPT_LOGVIRUS },
	{ "logspam", 		OPT_LOGSPAM },
	{ "logunwanted",	OPT_LOGUNWANTED },
	{ NULL, 0 }
};

enum tmpfiles_opt {
	TF_GRPRD = 0,
	TF_SETGRP
};

static struct token tmpfiles_tab[] = {
	{ "g+r",	TF_GRPRD },
	{ "setgrp",	TF_SETGRP },
	{ NULL,	0 }
};

int
get_token(struct token *token_tab, char *string, int deflt)
{
	int n;
	
	for (n = 0; token_tab[n].string != NULL; n++)
		if (!strcmp(string, token_tab[n].string))
			return token_tab[n].token;
			
	return deflt;
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-V] [-m] [-k] [-p port] [-d dir] [-T tmpfile-opt] [-l log-facility] [-e logile] [-s virus-strategy] [-U spam-strategy] [-c unwanted-strategy] [-S error-strategy] [-b backend] [-n recipient-notification] [-o notify-only] [-C configfile] [-P pidfile] [-t chroot-dir] [-u user] [-g group] [-f maxfiles] [-a spam-subject-prefix]", __progname);
#ifdef WITH_LDAP
	fprintf(stderr, " [-D binddn] [-h ldaphost] [-L ldapport] [-w bindpasswd] [-B searchbase]");
#endif
	fprintf(stderr, "\n");
	exit(1);
}

static int
decode_backend(char *s)
{
	char *p;
	int n;
	char *q, *r;
	
	if ((q = strdup(s)) == NULL)
		err(1, "memoy allocation error");
	
	nbe = 0;
	r = q;
	do {
		p = strsep(&r, ",");
		if (*p != '\0')
			++nbe;
	} while (r != NULL);
	
	free(q);
		
	if ((be = (struct backend *) malloc(nbe * sizeof(struct backend))) == NULL)
		err(1, "error allocating memory for backends");
	bzero(be, nbe * sizeof(struct backend));
	
	n = 0;
	do {
		p = strsep(&s, ",");
		if (*p != '\0') {
			if ((be[n].name = strdup(p)) == NULL)
				err(1, "memory allocation error");
			++n;
		}
	} while (s != NULL);
	
	
	return 0;
}

static int
smtp_vilter_init(void)
{
	FILE *fp;
	char field[MAXSTR];
	char value[MAXSTR];
	char *be_cfgfile;
	int n, be_found;
	
	if ((fp = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("using configuration from file %s", cfgfile);
			
		while (!read_conf(fp, field, sizeof(field), value, sizeof(value))) {
			switch (get_token(cmd_tab, field, TOK_UNKNOWN)) {
				case CMD_BACKEND:
					if (backend == NULL)
						decode_backend(value);
					break;
				case CMD_VIRUS_STRATEGY:
					if (virus_strategy == STRATEGY_UNSET) {
						virus_strategy = get_token(virus_strategy_tab, value, TOK_UNKNOWN);
						if (virus_strategy == TOK_UNKNOWN)
							errx(1, "unknown virus-strategy %s", value);
					}
					break;
				case CMD_ERROR_STRATEGY:
					if (error_strategy == STRATEGY_UNSET) {
						error_strategy = get_token(error_strategy_tab, value, TOK_UNKNOWN);
						if (error_strategy == TOK_UNKNOWN)
							errx(1, "unknown error-strategy %s", value);
					}
					break;
				case CMD_SPAM_STRATEGY:
					if (spam_strategy == STRATEGY_UNSET) {
						spam_strategy = get_token(spam_strategy_tab, value, TOK_UNKNOWN);
						if (spam_strategy == TOK_UNKNOWN)
							errx(1, "unknown spam-strategy %s", value);
					}
					break;
				case CMD_UNWANTED_STRATEGY:
					if (unwanted_strategy == STRATEGY_UNSET) {
						unwanted_strategy = get_token(unwanted_strategy_tab, value, TOK_UNKNOWN);
						if (unwanted_strategy == TOK_UNKNOWN)
							errx(1, "unknown unwanted-strategy %s", value);
					}
					break;
				case CMD_PORT:
					if (port == NULL)
						port = strdup(value);
					break;
				case CMD_TMPDIR:
					if (tmpdir == NULL)
						tmpdir = strdup(value);
					break;
				case CMD_PIDFILE:
					if (pidfile == NULL)
						pidfile = strdup(value);
					break;
				case CMD_LOG_FACILITY:
					if (log_facility == LOG_UNSET) {
						log_facility = get_token(lfacilities_tab, value, TOK_UNKNOWN);
						if (log_facility == TOK_UNKNOWN) {
							warnx("unknown log facility %s, resetting to LOG_MAIL", value);
							log_facility = LOG_MAIL;
						}
					}
					break;
				case CMD_LOGFILE:
					if (logfile == NULL)
						logfile = strdup(value);
					break;
				case CMD_CFGFILE:
					be_cfgfile = strchr(value, ':');
					if (be_cfgfile != NULL) {
						*be_cfgfile++ = '\0';
						be_found = 0;
						for (n = 0; n < nbe; n++) {
							if (!strcmp(be[n].name, value)) {
								if ((be[n].config_file = strdup(be_cfgfile)) == NULL)
									err(1, "memory allocation error, can't store config file %s for backend %s", be_cfgfile, be[n].name);
								be_found = 1;
								break;
							}
						}
						if (!be_found) {
							warnx("config-file: backend %s not found (hint: config-file commands must be listed after the backend command)", value);
						}
						
					} else
						warnx("config-file syntax error");
					break;
				case CMD_RECIPIENT_NOTIFICATION:
					recipient_notification = strdup(value);
					break;
				case CMD_NOTIFY_ONLY:
					notify_only = strdup(value);
					break;
				case CMD_OPTION:
					switch (get_token(option_tab, value, TOK_UNKNOWN)) {
						case OPT_MARKALL:
							markall = 1;
							break;
						case OPT_LOGVIRUS:
							logvirus = 1;
							break;
						case OPT_LOGSPAM:
							logspam = 1;
							break;
						case OPT_LOGUNWANTED:
							logunwanted = 1;
							break;
						default:
							warnx("unknown option %s", value);
					}
					break;
				case CMD_USER:
					if (user == NULL)
						user = strdup(value);
					break;
				case CMD_GROUP:
					if (group == NULL)
						group = strdup(value);
					break;
				case CMD_CHROOT:
					if (chrootdir == NULL)
						chrootdir = strdup(value);
					break;
				case CMD_TMPFILES:
					switch (get_token(tmpfiles_tab, value, TOK_UNKNOWN)) {
						case TF_GRPRD:
							tmpfiles_gr = 1;
							break;
						case TF_SETGRP:
							tmpfiles_setgrp = 1;
							break;
						default:
							warnx("unknown mode for tmpfiles: %s", value);
					}
					break;
				case CMD_MAXFILES:
					if (rlimit_nofile == -1)
						rlimit_nofile = atoi(value);
					break;
				case CMD_SPAM_SUBJECT_PREFIX:
					if (spamprefix == NULL)
						spamprefix = strdup(value);
					break;
#ifdef WITH_LDAP
				case CMD_LDAPHOST:
					if (ldaphost == NULL)
						ldaphost = strdup(value);
					break;
				case CMD_LDAPPORT:
					if (ldapport == -1)
						ldapport = atoi(value);
					break;
				case CMD_BINDDN:
					if (binddn == NULL)
						binddn = strdup(value);
					break;
				case CMD_BINDPASSWD:
					if (bindpasswd == NULL)
						bindpasswd = strdup(value);
					break;
				case CMD_SEARCHBASE:
					if (searchbase == NULL)
						searchbase = strdup(value);
					break;
				default:
					warnx("unknown command %s", field);
#endif
			}
		}
		fclose(fp);
	} else
		warn("can't open configuration file %s", cfgfile);

	return 0;
}

/* Allocate and initialize per-connection memory */

static sfsistat
connection_setup(SMFICTX *ctx)
{
	struct privdata *priv;
	struct connection *conn;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	int n;
	
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (verbose)
		warnx("connection setup");

	if ((priv = malloc(sizeof(struct privdata))) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to allocate memory for internal data, %m");
		return SMFIS_TEMPFAIL;
	}
	bzero(priv, sizeof(struct privdata));
	
	if ((conn = malloc(sizeof(struct connection))) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to allocate memory for connection data, %m");
		free(priv);
		return SMFIS_TEMPFAIL;
	}
	bzero(conn, sizeof(struct connection));
	
	priv->conn = conn;

	/* Setup private data for syslog, per backend */
	
	memcpy(&priv->sdata, &sdata, sizeof(struct syslog_data));
	
	if (verbose) priv->sdata.log_stat |= LOG_PERROR;	
	
	if ((priv->backend_data = malloc(sizeof(struct be_data) * nbe)) == NULL) {
		syslog_r(LOG_ERR, &priv->sdata, "unable to allocate memory for backend data, %m");
		return SMFIS_TEMPFAIL;
	}
	bzero(priv->backend_data, sizeof(struct be_data) * nbe);
	
	/* Setup syslog data for each backend */
	
	for (n = 0; n < nbe; n++)
		priv->backend_data[n].sdata = &priv->sdata;

	if (smfi_setpriv(ctx, priv)) {
		syslog_r(LOG_ERR, &sdata, "unable to store internal data, %m");
		free(priv->conn);
		free(priv->backend_data);
		free(priv);
		return SMFIS_TEMPFAIL;
	}
	
	return SMFIS_CONTINUE;
}

/* Allocate and initialize per-message memory */

static sfsistat
message_setup(SMFICTX *ctx)
{
	struct privdata *priv;
	struct message *msg;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (verbose)
		warnx("message setup");
	
	if ((priv = smfi_getpriv(ctx)) != NULL) {
	
		if ((msg = malloc(sizeof(struct message))) != NULL) {
			bzero(msg, sizeof(struct message));
			
			priv->msg = msg;
			return SMFIS_CONTINUE;
		} else {
			syslog_r(LOG_ERR, &sdata, "unable to allocate memory for message, %m");
			return SMFIS_TEMPFAIL;
		}
	} else
		syslog_r(LOG_ERR, &sdata, "unable to get internal data in message_setup(), %m");
		
	return SMFIS_TEMPFAIL;
}

/* Deallocate per-message memory and ressources*/

static void
message_cleanup(SMFICTX *ctx)
{
	struct privdata *priv;
	struct message *msg;
	
	struct syslog_data sdata = SYSLOG_DATA_INIT;
#ifdef WITH_LDAP
	struct recipient *r;
	struct domain *d;
#endif
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (verbose)
		warnx("message cleanup");
		
	if ((priv = smfi_getpriv(ctx)) != NULL) {
	
		if ((msg = priv->msg) != NULL) {
		
			if (msg->fp != NULL)
				fclose(msg->fp);

			if (!keep_tmpfiles) {
				if (unlink(msg->tfn))
					syslog_r(LOG_ERR, &sdata, "can't delete temp file %s, %m", msg->tfn);
			}
			
			if (msg->msgid != NULL)
				free(msg->msgid);
			if (msg->envfrom != NULL)
				free(msg->envfrom);
			if (msg->envto != NULL)
				free(msg->envto);
	#ifdef WITH_LDAP
			rf = NULL;
			while (!SLIST_EMPTY(&msg->rhead)) {
				r = SLIST_FIRST(&msg->rhead, entries);
				
				free(r->recipient);
				free(r->mailLocalAddress);
				if (r->spamprefix != NULL)
					free(r->spamprefix);
					
				SLIST_REMOVE_HEAD(&msg->rhead, entries);
				free(r);
			}
			
			while (!SLIST_EMPTY(&msg->dhead)) {
				d = SLIST_FIRST(&msg->dhead, entries);
				free(d->mailLocalDomain);
				if (d->spamprefix != NULL)
					free(d->spamprefix);
				SLIST_REMOVE_HEAD(&msg->dhead, entries);
				free(d);
			}
			

	#endif
			if (msg->hdrfrom != NULL)
				free(msg->hdrfrom);
			if (msg->hdrto != NULL)
				free(msg->hdrto);
			if (msg->subject != NULL)
				free(msg->subject);
			free(msg);
			priv->msg = NULL;
		}
	}
}

/* Deallocate per-connection memory */

static void
connection_cleanup(SMFICTX *ctx)
{
	struct privdata *priv;
	struct connection *conn;
	struct syslog_data sdata = SYSLOG_DATA_INIT;

	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (verbose)
		warnx("connection cleanup");
		
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		if (priv->conn != NULL) {
			conn = priv->conn;
			
			if (conn->hostname != NULL)
				free(conn->hostname);
				
			free(conn);
		}
		
		if (priv->backend_data != NULL)
			free(priv->backend_data);
		
		free(priv);
		smfi_setpriv(ctx, NULL);

	} else
		syslog_r(LOG_ERR, &sdata, "unable to get internal data in connection_cleanup()");
	
}

static sfsistat
smtp_vilter_connect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *hostaddr)
{
	struct privdata *priv;
	struct connection *conn;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	sfsistat retval;
	
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (verbose)
		warnx("connect from: %s", hostname);
	
	if ((retval = connection_setup(ctx)) != SMFIS_CONTINUE)
		return retval;
		
	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to get internal data, %m");
		return SMFIS_TEMPFAIL;
	}
	
	conn = priv->conn;
	
	if ((conn->hostname = strdup(hostname)) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to allocate memory for hostname %s, %m", hostname);
		return SMFIS_TEMPFAIL;
	}
	
	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_envfrom(SMFICTX *ctx, char **argv)
{
	struct privdata *priv;
	struct message *msg;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	int fd;
	int n;
	sfsistat retval;
	
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (verbose)
		warnx("envelope sender: %s", argv[0]);
	
	if ((retval = message_setup(ctx)) != SMFIS_CONTINUE)
		return retval;
		
	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to get internal data, %m");
		return SMFIS_TEMPFAIL;
	}
	
	msg = priv->msg;
	
	if ((msg->envfrom = strdup(argv[0])) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to allocate memory for envelope address %s, %m", argv[0]);
		return SMFIS_TEMPFAIL;
	}
	
	
	if (virus_strategy == STRATEGY_NOTIFY_RECIPIENT)
		msg->notify = 1;
		
	strlcpy(msg->tfn, tmpdir, sizeof(msg->tfn));
	strlcat(msg->tfn, "/vilter.XXXXXXXXXX", sizeof(msg->tfn));
	
	if ((fd = mkstemp(msg->tfn)) == -1) {
		syslog_r(LOG_ERR, &sdata, "unable to create temp file, %m");
		return SMFIS_TEMPFAIL;
	}
	
	if (tmpfiles_gr) {
		if (fchmod(fd, S_IRUSR | S_IWUSR | S_IRGRP)) {
			syslog_r(LOG_ERR, &sdata, "unable to change access mode of temp file, %m");
			close(fd);	
			unlink(msg->tfn);
			return SMFIS_TEMPFAIL;
		}
	}
	
	if (tmpfiles_setgrp) {
		if (fchown(fd, -1, getgid())) {
			syslog_r(LOG_ERR, &sdata, "unable to change group ownership of temp file, %m");
			close(fd);	
			unlink(msg->tfn);
			return SMFIS_TEMPFAIL;
		}
	}
		
	if ((msg->fp = fdopen(fd, "w")) == NULL) {
		syslog_r(LOG_ERR, &sdata, "unable to open temp file %s for writing, %m", msg->tfn);
		if (fd != -1) {
			close(fd);
			unlink(msg->tfn);
		}
		return SMFIS_TEMPFAIL;
	}		
		
	for (n = 0; n < nbe; n++) {
		if (be[n].be_new != NULL) {
			if (be[n].be_new(&priv->backend_data[n], argv[0], msg->fp)) {
				syslog_r(LOG_ERR, &sdata, "vilter-engine refuses to work on this message");
				/* vilter_cleanup(ctx); */
				return SMFIS_TEMPFAIL;
			}
		}
	}
#ifdef WITH_LDAP
	SLIST_INIT(&priv->rhead);
	SLIST_INIT(&priv->dhead);
#endif

	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_envrcpt(SMFICTX *ctx, char **argv)
{
	struct privdata *priv;
	struct message *msg;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
#ifdef WITH_LDAP
	struct recipient *r;
	struct domain *d;
	char *p;
	char *attrs[] = { "spamStrategy",
					  "virusStrategy",
					  "spamSubjectPrefix",
					  NULL 
					};
	char filter[1024];
	
#endif
	if (verbose) sdata.log_stat |= LOG_PERROR;

	if (verbose)
		warnx("envelope recipient: %s", argv[0]);
		
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;
		
		if (notify_only != NULL) {	/* ugly hack, to be honest... */
			if (strstr(argv[0], notify_only) == NULL)
				msg->notify = 0;
		}

#ifdef WITH_LDAP		
		/*
		 * Keep a list of all recipients and all domains and
		 * initiate an asynchroneous LDAP search for each
		 * recipient and each domain to retrieve per-user and
		 * per-domain configuration.
		 */
	
		if ((r = malloc(sizeof(struct recipient))) == NULL) {
			syslog_r(LOG_ERR, &sdata, "out of memory, %m");
			goto bailout;
		}
		bzero(r, sizeof(struct recipient));
		
		if ((r->recipient = strdup(argv[0])) == NULL) {
			syslog_r(LOG_ERR, &sdata, "out of memory, %m");
			goto bailout;
		}
		
		if ((r->mailLocalAddress = strdup(r->recipient + 1)) == NULL) {
			free(r->recipient);
			free(r);
			syslog_r(LOG_ERR, &sdata, "out of memory, %m");
			goto bailout;
		}
		
		if ((p = strrchr(r->mailLocalAddress, '>')) != NULL)
			*p = '\0';
		
		r->virus_strategy = r->spam_strategy = STRATEGY_UNSET;
		
		/* Search for per-user configuration */
		
		snprintf(filter, sizeof(filter), "(&(objectClass=SMTPVilterUserConf)(mailLocalAddress=%s))", r->mailLocalAddress);
		
		if ((r->msgid = ldap_search(ld, searchbase, LDAP_SCOPE_SUBTREE, filter, attrs, 0)) == -1) {
			free(r->recipient);
			free(r->mailLocalAddress);
			free(r);
			syslog_r(LOG_ERR, &sdata, "ldap_search failed");
			goto bailout;
		} else {
			SLIST_INSERT_HEAD(&msg->rhead, r, entries);
		}
		
		/* Search for per-domain configuration */
		
		if ((p = strchr(r->mailLocalAddress, '@')) == NULL) {
			syslog_r(LOG_ERR, &sdata, "strange email address %s", r->mailLocalAddress);
			goto bailout;
		}
		
		p++;
		
		SLIST_FOREACH(d, &msg->dhead, entries) {
			if (!strcmp(d->mailLocalDomain, p)) { /* Domain already searched for */
				r->domain = d;
				goto bailout;
			}
		}
		
		if ((d = malloc(sizeof(struct domain))) == NULL) {
			syslog_r(LOG_ERR, &sdata, "out of memory, %m");
			goto bailout;
		}
		bzero(d, sizeof(struct domain));
		
		if ((d->mailLocalDomain = strdup(p)) == NULL) {
			syslog_r(LOG_ERR, &sdata, "out of memory, %m");
			goto bailout;
		}
		d->virus_strategy = d->spam_strategy = STRATEGY_UNSET;

		snprintf(filter, sizeof(filter), "(&(objectClass=SMTPVilterDomainConf)(mailLocalDomain=%s))", d->mailLocalDomain);
		
		if ((d->msgid = ldap_search(ld, searchbase, LDAP_SCOPE_SUBTREE, filter, attrs, 0)) == -1) {
			free(d->mailLocalDomain);
			free(d);
			syslog_r(LOG_ERR, &sdata, "ldap_search failed");
			goto bailout;
		} else {
			SLIST_INSERT_HEAD(&msg->dhead, d, entries);
			r->domain = d;
		}
		
#endif
	} else
		syslog_r(LOG_ERR, &sdata, "unable to access internal data, %m");

#ifdef WITH_LDAP
bailout:
#endif
	return SMFIS_CONTINUE;
}

static sfsistat
vilter_header(SMFICTX *ctx, char *headerf, char *headerv)
{
	struct privdata *priv;
	struct message *msg;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	if (verbose) sdata.log_stat |= LOG_PERROR;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;
		
		if (fprintf(msg->fp, "%s: %s\r\n", headerf, headerv) >= 0) {
						
			if (!strncmp(headerf, "From", 4)) {
				if (msg->hdrfrom != NULL) {
					syslog_r(LOG_INFO, &sdata, "duplicate From: header");
					free(msg->hdrfrom);
					msg->hdrfrom = NULL;
				}
				
				if ((msg->hdrfrom = strdup(headerv)) == NULL) {
					syslog_r(LOG_ERR, &sdata, "unable to allocate memory for header from: address, %m");
					goto badluck;
				}
			} else if (!strncmp(headerf, "To", 2)) {
				if (msg->hdrto != NULL) {
					syslog_r(LOG_INFO, &sdata, "duplicate To: header");
					free(msg->hdrto);
					msg->hdrto = NULL;
				}

				if ((msg->hdrto = strdup(headerv)) == NULL) {
					syslog_r(LOG_ERR, &sdata, "unable to allocate memory for header to: address, %m");
					goto badluck;
				}
			} else if (!strncmp(headerf, "Message-Id", 10)) {
				if (msg->msgid != NULL) {
					syslog_r(LOG_INFO, &sdata, "duplicate message id");
					free(msg->msgid);
					msg->msgid = NULL;
				}

				if ((msg->msgid = strdup(headerv)) == NULL) {
					syslog_r(LOG_ERR, &sdata, "unable to allocate memory for message id, %m");
					goto badluck;
				}
			} else if (spamprefix != NULL && !strncmp(headerf, "Subject", 7)) {
				if (msg->subject != NULL) {
					syslog_r(LOG_INFO, &sdata, "duplicate subject");
					free(msg->subject);
					msg->subject = NULL;
				}
				
				if ((msg->subject = strdup(headerv)) == NULL) {
					syslog_r(LOG_ERR, &sdata, "unable to allocate memory for subject, %m");
					goto badluck;
				}
			}	
			
			return SMFIS_CONTINUE;
		} else
			syslog_r(LOG_ERR, &sdata, "write error, %m");
	} else
		syslog_r(LOG_ERR, &sdata, "unable to access internal data, %m");

badluck:		
	/* vilter_cleanup(ctx); */
	return SMFIS_TEMPFAIL;
}

static sfsistat
vilter_eoh(SMFICTX *ctx)
{
	struct privdata *priv;
	struct message *msg;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	if (verbose) sdata.log_stat |= LOG_PERROR;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;
		
		if (fprintf(msg->fp, "\r\n") >= 0)
			return SMFIS_CONTINUE;
		else 
			syslog_r(LOG_ERR, &sdata, "write error, %m");
	} else
		syslog_r(LOG_ERR, &sdata, "unable to access internal data, %m");
		
	/* vilter_cleanup(ctx); */
	return SMFIS_TEMPFAIL;
}

static sfsistat
vilter_body(SMFICTX *ctx, u_char *bodyp, size_t len)
{
	struct privdata *priv;
	struct message *msg;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	if (verbose) sdata.log_stat |= LOG_PERROR;
		
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;
		
		if (fwrite(bodyp, len, 1, msg->fp) == 1)
			return SMFIS_CONTINUE;
		else
			syslog_r(LOG_ERR, &sdata, "write error, %m");
	} else
		syslog_r(LOG_ERR, &sdata, "unable to access internal data, %m");

	/* vilter_cleanup(ctx); */
	return SMFIS_TEMPFAIL;
	
}

static void
mark_virus(SMFICTX *ctx, char *status, char *virus, char *backend)
{
	char headerf[MAXSTR];
	
	smfi_addheader(ctx, "X-SMTP-Vilter-Virus-Backend", backend);

	if (status != NULL) {
		smfi_addheader(ctx, "X-SMTP-Vilter-Status", status);
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Virus-Status", backend);
		smfi_addheader(ctx, headerf, status);
	}
	
	if (virus != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Virus-Name", backend);

		smfi_addheader(ctx, headerf, virus);
	}
}

static void
mark_spam(SMFICTX *ctx, char *status, char *backend, double score, double threshold)
{
	char hdr[32];
	char slvl[] = "************************************";
	int lvl;
	
	smfi_addheader(ctx, "X-SMTP-Vilter-Spam-Backend", backend);

	snprintf(hdr, sizeof(hdr), "%.1f", score);
	smfi_addheader(ctx, "X-Spam-Score", hdr);
	
	lvl = (int) rint(score);
	
	if (lvl > 0) {
		if (lvl < strlen(slvl))
			slvl[lvl] = '\0';
		
		smfi_addheader(ctx, "X-Spam-Level", slvl);
	}
	
	if (threshold > 0.0) {
		snprintf(hdr, sizeof(hdr), "%.1f", threshold);
		smfi_addheader(ctx, "X-Spam-Threshold", hdr);
		
		snprintf(hdr, sizeof(hdr), "%.1f", score / threshold);
		smfi_addheader(ctx, "X-Spam-Probability", hdr);
	}
	
	if (status != NULL)
		smfi_addheader(ctx, "X-Spam-Status", status);
}

static void
mark_unwanted(SMFICTX *ctx, char *status, char *reason, char *backend)
{
	char headerf[MAXSTR];

	smfi_addheader(ctx, "X-SMTP-Vilter-Unwanted-Backend", backend);

	if (status != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Unwanted-Status", backend);
		smfi_addheader(ctx, headerf, status);
	}
	
	if (reason != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Unwanted-Reason", backend);
		smfi_addheader(ctx, headerf, reason);
	}
}

static int
vilter_replacebody(SMFICTX *ctx, char *s)
{
	return smfi_replacebody(ctx, (u_char *) s, strlen(s));
}

static int
replace_msgbody(SMFICTX *ctx, char *fn)
{
	FILE *fp;
	char buf[1024];
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	
	if ((fp = fopen(fn, "r")) != NULL) {
		while (fgets(buf, sizeof(buf), fp)) {
			if (vilter_replacebody(ctx, buf) != MI_SUCCESS) {
				fclose(fp);
				goto discard;
			}
		}
		fclose(fp);
		return SMFIS_CONTINUE;				
	} else
		syslog_r(LOG_ERR, &sdata, "can not open replacement file %s, %m", fn);
	
discard:
	syslog_r(LOG_ERR, &sdata, "failed to replace message body");		
	return SMFIS_DISCARD;
}

static int
notify_recipient(SMFICTX *ctx, char *virus)
{
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	if (recipient_notification != NULL)
		return replace_msgbody(ctx, recipient_notification);
	
	/*
	 * If there is no recipient-notification file or there was an error
	 * processing it, fall through to the standard message.
	 */
	   						
	if (vilter_replacebody(ctx, "\r\n*** The message body has been deleted because it was infected with a virus ***\r\n") == MI_SUCCESS) {
		return SMFIS_CONTINUE;
	}

	syslog_r(LOG_ERR, &sdata, "failed to replace message body");		
	return SMFIS_DISCARD;
}

void
log_incident(struct privdata *priv, char *type, char *backend, char *value)
{
	time_t now;
	struct message *msg;
	struct connection *conn;
	
	msg = priv->msg;
	conn = priv->conn;
	
	if (logfp != NULL) {
		flockfile(logfp);
		time(&now);
		fprintf(logfp, "smtp-vilter incident report\n");
		fprintf(logfp, "timestamp: %s", ctime(&now));
		fprintf(logfp, "type: %s\n", type);
		fprintf(logfp, "peer: %s\n", (conn->hostname != NULL) ? conn->hostname : "n/a");
		fprintf(logfp, "envelope-sender: %s\n", (msg->envfrom != NULL) ? msg->envfrom : "n/a");
		/* fprintf(logfp, "envelope-recipient: %s\n", (priv->envto != NULL) ? priv->envto : "n/a"); */
		fprintf(logfp, "message-id: %s\n", (msg->msgid != NULL) ? msg->msgid : "n/a");

		fprintf(logfp, "from: %s\n", (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a");
		fprintf(logfp, "to: %s\n", (msg->hdrto != NULL) ? msg->hdrto : "n/a");
		fprintf(logfp, "subject: %s\n", (msg->subject != NULL) ? msg->subject : "n/a");
		fprintf(logfp, "backend: %s\n", backend);
		if (value != NULL)
			fprintf(logfp, "%s: %s\n", type, value);
		fprintf(logfp, "\n");
		fflush(logfp);
		funlockfile(logfp);
	}
}

static sfsistat
vilter_virus(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain)
{
	int retval = SMFIS_CONTINUE;
	char virus[MAXLEN];
	int scanresult;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	char scanfn[PATH_MAX];
	struct message *msg;
	
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	msg = priv->msg;
	
	bzero(virus, sizeof(virus));
		
	/* Scan for virus */
	
	strlcpy(scanfn, msg->tfn, sizeof(scanfn));
	scanresult = backend->be_scan.be_scan_virus(backend_data, scanfn, chrootdir, virus, sizeof(virus));
			
	switch(scanresult) {
			
		case SCAN_VIRUS:

			syslog_r(LOG_NOTICE, &sdata, "virus '%s' detected in message with id '%s' from '%s' (env '%s') to '%s' by backend %s",
				virus,
				(msg->msgid != NULL) ? msg->msgid : "n/a",
				(msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
				(msg->envfrom != NULL) ? msg->envfrom : "n/a",
				(msg->hdrto != NULL) ? msg->hdrto : "n/a",
				backend->name);

			if (logvirus)
				log_incident(priv, "virus", backend->name, virus);
			
			switch (virus_strategy) {
				case STRATEGY_MARK:
					syslog_r(LOG_INFO, &sdata, "marking message (but deliver anyway)");

					mark_virus(ctx, "infected", virus, backend->name);
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_REJECT:
					syslog_r(LOG_INFO, &sdata, "rejecting message");
					smfi_setreply(ctx, "554", "5.6.1", "Virus detected in the message");
					retval = SMFIS_REJECT;
					break;
				case STRATEGY_NOTIFY_RECIPIENT:
					if (msg->notify) {

						syslog_r(LOG_INFO, &sdata, "marking message and replacing message body");
						mark_virus(ctx, "cleaned", virus, backend->name);
						if ((retval = notify_recipient(ctx, virus)) == SMFIS_CONTINUE)
							*stop_chain = 1;
						
						/* Set the message content type to text/plain */

						if (smfi_chgheader(ctx, "Content-Type", 1, "text/plain")) {
							syslog_r(LOG_INFO, &sdata, "failed to set message content type to text/plain");
						}
						break;
					}
					/* FALLTRHOUGH */
				case STRATEGY_DISCARD:
				default:
					syslog_r(LOG_INFO, &sdata, "silently discarding message");
					retval = SMFIS_DISCARD;
					break;
			}
			break;
		case SCAN_OK:
			syslog_r(LOG_INFO, &sdata, "message contains no virus");

			if (markall)
				mark_virus(ctx, "clean", NULL, backend->name);
				
			if (strcmp(scanfn, msg->tfn)) {
				syslog_r(LOG_INFO, &sdata, "replace message body");
				replace_msgbody(ctx, scanfn);
				if (unlink(scanfn))
					syslog_r(LOG_ERR, &sdata, "can't delete temp file %s, %m", scanfn);
			}
			
			break;
		case SCAN_ERROR:
			syslog_r(LOG_ERR, &sdata, "error during virus scan of file %s", msg->tfn);

			switch (error_strategy) {
				case STRATEGY_MARK:
					syslog_r(LOG_INFO, &sdata, "marking message (but deliver anyway)");

					mark_virus(ctx, "unchecked", NULL, backend->name);
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_TEMPFAIL:
					syslog_r(LOG_INFO, &sdata, "temporarily failing message");
					retval = SMFIS_TEMPFAIL;
					break;
				case STRATEGY_DISCARD:
					/* FALLTHROUGH */
				default:
					syslog_r(LOG_INFO, &sdata, "discarding message");

					retval = SMFIS_DISCARD;
					break;
			}
			break;
	}
			
	return retval;
}

static sfsistat
vilter_spam(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain)
{
	int retval = SMFIS_CONTINUE;
	int scanresult;
	double score, threshold;
	char *subject;
	int len;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	struct message *msg;
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	msg = priv->msg;
	
	score = threshold = 0.0;
	
	/* Scan for spam */
	
	scanresult = backend->be_scan.be_scan_spam(backend_data, msg->tfn, chrootdir, &score, &threshold);

	switch(scanresult) {

		case SCAN_SPAM:
			syslog_r(LOG_NOTICE, &sdata, "message with id '%s' from '%s' (env '%s') to '%s' classified as spam by backend %s",
				(msg->msgid != NULL) ? msg->msgid : "n/a",
				(msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
				(msg->envfrom != NULL) ? msg->envfrom : "n/a",
				(msg->hdrto != NULL) ? msg->hdrto : "n/a",
				backend->name);

			if (logspam)
				log_incident(priv, "spam", backend->name, NULL);

			switch (spam_strategy) {
				case STRATEGY_DISCARD:
					syslog_r(LOG_INFO, &sdata, "discarding message");

					retval = SMFIS_DISCARD;
					break;
				case STRATEGY_REJECT:
					syslog_r(LOG_INFO, &sdata, "rejecting message");
					smfi_setreply(ctx, "554", "5.6.1", "Message considered spam");
					retval = SMFIS_REJECT;
					break;
				case STRATEGY_MARK:
					syslog_r(LOG_INFO, &sdata, "marking message (but deliver anyway)");
					mark_spam(ctx, "spam", backend->name, score, threshold);
					
					/* Idea:  Add a subject with spamprefix even if there is no
					   subject. */
					   
					if (spamprefix != NULL && msg->subject != NULL) {
						len = strlen(spamprefix) + strlen(msg->subject) + 2;
						subject = malloc(len);
						if (subject != NULL) {
							snprintf(subject, len, "%s %s", spamprefix, msg->subject);
							smfi_chgheader(ctx, "Subject", 1, subject);
							free(subject);
						} else
							syslog_r(LOG_ERR, &sdata, "no space to construct a new subject line, %m");
					}
					
					retval = SMFIS_CONTINUE;
					break;
			}

			break;
		case SCAN_OK:
			syslog_r(LOG_INFO, &sdata, "message is not spam");

			mark_spam(ctx, NULL, backend->name, score, threshold);
			break;
		case SCAN_ERROR:
			syslog_r(LOG_ERR, &sdata, "error during spam scan of file %s", msg->tfn);

			switch (error_strategy) {
				case STRATEGY_MARK:
					syslog_r(LOG_INFO, &sdata, "marking message (but deliver anyway)");

					mark_spam(ctx, "unchecked", backend->name, 0.0, 0.0);
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_TEMPFAIL:
					syslog_r(LOG_INFO, &sdata, "temporarily failing message");
					retval = SMFIS_TEMPFAIL;
					break;
				case STRATEGY_DISCARD:
					/* FALLTHROUGH */
				default:
					syslog_r(LOG_INFO, &sdata, "discarding message");

					retval = SMFIS_DISCARD;
					break;
			}
			break;
	}

	return retval;
}

static sfsistat
vilter_unwanted(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain)
{
	int retval = SMFIS_CONTINUE;
	char reason[MAXLEN];
	int scanresult;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	char scanfn[PATH_MAX];
	struct message *msg;
	
	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	msg = priv->msg;
	
	bzero(reason, sizeof(reason));
		
	/* Scan for unwanted content */
	
	strlcpy(scanfn, msg->tfn, sizeof(scanfn));
	scanresult = backend->be_scan.be_scan_unwanted(backend_data, scanfn, chrootdir, reason, sizeof(reason));
			
	switch(scanresult) {
			
		case SCAN_UNWANTED:

			syslog_r(LOG_NOTICE, &sdata, "unwanted content  detected in message with id '%s' from '%s' (env '%s') to '%s' by backend %s, reason %s",
				(msg->msgid != NULL) ? msg->msgid : "n/a",
				(msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
				(msg->envfrom != NULL) ? msg->envfrom : "n/a",
				(msg->hdrto != NULL) ? msg->hdrto : "n/a",
				backend->name, reason);

			if (logunwanted)
				log_incident(priv, "unwanted-content", backend->name, reason);

			switch (unwanted_strategy) {
				case STRATEGY_MARK:
					syslog_r(LOG_INFO, &sdata, "marking message (but deliver anyway)");

					mark_unwanted(ctx, "unwanted content", reason, backend->name);
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_REJECT:
					syslog_r(LOG_INFO, &sdata, "rejecting message");
					smfi_setreply(ctx, "554", "5.6.1", "Unwanted content detected in the message");
					retval = SMFIS_REJECT;
					break;
				case STRATEGY_NOTIFY_RECIPIENT:
					if (msg->notify) {

						syslog_r(LOG_INFO, &sdata, "marking message and replacing message body");
						mark_unwanted(ctx, "cleaned", reason, backend->name);
						retval = notify_recipient(ctx, reason);

						/* Set the message content type to text/plain */

						if (smfi_chgheader(ctx, "Content-Type", 1, "text/plain")) {
							syslog_r(LOG_INFO, &sdata, "failed to set message content type to text/plain");
						}
						break;
					}
					/* FALLTRHOUGH */
				case STRATEGY_DISCARD:
				default:
					syslog_r(LOG_INFO, &sdata, "silently discarding message");
					retval = SMFIS_DISCARD;
					break;
			}
			break;
		case SCAN_OK:
			syslog_r(LOG_INFO, &sdata, "message contains no unwanted content");

				
			if (strcmp(scanfn, msg->tfn)) {
				if (markall)
					mark_unwanted(ctx, "cleaned", NULL, backend->name);
					
			if (logunwanted)
				log_incident(priv, "unwanted-content", backend->name, reason);

				syslog_r(LOG_INFO, &sdata, "replace message body");
				replace_msgbody(ctx, scanfn);
				
				/* Make sure later backends get the new content */
				
				if (unlink(scanfn))
					syslog_r(LOG_ERR, &sdata, "can't delete temp file %s, %m", scanfn);
			} else if (markall)
				mark_unwanted(ctx, "clean", NULL, backend->name);
			
			break;
		case SCAN_ERROR:
			syslog_r(LOG_ERR, &sdata, "error during scan of file %s", msg->tfn);

			switch (error_strategy) {
				case STRATEGY_MARK:
					syslog_r(LOG_INFO, &sdata, "marking message (but deliver anyway)");

					mark_unwanted(ctx, "unchecked", NULL, backend->name);
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_TEMPFAIL:
					syslog_r(LOG_INFO, &sdata, "temporarily failing message");
					retval = SMFIS_TEMPFAIL;
					break;
				case STRATEGY_DISCARD:
					/* FALLTHROUGH */
				default:
					syslog_r(LOG_INFO, &sdata, "discarding message");

					retval = SMFIS_DISCARD;
					break;
			}
			break;
	}
			
	return retval;
}

#ifdef WITH_LDAP
static void
print_entries_with_values(LDAP *ld, LDAPMessage *result)
{
	LDAPMessage *e;
	BerElement *ber;
	char *dn, *attr;
	char **vals;
	int i;
	
	for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
		if ((dn = ldap_get_dn(ld, e)) != NULL) {
			warnx("dn: %s", dn);
			ldap_memfree(dn);
		}
		
		for (attr = ldap_first_attribute(ld, e, &ber); attr != NULL; attr = ldap_next_attribute(ld, e, ber)) {
			if ((vals = ldap_get_values(ld, e, attr)) != NULL) {
				for (i = 0; vals[i] != NULL; i++) {
					warnx("%s: %s", attr, vals[i]);
				}
				ldap_value_free(vals);
			}
			ldap_memfree(attr);
		}
		
		if (ber != NULL) {
			ber_free(ber, 0);
		}
	}
}
#endif

static sfsistat
vilter_eom(SMFICTX *ctx)
{
	struct privdata *priv;
	struct message *msg;
	int retval = SMFIS_CONTINUE;
	int n;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	struct be_data *backend_data;
	int stop_chain;
#ifdef WITH_LDAP
	struct recipient *r;
	struct domain *d;
	LDAPMessage *result;
	LDAPMessage *e;
	char **vals;
	int nentries, rc;	
#endif
	if (verbose) sdata.log_stat |= LOG_PERROR;

	
	/*
	 * If there is only one recipient, things are easy:  Take this
	 * recipients strategy settings, if present, else the domain settings,
	 * if these are absent, use the system default.
	 *
	 * If there is more than one recipient, things can get complicated.
	 * Check if all strategies are compatible, if they are, treat the
	 * message accordingly.  If one recipient has a discard strategy whereas
	 * others have a mark strategy, the recipient with discard is removed from the
	 * list of recipients.  The following table gives an idea how the
	 * situation is handled.
	 *
	 * R1		R2			Action
	 * any		the same	use strategy
	 * discard	mark		remove R1, then mark
	 * discard	notify		remove R1, then notify
	 * mark		notify		the only real conflict: notify
	 */
		 
	
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		fclose(msg->fp);
		msg->fp = NULL;
		
		smfi_addheader(ctx, "X-SMTP-Vilter-Version", VERSION);
		
		/* Scan content */
	
		for (n = stop_chain = 0; n < nbe && !stop_chain && retval == SMFIS_CONTINUE; n++) {
			backend_data = &priv->backend_data[n];
			switch (be[n].be_type()) {
				case SCAN_VIRUS:
					if (verbose)
						warnx("scanning for viruses using backend %s", be[n].name);
						
					retval = vilter_virus(&be[n], backend_data, ctx, priv, &stop_chain);
					break;
				case SCAN_SPAM:
					if (verbose)
						warnx("scanning for spam using backend %s", be[n].name);

					retval = vilter_spam(&be[n], backend_data, ctx, priv, &stop_chain);
					break;
				case SCAN_UNWANTED:
					if (verbose)
						warnx("scanning for unwanted content using backend %s", be[n].name);
					retval = vilter_unwanted(&be[n], backend_data, ctx, priv, &stop_chain);
					break;
				default:
					/* NOTREACHED */
					return SMFIS_TEMPFAIL;
			}
		}

#ifdef WITH_LDAP
		SLIST_FOREACH(r, &msg->rhead, entries) {
			warnx("recipient: %s (domain %s)", r->mailLocalAddress, r->domain->mailLocalDomain);
			nentries = 0;
			while ((rc = ldap_result(ld, r->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY) {
				nentries++;
				if (verbose)
					print_entries_with_values(ld, result);
				for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
					if ((vals = ldap_get_values(ld, e, "spamStrategy")) != NULL) {
						r->spam_strategy = get_token(spam_strategy_tab, vals[0], TOK_UNKNOWN);
						if (r->spam_strategy == TOK_UNKNOWN) {
							warnx("unknown spam strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
							r->spam_strategy = STRATEGY_UNSET;
						}
						ldap_value_free(vals);
					}

					if ((vals = ldap_get_values(ld, e, "virusStrategy")) != NULL) {
						r->virus_strategy = get_token(virus_strategy_tab, vals[0], TOK_UNKNOWN);
						if (r->virus_strategy == TOK_UNKNOWN) {
							warnx("unknown virus strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
							r->virus_strategy = STRATEGY_UNSET;
						}
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "spamSubjectPrefix")) != NULL) {
						r->spamprefix = strdup(vals[0]);
						ldap_value_free(vals);
					}
				}
				
				ldap_msgfree(result);
			}
		}
		SLIST_FOREACH(d, &msg->dhead, entries) {
			warnx("domain: %s", d->mailLocalDomain);
			nentries = 0;
			while ((rc = ldap_result(ld, d->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY) {
				nentries++;
				if (verbose)
					print_entries_with_values(ld, result);
				for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
					if ((vals = ldap_get_values(ld, e, "spamStrategy")) != NULL) {
						d->spam_strategy = get_token(spam_strategy_tab, vals[0], TOK_UNKNOWN);
						if (d->spam_strategy == TOK_UNKNOWN) {
							warnx("unknown spam strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
							d->spam_strategy = STRATEGY_UNSET;
						}
						ldap_value_free(vals);
					}

					if ((vals = ldap_get_values(ld, e, "virusStrategy")) != NULL) {
						d->virus_strategy = get_token(virus_strategy_tab, vals[0], TOK_UNKNOWN);
						if (d->virus_strategy == TOK_UNKNOWN) {
							warnx("unknown virus strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
							d->virus_strategy = STRATEGY_UNSET;
						}
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "spamSubjectPrefix")) != NULL) {
						d->spamprefix = strdup(vals[0]);
						ldap_value_free(vals);
					}
				}
				ldap_msgfree(result);
			}
		}	
#endif
	
		message_cleanup(ctx);
		
	} else {
		syslog_r(LOG_ERR, &sdata, "unable to access internal data, %m");
		retval = SMFIS_TEMPFAIL;
	}
	
	return retval;
}


static sfsistat
vilter_abort(SMFICTX *ctx)
{
	struct syslog_data sdata = SYSLOG_DATA_INIT;
#ifdef WITH_LDAP
	struct privdata *priv;
	struct recipient *r;
	struct domain *d;
	LDAPMessage *result;
	int rc;
#endif
	if (verbose) sdata.log_stat |= LOG_PERROR;

	syslog_r(LOG_INFO, &sdata, "message aborted");
	
#ifdef WITH_LDAP
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		SLIST_FOREACH(r, &priv->rhead, entries) {
			while ((rc = ldap_result(ld, r->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY)
				ldap_msgfree(result);
		}	
		SLIST_FOREACH(d, &priv->dhead, entries) {
			while ((rc = ldap_result(ld, d->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY)
				ldap_msgfree(result);
		}
	}
#endif
	
	message_cleanup(ctx);
	/* vilter_cleanup(ctx); */
	
	return SMFIS_CONTINUE;
}

static sfsistat
vilter_close(SMFICTX *ctx)
{
	struct privdata *priv;
	struct connection *conn;
	if (verbose) {
		if ((priv = smfi_getpriv(ctx)) != NULL) {
			conn = priv->conn;
			
			if (conn != NULL && conn->hostname != NULL) {
				warnx("closing connection with %s", conn->hostname);
				goto cleanup;
			}
		}
		
		warnx("closing connection");
	}
	
cleanup:
	connection_cleanup(ctx);
	return SMFIS_CONTINUE;
}
	
#ifdef WITH_LDAP
int
get_rebind_credentials(LDAP *ld, char **dnp, char **pwp, int *authmethodp, int freeit)
{
	warnx("get_rebind_credentials() called");
	return LDAP_SUCCESS;
}
#endif

int
main(int argc, char *argv[])
{
	smfiDesc_str desc;
	int ch;
	FILE *fp;
	struct stat statbuf;
	char libname[MAX_FILE];
	struct passwd *passwd;
	struct group *grp;
	uid_t uid;
	gid_t gid;
	struct rlimit rl;
	int n;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	int status;
	pid_t child_pid;
	time_t now, last_start;
	
	/* Set initial values */

	port = NULL;
	tmpdir = NULL;
	tmpfiles_gr = 0;
	tmpfiles_setgrp = 0;
	pidfile = NULL;
	user = NULL;
	group = NULL;
	chrootdir = NULL;
	cfgfile = CFGFILE;
	backend = NULL;
	virus_strategy = STRATEGY_UNSET;
	spam_strategy = STRATEGY_UNSET;
	error_strategy = STRATEGY_UNSET;
	log_facility = LOG_UNSET;
	logfile = NULL;
	recipient_notification = NULL;
	notify_only = NULL;
	markall = 0;
	rlimit_nofile = -1;
	spamprefix = NULL;
	logvirus = logspam = logunwanted = 0;
	keep_tmpfiles = 0;
#ifdef WITH_LDAP
	ldaphost = NULL;
	ldapport = -1;
	searchbase = NULL;
	binddn = NULL;
	bindpasswd = NULL;
#endif
	
	/* Process the commandline */

#ifdef WITH_LDAP	
	while ((ch = getopt(argc, argv, "vmp:d:T:s:S:U:b:l:e:n:o:C:P:t:u:g:Vf:ka:D:h:L:w:B:?")) != -1) {
#else
	while ((ch = getopt(argc, argv, "vmp:d:T:s:S:U:b:l:e:n:o:C:P:t:u:g:Vf:ka:?")) != -1) {
#endif
		switch (ch) {
			case 'v':
				++verbose;
				break;
			case 'm':
				markall = 1;
				break;
			case 'p':
				port = optarg;
				break;
			case 'd':
				tmpdir = optarg;
				break;
			case 'T':
				switch (get_token(tmpfiles_tab, optarg, TOK_UNKNOWN)) {
					case TF_GRPRD:
						tmpfiles_gr = 1;
						break;
					case TF_SETGRP:
						tmpfiles_setgrp = 1;
						break;
					default:
						errx(1, "unknown temp file option %s", optarg);
				}
				break;
			case 's':
				virus_strategy = get_token(virus_strategy_tab, optarg, TOK_UNKNOWN);
				if (virus_strategy == TOK_UNKNOWN)
					errx(1, "unknown virus-strategy %s", optarg);
				break;
			case 'S':
				error_strategy = get_token(error_strategy_tab, optarg, TOK_UNKNOWN);
				if (error_strategy == TOK_UNKNOWN)
					errx(1, "unknown error-strategy %s", optarg);
				break;
			case 'U':
				spam_strategy = get_token(spam_strategy_tab, optarg, TOK_UNKNOWN);
				if (error_strategy == TOK_UNKNOWN)
					errx(1, "unknow spam-strategy %s", optarg);
				break;
			case 'c':
				unwanted_strategy = get_token(unwanted_strategy_tab, optarg, TOK_UNKNOWN);
				if (unwanted_strategy == TOK_UNKNOWN)
					errx(1, "unknow unwanted-strategy %s", optarg);
				break;
			case 'b':
				backend = optarg;
				decode_backend(backend);
				break;
			case 'l':
				log_facility = get_token(lfacilities_tab, optarg, TOK_UNKNOWN);
				if (log_facility == TOK_UNKNOWN)
					errx(1, "unknown log facility %s", optarg);
				break;
			case 'e':
				logfile = optarg;
				break;
			case 'n':
				recipient_notification = optarg;
				break;
			case 'o':
				notify_only = optarg;
				break;
			case 'C':
				cfgfile = optarg;
				break;
			case 'P':
				pidfile = optarg;
				break;
			case 't':
				chrootdir = optarg;
				break;
			case 'u':
				user = optarg;
				break;
			case 'g':
				group = optarg;
				break;
			case 'V':
				printf("%s %s\n", __progname, VERSION);
#ifdef WITH_LDAP
				printf("(with LDAP support enabled)\n");
#endif
				exit(0);
			case 'f':
				rlimit_nofile = atoi(optarg);
				break; 
			case 'a':
				spamprefix = optarg;
				break;
			case 'k':
				keep_tmpfiles = 1;
				break;
#ifdef WITH_LDAP
			case 'D':
				binddn = optarg;
				break;
			case 'h':
				ldaphost = optarg;
				break;
			case 'L':
				ldapport = atoi(optarg);
				break;
			case 'w':
				bindpasswd = optarg;
				break;
			case 'B':
				searchbase = optarg;
				break;
#endif
			default:
				usage();
		}
	}
	
	/*
	argc -= optind;
	argv += optind;
	*/

	if (verbose) sdata.log_stat |= LOG_PERROR;
	
	/* Read config file */
	
	smtp_vilter_init();

	/* Set default values if some variables are not set */

	if (port == NULL)
		port = "unix:/var/run/smtp-vilter.sock";
	if (tmpdir == NULL)
		tmpdir = "/tmp";
	if (pidfile == NULL)
		pidfile = PIDFILE;
	if (backend == NULL)
		backend = "clamd";
	if (virus_strategy == STRATEGY_UNSET)
		virus_strategy = STRATEGY_DISCARD;
	if (spam_strategy == STRATEGY_UNSET)
		spam_strategy = STRATEGY_MARK;
	if (error_strategy == STRATEGY_UNSET)
		error_strategy = STRATEGY_TEMPFAIL;
	if (log_facility == LOG_UNSET)
		log_facility = LOG_MAIL;

#ifdef WITH_LDAP
	if (ldaphost == NULL)
		ldaphost = LDAPHOST;
	if (ldapport == -1)
		ldapport = LDAPPORT;
		
	if ((ld = ldap_init(ldaphost, ldapport)) == NULL)
		errx(1, "No directory server at %s, %m", ldaphost);
	
	/* Setup LDAP rebind, eventually continue with system defaults if
	   the initial bind fails? */
	   
	if (ldap_simple_bind_s(ld, binddn, bindpasswd) != LDAP_SUCCESS)
		errx(1, "Failed to bind to directory server as %s, %m", binddn);
	
	ldap_set_rebind_proc(ld, get_rebind_credentials, NULL);
	
	if (verbose)
		warnx("Directory server at %s server is up", ldaphost);
#endif

	openlog_r(__progname, verbose ? LOG_PID | LOG_PERROR : LOG_PID, log_facility, &sdata);

	/* Load the backends */

	for (n = 0; n < nbe; n++) {
		snprintf(libname, sizeof(libname), "libvilter-%s.so.%d", be[n].name, MAJOR);

		if (verbose)
			warnx("loading backend %s from file %s", be[n].name, libname);
			
		if ((be[n].dlhandle = dlopen(libname, RTLD_LAZY)) == NULL)
			err(1, "error loading backend %s (%s), %s", be[n].name, libname, dlerror());
		
#ifdef __ELF__
		be[n].be_init = dlsym(be[n].dlhandle, "vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "vilter_name")) == NULL)
			errx(1, "no function vilter_name in backend %s", be[n].name);

		be[n].be_new = dlsym(be[n].dlhandle, "vilter_new");	/* Optional */
		be[n].be_end = dlsym(be[n].dlhandle, "vilter_end");	/* Optional */
		
		if ((be[n].be_type = dlsym(be[n].dlhandle, "vilter_type")) == NULL)
			errx(1, "no function vilter_type in backend %s", be[n].name);
		
		switch (be[n].be_type()) {
			case SCAN_VIRUS:	
				if ((be[n].be_scan.be_scan_virus = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_SPAM:
				if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_UNWANTED:
				if ((be[n].be_scan.be_scan_unwanted = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			default:
				errx(1, "backend %s has an unknown vilter type", be[n].name);
		}
			

		be[n].be_exit = dlsym(be[n].dlhandle, "vilter_exit");
#else	/* assume a.out format */
		be[n].be_init = dlsym(be[n].dlhandle, "_vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "_vilter_name")) == NULL)
			errx(1, "no function vilter_name in backend %s", be[n].name);

		be[n].be_new = dlsym(be[n].dlhandle, "_vilter_new");	/* Optional */
		be[n].be_end = dlsym(be[n].dlhandle, "_vilter_end");	/* Optional */

		if ((be[n].be_type = dlsym(be[n].dlhandle, "_vilter_type")) == NULL)
			errx(1, "no function vilter_type in backend %s", be[n].name);
		
		switch (be[n].be_type()) {
			case SCAN_VIRUS:	
				if ((be[n].be_scan.be_scan_virus = dlsym(be[n].dlhandle, "_vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_SPAM:
				if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle, "_vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_UNWANTED:
				if ((be[n].be_scan.be_scan_unwanted = dlsym(be[n].dlhandle, "_vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			default:
				errx(1, "backend %s has an unknown vilter type", be[n].name);
		}
			

		be[n].be_exit = dlsym(be[n].dlhandle, "_vilter_exit");
#endif
		if (be[n].be_init != NULL) {
			if (be[n].be_init(be[n].config_file))
				errx(1, "error initializing backend %s", be[n].name);
		}
	}
				
	uid = getuid();
	gid = getgid();
		
	if (user != NULL) {
		if ((passwd = getpwnam(user)) != NULL) {
			uid = passwd->pw_uid;
			gid = passwd->pw_gid;
		} else
			err(1, "no such user '%s'", user);
		
		if (group != NULL) {
			if ((grp = getgrnam(group)) != NULL)
				gid = grp->gr_gid;
			else
				err(1, "no such group '%s'", group);
		}
	}
	
	if (!stat(pidfile, &statbuf))
		errx(1, "pid file %s exists, another copy running?", pidfile);

	if (!verbose) {
		if (daemon(0, 0))
			err(1, "can't run as daemon");
	}
	
	/* Fork into two processes, one to remove the PID file, one to do the
	   work. */
	
	do {
		child_pid = 0;
		if (!verbose)
			child_pid = fork();

		if (child_pid == 0) {
#ifndef LINUX	
			setproctitle(NULL);
#endif		
			/* Set ressource limits */

			if (rlimit_nofile > 0) {
				rl.rlim_cur = rl.rlim_max = rlimit_nofile;

				if (setrlimit(RLIMIT_NOFILE, &rl))
					err(1, "can set number of files to %d", rlimit_nofile);
			}

			if (logfile != NULL) {
				if ((logfp = fopen(logfile, "a")) == NULL)
					err(1, "can't open logfile %s", logfile);
			} else
				logfp = NULL;

			if ((fp = fopen(pidfile, "w")) == NULL) 
				err(1, "can't open pidfile %s", pidfile);
			if (fprintf(fp, "%d\n", getpid()) == -1)
				err(1, "can't write pid to pidfile %s", pidfile);
			if (fclose(fp))
				err(1, "error closing pidfile %s", pidfile);

			/* Chroot, if necessary */

			if (chrootdir) {
				if (getuid())
					errx(1, "must be started by root if using chroot");
				if (chroot(chrootdir))
					err(1, "can't chroot to %s", chrootdir);
				if (chdir("/"))
					err(1, "can't chdir to / after chroot to %s", chrootdir);
			}

			/* Change the user and group id */

			setgid(gid);
			setuid(uid);

			if (!getuid() || !getgid())
				errx(1, "must not be run as root");

			syslog_r(LOG_INFO, &sdata, "dropped privileges, running as %d:%d", uid, gid);

			/* Initialize sendmail milter */

			desc.xxfi_name = "smtp-vilter";
			desc.xxfi_version = SMFI_VERSION;
			desc.xxfi_flags = SMFIF_ADDHDRS | SMFIF_CHGHDRS;

			/*
			if (virus_strategy == STRATEGY_NOTIFY_RECIPIENT)
				desc.xxfi_flags |= SMFIF_CHGBODY;
			*/
			desc.xxfi_flags |= SMFIF_CHGBODY;

			desc.xxfi_connect = smtp_vilter_connect;	
			desc.xxfi_helo = NULL;
			desc.xxfi_envfrom = smtp_vilter_envfrom;
			desc.xxfi_envrcpt = smtp_vilter_envrcpt;
#ifdef WITH_LDAP
			desc.xxfi_flags |= SMFIF_DELRCPT;
			/* desc.xxfi_envrcpt = smtp_vilter_envrcpt; */
#else
			/*
			if (virus_strategy == STRATEGY_NOTIFY_RECIPIENT && notify_only != NULL)
				desc.xxfi_envrcpt = smtp_vilter_envrcpt;
			else
				desc.xxfi_envrcpt = NULL;
			*/
#endif
			desc.xxfi_header = vilter_header;
			desc.xxfi_eoh = vilter_eoh;
			desc.xxfi_body = vilter_body;
			desc.xxfi_eom = vilter_eom;
			desc.xxfi_abort = vilter_abort;
			desc.xxfi_close = vilter_close;

			if (smfi_register(desc) == MI_FAILURE)
				err(1, "failed to register with sendmail");	
		
			if (!strncasecmp(port, "unix:", 5) || !strncasecmp(port, "local:", 6))
				unlink(strchr(port, ':') + 1);
				
			smfi_setconn(port);
			if (smfi_main() == MI_FAILURE)
				syslog_r(LOG_ERR, &sdata, "failed to establish a connection on port %s", port);

			for (n = 0; n < nbe; n++) {
				if (be[n].be_exit != NULL)
					be[n].be_exit();
				free(be[n].name);
				dlclose(be[n].dlhandle);
			}
			free(be);

			syslog_r(LOG_INFO, &sdata, "exiting");
			fclose(logfp);

			if (verbose)
				warnx("exiting, please remove pidfile %s manually", pidfile);
			_exit(2);

		} else {
			time(&last_start);
			if (!verbose) {
				syslog_r(LOG_INFO, &sdata, "started child process with pid %d", child_pid);
#ifndef LINUX
				setproctitle("[priv]");
#endif
				wait(&status);
				time(&now);
				sleep(1);
			} else
				now = last_start;
			
		}

		if (unlink(pidfile))
			syslog_r(LOG_ERR, &sdata, "failed to unlink pidfile %s, %m", pidfile);
	} while (!verbose);
	
/*
	if (!verbose && (now - last_start <= 3))
		syslog_r(LOG_ERR, &sdata, "respawning to fast.  try verbose mode first");
*/
		
#ifdef WITH_LDAP
	ldap_unbind(ld);
#endif	
	closelog_r(&sdata);
	
	return 0;
}
