/*
 * Copyright (c) 2003, 2004 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <err.h>
#include <syslog.h>

#include "smtp-vilter.h"

#ifdef LINUX
#include "strlfunc.h"
#endif

#define	SAVSE_CONF	"/etc/smtp-vilter/savse.conf"

#define TIMEOUT 	60
#define MAXBUF		1024
#define MAXSTR		64
#define MAXTRIES	256

int savse_port;
char *savse_host;
int savse_tries;
int savse_timeout;
int chroot_scanrealpath;

int
vilter_init(char *cfgfile)
{
	FILE *fp;
	char field[MAXSTR];
	char value[MAXSTR];

	/* Default values */
	
	savse_port = 7777;
	if ((savse_host = strdup("localhost")) == NULL)
		err(1, "out of memory");
	savse_tries = 1;
	savse_timeout = TIMEOUT;
	chroot_scanrealpath = 0;
	
	if (cfgfile == NULL)
		cfgfile = SAVSE_CONF;
		
	if ((fp = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("using configuration from file %s", cfgfile);
			
		while (!read_conf(fp, field, sizeof(field), value, sizeof(value))) {
			if (!strcmp(field, "host")) {
				if ((savse_host = strdup(value)) == NULL)
					err(1, "out of memory");
				if (verbose)
					warnx("setting savse host to %s", savse_host);
			} else if (!strcmp(field, "port")) {
				savse_port = atoi(value);
				if (verbose)
					warnx("setting savse port to %d", savse_port);
			} else if (!strcmp(field, "tries")) {
				savse_tries = atoi(value);
				if (savse_tries < 1)
					savse_tries = 1;
				else if (savse_tries > MAXTRIES)
					savse_tries = MAXTRIES;
				if (verbose)
					warnx("setting nr of tries to %d", savse_tries);
			} else if (!strcmp(field, "timeout")) {
				savse_timeout = atoi(value);
				if (savse_timeout < 0)
					savse_timeout = 0;
				if (verbose)
					warnx("savse: setting timeout to %d", savse_timeout);
			} else if (!strcmp(field, "option")) {
				if (!strcmp(value, "chroot-scanrealpath")) {
					chroot_scanrealpath = 1;
					if (verbose)
						warnx("savse: scanning realpath when chrooted");
				} else
					warnx("savse: unknown option %s", value);
			} else if (verbose)
				warnx("unknown command %s=%s", field, value);
		}
		fclose(fp);
	} else if (verbose)
		warnx("configuration file %s for savse backend not found, using default values", cfgfile);
	
	return 0;
}

int
vilter_scan(struct be_data *priv, char *fn, char *chroot, char *virus, int namelen)
{
	int fd;
	int try;
	FILE *fp;
	struct sockaddr_in server_sockaddr;
	struct timeval tv;
	struct hostent *hostent;
	char buf[MAXBUF];
	int infection_count;
			
	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog_r(LOG_ERR, priv->sdata, "savse: unable to obtain network, %m");
		return SCAN_ERROR;
	}
  
	bzero((char *) &server_sockaddr, sizeof(server_sockaddr));
		
	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(savse_port);

	if (inet_aton(savse_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(savse_host);
		if (!hostent) {
			syslog_r(LOG_ERR, priv->sdata, "savse: unknown host: %s, %m", savse_host);
			return SCAN_ERROR;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
	}

	try = 0;
	while (connect(fd, (struct sockaddr *) &server_sockaddr, sizeof(server_sockaddr)) == -1) {
		++try;
		
		if (try == savse_tries) {
			syslog_r(LOG_ERR, priv->sdata, "savse: unable to connect socket, %m");
			return SCAN_ERROR;
		} else
			sleep(1);
	}

	if (savse_timeout > 0) {
		tv.tv_sec = savse_timeout;
		tv.tv_usec = 0;

		setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
		setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
	}	
	
	
	if (!(fp = fdopen(fd, "a+"))) {
		syslog_r(LOG_ERR, priv->sdata, "savse: can't open output, %m");
		return SCAN_ERROR;
	}
	
	if (!fgets(buf, sizeof(buf), fp)) {
		syslog_r(LOG_ERR, priv->sdata, "savse: no response from savse, %m");
		fclose(fp);
		return SCAN_ERROR;
	}
	
	if (strncmp(buf, "220", 3)) {
		syslog_r(LOG_ERR, priv->sdata, "savse: savse not ready, %s", buf);
		fclose(fp);
		return SCAN_ERROR;
	}

	if (!fgets(buf, sizeof(buf), fp)) {
		syslog_r(LOG_ERR, priv->sdata, "savse: no response from savse, %m");
		fclose(fp);
		return -SCAN_ERROR;
	}
	
	if (buf[0] != '2') {
		syslog_r(LOG_ERR, priv->sdata, "savse: savse version number mismatch");
		fclose(fp);
		return SCAN_ERROR;
	}

	if (chroot_scanrealpath && chroot != NULL)
		fprintf(fp, "Version 2\r\nAVSCANLOCAL\r\n%s/%s\r\n", chroot, *fn == '/' ? fn + 1 : fn);
	else
		fprintf(fp, "Version 2\r\nAVSCANLOCAL\r\n%s\r\n", fn);
	
	if (!fgets(buf, sizeof(buf), fp)) {
		syslog_r(LOG_ERR, priv->sdata, "savse: no response from savse, %m");
		fclose(fp);
		return SCAN_ERROR;
	}
	
	if (buf[0] != '2') {
		syslog_r(LOG_ERR, priv->sdata, "savse: error, %s", buf);
		fclose(fp);
		return SCAN_ERROR;
	}

	if (!fgets(buf, sizeof(buf), fp)) {
		syslog_r(LOG_ERR, priv->sdata, "savse: no response from savse, %m");
		fclose(fp);
		return SCAN_ERROR;
	}
	
	/* Bad coding practice on the next 20 lines... */
	
	if (buf[0] != '0') {
		
		fgets(buf, sizeof(buf), fp);	/* definitons date */
		fgets(buf, sizeof(buf), fp);	/* definitions version */
		fgets(buf, sizeof(buf), fp);	/* infection count */
		
		infection_count = atoi(buf);
		
		if (infection_count > 0) {
			fgets(buf, sizeof(buf), fp);	/* filename */
			fgets(buf, sizeof(buf), fp);	/* virusname */
			buf[strlen(buf) -2] = '\0';
			
			/* ignore the rest... */
			
			strlcpy(virus, buf, namelen);
		}
		
		fclose(fp);
		return SCAN_VIRUS;
	}
		
	fclose(fp);
	
	return SCAN_OK;
}

char *
vilter_name(void)
{
	return "Symantec AntiVirus Scan Engine, native protocol (savse)";
}

int
vilter_type(void)
{
	return SCAN_VIRUS;
}

void
vilter_exit(void)
{
	free(savse_host);
}
