/*
 * Copyright (c) 2003, 2004 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <err.h>
#include <syslog.h>

#include "smtp-vilter.h"

#ifdef LINUX
#include "strlfunc.h"
#endif

#define	CLAMD_CONF	"/etc/smtp-vilter/icap.conf"

#define TIMEOUT 	60
#define MAXBUF		1024
#define MAXSTR		64
#define MAXTRIES	256

#define SCANTYPE_FILESYSTEM	1
#define SCANTYPE_SOCKET		2

int clamd_port;
char *clamd_host;
int clamd_tries;
int clamd_timeout;
int scantype;
int chroot_scanrealpath;


int
vilter_init(char *cfgfile)
{
	FILE *fp;
	char field[MAXSTR];
	char value[MAXSTR];

	/* Default values */
			
	clamd_port = 3310;
	if ((clamd_host = strdup("localhost")) == NULL)
		err(1, "clamd: out of memory");
	clamd_tries = 1;
	clamd_timeout = 0;
	scantype = SCANTYPE_FILESYSTEM;
	chroot_scanrealpath = 0;
	
	if (cfgfile == NULL)
		cfgfile = CLAMD_CONF;
		
	if ((fp = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("clamd: using configuration from file %s", cfgfile);
			
		while (!read_conf(fp, field, sizeof(field), value, sizeof(value))) {
			if (!strcmp(field, "host")) {
				free(clamd_host);
				if ((clamd_host = strdup(value)) == NULL)
					err(1, "clamd: out of memory");
				if (verbose)
					warnx("clamd: setting clamd host to %s", clamd_host);
			} else if (!strcmp(field, "port")) {
				clamd_port = atoi(value);
				if (verbose)
					warnx("clamd: setting clamd port to %d", clamd_port);
			} else if (!strcmp(field, "tries")) {
				clamd_tries = atoi(value);
				if (clamd_tries < 1)
					clamd_tries = 1;
				else if (clamd_tries > MAXTRIES)
					clamd_tries = MAXTRIES;
				if (verbose)
					warnx("clamd: setting nr of tries to %d", clamd_tries);
			} else if (!strcmp(field, "timeout")) {
				clamd_timeout = atoi(value);
				if (clamd_timeout < 0)
					clamd_timeout = 0;
				if (verbose)
					warnx("clamd: setting timeout to %d", clamd_timeout);
			} else if (!strcmp(field, "scantype")) {
				if (!strcmp(value, "filesystem")) {
					scantype = SCANTYPE_FILESYSTEM;
					if (verbose)
						warnx("clamd: set scantype to filesystem");
				} else if (!strcmp(value, "socket")) {
					scantype = SCANTYPE_SOCKET;
					if (verbose)
						warnx("clamd: set scantype to socket");
				} else if (verbose)
					warnx("clamd: unknown scantype %s, using default", value);
			} else if (!strcmp(field, "option")) {
				if (!strcmp(value, "chroot-scanrealpath")) {
					chroot_scanrealpath = 1;
					if (verbose)
						warnx("clamd: scanning realpath when chrooted");
				} else
					warnx("clamd: unknown option %s", value);
			} else if (verbose)
				warnx("clamd: unknown command %s=%s", field, value);
		}
		fclose(fp);
	} else if (verbose)
		warnx("clamd: configuration file %s for clamd backend not found, using default values", cfgfile);
	
	return 0;
}

int
vilter_newmsg(struct be_data *priv, char *env_from, FILE *fp)
{
	/*
		Write "From <sender>" to the temporary file so that clamd
		recognizes the file as e-mail
	*/
	
	fprintf(fp, "From %s\r\n", env_from);
	
	return 0;
}

int
vilter_scan(struct be_data *priv, char *fn, char *chroot, char *virus, int namelen)
{
	int fd, datfd;
	int try;
	int nread;
	FILE *fp;
	unsigned short port;
	struct sockaddr_in server_sockaddr, data_sockaddr;
	struct timeval tv, *t;
	struct hostent *hostent;
	char buf[MAXBUF];
	char *p, *q;
		
	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog_r(LOG_ERR, priv->sdata, "clamd: unable to obtain network, %m");
		return SCAN_ERROR;
	}

  
	bzero((char *) &server_sockaddr, sizeof(server_sockaddr));
		
	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(clamd_port);

	if (inet_aton(clamd_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(clamd_host);
		if (!hostent) {
			syslog_r(LOG_ERR, priv->sdata, "clamd: unknown host: %s, %m", clamd_host);
			close(fd);
			return SCAN_ERROR;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
	}

	try = 0;
	
	while (connect(fd, (struct sockaddr *) &server_sockaddr, sizeof(server_sockaddr)) == -1) {
		++try;
		
		if (try == clamd_tries) {
			syslog_r(LOG_ERR, priv->sdata, "clamd: unable to connect socket, %m");
			close(fd);
			return SCAN_ERROR;
		} else
			sleep(1);
	}

	
	if (clamd_timeout > 0) {
		tv.tv_sec = clamd_timeout;
		tv.tv_usec = 0;
		t = &tv;
	} else
		t = NULL;
		
	switch (scantype) {
		case SCANTYPE_FILESYSTEM:
			if (chroot_scanrealpath && chroot != NULL) {
				snprintf(buf, sizeof(buf), "SCAN %s/%s\r\n", chroot, *fn == '/' ? fn + 1 : fn);
			} else
				snprintf(buf, sizeof(buf), "SCAN %s\r\n", fn);
			
			if (to_send(fd, buf, strlen(buf), 0, t) != strlen(buf)) {
				syslog_r(LOG_ERR, priv->sdata, "clamd: send error, %m");
				close(fd);
				return SCAN_ERROR;
			}
			break;
		case SCANTYPE_SOCKET:
			if (to_send(fd, "STREAM\r\n", 8, 0, t) != 8) {
				syslog_r(LOG_ERR, &sdata, "clamd: %m");
				close(fd);
				return SCAN_ERROR;
			}
			if ((nread = to_recv(fd, buf, sizeof(buf), 0, t)) == -1 || nread == 0) {
				syslog_r(LOG_ERR, priv->sdata, "clamd: no response to STREAM cmd, %m");
				close(fd);
				return SCAN_ERROR;
			}
			
			if (sscanf(buf, "PORT %hu\n", &port) != 1) {
				syslog_r(LOG_ERR, priv->sdata, "clamd: did not receive port number");
				close(fd);
				return SCAN_ERROR;
			}

			if (verbose)
				warnx("sending data to port %d", port);
				
			if ((datfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
				syslog_r(LOG_ERR, priv->sdata, "clamd: unable to obtain network, %m");
				close(fd);
				return SCAN_ERROR;
			}

			bzero((char *) &data_sockaddr, sizeof(data_sockaddr));

			data_sockaddr.sin_family = AF_INET;
			data_sockaddr.sin_port = htons(port);

			if (inet_aton(clamd_host, &data_sockaddr.sin_addr) == 0) {
				hostent = gethostbyname(clamd_host);
				if (!hostent) {
					syslog_r(LOG_ERR, priv->sdata, "clamd: unknown host: %s, %m", clamd_host);
					close(datfd);
					close(fd);
					return SCAN_ERROR;
				}
				data_sockaddr.sin_family = hostent->h_addrtype;
				memcpy(&data_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
			}

			try = 0;

			while (connect(datfd, (struct sockaddr *) &data_sockaddr, sizeof(data_sockaddr)) == -1) {
				++try;

				if (try == clamd_tries) {
					syslog_r(LOG_ERR, priv->sdata, "clamd: unable to connect socket, %m");
					close(datfd);
					close(fd);
					return SCAN_ERROR;
				} else
					sleep(1);
			}
			
			fp = fopen(fn, "r");
			
			while ((nread = fread(buf, 1, sizeof(buf), fp)) > 0)
				if (to_send(datfd, buf, nread, 0, t) != nread) {
					syslog_r(LOG_ERR, priv->sdata, "clamd: error sending file, %m");
					close(datfd);
					close(fd);
					fclose(fp);
					return SCAN_ERROR;
				}
			fclose(fp);	
			close(datfd);
			
			break;
		default:
			syslog_r(LOG_ERR, &sdata, "clamd: internal error, unknown scantype %d", scantype);
			close(fd);
			return SCAN_ERROR;
	}
		
	if ((nread = to_recv(fd, buf, sizeof(buf), 0, t)) == -1 || nread == 0) {
		syslog_r(LOG_ERR, priv->sdata, "clamd: no response from clamd, %m");
		close(fd);
		return SCAN_ERROR;
	}
	
	close(fd);
			
	if ((p = strrchr(buf, ' ')) == NULL) {
		syslog_r(LOG_ERR, priv->sdata, "clamd: unable to interpret result '%s'", buf);
		return SCAN_ERROR;
	} else {
		++p;
		
		if (!strncmp(p, "OK", 2)) {/* File is OK */
			if (verbose)
				warnx("message contains no virus");
			return SCAN_OK;
		} else if (!strncmp(p, "FOUND", 5)) {
			*--p = '\0';
			q = strrchr(buf, ' ') + 1;
			syslog_r(LOG_ERR, priv->sdata, "clamd: found '%s'", q);
			strlcpy(virus, q, namelen);
			return SCAN_VIRUS;
		} else if (!strncmp(p, "ERROR", 5)) {
			*--p = '\0';
			syslog_r(LOG_ERR, priv->sdata, "clamd: error '%s'", buf);
			return SCAN_ERROR;
		} else {
			syslog_r(LOG_ERR, priv->sdata, "clamd: unrecognized response: '%s'", p);
		}
 	}
		
	return SCAN_ERROR;
}

char *
vilter_name(void)
{
	return "ICAP Vilter (icap)";
}

int
vilter_type(void)
{
	return SCAN_VIRUS;
}

void
vilter_exit(void)
{
	free(clamd_host);
}
