/*	$Id: smtpsink.c,v 1.5 2006/06/24 11:22:48 mbalmer Exp $	*/

/*
 * Copyright (c) 2003, 2004, 2005, 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <ctype.h>
#include <dirent.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "pathnames.h"

#define SERVER_PORT	25
#define MAXLEN		1024

int verbose;

__dead void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-v] [-p port]\n", __progname);
	exit(1);
}

void
send_line(FILE *fp, char *line)
{	
	fprintf(fp, "%s\r\n", line);

	if (verbose)
		printf("-> %s\n", line);
}

void *
handle_connection(void *arg)
{
	int data = 0, terminate = 0;
	char buf[MAXLEN];
	FILE *fp = (FILE *)arg;

	send_line(fp, "220 SMTPSINK (all data will be lost)");

	do {
		if (!fgets(buf, sizeof(buf), fp))
			terminate = 1;
		else {
			if (verbose)
				printf("<- %s", buf);

			if (data) {
				if (strlen(buf) <= 3 && buf[0] == '.') {
					data = 0;
					send_line(fp, "250 ACCEPTED");
				}
			} else {
				if (!strncasecmp(buf, "QUIT", 4)) {
					send_line(fp, "221 BYE");
					terminate = 1;
				} else if (!(strncasecmp(buf, "RCPT TO", 7)))
					send_line(fp, "250 RCPT OK");
				else if (!(strncasecmp(buf, "DATA", 4))) {
					data = 1;
					send_line(fp, "354 SEND DATA");
				} else
					send_line(fp, "250 OK");
			}
		}
	} while (!terminate);

	fclose(fp);

	return NULL;
}

int
main(int argc, char *argv[])
{
	int fd, s;
	int len;
	volatile int true;
	struct sockaddr_in ec;
	struct sockaddr_in server_sockaddr;
	int port = SERVER_PORT;
	int ch;
	pthread_t tid;
	pthread_attr_t attr;
	FILE *fp;
	struct passwd *passwd;
	char *ep;

	while ((ch = getopt(argc, argv, "p:v")) != -1) {
		switch (ch) {
		case 'p':
			port = (int)strtol(optarg, &ep, 10);
			if (port <= 0 || *ep != '\0') {
				warnx("illegal number, -p argument -- %s",
				    optarg);
				usage();
			}
			if (port < 1024 && getuid())
				errx(1, "must be root to use port < 1024");
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
		}
	}

	argc -= optind;
	argv += optind;

	signal(SIGPIPE, SIG_IGN);

	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1)
		err(1, "unable to obtain network");
 
	if ((setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *)&true,
	    sizeof(true))) == -1)
		err(1, "setsockopt failed");

	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(port);
	server_sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  
	if (bind(s, (struct sockaddr *)&server_sockaddr,
	    sizeof(server_sockaddr)) == -1) 
		err(1, "bind to socket failed");

	/* Take precaution if we are running as root */
	if (!getuid()) {
		if ((passwd = getpwnam("nobody")) == NULL)
			err(1, "can't change to user nobody");

		if (chroot(ROOT_PATH))
			err(1, "can't chroot to %s", ROOT_PATH);

		if (chdir("/"))
			err(1, "can't chdir to /");

		if (setuid(passwd->pw_uid))
			err(1, "setuid failed");
	}

	if (listen(s, 16) == -1)
		err(1, "deaf");

	if (verbose)
		printf("smtpsink: listen on port %d for connections\n", port);

	while (1) {
 		len = sizeof(ec);
   
		if ((fd = accept(s, (void *)&ec, &len)) == -1)
			err(5, "unable to accept connection");

		if ((fp = fdopen(fd, "r+")) == NULL)
			err(5, "unable to open input");
		if (verbose)
			handle_connection(fp);
		else if (pthread_create(&tid, &attr, handle_connection, fp))
			err(1, "can't create thread");
	}
}
