%{
/*	$Id: parse.y,v 1.2 2006/06/06 06:43:00 mbalmer Exp $	*/

/*
 * Copyright (c) 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <syslog.h>

extern FILE	*yyin;
extern int	 yylineno;
extern char	*yytext;

extern char	*cfgfile;
extern char	*ldaphost;
extern int	 ldapport;
extern char	*ldapurl;
extern char	*searchbase;
extern char	*binddn;
extern char	*bindpasswd;
extern char	*spamheader;
extern char	*pagerattr;
extern char	*query;
extern int	 ldap_use_tls;
extern int	 log_facility;

static int	 yyerrcnt;

int	yyerror(const char *, ...);
int	yyparse(void);
int	yylex(void);
%}

%union {
	long number;
	char *string;
}

%token	LOGFAC LDAPHOST LDAPPORT LDAPURL BINDDN BINDPASSWD SEARCHBASE
%token	QUERY USETLS LOCAL0 LOCAL1 LOCAL2 LOCAL3
%token	LOCAL4 LOCAL5 LOCAL6 LOCAL7 COMMENT NEVER TRY ALWAYS
%token	DAEMON MAIL USER PAGERID SPAMHEADER
%token	<string>	NUMBER
%token	<string>	TEXT
%type	<number>	logfac

%%
statement	: /* empty */
		| statement '\n'
		| statement logfacility '\n'
		| statement ldaphost '\n'
		| statement ldapport '\n'
		| statement ldapurl '\n'
		| statement searchbase '\n'
		| statement query '\n'
		| statement binddn '\n'
		| statement bindpasswd '\n'
		| statement usetls '\n'
		| statement pagerid '\n'
		| statement spamheader '\n'
		;

ldaphost	: LDAPHOST '=' TEXT			{
			if (ldaphost == NULL)
				ldaphost = $3;
			else
				free($3);
		}
		;

ldapport	: LDAPPORT '=' NUMBER			{
			if (ldapport == -1) {
				const char *errstr;

				ldapport = (int)strtonum($3, 1LL, 65535LL,
				    &errstr);
				if (errstr)
					errx(1, "port number is %s:%s", errstr,
					    $3);
			}
			free($3);
		}
		;

ldapurl		: LDAPURL '=' TEXT			{
			if (ldapurl == NULL)
				ldapurl = $3;
			else
				free($3);
		}
		;

searchbase	: SEARCHBASE '=' TEXT		{
			if (searchbase == NULL)
				searchbase = $3;
			else
				free($3);
		}
		;

query		: QUERY '=' TEXT		{
			if (query == NULL)
				query = $3;
			else
				free($3);
		}
		;

binddn		: BINDDN '=' TEXT		{
			if (binddn == NULL)
				binddn = $3;
			else
				free($3);

		}
		;

bindpasswd	: BINDPASSWD '=' TEXT		{
			if (bindpasswd == NULL)
				bindpasswd = $3;
			else
				free($3);
		}
		;

pagerid		: PAGERID '=' TEXT		{
			if (pagerattr == NULL)
				pagerattr = $3;
			else
				free($3);
		}
		;

spamheader	: SPAMHEADER '=' TEXT		{
			if (spamheader == NULL && (*$3 == 'x' || *$3 == 'X'))
				spamheader = $3;
			else
				free($3);
		}
		;



logfacility	:	LOGFAC '=' logfac	{
				log_facility = $3;
		}
		;

logfac		:	LOCAL0		{ $$ = LOG_LOCAL0; }
		|	LOCAL1		{ $$ = LOG_LOCAL1; }
		|	LOCAL2		{ $$ = LOG_LOCAL2; }
		|	LOCAL3		{ $$ = LOG_LOCAL3; }
		|	LOCAL4		{ $$ = LOG_LOCAL4; }
		|	LOCAL5		{ $$ = LOG_LOCAL5; }
		|	LOCAL6		{ $$ = LOG_LOCAL6; }
		|	LOCAL7		{ $$ = LOG_LOCAL7; }
		|	DAEMON		{ $$ = LOG_DAEMON; }
		|	MAIL		{ $$ = LOG_MAIL; }
		|	USER		{ $$ = LOG_USER; }
		;
	
usetls		: USETLS '=' NEVER		{
			if (ldap_use_tls == -1)
				ldap_use_tls = 0;
		}
		| USETLS '=' TRY		{
			if (ldap_use_tls == -1)
				ldap_use_tls = 1;
		}
		| USETLS '=' ALWAYS		{
			if (ldap_use_tls == -1)
				ldap_use_tls = 2;
		}
		;
%%

void
smsmail_init(void)
{
	yylineno = 1;
	yyerrcnt = 0;
	if ((yyin = fopen(cfgfile, "r")) != NULL) {
		while (!feof(yyin))
			yyparse();
		fclose(yyin);
		if (yyerrcnt)
			errx(1, "configuration file contains errors, "
			    "terminating");
	}
}

int
yyerror(const char *fmt, ...)
{
	va_list		 ap;
	char		*nfmt;

	++yyerrcnt;
	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s, line %d: %s near '%s'",
	    cfgfile, yylineno, fmt, yytext) == -1)
		errx(1, "yyerror asprintf");
	va_end(ap);
	fprintf(stderr, "%s\n", nfmt);
	free(nfmt);
	return (0);
}
