/*
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2004 Brian Carrier.  All rights reserved
 *
 * gpt: GUID Partition Tables 
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mmtools; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

// @@@ FIx this reference
#include <inttypes.h>

#include "mm_tools.h"
#include "mymalloc.h"
#include "error.h"

#include "gpt.h"
#include "dos.h"


/* 
 * Process the partition table at the sector address 
 * 
 * It is loaded into the internal sorted list 
 */
static void 
gpt_load_table(MM_INFO *mm, daddr_t sect)
{
	gpt_head head;
	gpt_entry *ent;
	dos_sect dos_part;
	char *ent_buf;	/* buffer for each table entry */
	int ent_size, i;
	off_t ret;
	char *safe_str, *head_str, *tab_str;
	off_t addr = (off_t)sect * 512;

	if (verbose)
		fprintf(logfp,
		  "gpt_load_table: Sector: %lu\n", (ULONG)sect);

	/* seek to the table */
	if  (addr != (ret = lseek(mm->fd, addr, SEEK_SET))) {
		error ("Error seeking to table sector: %lu  Ret: %lu", 
		  (ULONG)addr, (ULONG)ret);
	}



	/* Read the DOS partition table to make sure it exists */
	if (sizeof(dos_part) != read (mm->fd, (char *)&dos_part, sizeof(dos_part))) {
		error ("Error reading DOS safety partition table sector %lu\n",
		  (ULONG)sect);
	}

	/* Sanity Check */
	if (guessu16(mm, dos_part.magic, DOS_MAGIC)) {
		error ("Missing DOS safety partition (invalid magic) (Sector: %lu)\n",
		   (ULONG)sect);
	}

	if (dos_part.ptable[0].ptype != GPT_DOS_TYPE) {
		error ("Missing DOS safety partition (invalid type in table: %d)", 
				dos_part.ptable[0].ptype);
	}
	safe_str = mymalloc(16);
	snprintf(safe_str, 16, "Safety Table");
	mm_part_add(mm, 0, 1, safe_str, -1, -1);


	/* Read the GPT header */
	if (sizeof(head) != read (mm->fd, (char *)&head, sizeof(head))) {
		error ("Error reading GPT Header structure: %lu",
		  (ULONG)sect+1);
	}

	if (getu64(mm, &head.signature) != GPT_HEAD_SIG) {
		error ("Invalid signature in GPT Header: %"PRIx64"\n",
		  getu64(mm, &head.signature));
	}

	head_str = mymalloc(16);
	snprintf(head_str, 16, "GPT Header");
	mm_part_add(mm, 1, (getu32(mm, &head.head_size_b)+511)/512, head_str, -1, -1);

	/* Allocate a buffer for each table entry */
	ent_size = getu32(mm, &head.tab_size_b);
	if (ent_size < sizeof(gpt_entry)) {
		error ("Header reports partition entry size of %d\n",
		  ent_size);
	}	
	ent_buf = mymalloc(ent_size);
	ent = (gpt_entry *)ent_buf;

	tab_str = mymalloc(20);
	snprintf(tab_str, 20, "Partition Table");
	mm_part_add(mm, getu32(mm, &head.tab_start_lba), 
	  (ent_size * getu32(mm, &head.tab_num_ent) + 511) / 512, 
	  tab_str, -1, -1);


	/* Process the partition table */
	if  ((getu64(mm, &head.tab_start_lba)*512) != 
	  (ret = lseek(mm->fd, getu64(mm, &head.tab_start_lba)*512, SEEK_SET))) {
		error ("Error seeking to partition table: %"PRIu64"  Ret: %lu", 
		  (ULONG)getu64(mm, &head.tab_start_lba)*512, (ULONG)ret);
	}
	

	for (i = 0; i < getu32(mm, &head.tab_num_ent); i++) {
		char *name;

		if (ent_size != read (mm->fd, ent_buf, ent_size)) {
			error ("Error reading GPT Table Entry: %d (offset: %"PRIu64")",
					i, getu64(mm, &head.tab_start_lba)*512 + i*ent_size);
		}

		if (verbose)
			fprintf (logfp, 
			  "gpt_load: %d  Starting Sector: %"PRIu64"  End: %"PRIu64" Flag: %"PRIx64"\n",
			  i, 
			  getu64(mm, ent->start_lba),
			  getu64(mm, ent->end_lba),
			  getu64(mm, ent->flags));


		if (getu64(mm, ent->start_lba) == 0)
			continue;


		name = mymalloc(72);
		uni2ascii(ent->name, 72, (u_int8_t *)name, 72);
		mm_part_add(mm, getu32(mm, ent->start_lba),
		  getu32(mm, ent->end_lba) - getu32(mm, ent->start_lba) + 1,
		  name, -1, i);
	}

	return;
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
gpt_part_walk(MM_INFO *mm, u_int16_t start, u_int16_t last, int flags, 
  MM_PART_WALK_FN action, char *ptr)
{
	MM_PART *part;
	int cnt = 0;

	if (start < mm->first_part || start > mm->last_part)
		error ("Invalid starting partition: %d", start);

	if (last < mm->first_part || last > mm->last_part)
		error ("Invalid ending partition: %d", last);

	part = mm->part_list;
	while ((part != NULL) && (cnt <= last)) {

		if (cnt >= start)
			action(mm, cnt, part, 0, ptr);

		part = part->next;
		cnt++;
	}

	return;
}


void 
gpt_close (MM_INFO *mm)
{
	mm_part_free(mm);
	close(mm->fd);
	free(mm);
}

MM_INFO *
gpt_open(const char *path, unsigned char type, daddr_t sect_offset)
{
	char *myname = "gpt_open";

	MM_INFO *mm = (MM_INFO *)mymalloc(sizeof(*mm));

	if (type != MM_GPT)
		error ("%s: Invalid media type for GPT tools", myname);

	if ((mm->fd = open(path, O_RDONLY)) < 0)
		error("%s: open %s: %m", myname, path);

	/* Use the supplied offset if given */
	if (sect_offset)
		mm->sect_offset = sect_offset;
	else
		mm->sect_offset = GPT_PART_OFFSET;

	/* inititialize settings */
	mm->mmtype = type;
	mm->str_type = "GUID Partition Table";
	mm->part_list = NULL;
	mm->flags = 0;

	/* Assign functions */
	mm->part_walk = gpt_part_walk;
	mm->close = gpt_close;

	/* Load the partitions into the sorted list */
	gpt_load_table(mm, mm->sect_offset);

	/* fill in the sorted list with the 'unknown' values */
	mm_part_unused(mm);

	return mm;
}
