/*
** The  Sleuth Kit 
**
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2003 Brian Carrier.  All rights reserved 
**
** TASK
** Copyright (c) 2002 Brian Carrier, @stake Inc.  All rights reserved
** 
** Copyright (c) 1997,1998,1999, International Business Machines          
** Corporation and others. All Rights Reserved.
*/

/* TCT */
/*++
 * NAME
 *	icat 1
 * SUMMARY
 *	copy files by inode number
 * SYNOPSIS
 * .ad
 * .fi
 *	\fBicat\fR [\fB-hHvV\fR] [\fB-f \fIfstype\fR]
 *		\fIdevice\fR \fIinode\fR ...
 * DESCRIPTION
 *	\fBicat\fR opens the named \fIdevice\fR and copies the files
 *	with the specified \fIinode\fR numbers to standard output.
 *
 *	 Arguments:
 * .IP "\fB-f \fIfstype\fR"
 *	Specifies the file system type. The default file system type
 *	is system dependent. With most UNIX systems the default type
 *	is \fBufs\fR (Berkeley fast file system). With Linux the default
 *	type is \fBext2fs\fR (second extended file system).
 * .IP \fB-h\fR
 *	Skip over holes in files, so that absolute address information
 *	is lost. This option saves space when copying sparse files.
 * .IP "\fB-H\fR (default)"
 *	Copy holes in files as null blocks, so that absolute address
 *	information is preserved. This option wastes space when copying
 *	sparse files.
 * .IP \fB-v\fR
 *	Enable verbose mode, output to stderr.
 * .IP \fB-V\fR
 *	Enable verbose mode, output to stdout.
 * .IP \fIdevice\fR
 *	Disk special file, or regular file containing a disk image.
 *	On UNIX systems, raw mode disk access may give better performance
 *	than block mode disk access.  LINUX disk device drivers support
 *	only block mode disk access.
 * .IP \fIinode\fR
 *	Inode number. \fBicat\fR concatenates the contents of all specified
 *	files.
 * BUGS
 *	\fBicat\fR should support more file system types. Right now, support
 *	is limited to \fBext2fs\fR when built on Linux, and \fBufs\fR when
 *	built on Solaris and BSD systems.
 * LICENSE
 *	This software is distributed under the IBM Public License.
 * AUTHOR(S)
 *	Wietse Venema
 *	IBM T.J. Watson Research
 *	P.O. Box 704
 *	Yorktown Heights, NY 10598, USA
 --*/

#include "fs_tools.h"
#include "error.h"
#include "ntfs.h"	// for data types

FILE   *logfp;
extern char *progname;

/* usage - explain and terminate */

static void 
usage()
{
    printf("usage: %s [-hHsvV] [-f fstype] device [inum[-typ[-id]] ...]\n", progname);
	printf("\t-h: Do not display holes\n");
	printf("\t-H: Display holes (default)\n");
	printf("\t-s: Display slack space as well\n");
	printf("\t-v: verbose to stderr\n");
	printf("\t-V: Print version\n");
    printf("\t-f fstype: Image file system type\n");
	printf("Supported file system types:\n");
	fs_print_types();

	exit(1);
}

/* Call back action for file_walk
 */
static u_int8_t
icat_action(FS_INFO *fs, DADDR_T addr, char *buf, int size, 
  int flags, char *ptr)
{
	if (size == 0)
		return WALK_CONT;

	if (fwrite(buf, size, 1, stdout) != 1)
		error("icat_action: write: %m");

	return WALK_CONT;
}

int     main(int argc, char **argv)
{
    FS_INFO *fs;
    char   	*cp;
    INUM_T  inum;
    int     flags = FS_FLAG_ALLOC | FS_FLAG_UNALLOC;
    int     ch;
    char   	*fstype = DEF_FSTYPE;
	FS_INODE *inode;

    progname = argv[0];

    while ((ch = getopt(argc, argv, "f:hHsvV")) > 0) {
	switch (ch) {
	default:
	    usage();
	case 'f':
	    fstype = optarg;
	    break;
	case 'h':
	    flags &= ~FS_FLAG_UNALLOC;
	    break;
	case 'H':
	    flags |= FS_FLAG_UNALLOC;
	    break;
	case 's':
	    flags |= FS_FLAG_SLACK;
		break;
	case 'v':
	    verbose++;
	    logfp = stderr;
	    break;
	case 'V':
	    print_version();
		exit(0);
	}
    }

    if (argc < optind + 2)
		usage();

    fs = fs_open(argv[optind], fstype);

    while (argv[++optind]) {
		int type = 0;
		int id = 0;
		char 	*dash;
		/* This is just the single inode usage */
		if ((dash = strchr(argv[optind], '-')) == NULL) {
			inum = STRTOUL(argv[optind], &cp, 0);
			if (*cp || cp == argv[optind])
				usage();
		}
		/* now we have the inum-type or inum-type-id format */
		else {
			char *dash2;
			*dash = '\0';
			dash++;

			if ((dash2 = strchr(dash, '-')) == NULL) {
				id = 0;
			}
			else {
				*dash2 = '\0';
				dash2++;

				id = STRTOUL(dash2, &cp, 0);
				if (*cp || cp == dash2)
					usage();
			}

			inum = STRTOUL(argv[optind], &cp, 0);
			if (*cp || cp == argv[optind])
				usage();

			type = STRTOUL(dash, &cp, 0);
			if (*cp || cp == dash)
				usage();
		}

		inode = fs->inode_lookup(fs, inum);
		if (!inode)
			error ("error getting inode");

		fs->file_walk(fs, inode, type, id, 
		  flags | FS_FLAG_CONT | FS_FLAG_ALLOC | FS_FLAG_UNALLOC, 
			  icat_action, "");

		fs_inode_free(inode);
    }
    fs->close(fs);
    exit(0);
}
