/*
** Copyright (c) 1986, 1994, 1996, 2000, 2002
**	Jeff Forys (jeff@forys.cranbury.nj.us).  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that: (1) Redistributions of
** source code must retain the above copyright notice, this list of
** conditions and the following disclaimer, (2) Redistributions in
** binary form must reproduce the above copyright notice, this list
** of conditions and the following disclaimer in the documentation
** and/or other materials provided with the distribution, (3) All
** advertising materials mentioning features or use of this software
** must display the following acknowledgment: ``This product includes
** software developed by Jeff Forys (jeff@forys.cranbury.nj.us).'', (4)
** The name of the author may not be used to endorse or promote products
** derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** HP-UX 6.5 support by Christos Zoulas (christos@deshaw.com)
*/

#ifndef lint
static char rcsid[] = "$Id: hpux-70.c,v 1.11 2002/03/21 18:20:42 forys Exp $";
#endif

#define	WOPR	/* some headers do magic when this is defined */

#define	NO_MEXTERN
#include "conf.h"
#undef	NO_MEXTERN

#include <sys/user.h>
#include <sys/proc.h>
#include <sys/rtprio.h>
 
#include <stdio.h>

/*
 * Define SigNames, NSig, and TtyDevDir here; they are used by other
 * routines and must be global.  Everyone seems to have their own
 * idea as to what NSIG should be.  Here, `NSig' is the number of
 * signals available, not counting zero.
 */
char *SigMap[] = { "0",
	"HUP", "INT", "QUIT", "ILL", "TRAP", "IOT",		/*  1 -  6 */
	"EMT", "FPE", "KILL", "BUS", "SEGV", "SYS",		/*  7 - 12 */
	"PIPE", "ALRM", "TERM", "USR1", "USR2", "CLD",		/* 13 - 18 */
	"PWR", "VTALRM", "PROF", "IO", "WINDOW", "STOP",	/* 19 - 24 */
	"TSTP", "CONT", "TTIN", "TTOU", "URG", "LOST",		/* 25 - 30 */
	"31", "DIL",						/* 31 - 32 */
};
int NSig = NSIG-1;

#define	SETCMD(dst,src,maxlen) {			\
	extern char *strrchr();				\
	if (maxlen > 0) src[maxlen] = '\0';		\
	dst = (dst = strrchr(src, '/')) ? ++dst: src;	\
}

static char *TtyDevDir = "/dev";

int	Skill;			/* set 1 if running `skill', 0 if `snice' */
int	PrioMin, PrioMax;	/* min and max process priorities */
int	SigPri;			/* signal to send or priority to set */
pid_T	MyPid;			/* pid of this process */
uid_T	MyUid;			/* uid of this process */
char	*ProgName;		/* program name */

/*
 * This is the machine-dependent initialization routine.
 *
 *   - The following global variables must be initialized:
 *     MyPid, MyUid, ProgName, Skill, PrioMin, PrioMax, SigPri
 *   - The working directory will be changed to that which contains the
 *     tty devices (`TtyDevDir'); this makes argument parsing go faster.
 *   - If possible, this routine should raise the priority of this process.
 */
void
MdepInit(pname)
	char *pname;
{
	extern char *rindex(), *SysErr();

	MyPid = (pid_T) getpid();
	MyUid = (uid_T) getuid();
	SETCMD(ProgName, pname, 0)

	/*
	 * If we are running as root, raise our priority to better
	 * catch runaway processes.
	 */
	if (MyUid == ROOTUID)
		(void) rtprio(MyPid, RTPRIO_MIN);

	/*
	 * Determine what we are doing to processes we find.  We will
	 * either send them a signal (skill), or renice them (snice).
	 */
	Skill = (strcmp(ProgName, "snice") != 0);

	/*
	 * chdir to `TtyDevDir' to speed up tty argument parsing.
	 */
	if (chdir(TtyDevDir) < 0) {
		fprintf(stderr, "%s: chdir(%s): %s\n", ProgName, TtyDevDir,
		        SysErr());
		exit(EX_SERR);
	}

	/*
	 * Set up minimum and maximum process priorities.
	 * Initialize SigPri to either default signal (`skill') or
	 * default priority (`snice').
	 */
	PrioMin = RTPRIO_MIN;
	PrioMax = RTPRIO_MAX;
	SigPri = Skill? SIGTERM: RTPRIO_RTOFF;
}

/*
 * Carry out an action on a particular process.  If this is `skill',
 * then send the process a signal, otherwise this is `snice' so change
 * it's priority.
 *
 * If 0 is returned, the operation was successful, otherwise -1 is
 * returned and `errno' set.
 */
int
MdepAction(pid)
	pid_T pid;
{
	if (Skill)
		return(kill((int)pid, SigPri));
	else
		return(rtprio((int)pid, SigPri));
}

/* disable regular expresions: regex(3) availability unknown */
NULL_REGEX_FUNCS

/*
 * Now, set up everything we need to write a GetProc() routine.
 */

#include <sys/file.h>
#include <sys/vm.h>

#include <machine/pte.h>

#ifdef hp9000s800
#include <nlist.h>
#endif

static char *kmemf =	"/dev/kmem";	/* window into kernel virtual memory */
static char *memf =	"/dev/mem";	/* window into physical memory */
static char *swapf =	"/dev/swap";	/* paging device */
static char *kernf =	"/hp-ux";	/* kernel image */
static int kmem = 0, mem = 0, swap = 0;

#ifdef hp9000s800
static struct nlist nl[] = {
	{ "nproc" },
#define	X_NPROC		0
	{ "proc" },
#define	X_PROC		1
	{ "ubase" },
#define	X_UBASE		2
	{ "npids" },
#define	X_NPIDS		3
	{ "" },
#define	X_LAST		4
};
#else
static struct nlist nl[] = {
	{ "_nproc" },
#define	X_NPROC		0
	{ "_proc" },
#define	X_PROC		1
	{ "" },
#define	X_LAST		2
};
#endif

static	int	nproc = -1;
static	struct	proc *procp;
#ifdef hp9000s800
static	int	npids;
static	struct user *ubase;
#endif

#define	NPROCS	32			/* number of procs to read at once */

#ifdef hp9000s800
static	char	*pidmap[] = { "swapper", "init", "pagedaemon",
		              "statdaemon", "syncdaemon" };
#else
static	char	*pidmap[] = { "swapper", "init", "pagedaemon" };
#endif
static	int	pidmapsiz = sizeof(pidmap) / sizeof(pidmap[0]);

extern	off_t lseek();

/*
 * GetProc()
 *
 * Fill in and return a `struct ProcInfo' with information about the
 * next process.  If no processes are left, return NULL.
 */
struct ProcInfo *
GetProc()
{
	extern char *SysErr();
	static struct user *GetUser();
	static struct proc procs[NPROCS], *procsp;
	static struct ProcInfo procinfo;
	register struct user *auser;
	register struct proc *aproc;
	static int thisproc = 0;

	/*
	 * If this is our first time here, open various files,
	 * and set up the nlist.
	 */
	if (nproc == -1) {
		char *errstr = "%s: %s: %s\n";
		int nfound;

		if ((kmem=open(kmemf, 0)) < 0) {	/* open kmem */
			fprintf(stderr, errstr, ProgName, kmemf, SysErr());
			exit(EX_SERR);
		}

		if ((mem=open(memf, 0)) < 0) {		/* open mem */
			fprintf(stderr, errstr, ProgName, memf, SysErr());
			exit(EX_SERR);
		}

		if ((swap=open(swapf, 0)) < 0) {	/* open swap device */
			fprintf(stderr, errstr, ProgName, swapf, SysErr());
			exit(EX_SERR);
		}

		if ((nfound=nlist(kernf, nl)) < 0) {	/* kernel name list */
			fprintf(stderr, errstr, ProgName, kernf,"no name list");
			exit(EX_SERR);
		}
		if (nfound != 0) {
			register int i;

			fprintf(stderr, "%s: nlist: unresolved symbols:",
			        ProgName);
			for (i = 0; i < X_LAST; i++)
				if (nl[i].n_type == 0)
					fprintf(stderr, " %s", nl[i].n_name);
			(void) putc('\n', stderr);
			exit(EX_SERR);
		}

		procp = (struct proc *)GetWord((off_t)nl[X_PROC].n_value);
		nproc = GetWord((off_t)nl[X_NPROC].n_value);
#ifdef hp9000s800
		npids = GetWord((off_t)nl[X_NPIDS].n_value);
		ubase = (struct user *)nl[X_UBASE].n_value;
#endif
	}

	/*
	 * Read in NPROCS proc structures at-a-time.  Decrement `nproc'
	 * by the number of proc structures we have read; when it reaches
	 * zero, we are finished (return NULL).
	 */
	do {
		while (thisproc == 0) {
			int nread;
			int psize;

			if (nproc == 0)
				return((struct ProcInfo *)NULL);

			thisproc = MIN(NPROCS, nproc);
			psize = thisproc * sizeof(struct proc);
			nproc -= thisproc;
			if (lseek(kmem, (off_t)procp, L_SET) == -1 ||
			    (nread = read(kmem, (char *)procs, psize)) < 0) {
				fprintf(stderr, "%s: read proc: %s\n",
				        ProgName, SysErr());
				return((struct ProcInfo *)NULL);
			} else if (nread != psize) {
				thisproc = nread / sizeof(struct proc);
				nproc = 0;
				fprintf(stderr, "%s: read proc: short read\n",
				        ProgName);
			}
			procsp = procs;
			procp += thisproc;
		}

		aproc = procsp++;
		thisproc--;

		if (aproc->p_stat != 0) {
			/*
			 * Before we go through the trouble of reading
			 * in the user struct, let's make sure this isn't
			 * a "zombie" or "exiting" process.  If it is,
			 * we have all the information we need; fill in
			 * procinfo and return.
			 */
			procinfo.pi_flags = 0;
			procinfo.pi_pid = (pid_T) aproc->p_pid;
			procinfo.pi_uid = (uid_T) aproc->p_uid;

			if (aproc->p_stat == SZOMB) {		/* zombie */
				static char *zombie = "<defunct>";
				procinfo.pi_flags |= PI_ZOMBIE;
				procinfo.pi_cmd = zombie;
			} else if (aproc->p_flag & SWEXIT) {	/* exiting */
				static char *exiting = "<exiting>";
				procinfo.pi_flags |= PI_SWEXIT;
				procinfo.pi_cmd = exiting;
			}

			if (procinfo.pi_flags)
				return(&procinfo);
			else
				auser = GetUser(aproc);
		}

	} while (aproc->p_stat == 0 || auser == NULL);

	/*
	 * We now have a process (`aproc') and a user (`auser').
	 * Fill in the rest of `procinfo'.
	 */
	if (auser->u_ttyp != 0) {	/* has a controlling tty */
		procinfo.pi_flags |= PI_CTLTTY;
		procinfo.pi_tty = (tty_T) auser->u_ttyd;
	}

	if (aproc->p_pid < pidmapsiz) {	/* special */
		procinfo.pi_cmd = pidmap[aproc->p_pid];
		procinfo.pi_flags |= PI_ASKUSR;
	} else				 /* set path-stripped command name */
		SETCMD(procinfo.pi_cmd, auser->u_comm, MAXCOMLEN)

	return(&procinfo);
}

#define	SKRD(file, src, dst, size)			\
	(lseek(file, (off_t)(src), L_SET) == -1) ||	\
	(read(file, (char *)(dst), (size)) != (size))

/*
 * GetWord(loc)
 *
 * Read in word at `loc' from kernel virtual memory.
 * If an error occurs, call exit(2) with EX_SERR.
 */
static int
GetWord(loc)
	off_t loc;
{
	int val;

	if (SKRD(kmem, loc, &val, sizeof(val))) {
		fprintf(stderr, "%s: can't read word at %lx in %s\n",
		        ProgName, (u_long)loc, kmemf);
		exit(EX_SERR);
	}
	return (val);
}

#ifdef hp9000s800
#define	SW_UADDR	dtob(aproc->p_swaddr+ctod(btoc(STACKSIZE*NBPG)))
#define	SW_UBYTES	sizeof(struct user)
#else
#define	SW_UADDR	dtob(aproc->p_swaddr)
#define	SW_UBYTES	ctob(btoc(sizeof(struct user)))
#endif

/*
 * GetUser(aproc)
 *
 * Read in the user struct for `aproc' and return a pointer to it.
 * If an error occurs, return NULL.
 */
static struct user *
GetUser(aproc)
	struct proc *aproc;
{
	static char *WarnMsg = "Warning: can't read ";
	static union {
		struct user user;
		char upgs[UPAGES][NBPG];
	} u;
#ifndef hp9000s800
	static struct pte uptes[UPAGES];
	register int i, nbytes;
#endif

	/*
	 * If process is not in core, we simply snarf it's user struct
	 * from the swap device.
	 */
	if ((aproc->p_flag & SLOAD) == 0) {
		if (SKRD(swap, SW_UADDR, &u.user, SW_UBYTES)) {
			if (Wflag)
				printf("%su for pid %d from %s\n",
				       WarnMsg, aproc->p_pid, swapf);
			return ((struct user *)NULL);
		}
		return (&u.user);
	}

#ifdef hp9000s800
	/*
	 * User areas are all in kernel VM starting at `ubase'.
	 * uvadd() uses `ubase' and `npids' to determine location.
	 */
#undef	pindx
#define	pindx(p)	((p)->p_ndx)
	if (SKRD(kmem, uvadd(aproc), &u.user, sizeof(struct user))) {
		if (Wflag)
			printf("%suser area for pid %d from %s\n",
			       WarnMsg, aproc->p_pid, kmemf);
		return((struct user *)NULL);
	}
#else
	/*
	 * Process is in core.  Follow p_addr to read in the page
	 * table entries that map the u-area and then read in the
	 * physical pages that comprise the u-area.
	 *
	 * If at any time, an lseek() or read() fails, print a warning
	 * message (if `Wflag' is set) and return NULL.
	 */
	if (SKRD(kmem, aproc->p_addr, uptes, sizeof(uptes))) {
		if (Wflag)
			printf("%suser pt for pid %d from %s\n",
			       WarnMsg, aproc->p_pid, kmemf);
		return ((struct user *)NULL);
	}

	nbytes = sizeof(struct user);
	for (i = 0; i < UPAGES && nbytes > 0; i++) {
		if (SKRD(mem, ptob(uptes[i].pg_pfnum), u.upgs[i], NBPG)) {
			if (Wflag)
				printf("%suser page %u for pid %d from %s\n",
				       WarnMsg, uptes[i].pg_pfnum,
				       aproc->p_pid, memf);
			return((struct user *)NULL);
		}
		nbytes -= NBPG;
	}
#endif
	return (&u.user);
}
