#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "../xposix/xposix.h"

static const char* progname;

static unsigned long int block_size;
static unsigned long int count;

static const char * const usage_msg = "\
\n\
Usage: %s <flags>\n\
\n\
    Flags:\n\
\n\
        bs=<block_size> : Set the block size.\n\
          count=<count> : Set the number of blocks to copy from stdin\n\
                          to stdout.\n\
                     -h : Print help.\n\
\n\
    About:\n\
\n\
        Version: 1.7.2\n\
        License: BSD\n\
         Author: Paul Serice\n\
         E-Mail: paul@serice.net\n\
            URL: http://www.serice.net/shunt/\n\
\n\
";

static void
show_usage(FILE* f)
{
    fprintf(f, usage_msg, progname);
}

static int
is_negative(const char* s)
{
    int rv = 0;
    for( ; *s != '\0' ; ++s ) {
        if( isspace((int)*s) ) {
            continue;
        }
        if( *s == '-' ) {
            rv = 1;
        }
        break;
    }
    return rv;
}

/* Return true if s converted without error into an unsigned long.
 * The unsigned long will be stored in n.  If the conversion failed,
 * false will be returned, and n will not be altered. */
static int
string_to_unsigned_long(const char* s, unsigned long int* n)
{
    char* eptr = NULL;
    int rv = 0;
    unsigned long int tmp = 0;
    if( (*s != '\0') && !is_negative(s)) {
        errno = 0;
        tmp = strtoul(s, &eptr, 0);
        if( (*eptr == '\0') && (errno == 0) ) {
            *n = tmp;
            rv = 1;
        }
    }
    return rv;
}

static void
decipher_command_line(int argc, char** argv)
{
    int i = 1;
    for( i = 1 ; i < argc ; ++i ) {
        const char* arg = argv[i];

        /*
         * Ignore empty arguments.
         */
        if( arg[0] == '\0' ) {
            continue;
        }

        /*
         * bs ==> block size
         */
        if( strncmp(arg, "bs=", 3) == 0 ) {
            if( !string_to_unsigned_long(arg + 3, &block_size)
                || (block_size == 0) )
            {
                fprintf(stderr,
                        "\n%s: Error: invalid block size: %s\n",
                        progname,
                        arg + 3);
                show_usage(stderr);
                exit(1);
            }
            continue;
        }

        /*
         * count
         */
        if( strncmp(arg, "count=", 6) == 0 ) {
            if( !string_to_unsigned_long(arg + 6, &count) ) {
                fprintf(stderr,
                        "\n%s: Error: invalid count: %s\n",
                        progname,
                        arg + 6);
                show_usage(stderr);
                exit(1);
            }
            continue;
        }

        /*
         * -h ==> help
         */
        if( strcmp(arg, "-h") == 0 ) {
            show_usage(stdout);
            exit(0);
        }

        /*
         * Warn if some other flag.
         */
        if( arg[0] == '-' ) {
            fprintf(stderr, "\n%s: Error: Invalid flag: %s\n", progname, arg);
            show_usage(stderr);
            exit(1);
        }

    }

}

int main(int argc, char* argv[])
{
    char* block = NULL;
    unsigned long int i = 0;
    unsigned long int rcount = 0;

    progname = xbasename(argv[0]);
    decipher_command_line(argc, argv);

    /* Make sure stdin and stdout are open as binary streams. */
    x_setmode(STDIN_FILENO, O_BINARY);
    x_setmode(STDOUT_FILENO, O_BINARY);

    /* Portably allocate the block. */
    block = xmalloc(block_size);

    /* Copy only the exact number of blocks from stdin to stdout.  The
     * idea is to avoid the buffering in the standard library. */
    for( i = 0 ; i < count ; ++i ) {
        /* I've written xread() and xwrite() to not return until they
         * have completed the entire read or write.  You have to keep
         * track of rcount in case the last block is short. */
        rcount = xread(STDIN_FILENO, (void*)block, block_size);
        xwrite(STDOUT_FILENO, (const void*)block, rcount);
    }

    free(block);

    return 0;
}
