/*	$Id: seal.c,v 1.4 2007/01/07 11:14:20 mbalmer Exp $ */

/*
 * Copyright (c) 2006 Marc Balmer <mbalmer@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>

#include <err.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "pathnames.h"

char *vnd_dev = "svnd2";
char *vnd_part = "a";
char *vnd_type = "ffs";
char *vnd_file = ".vnd";
char *mp1 = _PATH_MP1;
char *mp2 = _PATH_MP2;
int verbose = 0;

volatile sig_atomic_t quit = 0;

__dead void usage(void);
void sighdlr(int);
void seal(void);
void attach(char *);
void local_askpass(char *, size_t);
void handle(int);
void background(void);
void unseal(char *dev);
extern void seal_init(char *cfgfile);

enum modes {
	SEAL, UNSEAL, ATTACH, BACKGROUND
};

__dead void
usage(void)
{
	extern char *__progname;

	printf("usage: %s [-abuv] [-C path] [dev]\n", __progname);
	exit(1);
}

void
sighdlr(int signum)
{
	switch (signum) {
	case SIGTERM:
	case SIGINT:
		unlink(_PATH_SOCK);
		_exit(0);
	}
}

void
attach(char *dev)
{
	struct sockaddr_un addr;
	FILE *fp;
	int fd;

	fd = socket(AF_UNIX, SOCK_STREAM, 0);
	bzero(&addr, sizeof(addr));
	addr.sun_family = AF_UNIX;
	strlcpy(addr.sun_path, _PATH_SOCK, sizeof(addr.sun_path));
	if (connect(fd, (struct sockaddr *)&addr, sizeof(addr)))
		err(1, "can't connect");
	fp = fdopen(fd, "r+");
	if (fp == NULL)
		err(1, "can't fdopen");
	fprintf(fp, "attach %s\n", dev);
	fclose(fp);
	close(fd);
}

void
askpass(char *pass, size_t len)
{
	FILE *fp;
	char cmd[128];

	snprintf(cmd, sizeof(cmd), "%s \"Enter password for encrypted "
	    "filesystem\"", _PATH_SSH_ASKPASS);
	fp = popen(cmd, "r");
	if (fp == NULL)
		return;
	fgets(pass, len, fp);
	pclose(fp);
	if (strlen(pass))
		pass[strlen(pass) - 1] = '\0';
}

void
handle(int fd)
{
	FILE *fp;
	char cmd[128], dev[64];

	fp = fdopen(fd, "r+");
	if (fp == NULL)
		err(1, "can't fdopen");
	while (fgets(cmd, sizeof(cmd), fp)) {
		if (strlen(cmd) > 0)
			cmd[strlen(cmd) - 1] = '\0';
		if (!strncmp(cmd, "attach ", 7)) {
			if (verbose)
				syslog(LOG_INFO, "%s\n", cmd);
			strlcpy(dev, &cmd[7], sizeof(dev));
			unseal(dev);
		} else
			syslog(LOG_ERR, "unknown command %s", cmd);
	}
}

void
background(void)
{
	struct sockaddr_un saddr;
	struct sockaddr_storage addr;
	socklen_t len = sizeof(addr);
	int s, fd;

	s = socket(AF_UNIX, SOCK_STREAM, 0);
	bzero(&saddr, sizeof(saddr));
	saddr.sun_family = AF_UNIX;
	strlcpy(saddr.sun_path, _PATH_SOCK, sizeof(saddr.sun_path));
	if (bind(s, (struct sockaddr *)&saddr, sizeof(saddr)))
		err(1, "bind failed");
	if (listen(s, 2))
		err(1, "listen failed");

	chmod(_PATH_SOCK, S_IRWXU);

	if (daemon(0, 0))
		err(1, "can't daemonize");

	signal(SIGTERM, sighdlr);
	signal(SIGINT, sighdlr);

	while (!quit && (fd = accept(s, (struct sockaddr *)&addr, &len))
	    != -1) {
		len = sizeof(addr);
		handle(fd);
	}
}

void
unseal(char *dev)
{
	char cmd[128];
	char pass[128];
	FILE *fp;

	if (verbose)
		syslog(LOG_INFO, "unsealing %s\n", dev);
	snprintf(cmd, sizeof(cmd), "sudo mount -t %s /dev/%sa %s", vnd_type,
	    dev, mp1);
	system(cmd);
	sleep(1);
	askpass(pass, sizeof(pass));
	snprintf(cmd, sizeof(cmd), "sudo vnconfig -k %s %s/%s", vnd_dev, mp1,
	    vnd_file);
	fp = popen(cmd, "w");
	if (fp == NULL)
		syslog(LOG_ERR, "can't popen");
	sleep(1);
	fprintf(fp, "%s\n", pass);
	pclose(fp);
	bzero(pass, sizeof(pass));
	sleep(1);
	snprintf(cmd, sizeof(cmd), "sudo mount /dev/%sa %s", vnd_dev, mp2);
	system(cmd);
}

void
seal(void)
{
	char cmd[128];

	snprintf(cmd, sizeof(cmd), "sudo umount %s", mp2);
	system(cmd);
	snprintf(cmd, sizeof(cmd), "sudo vnconfig -u %s", vnd_dev);
	system(cmd);
	snprintf(cmd, sizeof(cmd), "sudo umount %s", mp1);
	system(cmd);
}

int
main(int argc, char *argv[])
{
	int ch, mode = SEAL;
	char *dev = NULL, *cfgfile = _PATH_CFGFILE;
	
	while ((ch = getopt(argc, argv, "a:bvC:")) != -1) {
		switch (ch) {
		case 'a':
			dev = optarg;
			mode = ATTACH;
			break;
		case 'b':
			mode = BACKGROUND;
			break;
		case 'u':
			mode = UNSEAL;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'C':
			cfgfile = optarg;
			break;
		default:
			usage();
		}
	}
	seal_init(cfgfile);

	switch (mode) {
	case SEAL:
		seal();
		break;
	case UNSEAL:
		unseal(dev);
		break;
	case ATTACH:
		attach(dev);
		break;
	case BACKGROUND:
		background();
		break;
	}
	return 0;
}
