/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _GNU_SOURCE
# define _GNU_SOURCE
#endif
#ifndef _REENTRANT
# define _REENTRANT
#endif

#include <smspdu.h>
#include "smscoding.h"
#include "smsudh.h"

#include <charsets.h>
#include <helper.h>
#include <gtincl.h>

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

static
char* sms_pdu_create_submit_frame (char* number,
				   struct smsopts* opts,
				   int header_present)
{
  uint8_t sca_len = 0;
  struct sms_number sca; //short message service center address
  uint8_t pdutype; //PDU type
  uint8_t mr; //message reference
  uint8_t da_len = 0;
  struct sms_number da; //destination address
  uint8_t pid = 0; //protocol identifier (normally 0)
  uint8_t dcs; //data coding scheme
  uint8_t vp_rel = 0xff; //validity period (0xff = max = 63 weeks)
  //char vp_abs[15]; //validity period (YYMMDDhhmmssTZ)

  unsigned int i;
  unsigned int k;
  char* frame = mem_alloc(176*2+1,1);
  char* temp;

  /* not yet supported
   * zero length means that the phone will use its default
   */
  struct_sms_number_init(&sca);
  sms_number_set(&sca,numtype(sca.digits),NULL);
  sca_len = strlen(sca.digits)/2;
  if (sca_len) {
    ++sca_len;
  }
  
  /* set PDU type to standard values
   */
  pdutype = 1; //message type SMS-SUBMIT
  pdutype |= (1<<4); //validity period format: relative
  if (opts->srr) {
    pdutype |= (1<<5); //status report request;
  }
  if (header_present) {
    pdutype |= (1<<6); //user data header indication
  }

  /* message reference must be zero for Siemens phones!
   */
  mr = 0;

  /* destination number
   */
  if (is_telephone_number(number) == 0) {
    fprintf(stderr,"%s: %s\n",_("Error"),_("No valid number specified."));
    return NULL;
  }
  struct_sms_number_init(&da);
  sms_number_set(&da,numtype(number),number);
  temp = sms_number_get(&da);
  if (temp == NULL || strcmp(temp,number) != 0) {
    fprintf(stderr,_("%s: sms number cannot have more than %d digits.\n"),_("Error"),sizeof(da.digits)-1);
    return NULL;
  }
  mem_realloc(temp,0);
  da_len = strlen(da.digits);

  /* support "flash-SMS" and unicode here
   */
  dcs = 0;
  if (opts->flash) {
    dcs &= 0xec; //clear all affected bits
    dcs |= 0x10; //set class 0 message (immediate display)
  }
  if (opts->unicode) {
    dcs &= 0xf3; //clear all affected bits
    dcs |= 0x08; //set unicode charset
  }

  //leave pid and vp_rel as is

  //create PDU frame
  sprintf(frame,"%02X",sca_len);
  if (sca_len) {
    sprintf(frame+2,"%02X%s",sms_number_get_toa(&sca),sca.digits);
  }
  sprintf(frame+strlen(frame),"%02X%02X%02X%02X",
	  pdutype,mr,da_len,sms_number_get_toa(&da));
  k = strlen(da.digits);
  for (i = 0; i < k; i += 2) {
    sprintf(frame+strlen(frame),"%c%c",
	    (i+1>=k && k%2) ? 'F' : da.digits[i+1],
	    da.digits[i]);
  }
  sprintf(frame+strlen(frame),"%02X%02X%02X",
	  pid,dcs,vp_rel);

  struct_sms_number_delete(&sca);
  struct_sms_number_delete(&da);

  return frame;
}

char** sms_pdu_create_submit (char* text, char* number,
			      struct smsopts* opts)
{
  ucs4char_t* wide_str = convert_from_system(text);
  char** pdu = sms_data_encode((opts->unicode) ? SMS_CHARSET_UCS2 : SMS_CHARSET_GSM,
			      NULL, wide_str);
  unsigned int i = 0;
  unsigned int k;
  char* frame;

  mem_realloc(wide_str,0);

  if (pdu == NULL) return NULL;

  frame = sms_pdu_create_submit_frame(number,opts,
				      (pdu[1] == NULL)? 0 : 1);
  if (frame == NULL) return NULL;

  k = strlen(frame);
  for (; pdu[i] != NULL; ++i) {
    pdu[i] = mem_realloc(pdu[i],(176*2)+1);
    memmove(pdu[i]+k,pdu[i],strlen(pdu[i])+1);
    memcpy(pdu[i],frame,k);
  }
  mem_realloc(frame,0);

  return pdu;
}
