/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMSPDU_H
#define SMSPDU_H

#include "charsets.h"
#include "timeincl.h"
#include "intincl.h"

/* various SMS options */
struct smsopts {
  //for sending
  char flash;
  char srr;
  char unicode;
  char direct;

  //for printing
  char* sort_mode;
};

#define SMS_NUMBER_TYPE_UNKNOWN  0
#define SMS_NUMBER_TYPE_INTERNAT 1
#define SMS_NUMBER_TYPE_NAT      2
#define SMS_NUMBER_TYPE_TEXT     5
#define SMS_NUMBER_PLAN_UNKNOWN  0
#define SMS_NUMBER_PLAN_ISDN     1  
#define SMS_NUMBER_TEXT_LEN   11
#define SMS_NUMBER_DIGITS_LEN 20
struct sms_number {
  struct { /* see ETSI 23.040, Ch. 9.1.2.5 */
    /* MSB is always 1! */
    unsigned int type :3; /* type of number */
    unsigned int plan :4; /* nummbering plan identification */
  } toa; /* type of address */

  char digits[SMS_NUMBER_DIGITS_LEN+1]; /* if toa.type != SMS_NUMBER_TYPE_TEXT */
  ucs4char_t* text;
};
void struct_sms_number_init (struct sms_number* s);
void struct_sms_number_delete (struct sms_number* s);
int sms_number_compare (const struct sms_number* s1, 
			const struct sms_number* s2);
void sms_number_set (struct sms_number* this, uint8_t toa, const char* number);
char* sms_number_get (const struct sms_number* this);
uint8_t sms_number_get_toa (const struct sms_number* this);

struct sms_pdu_time {
  enum {
    SMS_PDU_TIME_NONE = 0, /* ignore value */
    SMS_PDU_TIME_RELATIVE, /* value is relative */
    SMS_PDU_TIME_ABSOLUTE  /* value is absolute(GMT) */
  } format;
  time_t value;
};
void struct_sms_pdu_time_init (struct sms_pdu_time* s);
int sms_pdu_time_fill (struct sms_pdu_time* this,
			const char* format,
			const char* s, int tz);


enum sms_direction {
  SMS_INCOMING,
  SMS_OUTGOING
};

/* sms_direction implied by each value
 */
enum sms_pdu_type {
  SMS_TYPE_DELIVER,
  SMS_TYPE_SUBMIT_REPORT,
  SMS_TYPE_STATUS_REPORT,
  SMS_TYPE_DELIVER_REPORT,
  SMS_TYPE_SUBMIT,
  SMS_TYPE_COMMAND
};
char* sms_pdu_type_string (enum sms_pdu_type t);
/* type is the full PDU type field
 * but only the lowest two bits are really needed
 */
enum sms_pdu_type sms_pdu_type_get (enum sms_direction d, uint8_t type);


struct sms_pdu_options {
  enum sms_pdu_type type;
  unsigned int udh_present :1; /* needed by sms_udh_fill() */
  unsigned int mms :1; /* more messages to send indication */
  unsigned int rd :1; /* reject duplicate request */
  unsigned int rp :1; /* reply path request/indication */
  unsigned int sr :1; /* status report request/indication/qualifier */
};
void struct_sms_pdu_options_init (struct sms_pdu_options* s);

enum sms_encoding {
  SMS_CHARSET_GSM,
  SMS_CHARSET_UCS2,
  SMS_CHARSET_8BIT
};
enum sms_pdu_dcs_opt {
  SMS_DCS_OPT_NONE,
  SMS_DCS_OPT_CLASS, /* use class */
  SMS_DCS_OPT_IND    /* use indication */
};
struct sms_pdu_dcs {
  unsigned int dcs_present :1; /* needed for SMS-STATUS-REPORT (DCS is optional) */

  enum sms_encoding encoding;
  unsigned int compressed :1; /* =0 not compressed, !=0 compressed */
  unsigned int autodel :1; /* =0 normal mode, !=0 marked as auto-delete */

  enum sms_pdu_dcs_opt opttype;
  union {
    unsigned int class :2; /* 0-3, other values match SMS_DCS_OPT_NONE */
    struct {
      unsigned int sense :1; /* =0 inactive, !=0 active */
      enum {
	SMS_DCS_IND_STORE = 0,
	SMS_DCS_IND_DISCARD
      } group;
      enum {
	SMS_DCS_IND_OTHER = 0,
	SMS_DCS_IND_VOICE,
	SMS_DCS_IND_FAX,
	SMS_DCS_IND_EMAIL
      } type;
    } indication;
  } opt;
};
void struct_sms_pdu_dcs_init (struct sms_pdu_dcs* s);

struct sms_pdu_message_status {
  uint8_t s;
  struct sms_pdu_time t;
  struct sms_pdu_ud* message_parent;
  struct sms_pdu_data* status_parent;
};
void struct_sms_pdu_message_status_init(struct sms_pdu_message_status* s);

struct sms_pdu_ud_header {
  uint8_t type;
  uint8_t len;
  uint8_t data[137];
};
void struct_sms_pdu_ud_header_init (struct sms_pdu_ud_header* s);

struct sms_pdu_ud {
  /* a NULL terminated list */
  struct sms_pdu_ud_header** header;

  /* the text in the message */
  ucs4char_t* text;

  /* status of the message if a SMS-STATUS-REPORT can be matched
   * If the message type is SMS_STATUS_REPORT or no
   * SMS-STATUS-REPORT matches, this shall be NULL.
   * Never free this pointer from here, except that
   * ack->status_parent is NULL
   *
   * never free this from here
   */
  struct sms_pdu_message_status* ack;
};
void struct_sms_pdu_ud_init (struct sms_pdu_ud* s);
void struct_sms_pdu_ud_delete (struct sms_pdu_ud* s);

struct sms_pdu_data {
  struct sms_pdu_options options;

  /* message reference
   * This is not valid for all message types
   * (currently only SMS-SUBMIT and SMS-STATUS-REPORT)
   * and meaning depends on type, too.
   */
  uint8_t mref;

  struct sms_number address;

  uint8_t pid; /* protocol ID */
  struct sms_pdu_dcs scheme;

  struct sms_pdu_time timedata;

  uint16_t multipart_id; /* multipart short message id */
  uint8_t partnum; /* number of this part, count from 0 */

  /* ud is a list of size (parts*sizeof(*ud))
   * if parts is 0, ud should be NULL
   */  
  uint8_t parts;
  struct sms_pdu_ud* ud;

  /* service center status of a previously sent message
   * only valid for SMS-STATUS-REPORT
   */
  struct sms_pdu_message_status* status;
};
void struct_sms_pdu_data_init (struct sms_pdu_data* s);
void struct_sms_pdu_data_delete (struct sms_pdu_data* s);

struct sms_tpdu_data {
  struct sms_number sca; //service center address
  struct sms_pdu_data* pdu;
};
void struct_sms_tpdu_data_init (struct sms_tpdu_data* s);
void struct_sms_tpdu_data_delete (struct sms_tpdu_data* s);

struct sms_slot_data {
  int slot; /* >0 for true values */
  char tpdu[363]; /* as hexstring */
  enum sms_direction type;
};
void struct_sms_slot_data_init (struct sms_slot_data* s);

struct sms {
  struct sms_slot_data** encoded;
  struct sms_tpdu_data* decoded;
};
void struct_sms_init (struct sms* s,
		      struct sms_slot_data* encoded,
		      struct sms_tpdu_data* decoded);
void struct_sms_delete (struct sms* s);


#include <stdio.h>


/***************************
 * encoding short messages *
 ***************************/
/* create a new short message of type SMS-SUBMIT
 * return value is a TPDU allocated with malloc
 */
char** sms_pdu_create_submit (char* text, char* number, struct smsopts* opts);

/* both return length of pdu (pdu is part of tpdu)
 * This is needed as parameter for the at command for sending
 */
unsigned int sms_tpdu_len (enum sms_direction d,char* tpdu); //for TPDU (means: with SMSC field)
unsigned int sms_pdu_len (enum sms_direction d,char* pdu); //for PDU

/***************************
 * decoding short messages *
 ***************************/
/* STEP 1:
 * decode sms which is of type type
 * return value allocated with malloc
 */
struct sms_tpdu_data* sms_pdu_decode (enum sms_direction type,
				      struct sms_slot_data* sms);

/* STEP 2:
 * see struct_sms_init()
 */

/* STEP 3.1:
 * to match the status report messages to sent messages
 * The number of elements in list does not change with this call.
 * This MUST be done BEFORE merging concatenated message,
 * otherwise not all can be matched.
 *
 * If you don't need or want this, just omit it.
 */
void sms_statusreport_match (struct sms** list);

/* STEP 3.2:
 * to merge concatenated short messages
 * list may have less elements after this call
 *
 * If you don't need or want this, just omit it.
 */
int sms_multipart_merge (struct sms** list);

/* STEP 3.3:
 * order can currently be "type", "slot" or "type,slot"
 * The number of elements in list does not change with this call.
 * This can be done at any time in STEP 3.
 */
void sms_pdu_sort (struct sms** list, const char* order);

/* STEP 4:
 * print sms as text to fp
 */
void sms_pdu_print (FILE* fp, struct sms* sms);

#endif
