/*--------------------------------*-C-*---------------------------------*
 * File:	main.c
 *----------------------------------------------------------------------*
 * $Id: main.c,v 1.107 2000/02/27 04:57:52 mason Exp $
 *
 * All portions of code are copyright by their respective author/s.
 * Copyright (C) 1992      John Bovey, University of Kent at Canterbury <jdb@ukc.ac.uk>
 *				- original version
 * Copyright (C) 1994      Robert Nation <nation@rocket.sanders.lockheed.com>
 *				- extensive modifications
 * Copyright (C) 1995      Garrett D'Amore <garrett@netcom.com>
 * Copyright (C) 1997      mj olesen <olesen@me.QueensU.CA>
 *				- extensive modifications
 * Copyright (C) 1997,1998 Oezguer Kesim <kesim@math.fu-berlin.de>
 * Copyright (C) 1998,1999 Geoff Wing <gcw@pobox.com>
 *				- extensive modifications
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *---------------------------------------------------------------------*/

#include "../config.h"		/* NECESSARY */
#define INTERN			/* assign all global vars to me */
#include "rxvt.h"		/* NECESSARY */
#include "main.intpro"		/* PROTOS for internal routines */

#ifdef TTY_GID_SUPPORT
# include <grp.h>
#endif

/*----------------------------------------------------------------------*/
/* main() */
/* INTPROTO */
int
main(int argc, const char * const *argv)
{
    const char    **cmd_argv;

/*
 * Save and then give up any super-user privileges
 * If we need privileges in any area then we must specifically request it.
 * We should only need to be root in these cases:
 *  1.  write utmp entries on some systems
 *  2.  chown tty on some systems
 */
    privileges(SAVE);
    privileges(IGNORE);

    init_vars();
    cmd_argv = init_resources(argc, argv);

#if (MENUBAR_MAX)
    menubar_read(rs.menu);
#endif
    scrollbar_mapping(Options & Opt_scrollBar);

    Create_Windows(argc, argv);

    init_xlocale();

    scr_reset();		/* initialize screen */
    Gr_reset();			/* reset graphics */

#ifdef DEBUG_X
    XSynchronize(Xdisplay, True);
    XSetErrorHandler((XErrorHandler) abort);
#else
    XSetErrorHandler((XErrorHandler) xerror_handler);
#endif

    if (scrollbar_visible()) {
	Resize_scrollBar();
	XMapWindow(Xdisplay, scrollBar.win);
    }
#if (MENUBAR_MAX)
    if (menubar_visible())
	XMapWindow(Xdisplay, menuBar.win);
#endif
#ifdef TRANSPARENT
    if (Options & Opt_transparent)
	check_our_parents(True);
#endif
    XMapWindow(Xdisplay, TermWin.vt);
    XMapWindow(Xdisplay, TermWin.parent[0]);

    init_env();
    init_command(cmd_argv);

    main_loop();		/* main processing loop */
    return EXIT_SUCCESS;
}

/* ------------------------------------------------------------------------- *
 *                       SIGNAL HANDLING & EXIT HANDLER                      *
 * ------------------------------------------------------------------------- */
/*
 * Catch a SIGCHLD signal and exit if the direct child has died
 */
/* ARGSUSED */
/* EXTPROTO */
RETSIGTYPE
Child_signal(int unused)
{
    int             pid, save_errno = errno;

    do {
	errno = 0;
    } while ((pid = waitpid(-1, NULL, WNOHANG)) == -1 && errno == EINTR);

    if (pid == cmd_pid)
	exit(EXIT_SUCCESS);

    errno = save_errno;
    signal(SIGCHLD, Child_signal);
}
/*
 * Catch a fatal signal and tidy up before quitting
 */
/* EXTPROTO */
RETSIGTYPE
Exit_signal(int sig)
{
    signal(sig, SIG_DFL);
#ifdef DEBUG_CMD
    print_error("signal %d", sig);
#endif
    clean_exit();
    kill(getpid(), sig);
}

/* ARGSUSED */
/* INTPROTO */
XErrorHandler
xerror_handler(const Display *display, const XErrorEvent *event)
{
    print_error("XError: Request: %d . %d, Error: %d", event->request_code,
		event->minor_code, event->error_code);
    exit(EXIT_FAILURE);
    /* NOTREACHED */
}

/*----------------------------------------------------------------------*/
/*
 * Exit gracefully, clearing the utmp entry and restoring tty attributes
 * TODO: if debugging, this should free up any known resources if we can
 */
/* EXTPROTO */
void
clean_exit(void)
{
#ifdef DEBUG_SCREEN
    scr_release();
#endif
    privileged_ttydev(RESTORE);
    privileged_utmp(RESTORE);
#ifdef USE_XIM
    if (Input_Context != NULL) {
        XDestroyIC(Input_Context);
	Input_Context = NULL;
    }
#endif
}
/* ------------------------------------------------------------------------- *
 *                            PRIVILEGED OPERATIONS                          *
 * ------------------------------------------------------------------------- */
/* take care of suid/sgid super-user (root) privileges */
/* INTPROTO */
void
privileges(int mode)
{
#if ! defined(__CYGWIN32__)
# if !defined(HAVE_SETEUID) && defined(HAVE_SETREUID)
/* setreuid() is the poor man's setuid(), seteuid() */
#  define seteuid(a)	setreuid(-1, (a))
#  define setegid(a)	setregid(-1, (a))
#  define HAVE_SETEUID
# endif
# ifdef HAVE_SETEUID
    static uid_t    euid;
    static gid_t    egid;

    switch (mode) {
    case IGNORE:
    /*
     * change effective uid/gid - not real uid/gid - so we can switch
     * back to root later, as required
     */
	seteuid(getuid());
	setegid(getgid());
	break;
    case SAVE:
	euid = geteuid();
	egid = getegid();
	break;
    case RESTORE:
	seteuid(euid);
	setegid(egid);
	break;
    }
# else
    switch (mode) {
    case IGNORE:
	setuid(getuid());
	setgid(getgid());
    /* FALLTHROUGH */
    case SAVE:
    /* FALLTHROUGH */
    case RESTORE:
	break;
    }
# endif
#endif
}

#ifdef UTMP_SUPPORT
/* EXTPROTO */
void
privileged_utmp(int action)
{
    static int      next_action = SAVE;

    D_MAIN((stderr, "privileged_utmp(%c); waiting for: %c (pid: %d)", action, next_action, getpid()));
    if (next_action != action || action == IGNORE)
	return;
    if (ttydev == NULL || *ttydev == '\0')
	return;
    if (!(Options & Opt_utmpInhibit)) {
	privileges(RESTORE);
	if (action == SAVE) {
	    next_action = RESTORE;
	    makeutent(ttydev, rs.display_name);
	} else if (action == RESTORE) {
	    next_action = IGNORE;
	    cleanutent();
	}
	privileges(IGNORE);
    }
}
#endif

#ifndef __CYGWIN32__
/* EXTPROTO */
void
privileged_ttydev(int action)
{
    unsigned int    mode;
    gid_t           gid;
    static int      next_action = SAVE;
# ifndef RESET_TTY_TO_COMMON_DEFAULTS
    static struct stat ttyfd_stat;	/* original status of our tty */
# endif

    D_MAIN((stderr, "privileged_ttydev(%c); waiting for: %c (pid: %d)", action, next_action, getpid()));
    if (next_action != action || action == IGNORE)
	return;
    if (ttydev == NULL || *ttydev == '\0')
	return;
    if (changettyowner) {
	if (action == RESTORE) {
	    next_action = IGNORE;

	    privileges(RESTORE);
# ifndef RESET_TTY_TO_COMMON_DEFAULTS
	    chmod(ttydev, ttyfd_stat.st_mode);
	    chown(ttydev, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
# else
	    chmod(ttydev, (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP
				 | S_IROTH | S_IWOTH));
	    chown(ttydev, 0, 0);
# endif
	    privileges(IGNORE);
	} else if (action == SAVE) {
	    next_action = RESTORE;

# ifndef RESET_TTY_TO_COMMON_DEFAULTS
/* store original tty status for restoration clean_exit() -- rgg 04/12/95 */
	    if (lstat(ttydev, &ttyfd_stat) < 0) {	/* you lose out */
		next_action = IGNORE;
		return;
	    }
# endif

	    mode = S_IRUSR | S_IWUSR | S_IWGRP | S_IWOTH;
	    gid = getgid();
# ifdef TTY_GID_SUPPORT
	    {
		struct group   *gr = getgrnam("tty");

		if (gr) {
		/* change group ownership of tty to "tty" */
		    mode = S_IRUSR | S_IWUSR | S_IWGRP;
		    gid = gr->gr_gid;
		}
	    }
# endif				/* TTY_GID_SUPPORT */
	    privileges(RESTORE);
	    chown(ttydev, getuid(), gid);	/* fail silently */
	    chmod(ttydev, mode);
# ifdef HAVE_REVOKE
	    revoke(ttydev);
# endif
	    privileges(IGNORE);
	}
# ifndef RESET_TTY_TO_COMMON_DEFAULTS
	D_MAIN((stderr, "%s \"%s\": mode %03o, uid %d, gid %d", action == RESTORE ? "Restoring" : (action == SAVE ? "Saving" : "UNKNOWN ERROR for"), ttydev, ttyfd_stat.st_mode, ttyfd_stat.st_uid, ttyfd_stat.st_gid));
# endif
    }
}
#endif


/*----------------------------------------------------------------------*/
/* EXTPROTO */
void
szhints_set(void)
{
    int             x, y, flags;
    unsigned int    width, height;

    szHint.flags = PMinSize | PResizeInc | PBaseSize | PWinGravity;
    szHint.win_gravity = NorthWestGravity;
    szHint.min_aspect.x = szHint.min_aspect.y = 1;

    flags = (rs.geometry ? XParseGeometry(rs.geometry, &x, &y, &width, &height)
		         : 0);

    if (flags & WidthValue) {
	TermWin.ncol = width;
	szHint.flags |= USSize;
    }
    if (flags & HeightValue) {
	TermWin.nrow = height;
	szHint.flags |= USSize;
    }
    TermWin.width = TermWin.ncol * TermWin.fwidth;
    TermWin.height = TermWin.nrow * TermWin.fheight;
    szhints_recalc();

    if (flags & XValue) {
	if (flags & XNegative) {
	    x += (DisplayWidth(Xdisplay, Xscreen) - szHint.width
		  - 2 * TermWin.ext_bwidth);
	    szHint.win_gravity = NorthEastGravity;
	}
	szHint.x = x;
	szHint.flags |= USPosition;
    }
    if (flags & YValue) {
	if (flags & YNegative) {
	    y += (DisplayHeight(Xdisplay, Xscreen) - szHint.height
		  - 2 * TermWin.ext_bwidth);
	    if (szHint.win_gravity == NorthEastGravity)
		szHint.win_gravity = SouthEastGravity;
	    else
		szHint.win_gravity = SouthWestGravity;
	}
	szHint.y = y;
	szHint.flags |= USPosition;
    }
}

/* INTPROTO */
void
szhints_recalc(void)
{
    szHint.base_width = 2 * TermWin.int_bwidth;
    szHint.base_height = 2 * TermWin.int_bwidth;
    szHint.base_width += (scrollbar_visible() ? (SB_WIDTH + 2 * sb_shadow) : 0);
    szHint.base_height += (menubar_visible() ? menuBar_TotalHeight() : 0);
    szHint.width_inc = TermWin.fwidth;
    szHint.height_inc = TermWin.fheight;
    szHint.min_width = szHint.base_width + szHint.width_inc;
    szHint.min_height = szHint.base_height + szHint.height_inc;
    szHint.width = szHint.base_width + TermWin.width;
    szHint.height = szHint.base_height + TermWin.height;
}
/*----------------------------------------------------------------------*/
/*
 * Tell the teletype handler what size the window is.
 * Called after a window size change.
 */
/* EXTPROTO */
void
tt_winsize(int fd)
{
    struct winsize  ws;

    if (fd < 0)
	return;

    ws.ws_col = (unsigned short)TermWin.ncol;
    ws.ws_row = (unsigned short)TermWin.nrow;
    ws.ws_xpixel = ws.ws_ypixel = 0;
    ioctl(fd, TIOCSWINSZ, &ws);
}

/* EXTPROTO */
void
tt_resize(void)
{
    tt_winsize(cmd_fd);
}

/*----------------------------------------------------------------------*/
/* change_font() - Switch to a new font */
/*
 * init = 1   - initialize
 *
 * fontname == FONT_UP  - switch to bigger font
 * fontname == FONT_DN  - switch to smaller font
 */
/* EXTPROTO */
void
change_font(int init, const char *fontname)
{
    const char     *msg = "can't load font \"%s\"";
    int             fh, fw, recheckfonts;
    int             idx = 0;	/* index into rs.font[] */
    XFontStruct    *xfont;
    static char    *newfont[NFONTS];
    static int      fnum;		/* logical font number */
#ifndef NO_BOLDFONT
    static XFontStruct *boldFont;
#endif
#ifdef MULTICHAR_SET
    int             i;
    char           *c, *enc;
    char            tmpbuf[64];
#endif

#if (FONT0_IDX == 0)
# define IDX2FNUM(i)	(i)
# define FNUM2IDX(f)	(f)
#else
# define IDX2FNUM(i)	(i == 0 ? FONT0_IDX : (i <= FONT0_IDX ? (i-1) : i))
# define FNUM2IDX(f)	(f == FONT0_IDX ? 0 : (f < FONT0_IDX  ? (f+1) : f))
#endif
#define FNUM_RANGE(i)	(i <= 0 ? 0 : (i >= NFONTS ? (NFONTS-1) : i))

    if (init) {
#ifndef NO_BOLDFONT
	boldFont = NULL;
#endif
	fnum = FONT0_IDX;	/* logical font number */
    } else {
	switch (fontname[0]) {
	case '\0':
	    fnum = FONT0_IDX;
	    fontname = NULL;
	    break;

	/* special (internal) prefix for font commands */
	case FONT_CMD:
	    idx = atoi(fontname + 1);
	    switch (fontname[1]) {
	    case '+':		/* corresponds to FONT_UP */
		fnum += (idx ? idx : 1);
		fnum = FNUM_RANGE(fnum);
		break;

	    case '-':		/* corresponds to FONT_DN */
		fnum += (idx ? idx : -1);
		fnum = FNUM_RANGE(fnum);
		break;

	    default:
		if (fontname[1] != '\0' && !isdigit(fontname[1]))
		    return;
		if (idx < 0 || idx >= (NFONTS))
		    return;
		fnum = IDX2FNUM(idx);
		break;
	    }
	    fontname = NULL;
	    break;

	default:
	    if (fontname != NULL) {
	    /* search for existing fontname */
		for (idx = 0; idx < NFONTS; idx++) {
		    if (!STRCMP(rs.font[idx], fontname)) {
			fnum = IDX2FNUM(idx);
			fontname = NULL;
			break;
		    }
		}
	    } else
		return;
	    break;
	}
    /* re-position around the normal font */
	idx = FNUM2IDX(fnum);

	if (fontname != NULL) {
	    char           *name;

	    xfont = XLoadQueryFont(Xdisplay, fontname);
	    if (!xfont)
		return;

	    name = MALLOC(STRLEN(fontname + 1) * sizeof(char));

	    if (name == NULL) {
		XFreeFont(Xdisplay, xfont);
		return;
	    }
	    STRCPY(name, fontname);
	    if (newfont[idx] != NULL)
		FREE(newfont[idx]);
	    newfont[idx] = name;
	    rs.font[idx] = newfont[idx];
	}
    }
    if (TermWin.font)
	XFreeFont(Xdisplay, TermWin.font);

/* load font or substitute */
    xfont = XLoadQueryFont(Xdisplay, rs.font[idx]);
    if (!xfont) {
	print_error(msg, rs.font[idx]);
	rs.font[idx] = "fixed";
	xfont = XLoadQueryFont(Xdisplay, rs.font[idx]);
	if (!xfont) {
	    print_error(msg, rs.font[idx]);
	    goto Abort;
	}
    }
    TermWin.font = xfont;

#ifndef NO_BOLDFONT
/* fail silently */
    if (init && rs.boldFont != NULL)
	boldFont = XLoadQueryFont(Xdisplay, rs.boldFont);
#endif

/* alter existing GC */
    if (!init) {
	XSetFont(Xdisplay, TermWin.gc, TermWin.font->fid);
	menubar_expose();
    }

/* set the sizes */
    fw = get_fontwidest(TermWin.font);
    fh = TermWin.font->ascent + TermWin.font->descent;
    if (fw == TermWin.font->min_bounds.width)
	TermWin.fprop = 0;	/* Mono-spaced (fixed width) font */
    else
	TermWin.fprop = 1;	/* Proportional font */
    recheckfonts = !(fw == TermWin.fwidth && fh == TermWin.fheight);
    TermWin.fwidth = fw;
    TermWin.fheight = fh;

/* check that size of boldFont is okay */
#ifndef NO_BOLDFONT
    if (recheckfonts) {
	TermWin.boldFont = NULL;
	if (boldFont != NULL) {
	    fw = get_fontwidest(boldFont);
	    fh = boldFont->ascent + boldFont->descent;
	    if (fw <= TermWin.fwidth && fh <= TermWin.fheight)
		TermWin.boldFont = boldFont;
	    TermWin.bprop = !(fw == TermWin.fwidth /* && fh == TermWin.fheight */ );
	}
    }
#endif				/* NO_BOLDFONT */

#ifdef MULTICHAR_SET
    if (TermWin.mfont)
	XFreeFont(Xdisplay, TermWin.mfont);

/* load font or substitute */
    if (rs.mfont[idx] == NULL
	|| (xfont = XLoadQueryFont(Xdisplay, rs.mfont[idx])) == NULL) {
    /* TODO: improve this */
	i = 0;
	switch(encoding_method) {
	case GB:
	    c = "-*-r-*-%.2d-*-gb2312.1980-0";
	    enc = "GB";
	    break;
	case BIG5:
	    c = "-*-r-*-%.2d-*-big5-0";
	    enc = "BIG5";
	    break;
	case EUCJ:
	case SJIS:
	    c = "-*-%.2d-*-jisx0208*-*";
	    enc = "EUCJ/SJIS";
	    break;
	case EUCKR:
	    c = "-*-%.2d-*-ksc5601*-*";
	    enc = "KR";
	    break;
	default:
	    i = fh;		/* jump past next two sections */
	    break;
	}
	for ( ; i < fh / 2; i++) {
	    sprintf(tmpbuf, c, fh - i);
	    xfont = XLoadQueryFont(Xdisplay, tmpbuf);
	    if (xfont) {
		rs.mfont[idx] = MALLOC(STRLEN(tmpbuf) + 1);
		STRCPY(rs.mfont[idx], tmpbuf);
		break;
	    }
	}
	if (xfont == NULL && i != fh)
	    print_error("no similar multichar font: encoding %s; size %d",
			enc, fh);
    }
    TermWin.mfont = xfont;

    if (recheckfonts)
    /* XXX: This checks what? */
	if (TermWin.mfont != NULL) {
	    fw = get_fontwidest(TermWin.mfont);
	    fh = TermWin.mfont->ascent + TermWin.mfont->descent;
	    if (fw > (TermWin.fwidth * 2) || fh > TermWin.fheight)
		TermWin.mfont = NULL;
	    TermWin.mprop = !(fw == TermWin.fwidth /* && fh == TermWin.fheight */ );
	}
#endif				/* MULTICHAR_SET */

    /* Fontset setting is only valid when xlocale initialized. Since 
       init_xlocale() is called after here, so we don't set fontset when
       initialization, but let it set by init_xlocale() */
    if (init)
	setTermFontSet(-1);
    else
	setTermFontSet(idx);

    set_colorfgbg();

    TermWin.width = TermWin.ncol * TermWin.fwidth;
    TermWin.height = TermWin.nrow * TermWin.fheight;

    if (!init) {
	resize_all_windows();
	scr_touch(True);
    }
    return;
  Abort:
    print_error("aborting");	/* fatal problem */
    exit(EXIT_FAILURE);
#undef IDX2FNUM
#undef FNUM2IDX
#undef FNUM_RANGE
    /* NOTREACHED */
}
#ifdef STRICT_FONT_CHECKING
/* INTPROTO */
int
get_fontwidest(XFontStruct *f)
{
    int             i, cw, fw = 0;

    if (f->min_bounds.width == f->max_bounds.width)
	return f->min_bounds.width;
    if (f->per_char == NULL)
	return f->max_bounds.width;
    for (i = f->max_char_or_byte2 - f->min_char_or_byte2; --i >= 0; ) {
	cw = f->per_char[i].width;
	MAX_IT(fw, cw);
    }
    return fw;
}
#endif
/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/
/* xterm sequences - title, iconName, color (exptl) */
/* EXTPROTO */
void
set_title(const char *str)
{
#ifndef SMART_WINDOW_TITLE
    XStoreName(Xdisplay, TermWin.parent[0], str);
#else
    char           *name;

    if (XFetchName(Xdisplay, TermWin.parent[0], &name) == 0)
	name = NULL;
    if (name == NULL || STRCMP(name, str))
	XStoreName(Xdisplay, TermWin.parent[0], str);
    if (name)
	XFree(name);
#endif
}

/* EXTPROTO */
void
set_iconName(const char *str)
{
#ifndef SMART_WINDOW_TITLE
    XSetIconName(Xdisplay, TermWin.parent[0], str);
#else
    char           *name;

    if (XGetIconName(Xdisplay, TermWin.parent[0], &name))
	name = NULL;
    if (name == NULL || STRCMP(name, str))
	XSetIconName(Xdisplay, TermWin.parent[0], str);
    if (name)
	XFree(name);
#endif
}

#ifdef XTERM_COLOR_CHANGE
/* EXTPROTO */
void
set_window_color(int idx, const char *color)
{
    const char     *msg = "can't load color \"%s\"";
    XColor          xcol;
    int             i;

    if (color == NULL || *color == '\0')
	return;

/* handle color aliases */
    if (isdigit(*color)) {
	i = atoi(color);
	if (i >= 8 && i <= 15) {	/* bright colors */
	    i -= 8;
# ifndef NO_BRIGHTCOLOR
	    PixColors[idx] = PixColors[minBrightCOLOR + i];
	    goto Done;
# endif
	}
	if (i >= 0 && i <= 7) {	/* normal colors */
	    PixColors[idx] = PixColors[minCOLOR + i];
	    goto Done;
	}
    }
    if (!XParseColor(Xdisplay, Xcmap, color, &xcol)
	|| !rXAllocColor(Xdisplay, Xcmap, &xcol)) {
	print_error(msg, color);
	return;
    }
/* XStoreColor (Xdisplay, Xcmap, XColor*); */

/*
 * FIXME: should free colors here, but no idea how to do it so instead,
 * so just keep gobbling up the colormap
 */
# if 0
    for (i = Color_Black; i <= Color_White; i++)
	if (PixColors[idx] == PixColors[i])
	    break;
    if (i > Color_White) {
    /* fprintf (stderr, "XFreeColors: PixColors [%d] = %lu\n", idx, PixColors [idx]); */
	XFreeColors(Xdisplay, Xcmap, (PixColors + idx), 1,
		    DisplayPlanes(Xdisplay, Xscreen));
    }
# endif

    PixColors[idx] = xcol.pixel;

/* XSetWindowAttributes attr; */
/* Cursor cursor; */
  Done:
    if (idx == Color_bg && !(Options & Opt_transparent))
	XSetWindowBackground(Xdisplay, TermWin.vt, PixColors[Color_bg]);

/* handle Color_BD, scrollbar background, etc. */

    set_colorfgbg();
    {
	XColor          fg, bg;

	fg.pixel = PixColors[Color_pointer];
	XQueryColor(Xdisplay, Xcmap, &fg);
	bg.pixel = PixColors[Color_bg];
	XQueryColor(Xdisplay, Xcmap, &bg);
	XRecolorCursor(Xdisplay, TermWin_cursor, &fg, &bg);
    }
/* the only reasonable way to enforce a clean update */
    scr_poweron();
}
#else
# define set_window_color(idx,color)	((void)0)
#endif				/* XTERM_COLOR_CHANGE */
/*----------------------------------------------------------------------*/
/*
 * find if fg/bg matches any of the normal (low-intensity) colors
 */
/* INTPROTO */
void
set_colorfgbg(void)
{
    unsigned int    i;
    char           *p;
    int             fg = -1, bg = -1;
    static char     env_colorfgbg[] = "COLORFGBG=default;default;bg";

    for (i = Color_Black; i <= Color_White; i++) {
	if (PixColors[Color_fg] == PixColors[i]) {
	    fg = (i - Color_Black);
	    break;
	}
    }
    for (i = Color_Black; i <= Color_White; i++) {
	if (PixColors[Color_bg] == PixColors[i]) {
	    bg = (i - Color_Black);
	    break;
	}
    }

    p = STRCHR(env_colorfgbg, '=');
    p++;
    if (fg >= 0)
	sprintf(p, "%d", fg);
    else
	STRCPY(p, "default");
    p = STRCHR(p, '\0');
    *p++ = ';';
    if (bg >= 0) {
#ifdef XPM_BACKGROUND
	STRCPY(p, "default");
	p = STRCHR(p, '\0');
	*p++ = ';';
#endif
	sprintf(p, "%d", bg);
    } else
	STRCPY(p, "default");
    putenv(env_colorfgbg);

#ifndef NO_BRIGHTCOLOR
    colorfgbg = DEFAULT_RSTYLE;
    for (i = minCOLOR; i <= maxCOLOR; i++) {
	if (PixColors[Color_fg] == PixColors[i]
# ifndef NO_BOLDUNDERLINE
	    && PixColors[Color_fg] == PixColors[Color_BD]
# endif				/* NO_BOLDUNDERLINE */
    /* if we wanted boldFont to have precedence */
# if 0				/* ifndef NO_BOLDFONT */
	    && TermWin.boldFont == NULL
# endif				/* NO_BOLDFONT */
	    )
	    colorfgbg = SET_FGCOLOR(colorfgbg, i);
	if (PixColors[Color_bg] == PixColors[i])
	    colorfgbg = SET_BGCOLOR(colorfgbg, i);
    }
#endif
}
/*----------------------------------------------------------------------*/
/* 
 * Colour determination for low colour displays, routine from
 *     Hans de Goede <hans@highrise.nl>
 */

#define rSQR(x)		((float)(x)*(x))

/* EXTPROTO */
Status
rXAllocColor(Display *display, Colormap colormap, XColor *screen_in_out)
{
    int             res;

    if ((res = XAllocColor(display, colormap, screen_in_out)))
	return res;

    /* try again with closest match */
    if (Xdepth >= 4 && Xdepth <= 8) {
	int             i, numcol;
	int             best_pixel = 0;
	float           best_diff, diff;
	XColor         *colors;

	numcol = 0x01 << Xdepth;
	if ((colors = MALLOC(numcol * sizeof(XColor))) == NULL)
	    return 0;
      
	for (i = 0; i < numcol; i++)
	    colors[i].pixel = i;
      
	XQueryColors(display, colormap, colors, numcol);
	best_diff = rSQR(screen_in_out->red - colors[0].red)
		    + rSQR(screen_in_out->green - colors[0].green)
		    + rSQR(screen_in_out->blue - colors[0].blue);
	for (i = 1; i < numcol; i++) {
	    diff = rSQR(screen_in_out->red - colors[i].red)
		   + rSQR(screen_in_out->green - colors[i].green)
		   + rSQR(screen_in_out->blue - colors[i].blue);
	    if (diff < best_diff) {
		best_pixel = colors[i].pixel;
		best_diff = diff;
	    }
	}
	*screen_in_out = colors[best_pixel];
	FREE(colors);
	res = XAllocColor(display, colormap, screen_in_out);
    }
    return res;   
}

/*----------------------------------------------------------------------*/
/* window resizing - assuming the parent window is the correct size */
/* INTPROTO */
void
resize_subwindows(int width, int height)
{
    int             x = 0, y = 0;
#ifdef RXVT_GRAPHICS
    int             old_width = TermWin.width, old_height = TermWin.height;
#endif

    TermWin.width = TermWin.ncol * TermWin.fwidth;
    TermWin.height = TermWin.nrow * TermWin.fheight;

    szHint.width = width;
    szHint.height = height;

/* size and placement */
    if (scrollbar_visible()) {
	Resize_scrollBar();
	x = (SB_WIDTH + 2 * sb_shadow);	/* placement of vt window */
	width -= x;
	if (Options & Opt_scrollBar_right)
	    x = 0;		/* scrollbar on right so vt window at left */
    }
    { /* ONLYIF MENUBAR */
	if (menubar_visible()) {
	    y = menuBar_TotalHeight();	/* for placement of vt window */
	    Resize_menuBar(x, 0, width, y);
	}
    }
    XMoveResizeWindow(Xdisplay, TermWin.vt, x, y, width, height + 1);

#ifdef RXVT_GRAPHICS
    if (old_width)
	Gr_Resize(old_width, old_height);
#endif

    scr_clear();
    resize_pixmap();
    XSync(Xdisplay, False);
}

/* EXTPROTO */
void
resize_all_windows(void)
{
    szhints_recalc();
    XSetWMNormalHints(Xdisplay, TermWin.parent[0], &szHint);
    AddToCNQueue(szHint.width, szHint.height);
    XResizeWindow(Xdisplay, TermWin.parent[0], szHint.width, szHint.height);
    resize_window(szHint.width, szHint.height);
}

/*
 * Redraw window after exposure or size change
 * width/height are those of the parent
 */
/* EXTPROTO */
void
resize_window(unsigned int width, unsigned int height)
{
    int             new_ncol, new_nrow;
    static int      old_width, old_height = -1;

    new_ncol = (width - szHint.base_width) / TermWin.fwidth;
    new_nrow = (height - szHint.base_height) / TermWin.fheight;
    if (old_height == -1
	|| (new_ncol != TermWin.ncol)
	|| (new_nrow != TermWin.nrow)) {
	int             curr_screen = -1;

    /* scr_reset only works on the primary screen */
	if (old_height != -1) {	/* this is not the first time thru */
	    selection_clear();
	    curr_screen = scr_change_screen(PRIMARY);
	}
	TermWin.ncol = new_ncol;
	TermWin.nrow = new_nrow;

	resize_subwindows(width, height);
	scr_reset();

	if (curr_screen >= 0)	/* this is not the first time thru */
	    scr_change_screen(curr_screen);
    } else if (width != old_width || height != old_height)
	resize_subwindows(width, height);
    old_width = width;
    old_height = height;
}

/*
 * Set the width/height of the window in characters.  Units are pixels.
 * good for toggling 80/132 columns
 */
/* EXTPROTO */
void
set_widthheight(unsigned int width, unsigned int height)
{
    XWindowAttributes wattr;

    if (width == 0 || height == 0) {
	XGetWindowAttributes(Xdisplay, Xroot, &wattr);
	if (width == 0)
	    width = wattr.width	- szHint.base_width;
	if (height == 0)
	    height = wattr.height - szHint.base_height;
    }

    if (width != TermWin.width || height != TermWin.height) {
	width = szHint.base_width + width;
	height = szHint.base_height + height;

	AddToCNQueue(width, height);
	XResizeWindow(Xdisplay, TermWin.parent[0], width, height);
	resize_window(width, height);
#ifdef USE_XIM
	IMSetStatusPosition();
#endif
    }
}

/* -------------------------------------------------------------------- *
 * -                      X INPUT METHOD ROUTINES                     - * 
 * -------------------------------------------------------------------- */
#ifdef USE_XIM
/* INTPROTO */
void
setSize(XRectangle *size)
{
    size->x = TermWin.int_bwidth;
    size->y = TermWin.int_bwidth;
    size->width = Width2Pixel(TermWin.ncol);
    size->height = Height2Pixel(TermWin.nrow);
}

/* INTPROTO */
void
setColor(unsigned long *fg, unsigned long *bg)
{
    *fg = PixColors[Color_fg];
    *bg = PixColors[Color_bg];
}

/* Checking whether input method is running. */
/* INTPROTO */
Bool
IMisRunning(void)
{
    char           *p;
    Atom            atom;
    Window          win;
    char            server[IMBUFSIZ];

    /* get current locale modifier */
    if ((p = XSetLocaleModifiers(NULL)) != NULL) {
	STRCPY(server, "@server=");
	STRNCAT(server, &(p[4]), IMBUFSIZ - 9);	/* skip "@im=" */

	atom = XInternAtom(Xdisplay, server, False);
	win = XGetSelectionOwner(Xdisplay, atom);
	if (win != None)
	    return True;
    }
    return False;
}

/* EXTPROTO */
void
IMSendSpot(void)
{
    XPoint          spot;
    XVaNestedList   preedit_attr;

    if (Input_Context == NULL 
	|| !TermWin.focus
	|| !(input_style & XIMPreeditPosition)
	|| !(event_type == KeyPress
	     || event_type == SelectionNotify
	     || event_type == ButtonRelease
	     || event_type == FocusIn)
	|| !IMisRunning())
        return;

    setPosition(&spot);

    preedit_attr = XVaCreateNestedList(0, XNSpotLocation, &spot, NULL);
    XSetICValues(Input_Context, XNPreeditAttributes, preedit_attr, NULL);
    XFree(preedit_attr);
}

/* EXTPROTO */
void
setTermFontSet(int idx)
{
    char           *string;
    long            length;
    XFontSet	    prev_fontset;
    int		    success=0;

    if (idx < 0 || idx >= NFONTS)
	return;
    D_MAIN((stderr, "setTermFontSet()"));
    prev_fontset = TermWin.fontset;
    TermWin.fontset = NULL;

    length = 0;
    if (rs.font[idx])
	length += STRLEN(rs.font[idx]) + 1;
# ifdef MULTICHAR_SET
    if (rs.mfont[idx])
	length += STRLEN(rs.mfont[idx]) + 1;
# endif
    if (length == 0
	|| (string = MALLOC(length + 1)) == NULL)
	TermWin.fontset = NULL;
    else {
	int             missing_charsetcount;
	char          **missing_charsetlist, *def_string;

	string[0] = '\0';
	if (rs.font[idx]) {
	    STRCAT(string, rs.font[idx]);
	    STRCAT(string, ",");
	}
# ifdef MULTICHAR_SET
	if (rs.mfont[idx]) {
	    STRCAT(string, rs.mfont[idx]);
	    STRCAT(string, ",");
	}
# endif
	string[STRLEN(string) - 1] = '\0';
	TermWin.fontset = XCreateFontSet(Xdisplay, string,
					 &missing_charsetlist,
					 &missing_charsetcount,
					 &def_string);
	FREE(string);
	if (TermWin.fontset != NULL)
	    success = 1;
    }

    if (success) {
	if (prev_fontset != NULL)
	    XFreeFontSet(Xdisplay, prev_fontset);
    }
    else
	TermWin.fontset = prev_fontset;
}

/* INTPROTO */
void
setPreeditArea(XRectangle *preedit_rect, XRectangle *status_rect, XRectangle *needed_rect)
{
    preedit_rect->x = needed_rect->width
		      + (scrollbar_visible() && !(Options & Opt_scrollBar_right)
			 ? (SB_WIDTH + sb_shadow * 2) : 0);
    preedit_rect->y = Height2Pixel(TermWin.nrow - 1)
		      + ((menuBar.state == 1) ? menuBar_TotalHeight() : 0);

    preedit_rect->width = Width2Pixel(TermWin.ncol + 1) - needed_rect->width
			  + (!(Options & Opt_scrollBar_right)
			     ? (SB_WIDTH + sb_shadow * 2) : 0);
    preedit_rect->height = Height2Pixel(1);

    status_rect->x = (scrollbar_visible() && !(Options & Opt_scrollBar_right))
		      ? (SB_WIDTH + sb_shadow * 2) : 0;
    status_rect->y = Height2Pixel(TermWin.nrow - 1)
		     + ((menuBar.state == 1) ? menuBar_TotalHeight() : 0);

    status_rect->width = needed_rect->width ? needed_rect->width
					    : Width2Pixel(TermWin.ncol + 1);
    status_rect->height = Height2Pixel(1);
}

/* INTPROTO */
void
IMDestroyCallback(XIM xim, XPointer client_data, XPointer call_data)
{
    Input_Context = NULL;
    XRegisterIMInstantiateCallback(Xdisplay, NULL, NULL, NULL,
				   IMInstantiateCallback, NULL);
}

/* EXTPROTO */
void
IMInstantiateCallback(Display *display, XPointer client_data, XPointer call_data)
{
    char           *p, *s, buf[IMBUFSIZ], tmp[1024];
    char           *end, *next_s;
    XIM             xim = NULL;
    XIMStyles      *xim_styles = NULL;
    int             found;
    XPoint          spot;
    XRectangle      rect, status_rect, needed_rect;
    unsigned long   fg, bg;
    XVaNestedList   preedit_attr = NULL;
    XVaNestedList   status_attr = NULL;
    XIMCallback     ximcallback;

    if (Input_Context)
	return;

    ximcallback.callback = IMDestroyCallback;
    ximcallback.client_data = NULL;

    if (rs.inputMethod && *(rs.inputMethod)) {
	STRNCPY(tmp, rs.inputMethod, sizeof(tmp) - 1);
	for (s = tmp; *s; s = next_s + 1) {
	    for (; *s && isspace(*s); s++) ;
	    if (!*s)
		break;
	    for (end = s; (*end && (*end != ',')); end++) ;
	    for (next_s = end--; ((end >= s) && isspace(*end)); end--) ;
	    *(end + 1) = '\0';

	    if (*s) {
		STRCPY(buf, "@im=");
		STRNCAT(buf, s, IMBUFSIZ - 5);
		if ((p = XSetLocaleModifiers(buf)) != NULL && *p
		    && (xim = XOpenIM(Xdisplay, NULL, NULL, NULL)) != NULL)
		    break;
	    }
	    if (!*next_s)
		break;
	}
    }
/* try with XMODIFIERS env. var. */
    if (xim == NULL && (p = XSetLocaleModifiers("")) != NULL && *p)
	xim = XOpenIM(Xdisplay, NULL, NULL, NULL);

/* try with no modifiers base */
    if (xim == NULL && (p = XSetLocaleModifiers("@im=none")) != NULL && *p)
	xim = XOpenIM(Xdisplay, NULL, NULL, NULL);

    if (xim == NULL)
	return;
    XSetIMValues(xim, XNDestroyCallback, &ximcallback, NULL);

    if (XGetIMValues(xim, XNQueryInputStyle, &xim_styles, NULL)
	|| !xim_styles) {
	print_error("input method doesn't support any style");
	XCloseIM(xim);
	return;
    }
    STRNCPY(tmp, (rs.preeditType ? rs.preeditType
				 : "OverTheSpot,OffTheSpot,Root"),
	    sizeof(tmp) - 1);
    for (found = 0, s = tmp; *s && !found; s = next_s + 1) {
	unsigned short  i;

	for (; *s && isspace(*s); s++) ;
	if (!*s)
	    break;
	for (end = s; (*end && (*end != ',')); end++) ;
	for (next_s = end--; ((end >= s) && isspace(*end)); end--) ;
	*(end + 1) = '\0';

	if (!STRCMP(s, "OverTheSpot"))
	    input_style = (XIMPreeditPosition | XIMStatusNothing);
	else if (!STRCMP(s, "OffTheSpot"))
	    input_style = (XIMPreeditArea | XIMStatusArea);
	else if (!STRCMP(s, "Root"))
	    input_style = (XIMPreeditNothing | XIMStatusNothing);

	for (i = 0; i < xim_styles->count_styles; i++)
	    if (input_style == xim_styles->supported_styles[i]) {
		found = 1;
		break;
	    }
    }
    XFree(xim_styles);

    if (found == 0) {
	print_error("input method doesn't support my preedit type");
	XCloseIM(xim);
	return;
    }
    if ((input_style != (XIMPreeditNothing | XIMStatusNothing))
	&& (input_style != (XIMPreeditArea | XIMStatusArea))
	&& (input_style != (XIMPreeditPosition | XIMStatusNothing))) {
	print_error("This program does not support the preedit type");
	XCloseIM(xim);
	return;
    }
    if (input_style & XIMPreeditPosition) {
	setSize(&rect);
	setPosition(&spot);
	setColor(&fg, &bg);

	preedit_attr = XVaCreateNestedList(0, XNArea, &rect,
					   XNSpotLocation, &spot,
					   XNForeground, fg,
					   XNBackground, bg,
					   XNFontSet, TermWin.fontset,
					   NULL);
    } else if (input_style & XIMPreeditArea) {
	setColor(&fg, &bg);

    /*
     * The necessary width of preedit area is unknown
     * until create input context.
     */
	needed_rect.width = 0;

	setPreeditArea(&rect, &status_rect, &needed_rect);

	preedit_attr = XVaCreateNestedList(0, XNArea, &rect,
					   XNForeground, fg,
					   XNBackground, bg,
					   XNFontSet, TermWin.fontset,
					   NULL);
	status_attr = XVaCreateNestedList(0, XNArea, &status_rect,
					  XNForeground, fg,
					  XNBackground, bg,
					  XNFontSet, TermWin.fontset,
					  NULL);
    }
    Input_Context = XCreateIC(xim, XNInputStyle, input_style,
			      XNClientWindow, TermWin.parent[0],
			      XNFocusWindow, TermWin.parent[0],
			      XNDestroyCallback, &ximcallback,
			      preedit_attr ? XNPreeditAttributes : NULL,
			      preedit_attr,
			      status_attr ? XNStatusAttributes : NULL,
			      status_attr,
			      NULL);
    XFree(preedit_attr);
    XFree(status_attr);
    if (Input_Context == NULL) {
	print_error("Failed to create input context");
	XCloseIM(xim);
    }
    if (input_style & XIMPreeditArea)
	IMSetStatusPosition();
}

/* EXTPROTO */
void
IMSetStatusPosition(void)
{
    XRectangle      preedit_rect, status_rect, *needed_rect;
    XVaNestedList   preedit_attr, status_attr;

    if (Input_Context == NULL 
	|| !TermWin.focus
	|| !(input_style & XIMPreeditArea)
	|| !IMisRunning())
        return;

    /* Getting the necessary width of preedit area */
    status_attr = XVaCreateNestedList(0, XNAreaNeeded, &needed_rect, NULL);
    XGetICValues(Input_Context, XNStatusAttributes, status_attr, NULL);
    XFree(status_attr);

    setPreeditArea(&preedit_rect, &status_rect, needed_rect);

    preedit_attr = XVaCreateNestedList(0, XNArea, &preedit_rect, NULL);
    status_attr = XVaCreateNestedList(0, XNArea, &status_rect, NULL);

    XSetICValues(Input_Context,
		 XNPreeditAttributes, preedit_attr,
		 XNStatusAttributes, status_attr, NULL);

    XFree(preedit_attr);
    XFree(status_attr);
}
#endif				/* USE_XIM */

/*----------------------- end-of-file (C source) -----------------------*/
