/*
 * root-tail.c
 * --------------------------------------------------------
 * compile command: "gcc -o root-tail root-tail.c -lX11 -L/usr/X11/lib"
 */

/*------------------------- INCLUDES -------------------------*/

#include "config.h"
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>


/*---------------- Let's define signals functions -------------*/

static void reopen (int);
static void list_files (int);
static void force_refresh (int);
static void InstallSigHandler (void);
FILE *openLog (const char *);

/*------------------------ initalize variables -----------------*/

int screen, listlen = STD_HEIGHT, width = STD_WIDTH, ScreenWidth, ScreenHeight,
  win_x = LOC_X, win_y = LOC_Y, w = -1, h = -1, pid = -1, font_width = 0,
  font_height = 0, infocus = 0, shade = 0, frame = 0, reload = 0;
time_t lastreload;

char *fontName = USE_FONT;
typedef unsigned long Pixel;
Display *disp;
Window root;
GC WinGC;
char *dispname = NULL, *color = DEF_COLOR, *Geometry = 0, *command;

struct logfile_entry
  {
    char fname[255];		/* name of file                         */
    char desc[255];		/* alternative description              */
    FILE *f;			/* FILE struct associated with file     */
    Pixel color;		/* color to be used for printing        */
    struct logfile_entry *next;
  };

struct logfile_entry *loglist = NULL;
struct logfile_entry *loglist_tail = NULL;

struct linematrix
  {
    char *line;
    Pixel color;
  };


/*----------------------------- start code ---------------------*/

void
reopen (int signal)
{
  struct logfile_entry *e = NULL;

  printf ("Reopening files\n");
  for (e = loglist; e != NULL; e = e->next)
    {
      fclose (e->f);
      e->f = openLog (e->fname);
      if (e->f == NULL)
	{
	  printf ("WARNING: Cannot open %s\n", e->fname);
	  /*
	     exit(1);
	   */
	}
    }
  InstallSigHandler ();
}

void
list_files (int signal)
{
  struct logfile_entry *e = NULL;

  printf ("Files opened:\n");
  for (e = loglist; e != NULL; e = e->next)
    printf ("\t%s\n", e->fname);
  InstallSigHandler ();
}

void
force_refresh (int signal)
{
  XClearWindow (disp, root);
  InstallSigHandler ();
}

void
InstallSigHandler ()
{
  signal (SIGHUP, reopen);
  signal (SIGUSR1, list_files);
  signal (SIGUSR2, force_refresh);
}


Pixel
GetColor (char *ColorName)
{
  XColor Color;
  XWindowAttributes Attributes;
  XGetWindowAttributes (disp, root, &Attributes);
  Color.pixel = 0;
  if (!XParseColor (disp, Attributes.colormap, ColorName, &Color))
    fprintf (stderr, "can't parse %s\n", ColorName);
  else if (!XAllocColor (disp, Attributes.colormap, &Color))
    fprintf (stderr, "can't allocate %s\n", ColorName);
  return Color.pixel;
}

void
InitWindow ()
{
  XGCValues gcv;
  Font font;
  unsigned long gcm;
  XFontStruct *info;
  if (!(disp = XOpenDisplay (dispname)))
    {
      fprintf (stderr, "Error opening display %s.\n", dispname);
      exit (-1);
    }
  screen = DefaultScreen (disp);
  ScreenHeight = DisplayHeight (disp, screen);
  ScreenWidth = DisplayWidth (disp, screen);
  root = RootWindow (disp, screen);
  gcm = GCBackground;
  gcv.graphics_exposures = True;
  WinGC = XCreateGC (disp, root, gcm, &gcv);
  XMapWindow (disp, root);
  XSetForeground (disp, WinGC, GetColor (DEF_COLOR));
  font = XLoadFont (disp, fontName);
  XSetFont (disp, WinGC, font);
  info = XQueryFont (disp, font);
  font_width = info->max_bounds.width;
  font_height = info->max_bounds.ascent + info->max_bounds.descent;
  w = width * font_width;
  h = listlen * font_height;
  if (win_x < 0)
    win_x = win_x + ScreenWidth - w;
  if (win_y < 0)
    win_y = win_y + ScreenHeight - h;

#define EVENTMASK  ExposureMask \
		| SubstructureNotifyMask \
	        | VisibilityChangeMask  \
		| StructureNotifyMask \
		| FocusChangeMask \
		| PropertyChangeMask
  XSelectInput (disp, root, EVENTMASK);
}

int
lineinput (char *string, int width, FILE * f)
/*
 * This routine should read 'width' characters and not more. However,
 * we really want to read width + 1 charachters if the last char is a '\n',
 * which we should remove afterwards. So, read width+1 chars and ungetc
 * the last character if it's not a newline. This means 'string' must be
 * width + 2 wide!
 */
{
  int len;

  do {
    if (fgets (string, width, f) == NULL)		/* EOF or Error */
      return 1;

    len = strlen (string);
  } while (!len);
  
  if (string[len - 1] == '\n')
    string[len - 1] = '\0';		/* erase newline */
  else
    {
      ungetc (string[len - 1], f);
      string[len - 1] = '\0';
    }
  return 0;
}

FILE *
openLog (const char *name)
{
  FILE *f = fopen (name, "r");
  struct stat statbuf;
  off_t size;
  if (f == NULL)
    return NULL;
  stat (name, &statbuf);
  size = statbuf.st_size;
  if (size > (listlen++) * width)
    {
      char dummy[255];
      fseek (f, -((listlen++) * width), SEEK_END);

      /* 
       * the pointer might point halfway some line. Let's
       * be nice and skip this damaged line
       */
      lineinput (dummy, 255, f);
    }
  return f;
}

void
redraw (void)
/* redraw does a complete redraw, rather than an update (i.e. the area
 * gets cleared first)
 */
{
  XClearArea (disp, root, win_x - 2, win_y, w, h, True);
  /* the rest is handled by regular refresh()'es */
}

void
refresh (struct linematrix *lines, int miny, int maxy)
/* Just redraw everything without claring (i.e. after an EXPOSE event)
*/
{
  int offset = listlen * (font_height-1), lin;

  maxy += font_height*2;
  miny -= font_height*2;

  for (lin = listlen; lin--; )
    {
      offset -= (font_height - 1);

      if (offset < miny || offset >= maxy)
	continue;

      if (shade)
	{
	  XSetForeground (disp, WinGC, GetColor ("black"));
	  XDrawString (disp, root, WinGC, win_x - 2, win_y + offset - 2,
		       lines[lin].line, strlen (lines[lin].line));
	}
      XSetForeground (disp, WinGC, lines[lin].color);
      XDrawString (disp, root, WinGC, win_x, win_y + offset,
		   lines[lin].line, strlen (lines[lin].line));
    }
  if (frame)
    {
      XDrawLine (disp, root, WinGC, win_x, win_y, win_x + w, win_y);
      XDrawLine (disp, root, WinGC, win_x + w, win_y, win_x + w, win_y + h);
      XDrawLine (disp, root, WinGC, win_x + w, win_y + h, win_x, win_y + h);
      XDrawLine (disp, root, WinGC, win_x, win_y + h, win_x, win_y);
    }
}

void
loop ()
{
  struct logfile_entry *lastprinted = NULL;

  struct linematrix *lines = (struct linematrix *)
  malloc (listlen * sizeof (struct linematrix));

  int error, lin;

  char *temp = (char *) malloc (width + 2);

  Region region = XCreateRegion ();
  int miny = win_y + h, maxy = 0;

  XEvent xev;

  if (pid == 0)
    {
      pid = fork ();
      if (pid != 0)
	{
	  printf ("forking mode: pid=%d\n", pid);
	  exit (0);
	}
      close (0);
      close (2);
    }
  /* Install the Signal Handlers */
  InstallSigHandler ();

  /* Initialize linematrix
   */

  for (lin = 0; lin < listlen; lin++)
    {
      lines[lin].line = (char *) malloc (width);
      strcpy (lines[lin].line, "~");
      lines[lin].color = GetColor (color);
    }

  while ((error = lineinput (temp, width + 2, loglist->f)))
    {
      for (lin = 0; lin < (listlen - 1); lin++)
	{
	  strcpy (lines[lin].line, lines[lin + 1].line);
	  lines[lin].color = lines[lin + 1].color;
	}
      strcpy (lines[listlen - 1].line, temp);
    }

  redraw ();

  for (;;)
    {
      int linecnt = 0;
      struct logfile_entry *current;

      for (current = loglist; current != NULL; current = current->next)
	{
	  clearerr (current->f);
	  while (lineinput (temp, width + 2, current->f) == 0)
	    {
	      /* 
	       * print filename if any, and if last line was from
	       * different file
	       */
	      if (!(lastprinted && (lastprinted == current)) &&
		  current->desc[0])
		{
		  for (lin = 0; lin < (listlen - 1); lin++)
		    {
		      strcpy (lines[lin].line, lines[lin + 1].line);
		      lines[lin].color = lines[lin + 1].color;
		    }
		  sprintf (lines[listlen - 1].line, "[%s]",
			   current->desc);
		  lines[listlen - 1].color = current->color;
		  linecnt++;
		}
	      lastprinted = current;
	      linecnt++;
	      for (lin = 0; lin < (listlen - 1); lin++)
		{
		  strcpy (lines[lin].line, lines[lin + 1].line);
		  lines[lin].color = lines[lin + 1].color;
		}
	      strcpy (lines[listlen - 1].line, temp);
	      lines[listlen - 1].color = current->color;
	    }
	}

      if (linecnt > 0)		/* anything to update? */
	redraw ();
      else
        {
          XFlush (disp);
          if (!XPending (disp))
            {
              fd_set fdr;
              struct timeval to = { 3, 0 };

              FD_ZERO (&fdr);
              FD_SET (ConnectionNumber (disp), &fdr);
              select (ConnectionNumber (disp) + 1, &fdr, 0, 0, &to);
            }
        }

      /* we ignore possible errors due to windo resizing &c */
      while (XPending (disp))
	{
          XNextEvent (disp, &xev);
	  switch (xev.type)
	    {
	    case Expose:

	      {
		XRectangle r;
		r.x = xev.xexpose.x;
		r.y = xev.xexpose.y;
		r.width = xev.xexpose.width;
		r.height = xev.xexpose.height;
		XUnionRectWithRegion (&r, region, region);
                if (miny > r.y) miny = r.y;
                if (maxy < r.y+r.height) maxy = r.y+r.height;
	      }
	      break;
	    case MapNotify:
	    case UnmapNotify:
	      break;
	    case FocusOut:
	      infocus = 0;
	      break;
	    case FocusIn:
	      infocus = 1;
	      break;
	    default:
#ifdef DEBUGMODE
	      fprintf (stderr, "PANIC! Unknown event %d\n", xev.type);
#endif
	      break;
	    }
	  /* reload if requested */
	  if (reload && (lastreload + reload < time (NULL)))
	    {

	      system (command);
	      /* Why not use the same function?
	       * So we report and abort
	       * Maybe it's better to report and ignore
	       */

	      reopen (1);
	      lastreload = time (NULL);
	    }

	}

      if (!XEmptyRegion (region))
        {
          XSetRegion (disp, WinGC, region);
          refresh (lines, miny, maxy);
          XDestroyRegion (region);
          region = XCreateRegion ();
          maxy = 0; miny = win_y + h;
        }
    }
}

void
PrintHelp (char *myname)
{
  printf ("Usage: %s [options] file1[,color[,desc]] "
	  "[file2[,color[,desc]] ...]\n",
	  myname);
  printf (" -geometry geometry    [-g WIDTHxHEIGHT+X+Y                         ]\n");
  printf (" -color    color       [use color $color as default                 ]\n");
  printf (" -reload sec command   [reload after $sec and run command           ]\n");
  printf (" -font FONTSPEC        [(-fn) font to use.                          ]\n");
  printf (" -fork                 [fork into background                        ]\n");
  printf (" -shade                [add shading to font                         ]\n");
  printf ("\n");
  printf ("Example:\n%s -g 80x25+100+50 -font fixed /var/log/messages,green "
	  "/var/log/secure,red,'ALERT'\n", myname);
  exit (0);
}


int
main (int argc, char *argv[])
{

  int i;
  int filecount = 0;

  printf ("ROOT WINDOW TAIL v0.0.6\n"
	  "-- mjbaker@mtu.edu and irvdwijk@cs.vu.nl --\n");
  printf ("-- signal handling by andre@aleph.it\n");
  printf ("-- optimized refresh & many bugfixes by pcg@goof.com\n");


  /* window needs to be initialized before colorlookups can be done */

  InitWindow ();		// just a dummy to get the color lookups right

  lastreload = time (NULL);

  /*
   * More checking on the parameters would be nice
   */
#ifdef debug_expose
  frame = 1;
#endif

  for (i = 1; i < argc; i++)
    {
      if ((argv[i][0] == '-') && (argv[i][1] != 0))
	{
	  if (!strcmp (argv[i], "--?") ||
	      !strcmp (argv[i], "--help") ||
	      !strcmp (argv[i], "-h"))
	    PrintHelp (argv[0]);
	  else if (!strcmp (argv[i], "-g") || !strcmp (argv[i], "-geometry"))
	    {
	      i++;
	      XParseGeometry (argv[i], &win_x, &win_y,
			      &width, &listlen);
	    }
	  else if (!strcmp (argv[i], "-color"))
	    {
	      i++;
	      color = argv[i];
	    }
	  else if (!strcmp (argv[i], "-font") || !strcmp (argv[i], "-fn"))
	    {
	      i++;
	      fontName = argv[i];
	    }
	  else if (!strcmp (argv[i], "-fork"))
	    {
	      pid = 0;
	    }
	  else if (!strcmp (argv[i], "-reload"))
	    {
	      i++;
	      reload = atoi (argv[i]);
	      i++;
	      command = argv[i];
	    }

	  else if (!strcmp (argv[i], "-shade"))
	    shade = 1;
	  else if (!strcmp (argv[i], "-frame"))
	    frame = 1;
	  else
	    {
	      printf ("unknown option %s\n --help for help\n",
		      argv[i]);
	      exit (-1);
	    }
	}
      else
	/* it must be a filename */
	{
	  FILE *f;
	  struct logfile_entry *e;
	  char *fname, *desc, *fcolor = color, *sep;

	  /* 
	   * this is not foolproof yet (',' in filenames 
	   * are not allowed
	   */

	  fname = argv[i];
	  desc = fname;

	  if ((sep = strchr (argv[i], ',')))
	    {
	      *sep = '\0';
	      fcolor = sep + 1;

	      if ((sep = strchr (fcolor, ',')))
		{
		  *sep = '\0';
		  desc = sep + 1;
		}
	    }

	  if ((f = openLog (fname)) == NULL)
	    {
	      perror (fname);
	      exit (-1);
	    }
	  e = (struct logfile_entry *)
	    malloc (sizeof (struct logfile_entry));

	  strncpy (e->fname, fname, 255);
	  e->fname[255] = '\0';	/* just in case */
	  strncpy (e->desc, desc, 255);
	  e->desc[255] = '\0';	/* just in case */
	  e->f = f;
	  e->color = GetColor (fcolor);

	  e->next = NULL;

	  if (loglist == NULL)
	    loglist = e;
	  if (loglist_tail)
	    loglist_tail->next = e;
	  loglist_tail = e;
	  filecount++;
	}
    }


  InitWindow ();

  if (filecount == 0)
    {
      fprintf (stderr, "You did not specify any files to tail\n"
	       "use %s --help for help\n", argv[0]);
      exit (-1);
    }
  loop ();

  return 0;			/* to make gcc -Wall stop complaining */
}
