/*
 * 	Resolv
 * 	Copyright (c) 2001 - s23a <s23a@mail.ru> 
 * 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include "resolv.h"

/*
 * Misc variables 
 */
char address[IPRANGE_SIZE];	/* IP ranges */
char buf[IPRANGE_SIZE];		/* single ip address */
const char *progname;		/* programm name */
extern int h_errno;

/*
 * statistics variables 
 */
float resolv_count = 0;         /* total IP address resolved */
float timeout_count = 0;        /* total timeouts */
float search_count = 0;         /* total successful searchs */
float aliases_count = 0;        /* total aliasses found */

/* flags */
int fflag = 0;
int lflag = 0;
int sflag = 0;
int aflag = 0;
int dflag = 0;
int tflag = 0;
int iflag = 0;
int mflag = 0;
int eflag = 0;
int xflag = 0;
int verbose = 0;
int single = 0;
int t = 0;

int	extract_pattern(char *, FILE *);
int	search_pattern(char *, char *);
int	extract_addr(int *, FILE *);
void	sigalrm_handling();
int	check_addr(int *);
int	check_ipsingle(int *);
int	get_total_ips(int[]);
int	display_statistics(int[], int, int, FILE *);

/*
 * Start game :)
 */
int
main(argc, argv)
	int argc;
	char *argv[];
{
	struct hostent *host, *gethostbyaddr();
	char oval[4], *file = NULL, *pattern = NULL, **ptr;
	int final[8], opts, s, timeout, wait;
	register int acount, bcount, ccount, dcount, a_single, b_single, c_single, d_single;
	FILE *fildes, *sfile, *iplistfile, *plist;

	a_single = b_single = c_single = d_single = 0;
	acount = bcount = ccount = dcount = 0;
	progname = argv[0];

	while ((opts = getopt(argc, argv, "hf:s:al:dt:i:x:vme:")) != EOF) {
		switch (opts) {
		case 'f':
			if (strlen(optarg) > MAX_FILE)
				error(1);
			file = optarg;	/* log file descriptor */
			fflag = 1;
			break;
		case 's':
			if (strlen(optarg) > MAX_PATTERN)
				error(2);
			pattern = optarg; /* single searchs log file descriptor */
			sflag = 1;
			break;
		case 'a':
			aflag = 1;
			break;
		case 'l':
			if (strlen(optarg) > MAX_FILE)
				error(3);
			/* ip addresses file */ 
			if ((iplistfile = fopen(optarg, "r")) == NULL) {
				perror("open()");
				exit(-1);
			}
			lflag = 1;
			break;
		case 'd':
			dflag = 1;
			break;
		case 't':
			tflag = 1;
			timeout = atoi(optarg);
			break;
		case 'i':
			iflag = 1;
			wait = atoi(optarg);
			break;
		case 'x':
			if (strlen(optarg) > MAX_FILE)
				error(4);
			/* patterns list file */
			if ((plist = fopen(optarg, "r")) == NULL) {
				perror("open()");
				exit(-1);
			}
			xflag = 1;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'm':
			mflag = 1;
			break;
		case 'e':
			if (strlen(optarg) > FORMAT_SIZE)
				error(5);
			strncpy(vars.format, optarg, sizeof(vars.format));
			eflag = 1;
			break;
		case 'h':
		default:
			usage();
			break;
		}
	}

	if (argc == 1 || ((dflag && !fflag) && !mflag) ||
 		 (sflag && xflag) || (fflag && mflag)) 
		usage();

	/* if lflag,  don't pass IP address by command line */
	if (!lflag) {
		argc -= optind;
       		argv += optind;

		if (argc != 1)
			usage();

		/* check address size */
		if (strlen(*argv) > IPRANGE_SIZE)
			error(6);

		sscanf(*argv, "%s", address);
	}

	extract_addr(final, iplistfile);

	/* open log file */
	if (fflag) {
		if ((fildes = fopen(file, "a")) != 0) {
			if ((s = open("/dev/tty", O_RDWR)) != -1) {
				ioctl(s, TIOCNOTTY, (char*) NULL);
				close(s);
			}
		}
		else {
			perror("open()");
			exit(-1);
		}
	}
	if (sflag || xflag)
		if ((sfile = fopen(LOG_FILE, "a")) == 0) {
			perror("open()");
			exit(-1);
		}
	/* daemonize */
	if (dflag) {
		switch (fork()) {
		case 0:
			break;
		case -1:
			perror("fork()");
			exit(-1);
			break;
		default:
			exit(-1);
			break;
		}
		close(STDIN_FILENO);
		close(STDOUT_FILENO);
		close(STDERR_FILENO);
	}
	if (!tflag || !timeout)
		timeout = DEFAULT_TIMEOUT;
	if (!iflag)
		wait = DEFAULT_WAIT;

	signal(SIGALRM, (void (*)())sigalrm_handling);

	do {		
		if (mflag) {
			if ((fildes = fopen(buf, "a")) != 0) { 
	                        if ((s = open("/dev/tty", O_RDWR)) != -1) {
					ioctl(s, TIOCNOTTY, (char*) NULL);
					close(s);
				}       
			} 
	                else {
        	                perror("open()");
               	         	exit(-1);
               		}
		}        
		/* check for ip single */
		if (!eflag)
			single = check_ipsingle(final);

		resolv_count = 0;
		timeout_count = 0;
	
		if (final[1] == -1)
			++a_single;
		if (final[3] == -1)
			++b_single;
		if (final[5] == -1)
			++c_single;
		if (final[7] == -1)
			++d_single;	

		acount = final[0];
		if (a_single)
			goto A_Sing;	

	/* 1: A class [ AAA.xxx.xxx.xxx ] */
	while (final[0] <= final[1]) {
A_Sing:
		if (single) {	
			(void)fprintf(SFLAGS, SA, final[0], ((final[1] !=-1)? final[1]: 0));
			(void)fprintf(SFLAGS," |   |\n");
 		}

		bcount = final[2];
		if (b_single)    
	                goto B_Sing;

		/* 2: B class [ xxx.BBB.xxx.xxx ] */
		while (final[2] <= final[3]) {
B_Sing:			
			if (single) {
				(void)fprintf(SFLAGS, SB, final[0], final[2], 
					    ((final[3]!=-1)? final[3]: 0));
				(void)fprintf(SFLAGS," |   |   |\n");
			}
		
			ccount = final[4];
		        if (c_single) 
               	        	goto C_Sing;

			/* 3: C class [ xxx.xxx.CCC.xxx ] */
			while (final[4] <= final[5]) {
C_Sing:			
				if (single)
					(void)fprintf(SFLAGS, SC, final[0], final[2], final[4], 
					 			((final[5]!=-1)? final[5]: 0));

				dcount = final[6];
				if (d_single)
					goto D_Sing;
			
		/* 4: end */
		while (final[6] <= final[7]) {
D_Sing:					
			oval[0]=(unsigned char) final[0];
			oval[1]=(unsigned char) final[2];
			oval[2]=(unsigned char) final[4];
			oval[3]=(unsigned char) final[6];

			/* sleep 'wait' seconds */
			if (iflag) 
				sleep(wait);
			/* set timeout alarm signal */
			host = NULL;		
			alarm(timeout);
			host = gethostbyaddr(oval, 4, AF_INET);
			alarm(0);

			/* check for break timeout limit */
			if (host != NULL && t == 0) {	
				/* (1) print host name */
				if (!single) { 
					if(!eflag)
						(void)fprintf(SFLAGS, "%d.%d.%d.%d = %s\n", final[0], 
							final[2], final[4], final[6], host->h_name);
				}
				else
					(void)fprintf(SFLAGS, SD, STREE, final[0], final[2], 
							final[4], final[6], host->h_name);

				/* (2) search single pattern */
				if (sflag)
					if (search_pattern(pattern, host->h_name)) {
						fprintf(sfile, "%d.%d.%d.%d == %s\n", final[0], final[2], 
									final[4], final[6], host->h_name);
						(void)fflush(sfile);
						search_count += 1;

						if (eflag) {
							alarm(0);
							executing(host->h_name, fildes, final);
						}
						if (verbose && !eflag)
							(void)fprintf(SFLAGS,"%s found pattern!!\n",STREE);
					}

				 /* (3) searchs multiples patterns */				
				 if (xflag)
					if (!extract_pattern(host->h_name, plist)) {
						fprintf(sfile, "%d.%d.%d.%d == %s\n", final[0], final[2],
									final[4], final[6], host->h_name);
						fflush(sfile);
						search_count += 1;
						if (eflag) {
							alarm(0);
							executing(host->h_name, fildes, final);
						}
						else {
							if (single) {
								if (verbose && !eflag)
								fprintf(SFLAGS,"%s\tfound pattern!!\n",STREE);
							}
							else if (verbose && !eflag)
								(void)fprintf(SFLAGS," found pattern!!\n");
						}
					}

				/* (4) show alias */
				if (aflag) {
					ptr = host->h_aliases;
					while (*ptr != NULL) {
						if (single)
							fprintf(SFLAGS,"%s   + [alias] -> %s\n", STREE, *ptr);
						else
							fprintf(SFLAGS,"\t[alias] -> %s\n", *ptr);
						++ptr;
						if (*ptr == NULL)
							if (single)
								fprintf(SFLAGS, "%s\n",	STREE);
						aliases_count += 1;		
					}
				}
				if (fflag || mflag)
					(void)fflush(fildes);

				resolv_count += 1;
			}
			
			if (verbose) {
				if (t == 1) {
					if (single)
						fprintf(SFLAGS, SD, STREE, final[0], final[2], final[4],
										 final[6], "timeout");
					else	
						fprintf(SFLAGS, "%d.%d.%d.%d = %s\n", final[0], final[2], 
									final[4], final[6], "timeout");
					t = 0;
				}
				else if (h_errno == HOST_NOT_FOUND) 
					fprintf(SFLAGS, SD, STREE, final[0], final[2], final[4], 
								final[6], "host not found");
			}

			if (fflag || mflag)
				(void)fflush(fildes);

			final[6] += 1;
				}
				final[6] = dcount;
				if (single) 
	    			        fprintf(SFLAGS, " |   |   |\n");
				final[4] += 1;
			}
			final[4] = ccount;
			if (single)
		  	        fprintf(SFLAGS, " |   |\n");
			final[2] += 1;
		}
		final[2] = bcount;
		if (single)
 	 	        fprintf(SFLAGS, " |\n");
		final[0] += 1;

		}
		final[0] = acount;		
			
		display_statistics(final, timeout, wait, fildes);
		
		if (fflag || mflag)
			fflush(fildes);	
		if (mflag)
			fclose(fildes);
		
	} while (lflag? extract_addr(final, iplistfile) == NULL: 0);
	
	/* close all descriptors files */
	if (fflag)
		fclose(fildes);
	if (sflag || xflag)
		fclose(sfile);
	if (lflag)
		fclose(iplistfile);
	if (xflag)
		fclose(plist);

	return (0);
}


/*
 * Extract patterns from file, then call
 * search_pattern() function 
 */
int
extract_pattern(name, file)
	char *name;
	FILE *file;
{
	register int i;
	char pat[MAX_PATTERN];

	(void)bzero((char*)&pat, sizeof(pat));
	(void)fseek(file, 0L, SEEK_SET);

	while ((fgets(pat, sizeof(pat), file)) != NULL) {
		/* set '\0' character to end string */
		for (i = 0; i < MAX_PATTERN; i++)
			if (pat[i] == '\n')
				pat[i] = '\0';
		if (search_pattern(pat, name))
			return (0);
	}
	return (-1);
}						


/*
 * Searchs concordances betwen s1 (pattern) and s2 (hostname)
 */
int
search_pattern(s1, s2)
	char *s1, *s2;
{		
	register int i, j;
	int x, y, count, yes;

	i = j = x = y = count = 0;

	if (strlen(s1) > strlen(s2)) 
		error(7);

	x = ((strlen(s2) / strlen(s1)) -1);
	y = strlen(s2) % strlen(s1);

	for (i = 0; i < (strlen(s1) * x + 1 + y); i++, s2++) {
		yes = 1;
		for (j = 0; j < strlen(s1); j++)
			if (*(s1 + j) != *(s2 + j))
				yes = 0;
		if (yes)
			++count;
	}
	return (count);
}


/*
 * Convert a string to an int vector, that, 
 * contain ip ranges.
 */
int
extract_addr(fin, list)
	int *fin;
	FILE *list;
{
	char *p=NULL, a[4][8], *pa=NULL;
	register int i, j, num=0;

	(void)bzero((char*)&buf, sizeof(buf));
	(void)strncpy(buf, address, sizeof(address));

	if (lflag) {
		if (fgets(address, sizeof(address), list) == NULL)
			return (-1);

		(void)bzero((char*)&buf, sizeof(buf));		
		(void)snprintf(buf, strlen(address), "%s", address);
	}

	pa = (char*)a;
	(void)bzero((char*)a, 32);               

        for(i = 0; i < 8; i++)
                *(fin + i) = -1;
		/* empty buffer */	

	/* first division */
        for (i = 0, (p = strtok(address, ".")); p; (p = strtok(NULL, ".")), ++i) {
		if (i > 3)
			error(8);
	
		(void)strncpy((pa + (i * 8) + 0), p, 7);
		*(pa + (i * 8) + 8) = '\0'; /* set end */

		/* [1] check for non-digit char */
		for (j = 0; j < 8; j++)
			if ((*(pa +(i * 8) + j)) != '\0' && 
			    (*(pa +(i * 8) + j)) != '-' &&
			    (*(pa +(i * 8) + j)) != '\n') /* for -l option */
				if (isdigit((*(pa +(i * 8) + j))) == 0)
					error(8);
	}
	if (i < 4)
		error(8);

        p = NULL;              
	/* second division */   
        for (j = 0; j < 4; j++) {
                for (i = 0, (p = strtok((pa+(j*8)+0), "-")); p; (p = strtok(NULL,"-")), i++, num++) {
			if((num == 1) || (num == 3) || (num == 5))
                                if(i == 0) {
                                        *(fin + num) = -1;
                                        *(fin + (num+1)) = atoi(p);
                                        ++num;          
                                }       
                                else    
                                        *(fin + num) = atoi(p);
                	else            
                                *(fin + num) = atoi(p);	
                }
        }      

	check_addr(fin);
        return (0);
}


/* 
 * Increment timeout count variable
 */
void
sigalrm_handling()
{
	timeout_count += 1;
	t = 1;

	signal(SIGALRM, (void (*)())sigalrm_handling);
	return;
}


/*
 * Check the correct address ip or ip ranges format 
 */
int
check_addr(fin)
	int *fin;
{
	register int i, j;

	for (i = 0, j = 1; i < 8; i++, j += 2) {
		/* [2] check for size error */
                if (j <= 7)
                        if ((*(fin+j) != -1) && (*((fin+j) - 1) > *(fin+j)))
				error(8);
		/* [3] check for "x equal to y" error */		
		if (j <= 7)
			if ((*(fin+j) != -1) && (*((fin+j) - 1) == *(fin+j)))
				error(8);
		/* [4] check for range error */
		if (*(fin+i) < 0 || *(fin+i) > 255) {
			if (*(fin+i) == -1)
				continue;
				error(8);
		}
	}
	return (0);
}


/*
 * check for single ip
 */
int
check_ipsingle(fin)
	int *fin;
{
	if ((*(fin+1) == -1) && (*(fin+3) == -1) && 
	     (*(fin+5) == -1) && (*(fin+7) == -1))
		return (0);
	else
		return (-1);
}


/*
 * Get total ip addresses to scan
 */
int
get_total_ip(fin)
	int fin[8];
{
	int all = 0, yes = 0;
	register int i, j;

	if (fin[1] == -1 && fin[3] == -1 && 
	     fin[5] == -1 && fin[7] == -1)
		return (0);

	for (i = 7; i >= 1; i -= 2)
		if (fin[i] != -1) {
			yes = 1;
			for (j = 7; j >= i; j -= 2)
				if (fin[j] != -1 && j != i ) {
					all *= ((fin[i] + 1) - fin[i-1]);
					yes = 0;
					break;
				}
			if (yes)
				all += ((fin[i] + 1) - fin[i-1]);
		}
	return (all);
}                


/*
 * Show statistics information about network scan
 */
int
display_statistics(fin, timeout, wait, file)
	int fin[8], timeout, wait;
 	FILE *file;
{
        float total = 0;

	if ((total = get_total_ip(fin)) == 0)
		return (0);

	(void)fprintf(DESC,".-----------------------------------------------------.\n");
	(void)fprintf(DESC,"| Network Statistics                                  |\n");
	(void)fprintf(DESC,"+-----------------------------------------------------'\n");
	(void)fprintf(DESC,"| Ip range to scan:   %s\n", buf);
	(void)fprintf(DESC,"|\n");
	(void)fprintf(DESC,"| Successfull resolve         [%5.1f%%]\n", ((resolv_count * 100)
										 / total));
	(void)fprintf(DESC,"| Unsuccessfull resolve       [%5.1f%%]\n", (((total - (resolv_count
							 + timeout_count)) * 100) / total));
	(void)fprintf(DESC,"| Timeouts                    [%5.1f%%]\n", ((timeout_count * 100)
										 / total)); 
	(void)fprintf(DESC,"|\n");
	(void)fprintf(DESC,"| Total ips to resolve        [%6.0f]\n", total);
	(void)fprintf(DESC,"| Total successfull checks    [%6.0f]\n", resolv_count);
	(void)fprintf(DESC,"| Total unsuccessfull checks  [%6.0f]\n", (total - (resolv_count +
									   timeout_count)));
	(void)fprintf(DESC,"| Total timeouts              [%6.0f]\n", timeout_count);
	(void)fprintf(DESC,"| Total aliases found         [%6.0f]\n", aliases_count);
	(void)fprintf(DESC,"|\n");
	(void)fprintf(DESC,"| Total Successfull searchs   [%6.0f]\n", search_count);
	(void)fprintf(DESC,"|\n");
	(void)fprintf(DESC,"| Misc info:\n");
	(void)fprintf(DESC,"|		+ Total time to scan =\n");
	(void)fprintf(DESC,"|		+ String format = %s\n", vars.format);
	(void)fprintf(DESC,"|		+ Timeout set to     = %d seconds\n", timeout);
	(void)fprintf(DESC,"|		+ Wait second set to = %d seconds\n", wait);
	(void)fprintf(DESC,"'======================================================\n");

	return (0);
}
