/* 
 * Copyright (c) 2001 Secure Software Solutions
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

%{
#include <string.h>
#include "c-tokens.h"
#include "engine.h"

int lex_lineno = 1;

static int  identifier(void);
static int  string_const(void);
static void preprocessor(void);
static void cstyle_comment(void);
static void cppstyle_comment(void);
static void no_match(void);

#define YY_INPUT(buf, result, max_size)                                     \
    if (((result = fread(buf, 1, max_size, yyin)) == 0) && ferror(yyin))    \
        YY_FATAL_ERROR("input in flex scanner failed");                     \
    else {                                                                  \
        char *c, *end = (buf) + result - 1;                                 \
        for (c = (buf);  c < end;  c++) {                                   \
            if (*c == '\r') *c = ' ';                                       \
            if (*c == '\\' && *(c + 1) == '\n') {                           \
                memmove(c + 1, c + 2, end - c);                             \
                result--;                                                   \
                end--;                                                      \
                *c = '\r';                                                  \
            }                                                               \
        }                                                                   \
        if (*end == '\r') *end = ' ';                                       \
        if (*end == '\\') {                                                 \
            result--;                                                       \
            fseek(yyin, -1, SEEK_CUR);                                      \
        }                                                                   \
    }

%}

%%
"#"                     { preprocessor(); }
"/*"                    { cstyle_comment(); }
"//"                    { cppstyle_comment(); }

"auto"                  { return AUTO; }
"break"                 { return BREAK; }
"case"                  { return CASE; }
"char"                  { return CHAR; }
"const"                 { return CONST; }
"continue"              { return CONTINUE; }
"default"               { return DEFAULT; }
"do"                    { return DO; }
"double"                { return DOUBLE; }
"else"                  { return ELSE; }
"enum"                  { return ENUM; }
"extern"                { return EXTERN; }
"float"                 { return FLOAT; }
"for"                   { return FOR; }
"goto"                  { return GOTO; }
"if"                    { return IF; }
"int"                   { return INT; }
"long"                  { return LONG; }
"register"              { return REGISTER; }
"return"                { return RETURN; }
"short"                 { return SHORT; }
"signed"                { return SIGNED; }
"sizeof"                { return SIZEOF; }
"static"                { return STATIC; }
"struct"                { return STRUCT; }
"switch"                { return SWITCH; }
"typedef"               { return TYPEDEF; }
"union"                 { return UNION; }
"unsigned"              { return UNSIGNED; }
"void"                  { return VOID; }
"volatile"              { return VOLATILE; }
"while"                 { return WHILE; }

[a-zA-Z_]([a-zA-Z_]|[0-9]|\$|[\r])* { return identifier(); }

0[xX][a-fA-F0-9]+(u|U|l|L)*     { return HEX_CONST; }
0[0-9]+(u|U|l|L)*               { return OCT_CONST; }
[0-9]+(u|U|l|L)*                { return DEC_CONST; }
'(\\.|[^\\'])+'                 { return CHAR_CONST; }

[0-9]+[Ee][+-]?[0-9]+(f|F|l|L)*              { return FLOAT_CONST; }
[0-9]*"."[0-9]+([Ee][+-]?[0-9]+)?(f|F|l|L)*  { return FLOAT_CONST; }
[0-9]+"."[0-9]*([Ee][+-]?[0-9]+)?(f|F|l|L)*  { return FLOAT_CONST; }

\"(\\.|[^\\"])*\"       { return string_const(); }

">>="                   { return RIGHT_ASSIGN; }
"<<="                   { return LEFT_ASSIGN; }
"+="                    { return ADD_ASSIGN; }
"-="                    { return SUB_ASSIGN; }
"*="                    { return MUL_ASSIGN; }
"/="                    { return DIV_ASSIGN; }
"%="                    { return MOD_ASSIGN; }
"&="                    { return AND_ASSIGN; }
"^="                    { return XOR_ASSIGN; }
"|="                    { return OR_ASSIGN; }
">>"                    { return RIGHT_OP; }
"<<"                    { return LEFT_OP; }
"++"                    { return INC_OP; }
"--"                    { return DEC_OP; }
"->"                    { return PTR_OP; }
"&&"                    { return AND_OP; }
"||"                    { return OR_OP; }
"<="                    { return LE_OP; }
">="                    { return GE_OP; }
"=="                    { return EQ_OP; }
"!="                    { return NE_OP; }
";"                     { return ';'; }
"{"                     { return '{'; }
"}"                     { return '}'; }
","                     { return ','; }
":"                     { return ':'; }
"="                     { return '='; }
"("                     { return '('; }
")"                     { return ')'; }
"["                     { return '['; }
"]"                     { return ']'; }
"."                     { return '.'; }
"&"                     { return '&'; }
"!"                     { return '!'; }
"~"                     { return '~'; }
"-"                     { return '-'; }
"+"                     { return '+'; }
"*"                     { return '*'; }
"/"                     { return '/'; }
"%"                     { return '%'; }
"<"                     { return '<'; }
">"                     { return '>'; }
"^"                     { return '^'; }
"|"                     { return '|'; }
"?"                     { return '?'; }
 
[ \t\v\f]               { /* eat white space */ }
[\n\r]                  { lex_lineno++; }
.                       { no_match(); }

%%

int yywrap(void)
{
    return 1;
}

static
int identifier(void)
{
    char *  c;

    while ((c = strchr(yytext, '\r')) != (char *)NULL)
    {
        memmove(c, c + 1, strlen(c));
        lex_lineno++;
    }
    return IDENTIFIER;
}

static
int string_const(void)
{
    char *  c;

    while ((c = strchr(yytext, '\r')) != (char *)NULL)
    {
        memmove(c, c + 1, strlen(c));
        lex_lineno++;
    }
    return STRING_CONST;
}

static
void cstyle_comment(void)
{
    char    c;

    while ((c = input()) && c != -1)
    {
        if (c == '\n')
            lex_lineno++;
        while (c == '*')
        {
            if (!(c = input()) || c == -1)
                return;
            if (c == '\n')
                lex_lineno++;
            if (c == '/')
                return;
        }
    }
}

static
void cppstyle_comment(void)
{
    char    c;

    while ((c = input()) && c != -1)
    {
        if (c == '\n')
        {
            lex_lineno++;
            break;
        }
    }
}

static
void preprocessor(void)
{
    char    c;

    while ((c = input()) && c != -1)
    {
        if (c == '\n')
        {
            lex_lineno++;
            break;
        }

        /* handle multi-line comments beginning on a preprocessor line */
        if (c == '/')
        {
            if (!(c = input()) || c == -1)
                break;
            if (c == '*')
            {
                int save_lineno = lex_lineno;

                cstyle_comment();
                if (lex_lineno != save_lineno)
                    break;
                continue;
            }
            unput(c);
        }
    }
}

static
void no_match(void)
{
    fprintf(stderr, "%s:%d: warning: bad token `%s'\n", current_file, lex_lineno, yytext);
}
