/*
 * Placed into the public domain in 2002
 * by the author, Curt Sampson <cjs@cynic.net>.
 */

#ifndef FILELIST_H
#define FILELIST_H

/*
 *  We keep a (singly linked) list of files from which we are reading, and
 *  the ranges within each file. The ranges are always in blocksize units.
 */

struct filelist_entry {
    char	*path;		/* pathname to the file */
    int		fdesc;		/* descriptor when open */
    long	start_block;	/* the block from which we read */
    long	end_block;	/* one more than the last block we read */
    struct filelist_entry *next;	/* pointer to next in chain. */
};

struct filelist {
    int		blocksize;	/* in bytes */
    int		open_flags;	/* flags for open(2) */
    struct filelist_entry *head;	
    /* total of ((end_block - start_block) + 1) for all entries */
    long	total_blocks;
};
typedef struct filelist * FILELIST;

/*
 * Public methods.
 */

/*
 * Create an empty filelist.
 * The blocksize will be as specified, and when a file in the list must
 * be opened, it will be opened with flags open_flags.
 * This returns a FILELIST, or NULL on failure (with errno set).
 */
FILELIST filelist_create(int blocksize, int open_flags);

/* Delete a filelist. */
void filelist_delete(FILELIST);

/* Return the blocksize of a filelist. */
int filelist_blocksize(FILELIST);

/* Return the total number of blocks in the list. */
long filelist_totalblocks(FILELIST);

/*
 * Add a file to the filelist.
 * The filename should be a zero-terminated string with the name of the
 * file, optionally followed by a ":start-end" range, where start and
 * end are integers indicating the first and last blocks across which
 * to read. If this is not specified, the beginning and end of the file
 * (rounded down to the last full block) will be used (if they can be
 * determined).
 *
 * Return value:
 *	>= 0	add was successful
 *	-1	Error in errno.
 *	-2	Cannot determine size of file, and was not specified.
 *	-3	Invalid block range specified.
 */
int filelist_add(FILELIST, char *path);

/*
 * Given a block number between 0 and  filelist_totalblocks(), return
 * the filelist_entry that contains that block number.
 * This is an internal function exposed for testing purposes only.
 *
 * Returns:
 *	A pointer to a filelist_entry (don't modify this!) if the block
 *	number is valid, or NULL if it isn't. Also fills in *remainder
 *	with the number of blocks offset within the the file.
 */
struct filelist_entry * filelist_getblockentry(FILELIST, long block,
    long *offset_block);

/*
 * Given a block number between 0 and  filelist_totalblocks(), find the
 * file that constains that block, and reuturn a file descriptor that
 * has been seek'd to that block's location within that particular file.
 * This will automatically open the file for reading, if it's not already
 * open.
 *
 * Returns: a file descriptor, or -1 on failure.
 */
int filelist_getfiledesc(FILELIST, long block);

#endif /* !FILELIST_H */
