/*
 * Placed into the public domain in 2002
 * by the author, Curt Sampson <cjs@cynic.net>.
 */

#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>

#include "filelist.h"


FILELIST
filelist_create(int blocksize, int open_flags)
{
    struct filelist * list;

    list = malloc(sizeof(struct filelist));
    if (list != NULL) {
	list->blocksize = blocksize;
	list->open_flags = open_flags;
	list->head = NULL;
	list->total_blocks = 0;
    }
    return list;
}

void
filelist_delete(FILELIST list)
{
    struct filelist_entry *current, *next;

    if (list == NULL)
	return;

    current = list->head;
    while (current != NULL) {
	free(current->path);
	next = current->next;
	free(current);
	current = next;
    }
    free(list);
}

int
filelist_blocksize(FILELIST list)
{
    return list->blocksize;
}

long
filelist_totalblocks(FILELIST list)
{
    return list->total_blocks;
}

/*
 * This assumes that the start-end range is within the file size.
 */
static int
filelist_add_startend(FILELIST list, char *path, long start, long end)
{
    struct filelist_entry *entry, *newentry;

    if ((start < 0) || (start > end) || (end < 1))
	return -3;

    if (path == NULL) {
	errno = ENOENT;
	return -1;
    }

    if (access(path, R_OK) == -1)
	return -1;

    newentry = malloc(sizeof(struct filelist_entry));
    if (newentry == NULL) {
	errno = ENOMEM;
	return -1;
    }
    bzero(newentry, sizeof(struct filelist_entry));

    newentry->path = strdup(path);
    if (newentry->path == NULL) {
	errno = ENOMEM;
	free(newentry);
	return -1;
    }

    newentry->fdesc = -1;
    newentry->start_block = start;
    newentry->end_block = end;

    if (list->head == NULL) {
	list->head = newentry;
    } else {
	for (entry = list->head; entry->next != NULL; entry = entry->next)
	    ;
	entry->next = newentry;
    }
    list->total_blocks += newentry->end_block - newentry->start_block;

    return 0;
}

int
filelist_add(FILELIST list, char *path)
{
    struct stat statbuf;
    long start, end;
    char *realpath, *colon, *ptr;
    int retval;

    colon = strrchr(path, (int) ':');
    if (colon == NULL) {
	if (stat(path, &statbuf) == -1)
	    return -1;
	return filelist_add_startend(list, path, 0,
	    statbuf.st_size / list->blocksize);
    }

    realpath = strdup(path);
    *(strrchr(realpath, (int) ':')) = '\0';

    start = strtol(colon + 1, &ptr, 0);
    if (start == LONG_MAX || start < 0) {
	free(realpath);
	return -3;
    }

    if (*ptr == '-') {
	ptr++;
    } else {
	free(realpath);
	return -3;
    }

    end = strtol(ptr, &ptr, 0);
    if (end == LONG_MAX || end < 0 || end < start || *ptr != '\0') {
	free(realpath);
	return -3;
    }

    retval = filelist_add_startend(list, realpath, start, end);
    free(realpath);
    return retval;
}

struct filelist_entry *
filelist_getblockentry(FILELIST list, long block, long *offset_block)
{
    struct filelist_entry *entry;
    long length;

    if ((block < 0) || (block >= list->total_blocks))
	return NULL;

    if (list->head == NULL) {
	fprintf(stderr, "***** INTERNAL ERROR: filelist.c: list->head is NULL"
	    " but total_blocks > 0\n");
	return NULL;
    }

    entry = list->head;
    while (entry != NULL) {
	length = entry->end_block - entry->start_block;
	block -= length;
	if (block < 0) {
	    *offset_block = length + block + entry->start_block;
	    return entry;
	}
	entry = entry->next;
    }

    fprintf(stderr, "***** INTERNAL ERROR: filelist.c: total blocks in list"
	" is smaller than list->total_blocks\n");
    return NULL;
}

int
filelist_getfiledesc(FILELIST list, long block)
{
    struct filelist_entry * entry;
    long entry_offset;
    int fd;
    off_t offset;

    entry = filelist_getblockentry(list, block, &entry_offset);
    if (entry == NULL)
	return -1;

    fd = entry->fdesc;

    /*
     * Open the file, if it's not already open.
     * XXX Should have code here to deal with closing a file when too
     * many are open.
     */
    if (fd  < 0) {
	fd = open(entry->path, list->open_flags, 0777);
	if (fd == -1)
	    return -1;
	entry->fdesc = fd;
    }

    offset = lseek(fd, ((off_t) list->blocksize) * ((off_t) entry_offset),
	SEEK_SET);
    if (offset == -1)
	return -1;

    return fd;
}
