/***************************************************************************
 *   Copyright (C) 2005 by Simon Perreault   *
 *   nomis80@nomis80.org   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "quadkonsole.h"

#include "mousemovefilter.h"

#include <kdebug.h>
#include <kiconloader.h>
#include <klibloader.h>
#include <klocale.h>
#include <kparts/mainwindow.h>

#include <qaction.h>
#include <qapplication.h>
#include <qlayout.h>


QuadKonsole::QuadKonsole( int rows, int columns, bool clickfocus )
    : KParts::MainWindow( 0, "QuadKonsole" )
    , mFilter(0)
{
    if ( rows < 1 ) {
        kdError() << "Number of rows must be at last one." << endl;
        exit(1);
    }
    if ( columns < 1 ) {
        kdError() << "Number of columns must be at least one." << endl;
        exit(1);
    }

    QWidget* centralWidget = new QWidget( this, "central widget" );
    mLayout = new QGridLayout( centralWidget, rows, columns, 0, 1 );
    setCentralWidget(centralWidget);

    if ( !clickfocus ) {
        mFilter = new MouseMoveFilter(this);
    }

    // Try to find libkonsolepart
    mFactory = KLibLoader::self()->factory( "libkonsolepart" );
    if (mFactory) {
        // Create the parts
        for ( int i = 0; i < rows; ++i ) {
            for ( int j = 0; j < columns; ++j ) {
                mKonsoleParts.push_back( createPart(i,j) );
            }
        }
    }
    else {
        kdFatal() << "No libkonsolepart found !" << endl;
    }

    QAction* goRight = new QAction( QString::null, CTRL + SHIFT + Key_Right,
            this );
    connect( goRight, SIGNAL(activated()), this, SLOT(focusKonsoleRight()) );

    QAction* goLeft = new QAction( QString::null, CTRL + SHIFT + Key_Left, this
            );
    connect( goLeft, SIGNAL(activated()), this, SLOT(focusKonsoleLeft()) );

    QAction* goUp = new QAction( QString::null, CTRL + SHIFT + Key_Up, this );
    connect( goUp, SIGNAL(activated()), this, SLOT(focusKonsoleUp()) );

    QAction* goDown = new QAction( QString::null, CTRL + SHIFT + Key_Down, this
            );
    connect( goDown, SIGNAL(activated()), this, SLOT(focusKonsoleDown()) );

    setIcon( DesktopIcon("konsole") );
}

QuadKonsole::~QuadKonsole()
{
    for ( PartVector::const_iterator i = mKonsoleParts.begin(); i !=
            mKonsoleParts.end(); ++i ) {
        delete *i;
    }
}

KParts::ReadOnlyPart* QuadKonsole::createPart( int row, int column )
{
    // Create a part
    KParts::ReadOnlyPart* part =
        dynamic_cast<KParts::ReadOnlyPart*>( mFactory->create(this,
                    "konsolepart", "KParts::ReadOnlyPart") );
    connect( part, SIGNAL(destroyed()), SLOT(partDestroyed()) );

    // Make this part's widget activate on mouse over
    if (mFilter) {
        part->widget()->setMouseTracking(true);
        part->widget()->installEventFilter(mFilter);
    }

    // Add widget to layout
    part->widget()->reparent( centralWidget(), QPoint(0,0) );
    mLayout->addWidget( part->widget(), row, column );
    part->widget()->show();

    return part;
}

class PartDeletedEvent : public QCustomEvent
{
    public:
        PartDeletedEvent( int row, int column )
            : QCustomEvent( QEvent::User + 1 )
            , mRow(row)
            , mColumn(column)
        {
        }

        inline int row() const { return mRow; }
        inline int column() const { return mColumn; }

    private:
        int mRow, mColumn;
};

void QuadKonsole::partDestroyed()
{
    kdDebug() << k_funcinfo << endl;

    PartVector::iterator part = std::find( mKonsoleParts.begin(),
            mKonsoleParts.end(), sender() );
    if ( part != mKonsoleParts.end() ) {
        *part = 0;

        // A terminal emulator has been destroyed. Plan to start a new one.
        int index = part - mKonsoleParts.begin();
        int row = index / mLayout->numCols();
        int col = index % mLayout->numCols();

        QApplication::postEvent( this, new PartDeletedEvent(row, col) );
    }
}

void QuadKonsole::customEvent( QCustomEvent* e )
{
    kdDebug() << k_funcinfo << endl;

    if ( e->type() == QEvent::User + 1 ) {
        // Start a new part.
        PartDeletedEvent* pde = static_cast<PartDeletedEvent*>(e);
        mKonsoleParts[ pde->row() * mLayout->numCols() + pde->column() ] =
            createPart( pde->row(), pde->column() ); }
}

void QuadKonsole::focusKonsoleRight()
{
    for ( PartVector::iterator part = mKonsoleParts.begin(); part !=
            mKonsoleParts.end(); ++part ) {
        if ( (*part)->widget()->hasFocus() ) {
            ++part;
            if ( part == mKonsoleParts.end() ) {
                part = mKonsoleParts.begin();
            }
            (*part)->widget()->setFocus();
            break;
        }
    }
}

void QuadKonsole::focusKonsoleLeft()
{
    for ( PartVector::reverse_iterator part = mKonsoleParts.rbegin(); part !=
            mKonsoleParts.rend(); ++part ) {
        if ( (*part)->widget()->hasFocus() ) {
            ++part;
            if ( part == mKonsoleParts.rend() ) {
                part = mKonsoleParts.rbegin();
            }
            (*part)->widget()->setFocus();
            break;
        }
    }
}

void QuadKonsole::focusKonsoleUp()
{
    for ( int i = 0; i < mLayout->numRows(); ++i ) {
        for ( int j = 0; j < mLayout->numCols(); ++j ) {
            if ( mKonsoleParts[ i * mLayout->numRows() + j
                    ]->widget()->hasFocus() ) {
                if ( i == 0 ) {
                    i = mLayout->numRows() - 1;
                }
                else {
                    --i;
                }
                mKonsoleParts[ i * mLayout->numRows() + j
                    ]->widget()->setFocus();
                break;
            }
        }
    }
}

void QuadKonsole::focusKonsoleDown()
{
    for ( int i = 0; i < mLayout->numRows(); ++i ) {
        for ( int j = 0; j < mLayout->numCols(); ++j ) {
            if ( mKonsoleParts[ i * mLayout->numRows() + j
                    ]->widget()->hasFocus() ) {
                if ( i == mLayout->numRows() - 1 ) {
                    i = 0;
                }
                else {
                    ++i;
                }
                mKonsoleParts[ i * mLayout->numRows() + j
                    ]->widget()->setFocus();
                break;
            }
        }
    }
}

#include "quadkonsole.moc"
