/*
*  File      : qapp.cpp
*  Written by: alinden@gmx.de
*  Copyright : GPL
*
*  Subclassing for QApplication, contains the X11 event
*  filter and some static functions that are common to all
*  classes
*/

#include "defs.h"
#include "conf.h"
#include "keyboard.h"
#include "defaults.h"
#include "qapp.h"

// Toolbar items
Toolbar *qapp::tb;     // toolbar
pager   *qapp::pg;     // pager
winlist *qapp::wl;     // winlist
menu    *qapp::mn;     // menu
procbar *qapp::pb;     // procbar
apbar   *qapp::ap;     // apbox

QPixmap *qapp::leftwinpix;                  // window button pixmap
QPixmap *qapp::rightwinpix;                 // window button pixmap
QList <xwindow> qapp::clients;              // client list
QIntDict <xwindow> qapp::cwindows;          // client window lookup table
QIntDict <xwindow> qapp::pwindows;          // parent window lookup table
int qapp::next_x = 10;                      // autoplacement position counter
int qapp::next_y = defaults::tb_height+1;   // autoplacement position counter
int qapp::adesk = 0;                        // current desktop
Atom qapp::wm_protocols;
Atom qapp::wm_delete;
Atom qapp::wm_change_state;
Atom qapp::wm_state;
Atom qapp::wm_take_focus;
Atom qapp::wm_colormaps;
bool qapp::smode=FALSE;
QPalette *qapp::ipal;                       // window inactive palette
QPalette *qapp::upal;                       // window urgent palette
bool qapp::mrb = FALSE;                     // move/resize active
xwindow *qapp::focusclient = NULL;          // currently active client
QMap <QString, int> qapp::cprops;           // client flags for Property command
QMap <QString, int> qapp::apclients;        // Toolbar clients, position number
bool qapp::sighup = FALSE;                  // SIGHUP received
int qapp::servershapes;                     // server supports shapes

static int ShapeEventBase;                  // event base for shape extension
static bool rootptr = TRUE;                 // mouse pointer in root window

void sig_term(int);

qapp::qapp(int &argc, char **argv) : QApplication(argc, argv) 
{
	// get WM protocols required by ICCCM
	
	wm_protocols = XInternAtom(qt_xdisplay(), "WM_PROTOCOLS", FALSE); 
	wm_delete = XInternAtom(qt_xdisplay(), "WM_DELETE_WINDOW", FALSE);
	wm_change_state = XInternAtom(qt_xdisplay(), "WM_CHANGE_STATE", FALSE);
	wm_state = XInternAtom(qt_xdisplay(), "WM_STATE", FALSE);
	wm_take_focus = XInternAtom(qt_xdisplay(), "WM_TAKE_FOCUS", FALSE);
	wm_colormaps = XInternAtom(qt_xdisplay(), "WM_COLORMAP_WINDOWS", FALSE);

	// check if server supports nonrectangular windows
	
	int err;
	servershapes = XShapeQueryExtension(qt_xdisplay(), &ShapeEventBase, &err);
}

void qapp::setinactive(xwindow *client)  // set last active client to inactive
{
	if(client != focusclient)
	{
		if(focusclient != NULL && clients.find(focusclient) != -1)  // still there
			focusclient->setinactive();

		focusclient = client;
	}
}

void qapp::stopautofocus(void)
{
	if(focusclient != NULL && clients.find(focusclient) != -1)
		focusclient->stopautofocus();
}

void qapp::run_client(Window w)  // start new client
{
	xwindow *client;
	int apnumber;

	if((client = cwindows.find(w)) != NULL)
	{
		client->map();
	}
	else  // new client
	{
		XClassHint ch;
		if(XGetClassHint(qt_xdisplay(), w, &ch))
		{
			QString clname(ch.res_class);
			
			if(clname == "unnamed")  // workaround for some Qt versions
				clname = ch.res_name;
				
			XFree(ch.res_name);
			XFree(ch.res_class);


			if(! clname.isEmpty() && (apnumber = apclients[clname])) 
			{
				if(ap->add(w, apnumber, clname))  // add to toolbar
					return;
			}	
		}
		clients.append(new xwindow(w));
	}	
}

void qapp::install_colormap(Colormap cmap)
{
	static Colormap lastmap = None;

	if(cmap == lastmap)
		return;

	lastmap = cmap;
		
	if(cmap == None)
		cmap = DefaultColormap(qt_xdisplay(), DefaultScreen(qt_xdisplay()));
		
	XInstallColormap(qt_xdisplay(), cmap);
}

void qapp::read_cprops(void)  // read app defaults
{
	QString fname,cline,name,par;
	int flags;
	int apnum = 1;
	
	fname = get_cfile("appdefaults");
	
	if(fname.isNull())
		return;
	
	QFile istr(fname);
	
	if(! istr.open(IO_ReadOnly))
	{
		perror("cannot open appdefaults");
		return;
	}	
	cprops.clear();
	apclients.clear();
	
	while(! istr.atEnd())
	{
		istr.readLine(cline, 1024);
		QTextIStream si(&cline);
	
		si >> name;
		par = si.readLine();

		if(par.find("ToolBar") != -1)
		{
			apclients.insert(name, apnum++);
			continue;
		}
	
		flags = 0;	
		if(par.find("WindowListSkip") != -1)
			flags |= WindowListSkip;

		if(par.find("Sticky") != -1)
			flags |= Sticky;

		if(par.find("SmallFrame") != -1)
			flags |= SmallFrame;

		if(par.find("NoResize") != -1)
			flags |= NoResize;

		if(flags)
			cprops.insert(name, flags);

	}
	istr.close();

	// check for clients to update

	if(cprops.isEmpty())
		return;
		
	xwindow *client;
		
	for(client = clients.first(); client != NULL; client = clients.next())
	{
		flags = cprops[client->get_clientname()];
		if(flags != client->get_pflags())
			client->set_pflags(flags);
	}

	ap->remove();  // update clients on toolbar
}

QString qapp::get_cfile(char *name)  // get abs config file name
{
	QString fname(getenv("HOME"));
	
	if(! fname.isNull())  // user config dir
	{
		fname += "/.qlwm";
		QFileInfo fi(fname);

		if(! fi.isDir())
		{
			cerr << "WM: No user configuration (" << fname << "/)\n";
		}
		else
		{
			fname += '/';
			fname += name;

			QFileInfo fi(fname);
		
			if(fi.isReadable())
				return(fname);
		}
	}
	else cerr << "WM: cannot read HOME environment variable\n";
	
	fname = CONFDIR;   // system config dir
	fname += name;
	
	QFileInfo fi(fname);

	if(fi.isReadable())
		return(fname);
		
	perror((const char *)fname);
	fname = QString();
	return(fname);
}

void qapp::send_configurenotify(xwindow *client) 
{
	XConfigureEvent ce;

	ce.type = ConfigureNotify;
	ce.event = client->client_id();
	ce.window = ce.event;
	ce.x = client->x();
	ce.y = client->get_clienty();
	ce.width = client->width();
	ce.height = client->getcheight();
	ce.above = None;
	ce.border_width = 0;
	ce.override_redirect = 0;
	XSendEvent(qt_xdisplay(), ce.window, False, StructureNotifyMask, (XEvent *) &ce);
}

// event filter returns FALSE to pass event to qt's event handler

bool qapp::x11EventFilter(XEvent *event)  
{
	xwindow *client;
	Window w;
	XEvent ev;
	XConfigureRequestEvent *cev;
	XClientMessageEvent *mev;
	XCrossingEvent *xev;
	XCirculateRequestEvent *rev;
	XPropertyEvent *pev;
			
#ifdef DEBUGMSG		
#include "eventnames.h"
	if(event->type < 36)
		cerr << "Received: " << event_names[event->type] << " (WId:" << event->xany.window << ")\n";
#endif
	while(waitpid(-1, NULL, WNOHANG) > 0);

	if(sighup)
	{
		mn->readmenu();
		read_cprops();
		sighup = FALSE;
	}

	switch(event->type)
	{
		case DestroyNotify:
			w = event->xdestroywindow.window;

			if((client = cwindows.find(w)) != NULL)
			{
				clients.remove(client);
				pg->draw_pager();
				
				if(smode && client->isstate())
					keyboard::tscreen();  // turn off screen mode
			
				return TRUE;
			}	
			if(ap->remove(w))  // client on toolbar
				return TRUE;
				
			if(event->xdestroywindow.event != w)
				return TRUE;

			if(w == tb->winId() || w == pg->winId() || w == wl->winId() || w == mn->winId() || w == pb->winId())
				sig_term(SIGTERM);

			return FALSE;

		case MapNotify:
			if(event->xmap.event != event->xmap.window)
				return TRUE;
		
			if((client = pwindows.find(event->xmap.window)) != NULL)
				pg->add(client);  // add to pager

			return FALSE;

		case UnmapNotify:
			if((client = cwindows.find(event->xunmap.window)) != NULL)
			{
				if(event->xunmap.send_event)
				{
					// client requested transitions 
					// normal -> withdrawn
					// iconic -> withdrawn
					
					client->withdraw();
				}
				else
					client->unmap();
				
				return TRUE;	
			}
			if(event->xunmap.event != event->xunmap.window)
				return TRUE;

			if(pwindows.find(event->xunmap.window) != NULL)
				pg->draw_pager();
		
			return FALSE;

		case EnterNotify:
			xev = &event->xcrossing;
			
			if(event->xcrossing.window == qt_xrootwin())
			{
				stopautofocus();
				rootptr = TRUE;
			}
			else if(mrb == FALSE && (client = (xwindow *)widgetAt(xev->x_root, xev->y_root)) != NULL && 
			clients.find(client) != -1 && ((client = clients.current()) != focusclient || rootptr))
			{
				rootptr = FALSE;
				setinactive(client);  // old client to inactive, save new client
				
				if(xev->detail != NotifyInferior)
					client->startautofocus();
					
				client->setchildfocus(xev->time);
				client->setcmapfocus();
			}
			return FALSE;

		case ColormapNotify:
			if((client = cwindows.find(event->xcolormap.window)) != NULL)
			{
				client->setcmap(event->xcolormap.colormap);
				return TRUE;
			}	
			return FALSE;

		case PropertyNotify:
			pev = &event->xproperty;
			
			if((client = cwindows.find(pev->window)) != NULL)
			{
				if(pev->atom == XA_WM_NORMAL_HINTS)
				{
					client->get_wmnormalhints();
				}	
				else if(pev->atom == XA_WM_HINTS)
				{
					client->get_wmhints();
				}
				else if(pev->atom == XA_WM_NAME || pev->atom == XA_WM_ICON_NAME)
				{
					client->get_wmname();
				}
				else if(pev->atom == wm_colormaps)
				{
					client->get_colormaps();
					
					if(client == focusclient)
						client->setcmapfocus();
				}
				return TRUE;
			}
			return FALSE;

		case ConfigureNotify:
			if(event->xconfigure.event != event->xconfigure.window)
				return TRUE;
				
			if((client = pwindows.find(event->xconfigure.window)) != NULL)
			{
				pg->draw_pager();
				while(XCheckTypedEvent(qt_xdisplay(), ConfigureNotify, &ev));
			}
			return TRUE;

		case ReparentNotify:
			if((client = cwindows.find(event->xreparent.window)) != NULL &&
			event->xreparent.parent != client->winId())
			{
				clients.remove(client);
				pg->draw_pager();
			}	
			return TRUE;
		
		case ButtonPress:
			w = event->xbutton.window;
			
			if(w == qt_xrootwin())  // set focus to root window
				XSetInputFocus(qt_xdisplay(), w, RevertToPointerRoot, CurrentTime);

			if(w == tb->winId() || w == pb->winId() || w == ap->winId())
				XRaiseWindow(qt_xdisplay(), tb->winId());

			if(w == qt_xrootwin() || w == pg->winId())
				install_colormap(None);
				
			return FALSE;

		case ClientMessage:
			mev = &event->xclient;
			
			if(mev->message_type == wm_change_state && mev->format == 32 && 
			mev->data.l[0] == IconicState && (client = cwindows.find(mev->window)) != NULL)
				client->iconify();

			return TRUE;	

		case CirculateRequest:
			rev = &event->xcirculaterequest;
			
			if(rev->place == PlaceOnTop)
				XRaiseWindow(qt_xdisplay(), rev->window);
			else
				XLowerWindow(qt_xdisplay(), rev->window);
				
			return TRUE;

		case ConfigureRequest:
			cev = &event->xconfigurerequest;
			XWindowChanges wc;
			
			if((client = cwindows.find(cev->window)) != NULL)
			{
#ifdef DEBUGMSG
	cerr << "configure request to client (WId:" << client->winId() << ")\n";
#endif	
				if(cev->value_mask & (CWWidth|CWHeight|CWX|CWY))
				{
					if(smode && client->isstate())
						keyboard::tscreen(); 
						
					int cx,cy,cw,ch;

					if(cev->value_mask & CWWidth)
						cw = cev->width;
					else	
						cw = client->width();

					if(cev->value_mask & CWHeight)
						ch = cev->height;
					else
						ch = client->getcheight();

					if(cev->value_mask & CWX)
						cx = cev->x;
					else
						cx = client->x();

					if(cev->value_mask & CWY)
				    		cy = cev->y;
					else
						cy = client->y();
						
					client->resize_request(cx, cy, cw, ch);
					cev->value_mask &= ~(CWWidth|CWHeight|CWX|CWY);
				}
				
				if(! cev->value_mask)
					return TRUE;

				wc.width = client->width();
				wc.height = client->height();
				wc.x = client->x();
				wc.y = client->y();
				wc.border_width = 0;
				wc.sibling = cev->above;
				wc.stack_mode = cev->detail;

				XConfigureWindow(qt_xdisplay(), client->winId(), cev->value_mask, &wc);
				send_configurenotify(client);
			}
			else  // never mapped window
			{
				if(cev->window == tb->winId() || ap->client_exists(cev->window))  // deny requests on toolbar
					return TRUE;

#ifdef DEBUGMSG
	cerr << "configure request to unreparented window (WId:" << cev->window << ")\n";
#endif	

				wc.x = cev->x;
				wc.y = cev->y;
				wc.width = cev->width;
				wc.height = cev->height;
				cev->value_mask &= (CWX|CWY|CWWidth|CWHeight);
				
				XConfigureWindow(qt_xdisplay(), cev->window, cev->value_mask, &wc);	
			}
			return TRUE;
			
		case MapRequest:
			run_client(event->xmaprequest.window);
			return TRUE;
			
		case KeyPress:
			return(keyboard::keypress(&event->xkey));

		default:
			if(servershapes && event->type == (ShapeEventBase + ShapeNotify))
			{
				XShapeEvent *sev = (XShapeEvent *)event;
		
				if((client = cwindows.find(sev->window)) != NULL)
				{
					client->reshape();
					return TRUE;
				}	
			}
	}
	return FALSE;
}
