/************************** * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
*************************** * * * * * * * * * * * * **************************/

#include "config.h"
#include "accounts.h"
#include "transaction.h"
#include "qhaccacctman.h"

#include <time.h>
#include <iostream.h>

const int Transaction::NUM=          0;
const int Transaction::PAYEE=        1;
const int Transaction::MEMO=         2;
const int Transaction::INFOARRAYSIZE=3;
unsigned int Transaction::NEXTID=    1;

Transaction::Transaction( QDate d, QString _num, QString _payee,
			  QString _memo, float s ) : QObject(){
  init( _num, _payee, _memo );

  sum=s;
  date=d;
  reco=NO;
  owner=0;
}

Transaction::Transaction( const Transaction& copyMe ){
  if ( this==&copyMe ) return;
  init( copyMe.get( NUM ), copyMe.get( PAYEE ), copyMe.get( MEMO ) );

  sum=copyMe.getSum();
  date=copyMe.getDate();
  reco=copyMe.getReconciled();
  owner=copyMe.getAccount();  
}

void Transaction::init( QString n, QString p, QString m ){
  id=NEXTID++;
  loadArr=0;
  loadArrSize=0;

  info=new QString[INFOARRAYSIZE];
  info[NUM]=n;
  info[PAYEE]=p;
  info[MEMO]=m;

  splits=QHaccVector( 0, 1, 1 );
  splits.setNosort( true );
}

Transaction::~Transaction(){
#ifdef QDEBUG
  cout<<"    ~trans: deleting #"<<getID();
  if ( owner ) cout<<" from "<<owner->getName();
  cout<<endl;
#endif

  // before we go, clean out the splits
  dropSplits();

  delete [] info;
  if ( loadArr ) delete [] loadArr;
}

QDate Transaction::getDate() const { return date; }
bool Transaction::isReconciled( RecoType r) const { return reco==r; }
float Transaction::getSum() const { return sum; }
const QString Transaction::get( int i ) const { return QString( info[i] ); }
unsigned int Transaction::getID() const { return id; }
QHaccVector Transaction::getSplits() const{ return splits; }
Account * Transaction::getAccount() const { return owner; }
Transaction::RecoType Transaction::getReconciled() const { return reco; }

bool Transaction::isSingle() const { return splits.isEmpty(); }
bool Transaction::isSplit() const { return ( splits.length()>1 ); }

QString Transaction::getPairAName() const {
  if ( isSingle() ) return QString();
  if ( isSplit() ) return "SPLIT";
  return splits.getTrans( 0 )->getAccount()->getName();
}

QString Transaction::getPairAFName() const {
  if ( isSingle() ) return QString();
  if ( isSplit() ) return "SPLIT";
  return splits.getTrans( 0 )->getAccount()->getName( Account::FULL );
}

void Transaction::setAccount( Account * a ){
  owner=a;
  emit infoChanged();
}

void Transaction::setReconciled( RecoType b ){
  bool change=( reco!=b );
  if ( change ){
    reco=b;
    emit recChanged();
  }
}

void Transaction::setAll( const Transaction * copyMe ){
  set( MEMO, copyMe->get( MEMO ) );
  set( NUM, copyMe->get( NUM ) );
  set( PAYEE, copyMe->get( PAYEE ) );

  setDate( copyMe->getDate() );
  setSum( copyMe->getSum() );
  setReconciled( copyMe->getReconciled() );
}

void Transaction::set( int i, QString s ){
  bool change=( s!=info[i] );
  if ( change ){
    info[i]=QString( s );
    for ( int j=0; j<splits.length(); j++ )
      splits.getTrans( j )->set( i, s );
    emit infoChanged();
  }
}
void Transaction::setDate( QDate d ){
  bool change=( date!=d );
  if ( change ){
    date=d;
    for ( int j=0; j<splits.length(); j++ ) splits.getTrans( j )->setDate( d );
    emit dateChanged();
  }
}

void Transaction::setSum( float newSum ){
  // FIXME: just change the sums. Let the accounts take
  // care of the accounting! 
  // but maybe we need to connect the account to some signal first!
  if ( newSum!=sum ){
    float oldSum=sum;

    if ( isSplit() ){
      sum=0;
      for ( int j=0; j<splits.length(); j++ )
	sum+=splits.getTrans( j )->getSum();
    }
    else{
      sum=newSum;
      for ( int j=0; j<splits.length(); j++ ){
	Transaction * t=splits.getTrans( j );
	if ( t->getAccount()->isType( ASSET )==
	     getAccount()->isType( ASSET ) ) t->setSum( -newSum );
	else t->setSum( newSum );
      }
    }

    emit balanceChanged( oldSum, sum );
  }
}

Transaction::Transaction( QString rff ){
  // this constructor is used for creating 
  // a transaction from a startup file
  // It's important to read the info in the same
  // order it was written by getWriteableString
  int end=0,start=0;

  //id
  end=rff.find( "||", start );
  id=rff.mid( start, end-start ).toUInt();
  if ( id>=NEXTID ) NEXTID=id+1;
  start=end+2;
  
  // this is the number of splits
  end=rff.find( "||", start );
  loadArrSize=rff.mid( start, end-start ).toInt();
  start=end+2;

  // keep the trans AND account info for later
  loadArr=new unsigned int[loadArrSize*2];
  for ( int i=0; i<loadArrSize*2; i++ ){ 
    end=rff.find( "||", start );
    loadArr[i]=rff.mid( start, end-start ).toUInt();
    start=end+2;
  }

  //finally, start loading something
  info=new QString[INFOARRAYSIZE];
  for ( int i=0; i<INFOARRAYSIZE; i++){
    end=rff.find( "||", start );
    info[i]=rff.mid( start, end-start );
    start=end+2;
  }

  // this is the sum
  end=rff.find( "||", start );
  sum=rff.mid( start, end-start ).toFloat();
  start=end+2;

  //reconciled
  end=rff.find( "||", start );
  reco=( rff.mid( start, end-start ) == "Y" ? YES : NO );
  start=end+2;

  //date
  int d,m,y;
  end=rff.find( "/", start );
  m=rff.mid( start, end-start ).toInt();
  start=end+1;
  end=rff.find( "/", start );
  d=rff.mid( start, end-start ).toInt();
  y=rff.mid( end+1 ).toInt();
  date=QDate( y, m, d );
}

QString Transaction::getWriteableString() const {
  QString s, temp;

  //id
  s.append( temp.setNum( getID() ) );
  s.append( "||" );
  
  //number of splits
  s.append( temp.setNum( splits.length() ) );
  s.append( "||" );
  
  //split info
  for ( int i=0; i<splits.length(); i++ ) {
    s.append( temp.setNum( splits.getTrans( i )->getID() ) );
    s.append( "||" );
    s.append( temp.setNum( splits.getTrans( i )->getAccount()->getID() ) );
    s.append( "||" );
  }
  
  //info array
  for ( int i=0; i<INFOARRAYSIZE; i++) s+=info[i]+"||";
  
  //sum
  s.append( temp.setNum( sum, 'f', 2 ) );
  s.append( "||" );
  
  //reco
  if ( reco==YES ) s.append( "Y||" );
  else s.append( "N||" );
  
  //date
  s.append( temp.sprintf( "%02d/%02d/%4d", date.month(), 
			  date.day(), date.year() ) );
  return s;
}

void Transaction::load( QHaccAccountManager * manager ){
  // see if we've gotten all the splits yet
  if ( loadArrSize>splits.length() ){
    //if not, link up!
    for ( int i=0; i<loadArrSize*2; i=i+2 ){
      Account * a=manager->getByID( loadArr[i+1] );
      if ( a ){
	Transaction * t=a->find( loadArr[i] );
	if ( t ){
	  addSplit( t );
	  t->addSplit( this );
	}
      }
    }
    delete [] loadArr;
    loadArrSize=0;
    loadArr=0;
  }
}

void Transaction::addSplit( Transaction * t ){
  if ( !( splits.contains( t ) || t==this ) ) splits.add( t );
}

void Transaction::addSplits( QHaccVector& v ){
  splits.startLoad( v.length() );
  for ( int i=0; i<v.length(); i++ ) addSplit( v.getTrans( i ) );
}

void Transaction::dropSplits(){
  Account * temp=owner;
  owner=0;
#ifdef QDEBUG
    cout<<"\tkilling splits:"<<endl;
    for ( int i=0; i<splits.length(); i++ )
      cout<<"\t"<<splits.getTrans( i )->getID()<<endl;
#endif

  int nt=splits.length();
  for ( int i=0; i<nt; i++ ){
    Transaction * t=splits.getTrans( i );
    t->removeYourself();
  }
  splits.clear();
  
  owner=temp;
}

void Transaction::removeYourself(){
  // this should only be called by other transactions
  splits.clear();
  if ( owner ) owner->remTrans( this );
}

/*********************************************************/
/***************** MEMORIZED TRANSACTION *****************/
/*********************************************************/

MemorizedTransaction::MemorizedTransaction( const QString& rff )
  : Transaction( rff ){ 
  setAccount( 0 );
  setReconciled( NO );
}

void MemorizedTransaction::load( QHaccAccountManager * manager ){
  // create new transactions that pattern from the old transactions
  // in the split
  for ( int i=0; i<loadArrSize*2; i=i+2 ){
    Account * other=manager->getByID( loadArr[i+1] );
    if ( other ){
      Transaction * t=new Transaction( *( other->find( loadArr[i] ) ) );
      if ( t ){
	other->addTrans( t );
	t->addSplit( this );
	addSplit( t );
	t->setDate( QDate::currentDate() );
      }
    }
  }
  
  delete [] loadArr;
  loadArrSize=0;
  loadArr=0;
}
