/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999 Ryan F Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "config.h"
#include "accounts.h"
#include "qhaccview.h"
#include "transaction.h"
#include "qhaccrecwin.h"
#include "qhaccdialogs.h"

#ifdef QGRAPHS
#include "qhaccgrapher.h"
#endif

#include <iostream.h>

#include <qpixmap.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qscrollbar.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qcolordialog.h>
#include <qfontmetrics.h>

QHaccView::QHaccView( Account * a, QWidget * p, const char * n )
  : QFrame( p, n ){

  prefs=Prefs::getInstance();
  setFrameStyle( Panel|Sunken );
  setLineWidth( 2 );
  setFocusPolicy( StrongFocus );
  setMouseTracking( true );
  setIndex( -1 );

  copyTemp=0;
  numTrans=0;
  numViewableLines=0;
  lines=prefs->getLines();
  lineSize=QFontMetrics( prefs->getFont() ).lineSpacing();

  connect( prefs, SIGNAL( changedAltColor( QColor ) ),
	   this, SLOT( refresh() ) );

  QGridLayout * layout=new QGridLayout( this, 10, 12, 0 );
  scroller=new QScrollBar( QScrollBar::Vertical, this );
  layout->addMultiCellWidget( scroller, 0, 10, 12, 12 );
  connect( scroller, SIGNAL( valueChanged( int ) ), SLOT( vrepaint() ) );

  memPop=new QPopupMenu( this );
  connect( memPop, SIGNAL( activated( int ) ), SLOT( addMemTrans( int ) ) );

  transPop=new QPopupMenu( this );
  transPop->insertItem( "insert", this, SLOT( newTrans( ) ) );
  transPop->insertItem( "edit", this, SLOT( editTrans( ) ) );
  transPop->insertItem( "toggle reconciled", this, SLOT( recTrans() ) );
  transPop->insertItem( "delete", this, SLOT( remTrans() ) );
  transPop->insertSeparator();
  transPop->insertItem( "copy", this, SLOT( copyTrans() ), CTRL+Key_C );
  pasteID1=transPop->insertItem( "paste", this, SLOT( pasteTrans() ),
				CTRL+Key_P );
  transPop->setItemEnabled( pasteID1, false );
  transPop->insertItem( "memorize", this, SLOT( memTrans() ) );
  transPop->insertItem( "memorized", memPop );

  acctPop=new QPopupMenu( this );
  acctPop->insertItem( "new transaction", this, SLOT( newTrans() ) );
  pasteID2=acctPop->insertItem( "paste", this, SLOT( pasteTrans() ) );
  acctPop->setItemEnabled( pasteID2, false );
  acctPop->insertItem( "memorized", memPop );
#ifdef QGRAPHS
  acctPop->insertItem( "graph account", this, SLOT( chart() ) );  
  acctPop->insertItem( "account report", this, SLOT( report() ) );
#endif
  acctPop->insertItem( "change prefs", this, SLOT( newPrefs( ) ) );

  if ( a ) accountOpened( a );
  else setAccount( 0 );

  connect( prefs, SIGNAL( changedLines( int ) ),
	   this, SLOT( setLines( int ) ) );
  connect( prefs, SIGNAL( changedAltColor( QColor ) ),
	   this, SLOT( vrepaint() ) );
  connect( prefs, SIGNAL( changedMainColor( QColor ) ),
	   this, SLOT( vrepaint() ) );
}

QHaccView::~QHaccView(){
  delete transPop;
  delete acctPop;
  delete memPop;
  delete scroller;
}

void QHaccView::setIndex( int i ){ currIndex=( i<=numTrans ? i : numTrans );}
int QHaccView::getIndex() const {return currIndex;}
void QHaccView::vrepaint(){repaint( false );}

void QHaccView::setAcctPop( QPopupMenu * menu ){
  delete acctPop;
  acctPop=menu;
}

void QHaccView::setTransPop( QPopupMenu * menu ){
  delete transPop;
  transPop=menu;
}

void QHaccView::setAccount( Account * acct ){account=acct;}
Account * QHaccView::getAccount() const {return account;}

void QHaccView::accountOpened( Account * a ){  
  Account * acct=getAccount();
  copyTemp=0;
  transPop->setItemEnabled( pasteID1, false );
  acctPop->setItemEnabled( pasteID2, false );

  if ( acct != 0 ){
    // get rid of the old connections
    disconnect( acct, SIGNAL( balanceChanged( float ) ), 
		this, SLOT( refresh() ) );
    disconnect( acct, SIGNAL( needRefresh() ),
 		this, SLOT( refresh() ) );
    disconnect( acct, SIGNAL( removedTrans() ), 
		this, SLOT( refresh() ) );
    
    memPop->clear();
  }
  
  setAccount( a );
  // we need these connections to make sure that the 
  // display gets updated no matter where the account
  // changes (like in a reconcile window)
  if ( getAccount() ){
    connect( a, SIGNAL( balanceChanged( float ) ), 
	     this, SLOT( refresh() ) );
    connect( a, SIGNAL( needRefresh() ), 
	     this, SLOT( refresh() ) );
    connect( a, SIGNAL( removedTrans() ), 
	     this, SLOT( refresh() ) );
    //a->recalcBal();
    setIndex( a->getNumTrans()-1 );

    //set up the memorized window
    QHaccVector v=a->getMems();
    int l=v.length();
    for ( int i=0; i<l; i++ ){
      Transaction * t=v.getTrans( i );
      memPop->insertItem( t->get( Transaction::PAYEE )+
			" ("+QString( 0 ).setNum( t->getSum() )+")", i );
    }
    memPop->setEnabled ( l );
  }
  refresh();
  scroller->setValue( scroller->maxValue() );
  setIndex( numTrans );
}

void QHaccView::refresh(){
  numTrans=getNumTrans();
  
  //basically, make sure the scrollbar uses valid values
  if ( numTrans < numViewableLines ) scroller->setRange( 0, 0 );
  else scroller->setRange( 0, numTrans-numViewableLines+1 );
  if ( scroller->value() > scroller->maxValue() )
    scroller->setValue(scroller->maxValue() );

  vrepaint();
}

int QHaccView::getNumTrans() const { 
  return ( getAccount() ? getAccount()->getNumTrans() : 0 );
}

void QHaccView::addMemTrans( int i ){ getAccount()->addMemTrans( i ); }
void QHaccView::memTrans(){
  if ( hasTrans() ){
    bool good=getAccount()->memTrans( getTrans() );
    if ( good )
      memPop->insertItem( getTrans()->get( Transaction::PAYEE )+
			  " ("+QString( 0 ).setNum( getTrans()->getSum() )+")",
			  getAccount()->getMems().length()-1 );
    if ( !memPop->isEnabled() ) memPop->setEnabled( true );
  }
}

void QHaccView::copyTrans(){
  if ( hasTrans() ){
    copyTemp=getTrans();
    transPop->setItemEnabled( pasteID1, true );
    acctPop->setItemEnabled( pasteID2, true );
  }
}

void QHaccView::pasteTrans(){
  if ( copyTemp ) getAccount()->addCloneTransFrom( copyTemp );
}

void QHaccView::drawContents( QPainter * p ){
  int h=height(), w=getUseableWidth();
  int viewStart=scroller->value();
  QPixmap off( size() );
  QPainter offPainter( &off );
  QBrush qb( prefs->getMainColor() );

  offPainter.setBrush( qb );
  offPainter.fillRect( 0, 0, w, h, qb );

  if ( getAccount() ){
    int currLine=lineSpacing();
    
    for (int i=0;i<numViewableLines+1;i++){
      if ( i%2 ) qb.setColor( prefs->getMainColor() );
      else qb.setColor( prefs->getAltColor() );
      
      offPainter.fillRect( 0, currLine, w, currLine+lineSpacing(), qb );
      currLine+=lineSpacing();
    }
    
    QFont f=font();
    currLine=lineSize; 
    // don't use lineSpacing() here, because we want to align text
    // starting at the top of the given area
    
    QPen blacker( "black" );
    for ( int i=viewStart; i<viewStart+numViewableLines; i++ ){    
      if ( i==getIndex() ) f.setBold( true );
      else f.setBold( false );
      offPainter.setFont( f );
      offPainter.setPen( blacker );

      if ( i<numTrans ) {  
	drawTrans( getTrans( i ), &offPainter, currLine, w );
	currLine+=lineSpacing();
      }
      else if ( i==numTrans ) offPainter.drawText( 10, currLine, "<new>" );
    }
  }
  else{
    QString s( "No Account Selected" );
    int wl=QFontMetrics( prefs->getFont() ).boundingRect( s ).width();
    offPainter.drawText( ( w-wl )/2, h/2, s );
  }

  p->drawPixmap( 0, 0, off );
}

void QHaccView::drawTrans( const Transaction * t, QPainter * p,
			   int pstart, int w ){
  QDate d=t->getDate();
  float f=t->getSum();
  int pw=w-70, pos=pstart;
  QFontMetrics fm( prefs->getFont() );
  int fh=fm.height(), ph=pos-fh+1;

  //FIX ME: Way to much casting and multiplications!
  p->drawText( 5, ph, ( int )( pw*0.1 ), pos, AlignLeft,
           t->get(Transaction::NUM ) );
  
  p->drawText( ( int )( pw*0.12 ), ph, ( int )( pw*0.10 ), pos, AlignLeft,
	       prefs->getDateString( d ) );

  if ( lines==1 ){
    p->drawText( ( int )( pw*0.25 ), ph, ( int )( pw*0.35 ), pos, AlignLeft,
         t->get( Transaction::PAYEE ) );
    p->drawText( ( int )( pw*0.62 ), ph, ( int )( pw*0.33 ), pos, AlignLeft,
         t->get( Transaction::MEMO ) );
  }
  else{
    p->drawText( ( int )( pw*0.25 ), ph, ( int )( pw*0.70 ), pos, AlignLeft,
         t->get( Transaction::PAYEE ) );
  }

  p->drawText( pw, pos, QString( 0 ).setNum( f, 'f', 2 ) );
  if ( t->isReconciled() ) p->drawText( w-10, pos, "R" );

  if ( lines > 1 ){
    pos+=lineSize;
    ph=pos-fh;
    p->drawText( 5, ph, (int)( pw*0.45 ), pos, AlignLeft, t->getPairAName() );
    p->drawText( ( int )( pw*0.5 ), ph, ( int )( pw*0.45 ), pos, AlignLeft,
         t->get( Transaction::MEMO ) );
  }

  p->setPen( QPen( "lightgrey" ) );

  int starthere=pstart-fh;
  p->drawLine( ( int )pw*0.11, starthere, ( int )pw*0.11, pstart );
  p->drawLine( ( int )pw*0.23, starthere, ( int )pw*0.23, pstart );
  p->drawLine( ( int )pw*0.96, starthere, ( int )pw*0.96, pstart );

  if ( lines>1 ){
    p->drawLine( 0, pstart, w, pstart );
    p->drawLine( ( int )pw*0.49, pstart, ( int )pw*0.49, pos );
  }
  else p->drawLine( ( int )pw*0.61, starthere, ( int )pw*0.61, pstart );
}

int QHaccView::lineSpacing() const{return lineSize*lines;}
int QHaccView::getUseableWidth() const {return width()-scroller->width();}
int QHaccView::getHeight() const {return numTrans*lineSize;}
void QHaccView::resizeEvent(QResizeEvent *){
  numViewableLines=height()/lineSpacing();
  refresh();
}

void QHaccView::mouseMoveEvent(QMouseEvent * qme){
  if ( getAccount() ){
    int i=getIndex();
    int y=qme->y()/lineSpacing()+scroller->value();
    if ( i!=y ){
      setIndex( y );
      vrepaint();
    }
  } 
}

void QHaccView::mouseReleaseEvent(QMouseEvent * qme){
  if ( qme->state()!=QMouseEvent::RightButton && getAccount() ) editTrans();
}

void QHaccView::mousePressEvent( QMouseEvent * qme ){
  if ( qme->button() == QMouseEvent::RightButton ){
    if ( hasTrans() ){
      if ( transPop != 0 ) transPop->popup( QCursor::pos() );
    }
    else if ( acctPop != 0 ) acctPop->popup( QCursor::pos() );
  }
}

void QHaccView::remTrans(){
  if ( hasTrans() ){
    Transaction * killme=getTrans();
    if ( killme==copyTemp ){ // you can't paste a deleted transaction
      copyTemp=0;
      transPop->setItemEnabled( pasteID1, false );
      acctPop->setItemEnabled( pasteID2, false );
    }

    getAccount()->remTrans( killme );

    // make sure we clean up the memorized transaction menu
    // if we remove the trans it's based on
    memPop->clear();
    QHaccVector v=getAccount()->getMems();
    int l=v.length();
    for ( int i=0; i<l; i++ ){
      Transaction * t=v.getTrans( i );
      memPop->insertItem( t->get( Transaction::PAYEE )+
			  " ("+QString( 0 ).setNum( t->getSum() )+")", i );
      
    }
    memPop->setEnabled( l );
  }
}

void QHaccView::newTrans(){
  QPoint p( 0, ( getIndex()-scroller->value() ) * lineSpacing() );
  p=mapToGlobal( p );
  QTransDlg qtd( getAccount(), ( prefs->getLines()==2 ), QRect( p.x(), p.y(), getUseableWidth(), 25 ), this );
  qtd.exec();
}

void QHaccView::editTrans(){
  if ( !hasTrans() ) newTrans();
  else{
    QPoint p( 0, ( getIndex()-scroller->value() ) * lineSpacing() );
    p=mapToGlobal( p );
    QTransDlg qtd( getTrans(), ( prefs->getLines()==2 ),
		   QRect( p.x(), p.y(), getUseableWidth(), 25 ), this );
    qtd.exec();
  }
}

void QHaccView::recTrans(){
  Transaction * t=getTrans();
  if ( t->isReconciled() ) t->setReconciled( Transaction::NO );
  else t->setReconciled( Transaction::YES );
}

void QHaccView::setLines( int i ) {
  lines=i;

  lineSize=QFontMetrics( prefs->getFont() ).lineSpacing();
  numViewableLines=height()/lineSpacing();
  refresh();
}

bool QHaccView::hasTrans() const {return currIndex>-1 && currIndex<numTrans;}
Transaction * QHaccView::getTrans( int i ) const {
  if ( i==-1 ){
    if ( hasTrans() ) return getAccount()->getTrans( currIndex );
    else return 0;
  }
  return getAccount()->getTrans( i );
}

int QHaccView::spacePressed(){
  editTrans();
  return getIndex();
}
int QHaccView::delPressed(){
  remTrans();
  int newIdx=getIndex()-1;

  if ( newIdx-1 < scroller->value() )
    scroller->setValue( scroller->value()-( numViewableLines/2 ) );
  return newIdx;
}
int QHaccView::upPressed(){
  int newIdx=getIndex()-1;
  if ( newIdx-1 < scroller->value() )
    scroller->setValue( scroller->value()-( numViewableLines/2 ) );
  return newIdx;
}

int QHaccView::downPressed(){
  int newIdx=getIndex()+1;
  if ( newIdx+2 > ( scroller->value() + numViewableLines ) )
    scroller->setValue( scroller->value()+( numViewableLines/2 ) );
  return newIdx;
}

void QHaccView::chart(){
#ifdef QGRAPHS
  QHaccGrapher * qg=new QHaccGrapher( QHaccGrapher::GRAPH, this );  
  qg->show();
  qg->accountOpened( getAccount() );
#endif
}

void QHaccView::report(){
#ifdef QGRAPHS
  QHaccGrapher * qg=new QHaccGrapher( QHaccGrapher::REPORT, this );  
  qg->show();
  qg->accountOpened( getAccount() );
#endif
}

void QHaccView::keyReleaseEvent( QKeyEvent * qke ){
  int key=qke->key();
  if ( key==Key_Space || key==Key_Up || key==Key_Down || key==Key_Delete ){
    int newIdx=0;
    
    if ( key == Key_Space ) newIdx=spacePressed();                // SPACE BAR
    else if ( key == Key_Delete ) newIdx=delPressed();            // DELETE!
    else {
      if ( key == Key_Up && getIndex() >= 0 ) newIdx=upPressed(); // UP KEY
      else newIdx=downPressed();                                  // DOWN KEY 
    }
    setIndex( newIdx );
    vrepaint();
  }
  else qke->ignore();
}

// reimplement these to stop the flicker when focus shifts in or out
void QHaccView::focusInEvent ( QFocusEvent * ){}
void QHaccView::focusOutEvent( QFocusEvent * ){}
void QHaccView::newPrefs(){ emit changePrefs(); }


/* * * * * * * * * * * * * * * */
/*    QHACCRECVIEW METHODS     */
/* * * * * * * * * * * * * * * */

QHaccRecView::QHaccRecView( Account * a, QWidget * p, const char * n )
  : QHaccView( a, p, n ){
  
  QPopupMenu * newAcctPop=new QPopupMenu( this );
  newAcctPop->insertItem( "new transaction", this, SLOT( newTrans() ) );
  setAcctPop( newAcctPop );
  limit=QDate::currentDate();
  lines=1;
}

QHaccRecView::~QHaccRecView(){ trans.clear(); }

void QHaccRecView::mouseReleaseEvent( QMouseEvent * qme ){
  if ( hasTrans() ) recTrans();
  else newTrans();
}

void QHaccRecView::drawTrans( const Transaction * t, QPainter * p,
			   int pstart, int w ){
  QHaccView::drawTrans( t, p, pstart, w );

  if ( !t->isReconciled( Transaction::NO ) ){
    p->setPen( QPen( "black" ) );
    p->drawText( w-10, pstart, "R" );
  }
}

void QHaccRecView::setDate( QDate date ){
  limit=date;
  refresh();
}

void QHaccRecView::recTrans(){
  Transaction * t=getTrans();

  if ( t->isReconciled( Transaction::MAYBE ) )
    t->setReconciled( Transaction::NO );
  else t->setReconciled( Transaction::MAYBE );

  refresh();
}

void QHaccRecView::refresh(){
  if ( getAccount() ){
    float sum, unsum;
    trans=getAccount()->getUnreconciled( sum, unsum, limit );
    emit changeTrans( trans, sum, unsum );
  }
  QHaccView::refresh();
}

Transaction * QHaccRecView::getTrans( int i ) const {
  if ( i==-1 ){
    if ( getIndex() != -1 ) return trans.getTrans( getIndex() );
    return 0;
  }
  return trans.getTrans( i );
}

void QHaccRecView::addedTrans( Transaction * t ){
  t->setReconciled( Transaction::MAYBE );
  refresh();
}

int QHaccRecView::spacePressed(){
  int newIdx=getIndex();
  if ( hasTrans() ) {
    recTrans();
    //if we just reconciled a transaction, move the highlighted line down one
    if ( getTrans()->isReconciled( Transaction::MAYBE ) ) newIdx++;
  }
  else newTrans();
  
  // we're moving the highlighting, so we might have to recenter
  if ( newIdx+2 > ( scroller->value() + numViewableLines ) )
    scroller->setValue( scroller->value()+( numViewableLines/2 ) );

  return newIdx;
}

int QHaccRecView::getNumTrans() const { return trans.length(); }
