/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************* * * * * * * * * * * * * ***************************/

#include "prefs.h"
#include "accounts.h"
#include "qhaccview.h"
#include "transaction.h"
#include "qhaccrecwin.h"
#include "qhacclineedits.h"

#include <qfont.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>

QHaccRecWin::QHaccRecWin( Account * a, QWidget * p, const char * n )
  : QWidget( p, n, WDestructiveClose ){

  acct=0;
  tView=new QHaccRecView( 0, this );
  prefs=Prefs::getInstance();
  firstRun=true;

  QGridLayout * layout=new QGridLayout( this, 50, 50, 5 );
  //setFontPropagation( AllChildren );
  //setFont( prefs->getFont() );

  connect( prefs, SIGNAL( changedSymbol( QString ) ), SLOT( refresh() ) );

  enddate=new QHaccDateEdit( QDate(), this );
  enddate->setAlignment( AlignCenter );
  obal=new QHaccLineEdit( this );
  obal->setAlignment( AlignCenter );
  ebal=new QHaccLineEdit( this );
  ebal->setAlignment( AlignCenter );
  lblDiff=new QLabel( "Difference: ", this );

  QPushButton * ok=new QPushButton( "Finished", this );
  QPushButton * cancel=new QPushButton( "Cancel", this );
  connect ( ok, SIGNAL( clicked() ), SLOT( accept() ) );
  connect ( cancel, SIGNAL( clicked() ), SLOT( cancel() ) );

  connect( enddate, SIGNAL( dateChanged( QDate ) ), SLOT( changeDate() ) );
  connect( obal, SIGNAL( lostFocus() ), SLOT( refresh() ) );
  connect( ebal, SIGNAL( lostFocus() ), SLOT( refresh() ) );

  connect( obal, SIGNAL( returnPressed() ), SLOT( refresh() ) );
  connect( ebal, SIGNAL( returnPressed() ), SLOT( refresh() ) );

  connect( tView, SIGNAL( changeTrans( QHaccVector, float, float ) ),
	   SLOT( changeSums( QHaccVector, float, float ) ) );

  int colS=1, colE=12, row=1;

  layout->addMultiCellWidget( new QLabel( "End Date", this ),
			      row, row, colS, colE, AlignCenter );
  layout->addMultiCellWidget( enddate, row+1, row+4, colS, colE, AlignCenter );
  row+=5;

  layout->addMultiCellWidget( new QLabel( "Opening Balance", this ),
			      row, row, colS, colE, AlignCenter );
  layout->addMultiCellWidget( obal, row+1, row+4, colS, colE, AlignCenter );
  row+=5;

  layout->addMultiCellWidget( new QLabel( "Closing Balance", this ),
			      row, row, colS, colE, AlignCenter );
  layout->addMultiCellWidget( ebal, row+1, row+4,
			      colS, colE, AlignCenter );
  row+=5;

  layout->addMultiCellWidget( lblDiff, row, row, colS, colE, AlignCenter );
  row++;
  layout->addMultiCellWidget( ok, row, row, colS, colE );
  row++;
  layout->addMultiCellWidget( cancel, row, row, colS, colE );
  

  layout->addMultiCellWidget( tView, 0, 35, 13, 49 );
  
  resize( 600, 400 );
  accountOpened( a );
}

QHaccRecWin::~QHaccRecWin(){}

void QHaccRecWin::changeSums( QHaccVector trans, float s, float u ){
  origUnsum=u;
  unsum=u;

  // first, make sure all the MAYBE transactions count as reconciled
  for ( int i=0; i<trans.length(); i++ ){
    Transaction * tr=trans.getTrans( i );
    if ( tr->isReconciled( Transaction::MAYBE ) ) unsum-=tr->getSum();
  }

  if ( firstRun ){
    // only reset an opening balance on the first run because the
    // user might have specified a different opening balance already
    obal->setText( QString().setNum( s, 'f', 2 ) );
    firstRun=false;
  }

  refresh();
}



void QHaccRecWin::refresh(){
  //recalc the difference of the statement to the trans in the account
  QString s;
  float diff=origUnsum-unsum+obal->text().toFloat()-ebal->text().toFloat();

  if ( diff<0 ){
    s.append( "-" );
    diff=0-diff; //always get a positive difference for display
  }
  s.append( prefs->getSymbol() );
  lblDiff->setText( "Difference "+s+s.setNum( diff, 'f', 2 ) );
}

void QHaccRecWin::changeDate(){
  firstRun=true;
  tView->setDate( enddate->getDate() );
}

void QHaccRecWin::accept(){
  if ( acct ) acct->alterReconciled( Account::UPGRADE );
  QWidget::close( true );  
}

void QHaccRecWin::hide(){
  if ( acct ) acct->stopReco();
  QWidget::hide();
}

void QHaccRecWin::cancel(){
  if ( acct ) acct->alterReconciled( Account::DOWNGRADE );
  QWidget::close( true );
}

void QHaccRecWin::accountOpened( Account * a ){
  if ( acct ){
    disconnect ( acct, SIGNAL( addedTrans( Transaction * ) ),
		 tView, SLOT( addedTrans( Transaction * ) ) );
    disconnect ( acct, SIGNAL( removedTrans() ), tView, SLOT( refresh() ) );
  }
  
  acct=a;
  tView->accountOpened( acct );

  if ( acct ){
    acct->startReco();
    setCaption( "Reconciling "+a->getName() );

    connect ( acct, SIGNAL( addedTrans( Transaction * ) ),
	      tView, SLOT( addedTrans( Transaction * ) ) );
    connect ( acct, SIGNAL( removedTrans() ), tView, SLOT( refresh() ) );
  }
  else hide();
}
