/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************* * * * * * * * * * * * * ***************************/

#include "prefs.h"
#include "accounts.h"
#include "qhaccgrapher.h"
#include "qhaccacctman.h"
#include "qhacclineedits.h"

#include <iostream.h>
#include <stdlib.h>
#include <time.h>

#include <qlabel.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qtextview.h>
#include <qcombobox.h>
#include <qpushbutton.h>

/********************/
/**   GRAPHTOPPER  **/
/********************/

GraphTopper::GraphTopper( Reporter * g, QWidget * p, const char * n )
  : QFrame( p, n ){
  graph=false;
  init( g );
}

GraphTopper::GraphTopper( Grapher * g, QWidget * p, const char * n )
  : QFrame( p, n ){
  graph=true;
  init( g );
}

void GraphTopper::init( QHaccGraphFrame * s ){
  slave=s;

  Prefs * prefs=Prefs::getInstance();

  manager=QHaccAccountManager::getInstance();

  connect( prefs, SIGNAL( changedAltColor( QColor ) ),
	   this, SLOT( changeColor( QColor ) ) );
  changeColor( prefs->getAltColor() );

  QDate ds=QDate::currentDate(), de=ds;
  ds.setYMD( ds.year(), 1, 1 ); //start at first of year...
  de=de.addDays( 1+de.daysInMonth()-de.day() ); //...through the current month

  start=new QHaccDateEdit( ds, this );
  end=new QHaccDateEdit( de, this );
  start->setAlignment( AlignCenter );
  end->setAlignment( AlignCenter );
 
  QGridLayout * layout=new QGridLayout( this, 2, 10, 5 );
  QLabel * lbl=new QLabel( "start", this );
  layout->addMultiCellWidget( lbl, 0, 0, 0, 0, AlignCenter );
  layout->addMultiCellWidget( start, 1, 1, 0, 0 );

  lbl=new QLabel( "end", this );
  layout->addMultiCellWidget( lbl, 0, 0, 1, 1, AlignCenter );
  layout->addMultiCellWidget( end, 1, 1, 1, 1 );

  lbl=new QLabel( "account", this );
  combo=new QComboBox( false, this );
  combo->insertStrList( manager->getNames() );
  layout->addMultiCellWidget( lbl, 0, 0, 2, 2, AlignCenter );
  layout->addMultiCellWidget( combo, 1, 1, 2, 2 );

  if ( graph ){
    lbl=new QLabel( "graph type", this );
    typer=new QComboBox( false, this );
    typer->insertItem( "Line" );
    typer->insertItem( "Bar" );
    typer->insertItem( "Two Bar" );
    if ( prefs->getLines()==2 ) typer->insertItem( "Pie" );
    
    if ( prefs->getGraphType()==BAR ) typer->setCurrentItem( 1 );
    else if ( prefs->getGraphType()==TWOBAR ) typer->setCurrentItem( 2 );
    else if ( prefs->getGraphType()==PIE ) typer->setCurrentItem( 3 );
    else typer->setCurrentItem( 0 );
    
    layout->addMultiCellWidget( lbl, 0, 0, 3, 3, AlignCenter );
    layout->addMultiCellWidget( typer, 1, 1, 3, 3 );

    connect( typer, SIGNAL( activated( int ) ), SLOT( repaint() ) );
  }
  else{
    char * types[]={ "<Assets>", "<Liabilities>", "<Equity>",
		     "<Expenses>", "<Revenues>"};
    for ( int i=0; i<5; i++ ) combo->insertItem( types[i] );
  }

  connect( prefs, SIGNAL( changedLines( int ) ), SLOT( changeEntry( int ) ) );
  connect( start, SIGNAL( dateChanged( QDate ) ), SLOT( repaint() ) );
  connect( end, SIGNAL( dateChanged( QDate ) ), SLOT( repaint() ) );
  connect( combo, SIGNAL( activated( int ) ), SLOT( repaint() ) );

  srand( time( NULL ) );
  accountOpened( NULL );
}

GraphTopper::~GraphTopper(){}
void GraphTopper::accountOpened( const Account * a ){
  if ( a ){
    for ( int i=0; i<combo->count(); i++ )
      if ( combo->text( i )==a->getName() ) combo->setCurrentItem( i );
    repaint();
  }
}
void GraphTopper::changeColor( QColor c ){ setPalette( c ); }
void GraphTopper::changeEntry( int i ){ 
  if ( typer->count()==4 ){if ( i==1 ) typer->removeItem( 3 );}
  else typer->insertItem( "Pie" );
}

void GraphTopper::repaint(){
  QDate starter=start->getDate(), ender=end->getDate();
  QString title;

  if ( starter>ender ){
    QDate date=ender;
    ender=starter;
    starter=date;
  }

  QString aname=combo->currentText();
  Account ** accts=0;
  GraphType type=LINE;
  unsigned int num=0;

  if ( graph ){
    type=( GraphType ) typer->currentItem();
    accts=new Account * [1];
    accts[0]=manager->get( aname );
    num=1;
  }
  else{
    QList <Account> alist;
    alist.setAutoDelete( false );

    if ( aname=="<Assets>" ) {
      alist=manager->getAccounts( ASSET );
      title="Assets";
    }
    else if ( aname=="<Liabilities>" ) {
      alist=manager->getAccounts( LIABILITY );
      title="Liabilities";
    }
    else if ( aname=="<Equity>" ) {
      alist=manager->getAccounts( EQUITY );
      title="Equity Accounts";
    }
    else if ( aname=="<Expenses>" ) {
      alist=manager->getAccounts( EXPENSE );
      title="Expenses";
    }
    else if ( aname=="<Revenues>" ) {
      alist=manager->getAccounts( REVENUE );
      title="Revenues";
    }
    else{
      alist.append( manager->get( aname ) );
      title=aname;
    }

    num=alist.count();
    accts=new Account * [num];
    for ( unsigned int i=0; i<  num; i++ ) accts[i]=alist.at( i );
  }

  slave->graphit( title, accts, num, starter, ender, type );
  delete []  accts;
}

/* * * * * * * * * * */
/* Base for Graphers */
/* * * * * * * * * * */
QHaccGraphFrame::QHaccGraphFrame( QWidget * p, const char * n )
  : QFrame( p, n ){
  Prefs * prefs=Prefs::getInstance();

  connect( prefs, SIGNAL( changedMainColor( QColor ) ),
	   this, SLOT( changeColor( QColor ) ) );
  changeColor( prefs->getMainColor() );
}
QHaccGraphFrame::~QHaccGraphFrame(){}
void QHaccGraphFrame::changeColor( QColor c ){ setBackgroundColor( c ); }
void QHaccGraphFrame::graphit( QString heading, Account ** accounts,
			       unsigned int num, QDate starter,
			       QDate ender, GraphType type ){
  title=heading;
  prepareData( accounts, num, starter, ender, type );
  repaint();
}

/****************/
/**   GRAPHER  **/
/****************/

Grapher::Grapher( QWidget * p, const char * n ) : QHaccGraphFrame( p, n ){
  prefs=Prefs::getInstance();

  headings=new QString[0];
  points=new float[0];
  sums=new float[0];  
  partitions=0;
  type=LINE;

  // get some random colors to draw into graphs
  // i picked 12 so a big graph will repeat colors every year
  // it's neccessary to keep track of these colors because
  // partial repaints will look very bad otherwise  
  colors=new QColor[12];
  for ( int i=0; i<12; i++ )
    colors[i]=QColor( rand()%100+150, rand()%100+150, rand()%100+150 );
}

Grapher::~Grapher(){
  delete [] sums;
  delete [] points;
  delete [] colors;
  delete [] headings;
}

void Grapher::prepareData( Account ** accounts, unsigned int, QDate starter,
			   QDate ender, GraphType ty ){

  //get rid of the old information
  delete [] sums;
  delete [] points;
  delete [] headings;
  partitions=0;

  if ( !accounts ) return;

  type=ty;
  QHaccVector v=accounts[0]->getSubset( starter, ender );
  QDate date=starter;
  QDate ndate=starter.addDays( starter.daysInMonth() );
  int arraySize=0;

  if ( type!=PIE ){
    QDate cycletime=starter;
    while ( cycletime<ender ){
      // figure out how many points the graph will have
      partitions++;
      cycletime=cycletime.addDays( cycletime.daysInMonth() );
    }
    // the twobar graph is a bar graph with increases and decreases, so
    // it has twice as many numbers as a bar or line graph
    if ( type==TWOBAR ) arraySize=partitions+partitions;
    else arraySize=partitions;
    
    headings=new QString[arraySize];
    sums=new float[arraySize];
    
    for ( int i=0; i<partitions; i++ ){
      //zero out the array before we begin
      sums[i]=0;
      if ( type==TWOBAR ) sums[i+partitions]=0;
      
      headings[i]=prefs->getDateString( date ).left( 5 )+" - "
	+prefs->getDateString( ndate ).left( 5 );
      
      //get the sum for this month's transactions
      QHaccVector v2=v.getSubset( date, ndate );
      for ( int j=0; j<v2.length(); j++ ){
	float sum=v2.getTrans( j )->getSum();

	if ( type==TWOBAR ){
	  if ( sum<0 ) sums[i+partitions]-=sum; //we only want positive values
	  else sums[i]+=sum;
	}
	else{
	  if ( accounts[0]->isType( ASSET ) ) sums[i]+=sum;
	  else sums[i]-=sum;
	}
      }
      
      date=ndate;
      ndate=ndate.addDays( ndate.daysInMonth() ); //go to the next month
      if ( ndate>ender ) ndate=ender; //but don't pass the last specified day
    }
  }
  else{
    QList<Account> accts;
    accts.setAutoDelete( false );
    // we're going to do two passes here...one to figure out which accounts
    // to graph, and one to get the amounts to graph
    for ( int i=0; i<v.length(); i++ ){
      //figure out how many different double-entry accounts there are
      Transaction * t=v.getTrans( i );
      QHaccVector v2=t->getSplits();
      
      for ( int j=0; j<v2.length(); j++ ){
	Account * act=v2.getTrans( j )->getAccount();
	if ( !accts.containsRef( act ) ) accts.append( act );
      }
    }
    
    arraySize=accts.count();
    headings=new QString[arraySize];
    sums=new float[arraySize];
    
    for ( int i=0; i<arraySize; i++ ) {
      sums[i]=0;
      headings[i]=accts.at( i )->getName();
    }
    
    for ( int i=0; i<v.length(); i++ ){
      Transaction * t=v.getTrans( i );
      QHaccVector v2=t->getSplits();
      
      for ( int j=0; j<v2.length(); j++ ){
	Transaction * t2=v2.getTrans( j );
	if ( t->isSplit() )
	  sums[accts.findRef( t2->getAccount() )]+=t2->getSum();
	else sums[accts.findRef( t2->getAccount() )]+=t->getSum();
      }
    }

    for ( int i=0; i<arraySize; i++ ) if ( sums[i]<0 ) sums[i]=0-sums[i];
  }

  points=new float[arraySize];
  int hVali=0, lVali=0;
  float total=0;

  for ( int i=0; i<arraySize; i++ ){
    points[i]=sums[i];
    if ( type==PIE ) total+=sums[i];
    if ( sums[i]<sums[lVali] ) lVali=i;
    if ( sums[i]>sums[hVali] ) hVali=i;
  }
  
  //now, set up the points we'll need to do the actual graph
  int h=getHeight();
  if ( type!=PIE ){
    float hVal=points[hVali], lVal=points[lVali];
    if ( lVal>0 ) lVal=0;
    if ( hVal<0 ) hVal=0;
    
    float factor=( hVal-lVal )/( h*0.9 ); //0.9 is wiggle room
    for ( int i=0; i<arraySize; i++ ) points[i]=points[i]/factor;
    
    zero=h-( int )( ( 0-lVal )/factor );
  }
  else{
    zero=( int )( h*0.1 );
    for ( int i=0; i<arraySize; i++ ) points[i]=points[i]/total;
    partitions=arraySize;
  }
}

void Grapher::drawContents( QPainter * p ){
  if ( !partitions ) {
    p->fillRect( 0, 0, size().width(), size().height(), backgroundColor() );
    p->drawText( 0, 0, size().width(), size().height(), AlignCenter,
		 "No Transactions" );
    return;
  }

  QPixmap off( size() );
  QPainter offPainter( &off );
  QBrush qb( "black" );
  offPainter.fillRect( 0, 0, size().width(), size().height(),
		       backgroundColor() );

  // w is the increment of width per index
  int w=width()/partitions;
  drawColors( &offPainter, w );

  int halfw=w/2, wspot=halfw, hspot=0, hspoto=0;
  QFontMetrics fm=fontMetrics();
  QString temp;

  switch ( type ){
  case BAR:
    for ( int i=0; i<partitions; i++ ){      
      hspot=( int )( zero-points[i] );
      
      
      //if the column label is too long, just put a placeholder
      if ( fm.width( headings[i] ) > w*2 ) temp="|";
      else temp=headings[i];
      
      //column labels
      offPainter.drawText( wspot-halfw, height()-2.5*fm.height(), w,
			   2.5*fm.height(), AlignTop|AlignHCenter|WordBreak,
			   temp );
      
      // write dollar amounts in boxes
      temp=QString( 0 ).setNum( sums[i], 'f', 2 );
      int swidth=fm.width( temp );
			   
      offPainter.drawText( wspot-swidth/2, hspot-5, temp );
      
      wspot+=w;
    }
    
    offPainter.setPen( QPen( "darkgray" ) );
    offPainter.drawLine( 0, zero, width(), zero );
    
    break;
  case TWOBAR:
    for ( int i=0; i<partitions; i++ ){      
      hspot=( int )( zero-points[i] );
      int hspoto=( int )( zero-points[i+partitions] );
      
      
      //if the column label is too long, just put a placeholder
      if ( fm.width( headings[i] ) > w*2 ) temp="|";
      else temp=headings[i];
      
      //column labels
      offPainter.drawText( wspot-halfw, height()-2.5*fm.height(), w,
			   2.5*fm.height(), AlignTop|AlignHCenter|WordBreak,
			   temp );
      
      // write dollar amounts in boxes
      temp=QString( 0 ).setNum( sums[i], 'f', 2 );
      int swidth=fm.width( temp );			   
      offPainter.drawText( ( w*i+10 ), hspot-5, temp );

      //negative values show up as positive values, so put a minus sign first
      temp="-"+QString( 0 ).setNum( sums[i+partitions], 'f', 2 );
      swidth=fm.width( temp );
      offPainter.drawText( ( w*( i+1 )-10 )-swidth, hspoto-5, temp );
      
      wspot+=w;
    }
    
    offPainter.setPen( QPen( "darkgray" ) );
    offPainter.drawLine( 0, zero, width(), zero );
    
    break;
  case LINE:
    for ( int i=0; i<partitions; i++ ){      
      hspot=( int )( zero-points[i] );
      // draw a line from the last sum to this sum
      if ( i>0 ) {
	hspoto=( int ) ( zero-points[i-1] );
	offPainter.drawLine( wspot-w, hspoto, wspot, hspot );
      }
      
      //if the column label is too long, just put a placeholder
      if ( fm.width( headings[i] ) > w*2 ) temp="|";
      else temp=headings[i];
      
      // write column labels
      offPainter.drawText( wspot-(w/2), height()-2.5*fm.height(), w,
			   2.5*fm.height(), AlignTop|AlignHCenter|WordBreak,
			   temp );
      

      // put a little box on the dollar amount
      offPainter.fillRect( wspot-3, hspot-3, 6, 6, qb );
      // write the values next to the sum
      offPainter.drawText( wspot+5, hspot,
			   QString( 0 ).setNum( sums[i], 'f', 2 ) );
      
      wspot+=w;
    }
    
    offPainter.setPen( QPen( "darkgray" ) );
    offPainter.drawLine( 0, zero, width(), zero );
    break;
  case PIE:
    w=2*width()/( partitions+1 );
    wspot=10;
    int l=( partitions+1 )/2;
    int hspot=height()-3*fm.height();
    int hspoto=( int )( height()-1.5*fm.height() );

    for ( int i=0; i<l; i++ ){
      offPainter.fillRect( wspot, hspot, 15, 15, colors[i] );
      offPainter.drawText( wspot+17, hspot, w-17, fm.height(), AlignLeft,
			   headings[i] );
      offPainter.drawText( wspot, hspot+fm.height(),
			   QString( 0 ).setNum( points[i]*100, 'f', 0 ) );
      
      if ( i+l<partitions ){
	offPainter.fillRect( wspot, hspoto, 15, 15, colors[i+l] );
	offPainter.drawText( wspot+17, hspoto, w-17, fm.height(), AlignLeft, 
			     headings[i+l] );
	offPainter.drawText( wspot, hspoto+fm.height(),
			     QString( 0 ).setNum( points[i+l]*100, 'f', 0 ) );
      }
      
      wspot+=w;
    }

    break;
  }

  p->drawPixmap( 0, 0, off );
}

int Grapher::getHeight() const { return height()-3*fontMetrics().height(); }
void Grapher::drawColors( QPainter * p, int w ){
  QFontMetrics fm=fontMetrics();
  Decoration decs=prefs->getDecoration();
  QPen pen=p->pen(), pen2( "gray" );

  switch( type ){
  case LINE:
    for ( int i=0; i<partitions; i++ ){
      switch( decs ){
      case GOBS:
	p->fillRect( w*i, 0, w*( i+1 ), height()-2.5*fm.height(),
		     QBrush( colors[i%12].light( 120 ) ) );
      case MORE:
	p->fillRect( w*i, height()-2.5*fm.height(), w*( i+1 ),
		     2.5*fm.height(),
		     QBrush( colors[i%12] ) );
      case SOME:
	p->setPen( pen2 );
	p->drawLine( w*i, 0, w*i, height() );
	break;
      default:
	break;
      }
    }
    break;
  case BAR:
    for ( int i=0; i<partitions; i++ ){
      int hspot=( int )( zero-points[i] );

      QRect rect( w*i+5, hspot, w-10, zero-hspot );
      
      if ( decs>SOME )
	p->fillRect( rect, QBrush( colors[i%12].light( 110 ) ) );
      p->drawRect( rect );
    }
    break;
  case TWOBAR:
    for ( int i=0; i<partitions; i++ ){
      int hspot=( int )( zero-points[i] );
      int hspot2=( int )( zero-points[i+partitions] );

      QRect rect1( w*i+10, hspot, w/2, zero-hspot );
      QRect rect2( w*i+(w/2)-10, hspot2, w/2, zero-hspot2 );
      
      if ( decs>SOME )
	p->fillRect( rect1, QBrush( colors[i%12].light( 110 ) ) );
      p->drawRect( rect1 );

      if ( decs>SOME )
	p->fillRect( rect2, QBrush( colors[i%12].dark( 110 ) ) );
      p->drawRect( rect2 );
    }
    break;

  case PIE:
    int limit=partitions;

    // sort everybody
    for ( int i=0; i<partitions; i++ ){
      for ( int j=0; j<partitions; j++ ){
	if ( sums[i]>sums[j] ){

	  QString thead=headings[i];
	  headings[i]=headings[j];
	  headings[j]=thead;
	  
	  float tsum=sums[i];
	  sums[i]=sums[j];
	  sums[j]=tsum;

	  float tpoi=points[i];
	  points[i]=points[j];
	  points[j]=tpoi;

	}
      }
    }

    for ( int i=0; i<partitions; i++ ){
      // we're going to merge the small groups
      if ( points[i]<0.01 && limit>i ) limit=i;
    }

    // if more than 12 groups, or sums<1%, merge
    if ( limit+1>partitions ) limit=partitions;
    if ( partitions>12 || limit+1<partitions ){
      headings[limit]="Other";

      for ( int i=limit+1; i<partitions; i++ ){
	sums[limit]+=sums[i];
	points[limit]+=points[i];
      }
      partitions=limit+1;
    }

    w=width()-10;
    int h=getHeight();
    int hspoto=0;
    
    for ( int i=0; i<partitions; i++ ){
      p->setBrush( colors[i%12] );
      int hspot=( int )( points[i]*5760 ); //360*16=5760
      p->drawPie( 10, 10, w-20, h-20, hspoto, hspot );
      hspoto+=hspot;
    }

    break;
  }
  p->setPen( pen );
}

/****************/
/**  REPORTER  **/
/****************/
Reporter::Reporter( QWidget * p, const char * n ) : QHaccGraphFrame( p, n ){
  QGridLayout * layout=new QGridLayout( this, 1, 1 );
  display=new QTextView( this );
  layout->addWidget( display, 0, 0 );
  
  display->setPaper( QBrush( Prefs::getInstance()->getMainColor() ) );
}
Reporter::~Reporter(){}

QString Reporter::makeReport( const QString& title,
			      QHaccVector& transactions ){
  QString text;
  if ( transactions.isEmpty() ) text="Nothing Selected";
  else{
    float incrs=0; // sum of increases
    float decrs=0; // sum of decreases
    int nincr=0;   // number of increases
    int ndecr=0;   // number of decreases
    unsigned int longPay=0; // we'll use this for layout
    int arrSize=0;

    transactions.sortOnName( true );
    text=title+": "+QString( 0 ).setNum( transactions.length() );
    if ( transactions.length()==1 ) text+=" transaction";
    else text+=" transactions";

    // do two runs: the first to figure out formatting and such,
    // and the second to start printing stuff
    QString lastPayee;
    for ( int i=0; i<transactions.length(); i++ ){
      const Transaction * t=transactions.getTrans( i );
      QString payee=t->get( Transaction::PAYEE );
      float f=t->getSum();
      
      if( payee!=lastPayee ){
	arrSize++;
	lastPayee=payee;
	if ( payee.length()>longPay ) longPay=payee.length();
      }

      if ( f<0 ){
	decrs-=f;
	ndecr++;
      }
      else{
	incrs+=f;
	nincr++;
      }
    }

    QString names[arrSize];
    float sums[arrSize];
    int counts[arrSize];
    
    lastPayee="";
    int cIndex=-1;
    for ( int i=0; i<transactions.length(); i++ ){
      const Transaction * t=transactions.getTrans( i );
      QString payee=t->get( Transaction::PAYEE );
      float f=t->getSum();
      
      if( payee==lastPayee ){
	sums[cIndex]+=f;
	counts[cIndex]++;
      }
      else{
	cIndex++;
	sums[cIndex]=f;
	counts[cIndex]=1;
	names[cIndex]=payee;
	lastPayee=payee;
      }
    }

    QString pa; 
    for ( int i=0; i<arrSize; i++ ){
      pa=names[i];
      pa.append( "("+QString().setNum( counts[i] )+")" );
      //pad payees
      for ( unsigned int j=pa.length(); j<longPay+10; j++ ) pa.append( " " );
      
      //put everything in out master string
      text.append( "\n\t"+pa+QString( 0 ).setNum( sums[i], 'f', 2 ) );
    }
  }
  return text;
}


void Reporter::prepareData( Account ** accts, unsigned int num, QDate starter,
			    QDate ender, GraphType ){
  setFont( QFont( "courier" ) );

  transactions.clear();
  transactions.startLoad();
  if ( accts ){
    for( unsigned int i=0; i<num; i++ ){
      QHaccVector v=accts[i]->getSubset( starter, ender );
      for( int j=0; j<v.length(); j++ ) transactions.add( v.getTrans( j ) );
    }
  }
  transactions.stopLoad();

  QString text=makeReport( title, transactions );
  display->setText( text );
}

/*********************/
/**   QHACCGRAPHER  **/
/*********************/

QHaccGrapher::QHaccGrapher( GType typer, QWidget * p, const char * n )
  : QDialog( p, n ){

  QGridLayout * layout=new QGridLayout( this, 10, 1, 1 );

  QPushButton * cl=new QPushButton( "Dismiss", this );

  QHaccGraphFrame * gr=0;
  if ( typer==GRAPH ) {
    gr=new Grapher( this );
    gt=new GraphTopper( ( Grapher * ) gr, this );
  }
  else {
    gr=new Reporter( this );
    gt=new GraphTopper( ( Reporter * ) gr, this );
  }

  connect( cl, SIGNAL( clicked() ), SLOT( close() ) );

  layout->addMultiCellWidget( gt, 0, 0, 0, 0 );
  layout->addMultiCellWidget( gr, 1, 8, 0, 0 );
  layout->addMultiCellWidget( cl, 9, 9, 0, 0 );
  setCaption( "QHacc Graph" );
  resize( 400, 450 );
}

QHaccGrapher::~QHaccGrapher(){}
void QHaccGrapher::close(){ QDialog::close( true ); }
void QHaccGrapher::accountOpened( const Account * a ) {gt->accountOpened( a );}
