/************************** * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
*************************** * * * * * * * * * * * * **************************/

#ifndef _ACCOUNTS_H
#define _ACCOUNTS_H

#include "globals.h"
#include "transaction.h"
#include "qhaccvector.h"

#include <iostream.h>

#include <qfile.h>
#include <qstring.h>
#include <qstrlist.h>
#include <qdatetime.h>
#include <qtextstream.h>


/**************************************/
/******** Account definitions *********/
/**************************************/

class Account:public QObject{
  Q_OBJECT
    public:

  const static int INSTITUTION, INSTADDR1, INSTADDR2, INSTCITY, INSTSTATE,
    INSTZIP, INSTPHONE, INSTFAX, INSTEMAIL, INSTCONTACT, ID, NAME,
    INFOARRAYSIZE, UPGRADE, DOWNGRADE, OPENING, CURRENT, FULL, SHORT;

  Account( const QString * =0, unsigned int =0, float =0, bool = false );
  virtual ~Account();

  friend ostream& operator<<( ostream &, const Account& );

  QString * getInfo() const;
  float getBal( int ) const;
  QString getName( int =0 ) const;
  QString getDefaultTransType() const;
  QString getTransType();  
  unsigned int getID() const;
  QStrList getPayees() const;
  Transaction * getTrans( int );
  const Account * getParent() const;
  int getNumTrans( QDate =QDate() ) const;
  QHaccVector getUnreconciled( float &, float &, QDate =QDate() ) const;
  QHaccVector getSubset( QDate, QDate ) const;
  QHaccVector getMems() const;

  void setOBal( float );
  void setTaxable( bool );
  void setDefaultTransType( QString );
  void setInfo( const QString * );
  void setParent( const Account * );
  void linkToParent( QHaccAccountManager * );

  bool isTaxable() const;
  bool isTopLevel() const;
  bool isInReco() const;

  int levelsDeep() const;
  Transaction * find( unsigned int ) const;
  int index( const Transaction * ) const;
  void startLoad( int );
  void stopLoad();
  void remTrans( int );
  QHaccVector prune( QDate );
  void alterReconciled( int );
  void addMemTrans( int );
  void addCloneTransFrom( Transaction * t ); 
  void loadMems( const QString& );
  void startReco();
  void stopReco();

  bool memTrans( Transaction * );
  bool writeToFile( const QString& );

  virtual bool writeMoreToFile( QFile& );
  virtual bool isType( AccountType ) const =0;
  virtual bool isLeftPlusAccount() const =0;
  virtual AccountType getType() const =0;
  virtual float getModifiedSumFor( float, const Account * ) const;

  static bool writeTrans( const QHaccVector&, const QString& );
  static Account * loadFromFile( const QString& );
  static Account * create( AccountType, unsigned int =0, const QString * =0,
			   float =0, bool =false );
  // we're making a new trans
  Transaction * createTrans( QDate =QDate(), QString =QString(),
			     QString =QString(), QString =QString(),
			     float =0,
			     Transaction::RecoType =Transaction::NO );
  // we're adding a split 
  Transaction * createBalancedTrans( Transaction *, float =0 );

 protected:
  QString * info, defaultTransType;
  float openBal, currBal;
  QHaccVector transactions, memtransactions;
  QStrList payees;
  int nextTranID;
  const Account * parent;
  unsigned int parentID;

 private:
  bool loading, taxable, beingRecod;
  unsigned int id;
  static unsigned int highID;

  public slots:
    void recalcBal();
  void newBal( float, float );
  // this trans already has a parent acct
  void addTrans( Transaction * );  
  void remTrans( Transaction * );

  protected slots:
  void refreshSignal();

 signals:
  void removedTrans();
  void addedTrans( Transaction * );
  void balanceChanged( float );
  void needRefresh();
  void changedParent();
};

/*****************************************/
/*******    Other   definitions    *******/
/*****************************************/

class LeftPlusAccount : public Account{
  Q_OBJECT
    public:
  LeftPlusAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~LeftPlusAccount();

  virtual bool isLeftPlusAccount() const;
};

/*****************************************/

class RightPlusAccount : public Account{
  Q_OBJECT
    public:
  RightPlusAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~RightPlusAccount();

  virtual bool isLeftPlusAccount() const;
};

/*****************************************/

class AssetAccount : public LeftPlusAccount{
  Q_OBJECT
    public:
  AssetAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~AssetAccount();

  AccountType getType() const;
  bool isType( AccountType ) const;
};

/*****************************************/

class LiabilityAccount : public RightPlusAccount{
  Q_OBJECT
    public:
  LiabilityAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~LiabilityAccount();

  AccountType getType() const;
  bool isType( AccountType ) const;
};

/*****************************************/

class EquityAccount : public RightPlusAccount{
  Q_OBJECT
    public:
  EquityAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~EquityAccount();

  AccountType getType() const;
  bool isType( AccountType ) const;
};

/*****************************************/

class ExpenseAccount : public EquityAccount{
  Q_OBJECT
    public:
  ExpenseAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~ExpenseAccount();

  AccountType getType() const;
  bool isType( AccountType ) const;

  virtual bool isLeftPlusAccount() const;
};

/*****************************************/

class RevenueAccount : public EquityAccount{
  Q_OBJECT
    public:
  RevenueAccount( const QString *, unsigned int =0, float =0, bool =false );
  virtual ~RevenueAccount();

  AccountType getType() const;
  bool isType( AccountType ) const;
};

/*****************************************/

#endif
