/* Direct libxmms bindings for Python
   Copyright (c) 2002 Florent Rougon

This file is part of PyXMMS.

PyXMMS is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

PyXMMS is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with PyXMMS; see the file COPYING. If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <Python.h>             /* Must come first */
#include <string.h>
#include <xmms/xmmsctrl.h>
#include <glib.h>

/* Small caveat: xmms uses glib. Some functions of the libxmms require glib
   types such as GList. We have to use glib functions like g_list_append to
   build them.

   The problem is that these functions don't report possible (though quite
   rare) errors like memory exhaustion, so we have no way to raise a Python
   exception in those cases. */


/* Small XMMS 1.2.6 bug */
void xmms_remote_show_about_box(gint session);

static PyObject *func_void_glist(void (*func)(gint, GList *),
                                 PyObject *self, PyObject *args);


#define PYXMMS_GBOOL(val) ((val) ? TRUE : FALSE)

#define PYXMMS_FUNC_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
\
    if (!PyArg_ParseTuple(args, "|i", &session)) \
        return NULL; \
\
    return Py_BuildValue("i", (int) func(session)); \
}

#define PYXMMS_FUNC_GFLOAT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
\
    if (!PyArg_ParseTuple(args, "|i", &session)) \
        return NULL; \
\
    return Py_BuildValue("f", (float) func(session)); \
}

#define PYXMMS_FUNC_VOID_GFLOAT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    float float_arg; \
\
    if (!PyArg_ParseTuple(args, "f|i", &float_arg, &session)) \
        return NULL; \
\
    func(session, (gfloat) float_arg); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_VOID_GINT_GFLOAT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int int_arg; \
    float float_arg; \
\
    if (!PyArg_ParseTuple(args, "if|i", &int_arg, &float_arg, &session)) \
        return NULL; \
\
    func(session, (gint) int_arg, (gfloat) float_arg); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_GFLOAT_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int int_arg; \
\
    if (!PyArg_ParseTuple(args, "i|i", &int_arg, &session)) \
        return NULL; \
\
    return Py_BuildValue("f", (float) func(session, int_arg)); \
}

#define PYXMMS_FUNC_GINT_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int int_arg; \
\
    if (!PyArg_ParseTuple(args, "i|i", &int_arg, &session)) \
        return NULL; \
\
    return Py_BuildValue("i", (int) func(session, int_arg)); \
}

#define PYXMMS_FUNC_VOID_GLIST(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    return func_void_glist(func, self, args); \
}

#define PYXMMS_FUNC_VOID_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int int_arg; \
\
    if (!PyArg_ParseTuple(args, "i|i", &int_arg, &session)) \
        return NULL; \
\
    func(session, (gint) int_arg); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_VOID_GCHARP(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    char *str; \
\
    if (!PyArg_ParseTuple(args, "s|i", &str, &session)) \
        return NULL; \
\
    func(session, (gchar *) str); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_VOID_GCHARP_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    char *str; \
    int int_arg; \
\
    if (!PyArg_ParseTuple(args, "si|i", &str, &int_arg, &session)) \
        return NULL; \
\
    func(session, (gchar *) str, (int) int_arg); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_VOID_GINT_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int int_arg1; \
    int int_arg2; \
\
    if (!PyArg_ParseTuple(args, "ii|i", &int_arg1, &int_arg2, &session)) \
        return NULL; \
\
    func(session, (gint) int_arg1, (gint) int_arg2); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_VOID(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
\
    if (!PyArg_ParseTuple(args, "|i", &session)) \
        return NULL; \
\
    func(session); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_VOID_GBOOLEAN(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int bool; \
\
    if (!PyArg_ParseTuple(args, "i|i", &bool, &session)) \
        return NULL; \
\
    func(session, PYXMMS_GBOOL(bool)); \
\
    Py_INCREF(Py_None); \
    return Py_None; \
}

#define PYXMMS_FUNC_GBOOLEAN(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
\
    if (!PyArg_ParseTuple(args, "|i", &session)) \
        return NULL; \
\
    return Py_BuildValue("i", (func(session) == TRUE) ? 1 : 0); \
}

#define PYXMMS_FUNC_GCHARP(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    PyObject *res; \
    gchar *str; \
\
    if (!PyArg_ParseTuple(args, "|i", &session)) \
        return NULL; \
\
    res = Py_BuildValue("s", (char *) (str = func(session))); \
    g_free(str); \
    return res; \
}

#define PYXMMS_FUNC_GCHARP_GINT(wrap, func) \
static PyObject *wrap(PyObject *self, PyObject *args) \
{ \
    int session = 0; \
    int int_arg; \
    PyObject *res; \
    gchar *str; \
\
    if (!PyArg_ParseTuple(args, "i|i", &int_arg, &session)) \
        return NULL; \
\
    res = Py_BuildValue("s", (char *) (str = func(session, int_arg))); \
    g_free(str); \
    return res; \
}

/* Function for local use */
static PyObject *func_void_glist(void (*func)(gint, GList *),
                                 PyObject *self, PyObject *args)
{
    int session = 0;
    PyObject *seq, *item;
    PyObject *fast_seq = NULL;
    int seq_len;
    char *msg_not_a_sequence = "first argument must be a sequence";
    guint i;
    char *py_internal_str;
    gchar *string;
    GList *strings = NULL;
    GList *listelt;
    
    if (!PyArg_ParseTuple(args, "O|i", &seq, &session))
        return NULL;

    if ((fast_seq = PySequence_Fast(seq, msg_not_a_sequence)) == NULL)
        return NULL;
    
    /* Would be OK on Python >= 2.2
    for (i=0; i < PySequence_Fast_GET_SIZE(fast_seq); i++)
    */

    if ((seq_len = PySequence_Size(seq)) == -1)
        goto error;
        
    for (i=0; i < seq_len; i++)
        {
            /* Retreive a string from the sequence */
            item = PySequence_Fast_GET_ITEM(fast_seq, i);
            if ((py_internal_str = PyString_AsString(item)) == NULL)
                goto error;

            /* In case xmms would write to py_internal_str ... */
            if ((string = strdup(py_internal_str)) == NULL)
                {
                    PyErr_SetFromErrno(PyExc_MemoryError);
                    goto error;
                }

            strings = g_list_append(strings, string);
        }
    
    /* And now, the call you've been all waiting for... */
    (*func)(session, strings);

    /* Free the strings */
    for (listelt = strings; listelt != NULL; listelt = g_list_next(listelt))
        free(listelt->data);
    /* Free the list */
    g_list_free(strings);

    Py_DECREF(fast_seq);

    Py_INCREF(Py_None);
    return Py_None;

  error:
    Py_XDECREF(fast_seq);
    /* The exception must be set at this point */
    return NULL;
}

/* Here follow the libxmms bindings */

/* XXXwatch */
static char playlist_doc[] =
"Set the playlist contents or enqueue files/URLs in it.

playlist(seq, enqueue, session=0) -> None

seq     -- a sequence of files/URLs
enqueue -- boolean (0 = false, 1 = true); false does not work in
           xmms 1.2.6.
session -- the XMMS session to act on";

static PyObject *pywrap_xmms_remote_playlist(PyObject *self,
                                             PyObject *args)
{
    int session = 0;
    int enqueue;
    PyObject *seq, *item;
    PyObject *fast_seq = NULL;
    int seq_len;
    char **str_list;
    guint i;
    char *str;
    
    if (!PyArg_ParseTuple(args, "Oi|i", &seq, &enqueue, &session))
        return NULL;

    if ((fast_seq = PySequence_Fast(seq,
                                    "first argument of playlist_add is not a "
                                    "sequence")) == NULL)
        return NULL;
    
    if ((seq_len = PySequence_Size(seq)) == -1)
        goto error;
        
    if ((str_list = malloc(seq_len * sizeof(char *))) == NULL)
        {
            PyErr_SetFromErrno(PyExc_MemoryError);
            goto error;
        }

    for (i=0; i < seq_len; i++)
        {
            item = PySequence_Fast_GET_ITEM(fast_seq, i);
            if ((str = PyString_AsString(item)) == NULL)
                goto error;

            /* In case xmms would write to the string... */
            if ((str_list[i] = strdup(str)) == NULL)
                {
                    PyErr_SetFromErrno(PyExc_MemoryError);
                    goto error;
                }
        }
    
    /* XXXwatch It seems that xmms 1.2.6 doesn't like to get enqueue = FALSE
       here (it _never_ happens in xmms 1.2.6 code)
    */
/*     printf("session: %d\nseq_len: %d\nenqueue: %d\n", */
/*            session, seq_len, enqueue); */
/*     for (i=0; i < seq_len; i++) */
/*         printf("%s\n", str_list[i]); */

    xmms_remote_playlist(session, str_list, seq_len, PYXMMS_GBOOL(enqueue));

    for (i=0; i < seq_len; i++)
        free(str_list[i]);

    free(str_list);
    
    Py_DECREF(fast_seq);

    Py_INCREF(Py_None);
    return Py_None;

  error:
    Py_XDECREF(fast_seq);
    /* The exception must be set at this point */
    return NULL;
}

/* XXX find the purpose of this function */
static char get_version_doc[] =
"Get version (of what?).

get_version(session=0) -> version of what? (integer)

The code fragment that answers to this request in XMMS 1.2.6 is:

  case CMD_GET_VERSION:
          ctrl_write_gint(pkt->fd, 0x09a3);
          ...
If you understand this better than I, please let me know.";

PYXMMS_FUNC_GINT(pywrap_xmms_remote_get_version, xmms_remote_get_version)

static char play_files_doc[] =
"Set the playlist and play.

play_files(seq, session=0) -> None

Clear the playlist, add the specified files/URLs to it and start
playing.

seq     -- a sequence of files/URLs
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GLIST(pywrap_xmms_remote_play_files,
                       xmms_remote_play_files)

static char playlist_add_doc[] =
"Add files/URLs to the playlist.

playlist_add(seq, session=0) -> None

seq     -- a sequence of files/URLs
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GLIST(pywrap_xmms_remote_playlist_add,
                       xmms_remote_playlist_add)

static char playlist_delete_doc[] =
"Delete one element of the playlist.

playlist_delete(index, session=0) -> None

index   -- the index (starting at 0) of the playlist entry to
           delete
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT(pywrap_xmms_remote_playlist_delete,
                      xmms_remote_playlist_delete)

static char play_doc[] =
"Play the current playlist entry.

play(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_play, xmms_remote_play)

static char pause_doc[] =
"Pause.

pause(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_pause, xmms_remote_pause)

static char stop_doc[] =
"Stop.

stop(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_stop, xmms_remote_stop)

static char play_pause_doc[] =
"Play or pause (toggle).

play_pause(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_play_pause, xmms_remote_play_pause)

static char is_playing_doc[] =
"Tell whether XMMS is playing (= not stopped).

is_playing(session=0) -> boolean (0 = false, 1 = true)

When in pause, XMMS is considered to be playing for this
function.

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_playing, xmms_remote_is_playing)

static char is_paused_doc[] =
"Tell whether XMMS is paused.

is_paused(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_paused, xmms_remote_is_paused)

static char get_playlist_pos_doc[] =
"Get the current playlist position.

get_playlist_pos(session=0) -> position (integer)

The count starts at 0 (first playlist entry).

session -- the XMMS session to contact";

PYXMMS_FUNC_GINT(pywrap_xmms_remote_get_playlist_pos,
                 xmms_remote_get_playlist_pos)

static char set_playlist_pos_doc[] =
"Set the current playlist position.

set_playlist_pos(position, session=0) -> None

pos     -- the target playlist index (counting from 0)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT(pywrap_xmms_remote_set_playlist_pos,
                      xmms_remote_set_playlist_pos)

static char get_playlist_length_doc[] =
"Get the playlist length (number of files/URLs).

get_playlist_length(session=0) -> length (integer)

session -- the XMMS session to contact";

PYXMMS_FUNC_GINT(pywrap_xmms_remote_get_playlist_length,
                 xmms_remote_get_playlist_length)

static char playlist_clear_doc[] =
"Clear the playlist.

playlist_clear(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_playlist_clear,
                 xmms_remote_playlist_clear)

static char get_output_time_doc[] =
"Get the time since the beginning of the current playlist entry.

get_output_time(session=0) -> integer value in ms

session -- the XMMS session to contact";

PYXMMS_FUNC_GINT(pywrap_xmms_remote_get_output_time,
                 xmms_remote_get_output_time)

static char jump_to_time_doc[] =
"Jump to a given time since the beginning of the current playlist entry.

jump_to_time(time, session=0) -> None

time    -- time to jump to, in ms (integer)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT(pywrap_xmms_remote_jump_to_time,
                      xmms_remote_jump_to_time)

static char get_volume_doc[] =
"Get left and right volumes.

get_volume(session=0) -> (left volume, right volume)

The volumes are integers.

session -- the XMMS session to contact";

static PyObject *pywrap_xmms_remote_get_volume(PyObject *self, PyObject *args)
{
    int session = 0;
    int vl, vr;

    if (!PyArg_ParseTuple(args, "|i", &session))
        return NULL;

    xmms_remote_get_volume(session, &vl, &vr);

    return Py_BuildValue("(ii)", vl, vr);
}

static char get_main_volume_doc[] =
"Get the greater of left and right volumes.

get_main_volume(session=0) -> volume (integer)

session -- the XMMS session to contact";

PYXMMS_FUNC_GINT(pywrap_xmms_remote_get_main_volume,
                 xmms_remote_get_main_volume)

static char get_balance_doc[] =
"Get the balance value.

get_balance(session=0) -> balance (integer)

session -- the XMMS session to contact";

PYXMMS_FUNC_GINT(pywrap_xmms_remote_get_balance, xmms_remote_get_balance)

static char set_volume_doc[] =
"Set left and right volumes.

set_volume(left_volume, right_volume, session=0) -> None

left_volume  -- left volume (integer)
right_volume -- right volume (integer)
session      -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT_GINT(pywrap_xmms_remote_set_volume,
                           xmms_remote_set_volume)

static char set_main_volume_doc[] =
"Set the \"main\" volume.

set_main_volume(volume, session=0) -> None

The left and right volumes are calculated (and set) from the
supplied `volume' argument and the current balance.

volume  -- \"main\" volume (integer)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT(pywrap_xmms_remote_set_main_volume,
                      xmms_remote_set_main_volume)

static char set_balance_doc[] =
"Set the balance.

set_balance(balance, session=0) -> None

balance -- balance value (integer)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT(pywrap_xmms_remote_set_balance,
                      xmms_remote_set_balance)

static char get_skin_doc[] =
"Get the current skin file.

get_skin(session=0) -> absolute file name for the skin (string)

session -- the XMMS session to contact";

PYXMMS_FUNC_GCHARP(pywrap_xmms_remote_get_skin, xmms_remote_get_skin)

static char set_skin_doc[] =
"Set the skin from a specified file.

set_skin(skin_file, session=0) -> None

skin_file -- the target skin file
session   -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GCHARP(pywrap_xmms_remote_set_skin, xmms_remote_set_skin)

static char get_playlist_file_doc[] =
"Get the file name corresponding to a given entry in the playlist.

get_playlist_file(index, session=0) -> absolute filename (string)

index   -- index (starting at 0) of the playlist entry
session -- the XMMS session to contact";

PYXMMS_FUNC_GCHARP_GINT(pywrap_xmms_remote_get_playlist_file,
                        xmms_remote_get_playlist_file)

static char get_playlist_title_doc[] =
"Get the title of a given entry in the playlist.

get_playlist_title(index, session=0) -> title (string)

index   -- index (starting at 0) of the playlist entry
session -- the XMMS session to contact";

PYXMMS_FUNC_GCHARP_GINT(pywrap_xmms_remote_get_playlist_title,
                        xmms_remote_get_playlist_title)
    
static char get_playlist_time_doc[] =
"Get the duration of a given entry in the playlist.

get_playlist_time(index, session=0) -> duration in ms (integer)

index   -- index (starting at 0) of the playlist entry
session -- the XMMS session to contact";

PYXMMS_FUNC_GINT_GINT(pywrap_xmms_remote_get_playlist_time,
                      xmms_remote_get_playlist_time)

static char get_info_doc[] =
"Get information about the current playlist entry.

get_info(session=0) -> (rate, frequency, number_of_channels)

The elements of the returned tuple are all integers.

session -- the XMMS session to contact";

static PyObject *pywrap_xmms_remote_get_info(PyObject *self, PyObject *args)
{
    int session = 0;
    int rate, freq, nch;

    if (!PyArg_ParseTuple(args, "|i", &session))
        return NULL;

    xmms_remote_get_info(session, &rate, &freq, &nch);

    return Py_BuildValue("(iii)", rate, freq, nch);
}

static char pl_win_toggle_doc[] =
"Hide or show the playlist window.

pl_win_toggle(display, session=0) -> None

This is not a real toggle function: you have to specify the
`display' argument.

display -- 1 to display the window, 0 to hide it
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GBOOLEAN(pywrap_xmms_remote_pl_win_toggle,
                          xmms_remote_pl_win_toggle)

static char eq_win_toggle_doc[] =
"Hide or show the equalizer window.

eq_win_toggle(display, session=0) -> None

This is not a real toggle function: you have to specify the
`display' argument.

display -- 1 to display the window, 0 to hide it
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GBOOLEAN(pywrap_xmms_remote_eq_win_toggle,
                          xmms_remote_eq_win_toggle)

static char main_win_toggle_doc[] =
"Hide or show the main window.

main_win_toggle(display, session=0) -> None

This is not a real toggle function: you have to specify the
`display' argument.

display -- 1 to display the window, 0 to hide it
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GBOOLEAN(pywrap_xmms_remote_main_win_toggle,
                          xmms_remote_main_win_toggle)

static char is_main_win_doc[] =
"Tell whether the main window is visible.

is_main_win(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_main_win, xmms_remote_is_main_win)

static char is_pl_win_doc[] =
"Tell whether the playlist window is visible.

is_pl_win(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_pl_win, xmms_remote_is_pl_win)

static char is_eq_win_doc[] =
"Tell whether the equalizer window is visible.

is_eq_win(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_eq_win, xmms_remote_is_eq_win)

static char show_prefs_box_doc[] =
"Show the preferences dialog box.

show_prefs_box(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_show_prefs_box,
                 xmms_remote_show_prefs_box)

/* XXX XMMS weird code here */
static char toggle_aot_doc[] =
"Set the always-on-top flag for the main window.

toggle_aot(always, session=0) -> None

The implementation of this function in XMMS 1.2.6 is weird,
I cannot tell you more about `always' than \"it seems not
to be used\".

always  -- should be a boolean (0 or 1)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GBOOLEAN(pywrap_xmms_remote_toggle_aot,
                          xmms_remote_toggle_aot)

static char show_about_box_doc[] =
"Show about box.

show_about_box(session=0) -> None

Does not to work in XMMS 1.2.6.

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_show_about_box,
                 xmms_remote_show_about_box)

static char eject_doc[] =
"Show the eject dialog box.

eject(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_eject, xmms_remote_eject)

static char playlist_prev_doc[] =
"Jump to the previous entry in the playlist.

playlist_prev(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_playlist_prev, xmms_remote_playlist_prev)

static char playlist_next_doc[] =
"Jump to the next entry in the playlist.

playlist_next(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_playlist_next, xmms_remote_playlist_next)

static char playlist_add_url_string_doc[] =
"Append a file/URL to the playlist.

playlist_add_url_string(string, session=0) -> None

string  -- an absolute file name or a URL
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GCHARP(pywrap_xmms_remote_playlist_add_url_string,
                        xmms_remote_playlist_add_url_string)

static char playlist_ins_url_string_doc[] =
"Insert a file/URL at a given position in the playlist.

playlist_ins_url_string(string, pos, session=0) -> None

string  -- an absolute file name or a URL
pos     -- index (starting at 0) of the playlist entry before
           which the entry for `string' will be inserted
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GCHARP_GINT(pywrap_xmms_remote_playlist_ins_url_string,
                             xmms_remote_playlist_ins_url_string)

static char is_running_doc[] =
"Tell whether the specified XMMS session is running.

is_running(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_running, xmms_remote_is_running)

static char toggle_repeat_doc[] =
"Toggle repeat mode.

toggle_repeat(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_toggle_repeat, xmms_remote_toggle_repeat)

static char toggle_shuffle_doc[] =
"Toggle shuffle mode.

toggle_shuffle(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_toggle_shuffle, xmms_remote_toggle_shuffle)

static char is_repeat_doc[] =
"Tell whether repeat mode is active.

is_repeat(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_repeat, xmms_remote_is_repeat)

static char is_shuffle_doc[] =
"Tell whether shuffle mode is active.

is_shuffle(session=0) -> boolean (0 = false, 1 = true)

session -- the XMMS session to contact";

PYXMMS_FUNC_GBOOLEAN(pywrap_xmms_remote_is_shuffle, xmms_remote_is_shuffle)

static char get_eq_doc[] =
"Get the equalizer settings.

get_eq(session=0) -> (preamp, (band0, band1, ... band9))

preamp, band0, ... band9 are all floats (gains in dB).

session -- the XMMS session to contact";

static PyObject *pywrap_xmms_remote_get_eq(PyObject *self, PyObject *args)
{
    int session = 0;
    float preamp;
    float *bands;
    PyObject *res;

    if (!PyArg_ParseTuple(args, "|i", &session))
        return NULL;

    xmms_remote_get_eq(session, (gfloat *) &preamp, (gfloat **) &bands);

    res = Py_BuildValue("(f(ffffffffff))", preamp,
                        bands[0],
                        bands[1],
                        bands[2],
                        bands[3],
                        bands[4],
                        bands[5],
                        bands[6],
                        bands[7],
                        bands[8],
                        bands[9]);
    g_free(bands);
    return res;
}

static char get_eq_preamp_doc[] =
"Get the equalizer preamp value.

get_eq_preamp(session=0) -> preamp in dB (float)

session -- the XMMS session to contact";

PYXMMS_FUNC_GFLOAT(pywrap_xmms_remote_get_eq_preamp,
                   xmms_remote_get_eq_preamp)

static char get_eq_band_doc[] =
"Get the equalizer setting for a given band.

get_eq_band(band, session=0) -> gain in dB (float)

band    -- integer between 0 and 9 (both inclusive), specifying
           the equalizer band from which the setting is to be
           retrieved. 0 is for the lowest frequency and 9 for the
           highest one.
session -- the XMMS session to contact";

PYXMMS_FUNC_GFLOAT_GINT(pywrap_xmms_remote_get_eq_band,
                        xmms_remote_get_eq_band)

static char set_eq_doc[] =
"Set all equalizer settings (preamp and all bands).

set_eq(preamp, seq, session=0) -> None

preamp  -- gain in dB (float)
seq     -- a sequence of 10 floats, which are the gains in dB of
           for the 10 frequency bands predefined in XMMS
           (first = lowest frequency, last = highest frequency)
session -- the XMMS session to act on";

static PyObject *pywrap_xmms_remote_set_eq(PyObject *self, PyObject *args)
{
    int session = 0;
    PyObject *seq, *item;
    PyObject *fast_seq = NULL;
    int seq_len;
    float preamp;
    float *bands;
    guint i;
    
    if (!PyArg_ParseTuple(args, "fO|i", &preamp, &seq, &session))
        return NULL;

    if ((fast_seq = PySequence_Fast(seq,
                                    "first argument of set_eq is not a "
                                    "sequence")) == NULL)
        return NULL;
    
    if ((seq_len = PySequence_Size(seq)) == -1)
        goto error;
        
    if ((bands = malloc(seq_len * sizeof(float))) == NULL)
        {
            PyErr_SetFromErrno(PyExc_MemoryError);
            goto error;
        }

    for (i=0; i < seq_len; i++)
        {
            item = PySequence_Fast_GET_ITEM(fast_seq, i);
            bands[i] = (float) PyFloat_AsDouble(item);
            if (PyErr_Occurred())
                goto error;
        }
    
    xmms_remote_set_eq(session, (gfloat) preamp, (gfloat *) bands);

    free(bands);
    
    Py_DECREF(fast_seq);

    Py_INCREF(Py_None);
    return Py_None;

  error:
    Py_XDECREF(fast_seq);
    /* The exception must be set at this point */
    return NULL;
}

static char set_eq_preamp_doc[] =
"Set the equalizer preamp value.

set_eq_preamp(preamp, session=0) -> None

preamp  -- gain in dB (float)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GFLOAT(pywrap_xmms_remote_set_eq_preamp,
                        xmms_remote_set_eq_preamp)

static char set_eq_band_doc[] =
"Set the equalizer setting for a given band.

set_eq_band(band, gain, session=0) -> None

band    -- number of the band (integer between 0 and 9, both
           inclusive, specifying the equalizer band from which
           the setting is to be set; 0 is for the lowest
           frequency and 9 for the highest one).
gain    -- value of the gain to set (float)
session -- the XMMS session to act on";

PYXMMS_FUNC_VOID_GINT_GFLOAT(pywrap_xmms_remote_set_eq_band,
                             xmms_remote_set_eq_band)

static char quit_doc[] =
"Quit the specified XMMS session.

quit(session=0) -> None

session -- the XMMS session to act on";

PYXMMS_FUNC_VOID(pywrap_xmms_remote_quit, xmms_remote_quit)


/* (defun flo-insert-python-method-def (func doc) */
/*   (interactive "sFunction: \nsDoc string: ") */
/*   (insert (format "    {\"%s\", pywrap_xmms_remote_%s, */
/*      METH_VARARGS, \"%s.\"},\n" func func doc))) */

/* (global-set-key [f1] 'flo-insert-python-method-def) */

static PyMethodDef pyxmms_methods[] =
{
    {"playlist", pywrap_xmms_remote_playlist,
     METH_VARARGS, playlist_doc},
    {"get_version", pywrap_xmms_remote_get_version,
     METH_VARARGS, get_version_doc},
    {"play_files", pywrap_xmms_remote_play_files,
     METH_VARARGS, play_files_doc},
    {"playlist_add", pywrap_xmms_remote_playlist_add,
     METH_VARARGS, playlist_add_doc},
    {"playlist_delete", pywrap_xmms_remote_playlist_delete,
     METH_VARARGS, playlist_delete_doc},
    {"play", pywrap_xmms_remote_play,
     METH_VARARGS, play_doc},
    {"pause", pywrap_xmms_remote_pause,
     METH_VARARGS, pause_doc},
    {"stop", pywrap_xmms_remote_stop,
     METH_VARARGS, stop_doc},
    {"play_pause", pywrap_xmms_remote_play_pause,
     METH_VARARGS, play_pause_doc},
    {"is_playing", pywrap_xmms_remote_is_playing,
     METH_VARARGS, is_playing_doc},
    {"is_paused", pywrap_xmms_remote_is_paused,
     METH_VARARGS, is_paused_doc},
    {"get_playlist_pos", pywrap_xmms_remote_get_playlist_pos,
     METH_VARARGS, get_playlist_pos_doc},
    {"set_playlist_pos", pywrap_xmms_remote_set_playlist_pos,
     METH_VARARGS, set_playlist_pos_doc},
    {"get_playlist_length", pywrap_xmms_remote_get_playlist_length,
     METH_VARARGS, get_playlist_length_doc},
    {"playlist_clear", pywrap_xmms_remote_playlist_clear,
     METH_VARARGS, playlist_clear_doc},
    {"get_output_time", pywrap_xmms_remote_get_output_time,
     METH_VARARGS, get_output_time_doc},
    {"jump_to_time", pywrap_xmms_remote_jump_to_time,
     METH_VARARGS, jump_to_time_doc},
    {"get_volume", pywrap_xmms_remote_get_volume,
     METH_VARARGS, get_volume_doc},
    {"get_main_volume", pywrap_xmms_remote_get_main_volume,
     METH_VARARGS, get_main_volume_doc},
    {"get_balance", pywrap_xmms_remote_get_balance,
     METH_VARARGS, get_balance_doc},
    {"set_volume", pywrap_xmms_remote_set_volume,
     METH_VARARGS, set_volume_doc},
    {"set_main_volume", pywrap_xmms_remote_set_main_volume,
     METH_VARARGS, set_main_volume_doc},
    {"set_balance", pywrap_xmms_remote_set_balance,
     METH_VARARGS, set_balance_doc},
    {"get_skin", pywrap_xmms_remote_get_skin,
     METH_VARARGS, get_skin_doc},
    {"set_skin", pywrap_xmms_remote_set_skin,
     METH_VARARGS, set_skin_doc},
    {"get_playlist_file", pywrap_xmms_remote_get_playlist_file,
     METH_VARARGS, get_playlist_file_doc},
    {"get_playlist_title", pywrap_xmms_remote_get_playlist_title,
     METH_VARARGS, get_playlist_title_doc},
    {"get_playlist_time", pywrap_xmms_remote_get_playlist_time,
     METH_VARARGS, get_playlist_time_doc},
    {"get_info", pywrap_xmms_remote_get_info,
     METH_VARARGS, get_info_doc},
    {"pl_win_toggle", pywrap_xmms_remote_pl_win_toggle,
     METH_VARARGS, pl_win_toggle_doc},
    {"eq_win_toggle", pywrap_xmms_remote_eq_win_toggle,
     METH_VARARGS, eq_win_toggle_doc},
    {"main_win_toggle", pywrap_xmms_remote_main_win_toggle,
     METH_VARARGS, main_win_toggle_doc},
    {"is_main_win", pywrap_xmms_remote_is_main_win,
     METH_VARARGS, is_main_win_doc},
    {"is_pl_win", pywrap_xmms_remote_is_pl_win,
     METH_VARARGS, is_pl_win_doc},
    {"is_eq_win", pywrap_xmms_remote_is_eq_win,
     METH_VARARGS, is_eq_win_doc},
    {"show_prefs_box", pywrap_xmms_remote_show_prefs_box,
     METH_VARARGS, show_prefs_box_doc},
    {"toggle_aot", pywrap_xmms_remote_toggle_aot,
     METH_VARARGS, toggle_aot_doc},
    {"show_about_box", pywrap_xmms_remote_show_about_box,
     METH_VARARGS, show_about_box_doc},
    {"eject", pywrap_xmms_remote_eject,
     METH_VARARGS, eject_doc},
    {"playlist_prev", pywrap_xmms_remote_playlist_prev,
     METH_VARARGS, playlist_prev_doc},
    {"playlist_next", pywrap_xmms_remote_playlist_next,
     METH_VARARGS, playlist_next_doc},
    {"playlist_add_url_string", pywrap_xmms_remote_playlist_add_url_string,
     METH_VARARGS, playlist_add_url_string_doc},
    {"playlist_ins_url_string", pywrap_xmms_remote_playlist_ins_url_string,
     METH_VARARGS, playlist_ins_url_string_doc},
    {"is_running", pywrap_xmms_remote_is_running,
     METH_VARARGS, is_running_doc},
    {"toggle_repeat", pywrap_xmms_remote_toggle_repeat,
     METH_VARARGS, toggle_repeat_doc},
    {"toggle_shuffle", pywrap_xmms_remote_toggle_shuffle,
     METH_VARARGS, toggle_shuffle_doc},
    {"is_repeat", pywrap_xmms_remote_is_repeat,
     METH_VARARGS, is_repeat_doc},
    {"is_shuffle", pywrap_xmms_remote_is_shuffle,
     METH_VARARGS, is_shuffle_doc},
    {"get_eq", pywrap_xmms_remote_get_eq,
     METH_VARARGS, get_eq_doc},
    {"get_eq_preamp", pywrap_xmms_remote_get_eq_preamp,
     METH_VARARGS, get_eq_preamp_doc},
    {"get_eq_band", pywrap_xmms_remote_get_eq_band,
     METH_VARARGS, get_eq_band_doc},
    {"set_eq", pywrap_xmms_remote_set_eq,
     METH_VARARGS, set_eq_doc},
    {"set_eq_preamp", pywrap_xmms_remote_set_eq_preamp,
     METH_VARARGS, set_eq_preamp_doc},
    {"set_eq_band", pywrap_xmms_remote_set_eq_band,
     METH_VARARGS, set_eq_band_doc},
    {"quit", pywrap_xmms_remote_quit,
     METH_VARARGS, quit_doc},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static char _xmms_doc[] =
"Python interface to XMMS -- internal module.

This module is an internal part of PyXMMS and should not be used
directly by \"user\" modules. It contains the direct interface to
libxmms. \"User\" modules should use the xmms module instead.";

void init_xmms(void)            /* Executed on the first import */
{
    Py_InitModule3("_xmms", pyxmms_methods, _xmms_doc);
}
