#include <config.h>

#include "ftpd.h"
#include "dynamic.h"
#include "ftpwho-update.h"
#include "globals.h"
#include "messages.h"
#ifdef WITH_DIRALIASES
# include "diraliases.h"
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

static void antiidle(void)
{
    if (noopidle == (time_t) -1) {
        noopidle = time(NULL);
    } else {
        if ((time(NULL) - noopidle) > (time_t) idletime_noop) {
            die(421, LOG_INFO, MSG_TIMEOUT_NOOP, (unsigned long) idletime_noop);
        }
    }    
}

/* 
 * Introduce a random delay, to avoid guessing existing user names by
 * mesuring delay. It's especially true when LDAP is used.
 * No need to call usleep2() because we are root at this point.
 */

static void randomdelay(void)
{
    usleep(rand() % 15000UL);          /* dummy... no need for arc4 */
}

/* 
 * Simple but fast command-line reader. We break the FTP protocol here,
 * because we deny access to files with strange characters in their name.
 * Now, I seriously doubt that clients should be allowed to upload files
 * with carriage returns, bells, cursor moves and other fancy stuff in the
 * names. It can indirectly lead to security flaws with scripts, it's
 * annoying for the sysadmin, it can be a client error, it can bring unexpected
 * results on some filesystems, etc. So control chars are replaced by "_".
 * Better be safe than 100% RFC crap compliant but unsafe. If you really want
 * RFC compliance, define RFC_CONFORMANT_PARSER. But I will hate you.
 * 
 * RFC_CONFORMANT_LINES is another thing that clients should implement
 * properly (and it's trivial to do) : lines must be ended with \r\n .
 * Guess what ? 
 * Some broken clients are just sending \n ... Grrrrrrrrrrrr !!!!!!!!!!!!!!!
 * 
 * -Frank.
 */

int sfgets(void)
{
    fd_set rs;
    struct timeval tv;
    ssize_t readen;
    signed char seen_r = 0;
    static size_t scanned;
    static size_t readend;
    
    if (scanned > (size_t) 0U) {       /* support pipelining */
        readend -= scanned;        
        memmove(cmd, cmd + scanned, readend);   /* safe */
        scanned = (size_t) 0U;
    }
    tv.tv_sec = idletime;
    tv.tv_usec = 0;        
    FD_ZERO(&rs);
    while (scanned < cmdsize) {
        if (scanned >= readend) {      /* nothing left in the buffer */
            FD_SET(0, &rs);
            while (select(1, &rs, NULL, NULL, &tv) == -1 && errno == EINTR);
            if (FD_ISSET(0, &rs) == 0) {
                return -1;
            }
            if (readend >= cmdsize) {
                break;
            }
            while ((readen = read(0, cmd + readend, cmdsize - readend))
                   < (ssize_t) 0 && errno == EINTR);
            if (readen <= (ssize_t) 0) {
                return -2;
            }
            readend += readen;
            if (readend > cmdsize) {
                return -2;
            }
        }
#ifdef RFC_CONFORMANT_LINES
        if (seen_r != 0) {
#endif
            if (cmd[scanned] == '\n') {
#ifndef RFC_CONFORMANT_LINES
                if (seen_r != 0) {
#endif
                    cmd[scanned - 1U] = 0;
#ifndef RFC_CONFORMANT_LINES
                } else {
                    cmd[scanned] = 0;
                }
#endif
                if (++scanned >= readend) {   /* non-pipelined command */
                    scanned = readend = (size_t) 0U;
                }
                return 0;
            }
            seen_r = 0;
#ifdef RFC_CONFORMANT_LINES
        }
#endif
        if (((unsigned char) cmd[scanned]) < 32U) {
            if (cmd[scanned] == '\r') {
                seen_r = 1;
            }
#ifdef RFC_CONFORMANT_PARSER                   /* disabled by default, intentionnaly */
            else if (cmd[scanned] == 0) {
                cmd[scanned] = '\n';
            }
#else
            /* replace control chars with _ */
            cmd[scanned] = '_';                
#endif
        }
        scanned++;
    }
    die(421, LOG_WARNING, MSG_LINE_TOO_LONG);   /* don't remove this */
    
    return 0;                         /* to please GCC */
}

/* Replace extra spaces before and after a string with '_' */

#ifdef MINIMAL
# define revealextraspc(X) (X)
#else
static char *revealextraspc(char * const s_)
{
    register unsigned char *s = (unsigned char *) s_;
    register unsigned char *sn;
    
    if (s == NULL) {
        return s_;
    }
    simplify(s);
    while (*s != 0U && isspace(*s)) {
        *s++ = '_';
    }
    if (*s == 0U) {
        return s_;
    }
    sn = s;
    do {
        sn++;
    } while (*sn != 0U);
    do {
        sn--;        
        if (!isspace(*sn)) {
            break;
        }
        *sn = '_';
    } while (sn != s);
    
    return s_;
}
#endif

void parser(void)
{
    char *arg;
#ifndef MINIMAL
    char *sitearg;
#endif
    size_t n;

    for (;;) {
        xferfd = -1;
        if (state_needs_update != 0) {
            state_needs_update = 0;
            setprogname("pure-ftpd (IDLE)");
#ifdef FTPWHO
            if (shm_data_cur != NULL) {
                ftpwho_lock();
                shm_data_cur->state = FTPWHO_STATE_IDLE;
                *shm_data_cur->filename = 0;
                ftpwho_unlock();
            }
#endif
        }
        doreply();
        alarm(idletime * 2);
        switch (sfgets()) {
        case -1:
            die(421, LOG_INFO, MSG_TIMEOUT_PARSER);            
        case -2:
            return;
        }
#ifdef DEBUG
        if (debug != 0) {
            addreply(0, "%s", cmd);
        }
#endif
        n = (size_t) 0U;
        while ((isalpha((unsigned char) cmd[n]) || cmd[n] == '@') &&
               n < cmdsize) {
            cmd[n] = tolower((unsigned char) cmd[n]);
            n++;
        }
        if (n >= cmdsize) {            /* overparanoid, it should never happen */
            die(421, LOG_WARNING, MSG_LINE_TOO_LONG);
        }
        if (n == (size_t) 0U) {
            nop:
            addreply(500, "?");
            continue;
        }
#ifdef SKIP_COMMAND_TRAILING_SPACES        
        while (isspace((unsigned char) cmd[n]) && n < cmdsize) {
            cmd[n++] = 0;
        }
        arg = cmd + n;        
        while (cmd[n] != 0 && n < cmdsize) {
            n++;
        }
        n--;
        while (isspace((unsigned char) cmd[n])) {
            cmd[n--] = 0;
        }
#else
        if (cmd[n] == 0) {
            arg = cmd + n;
        } else if (isspace((unsigned char) cmd[n])) {
            cmd[n] = 0;
            arg = cmd + n + 1;
        } else {
            goto nop;
        }
#endif
        if (logging != 0) {
#ifdef DEBUG
            logfile(LOG_DEBUG, MSG_DEBUG_COMMAND " [%s] [%s]",
                   cmd, arg);
#else
            logfile(LOG_DEBUG, MSG_DEBUG_COMMAND " [%s] [%s]",
                   cmd, strcmp(cmd, "pass") ? arg : "<*>");
#endif
        }
        /*
         * antiidle() is called with dummy commands, usually used by clients
         * who are wanting extra idle time. We give them some, but not too much.
         * When we jump to wayout, the idle timer is not zeroed. It means that
         * we didn't issue an 'active' command like RETR.
         */
        
#ifndef MINIMAL
        if (!strcmp(cmd, "noop") || !strcmp(cmd, "allo")) {
            antiidle();
            donoop();
            goto wayout;
        }
#endif
        if (!strcmp(cmd, "user")) {
            douser(arg);
        } else if (!strcmp(cmd, "pass")) {
            if (guest == 0) {
                randomdelay();
            }
            dopass(arg);
        } else if (!strcmp(cmd, "quit")) {
            addreply(221, MSG_GOODBYE,
                     (unsigned long long) ((uploaded + 1023ULL) / 1024ULL),
                     (unsigned long long) ((downloaded + 1023ULL) / 1024ULL));
            return;
        } else if (!strcmp(cmd, "syst")) {
            antiidle();
            addreply(215, "UNIX Type: L8");
            goto wayout;
        } else if (!strcmp(cmd, "auth") || !strcmp(cmd, "adat")) {
            /* RFC 2228 Page 5 Authentication/Security mechanism (AUTH) */
            addreply(502, MSG_AUTH_UNIMPLEMENTED);
        } else if (!strcmp(cmd, "type")) {
            antiidle();
            dotype(arg);
            goto wayout;
        } else if (!strcmp(cmd, "mode")) {
            antiidle();                
            domode(arg);
            goto wayout;
#ifndef MINIMAL
        } else if (!strcmp(cmd, "feat")) {
            dofeat();
            goto wayout;
#endif
        } else if (!strcmp(cmd, "stru")) {
            dostru(arg);
            goto wayout;
#ifndef MINIMAL
        } else if (!strcmp(cmd, "help")) {
            goto help_site;
#endif
#ifdef DEBUG
        } else if (!strcmp(cmd, "xdbg")) {
            debug++;
            addreply(200, MSG_XDBG_OK, debug);
            goto wayout;
#endif            
        } else if (loggedin == 0) {            
            /* from this point, all commands need authentication */
            addreply(530, MSG_NOT_LOGGED_IN);
            goto wayout;
        } else {
            if (!strcmp(cmd, "cwd") || !strcmp(cmd, "xcwd")) {
                antiidle();
                docwd(arg);
                goto wayout;
            } else if (!strcmp(cmd, "port")) {
                doport(arg);
#ifndef MINIMAL
            } else if (!strcmp(cmd, "eprt")) {
                doeprt(arg);
#endif
            } else if (disallow_passive == 0 && 
                       (!strcmp(cmd, "pasv") || !strcmp(cmd, "p@sw"))) {
                dopasv(0);
            } else if (disallow_passive == 0 && 
                       (!strcmp(cmd, "epsv") && 
                       (broken_client_compat == 0 ||
                        STORAGE_FAMILY(ctrlconn) == AF_INET6))) {
                if (!strcasecmp(arg, "all")) {
                    epsv_all = 1;
                    addreply(220, MSG_ACTIVE_DISABLED);
                } else if (!strcmp(arg, "2") && !v6ready) {
                    addreply(522, MSG_ONLY_IPV4);
                } else {
                    dopasv(1);
                }
#ifndef MINIMAL            
            } else if (disallow_passive == 0 && !strcmp(cmd, "spsv")) {
                dopasv(2);
#endif
            } else if (!strcmp(cmd, "pwd") || !strcmp(cmd, "xpwd")) {
                antiidle();
                addreply(257, "\"%s\" " MSG_IS_YOUR_CURRENT_LOCATION, wd);
                goto wayout;                
            } else if (!strcmp(cmd, "cdup") || !strcmp(cmd, "xcup")) {
                docwd("..");
            } else if (!strcmp(cmd, "retr")) {
                if (*arg != 0) {
                    doretr(arg);
                } else {
                    addreply(501, MSG_NO_FILE_NAME);
                }
            } else if (!strcmp(cmd, "rest")) {
                antiidle();
                if (*arg != 0) {
                    dorest(arg);
                } else {
                    addreply(501, MSG_NO_RESTART_POINT);
                    restartat = (off_t) 0;
                }
                goto wayout;
            } else if (!strcmp(cmd, "dele")) {
                if (*arg != 0) {
                    dodele(arg);
                } else {
                    addreply(501, MSG_NO_FILE_NAME);
                }
            } else if (!strcmp(cmd, "stor")) {
                arg = revealextraspc(arg);
                if (*arg != 0) {
                    dostor(arg, 0, autorename);
                } else {
                    addreply(501, MSG_NO_FILE_NAME);
                }
            } else if (!strcmp(cmd, "appe")) {
                arg = revealextraspc(arg);
                if (*arg != 0) {
                    dostor(arg, 1, 0);
                } else {
                    addreply(501, MSG_NO_FILE_NAME);
                }
#ifndef MINIMAL
            } else if (!strcmp(cmd, "stou")) {
                dostou();
#endif
#ifndef DISABLE_MKD_RMD
            } else if (!strcmp(cmd, "mkd") || !strcmp(cmd, "xmkd")) {
                arg = revealextraspc(arg);
                if (*arg != 0) {
                    domkd(arg);
                } else {
                    addreply(501, MSG_NO_DIRECTORY_NAME);
                }
            } else if (!strcmp(cmd, "rmd") || !strcmp(cmd, "xrmd")) {
                if (*arg != 0) {
                    dormd(arg);
                } else {
                    addreply(550, MSG_NO_DIRECTORY_NAME);
                }
#endif
#ifndef MINIMAL
            } else if (!strcmp(cmd, "stat")) {
                if (*arg != 0) {
                    opt_l = 1;
                    modern_listings = 0;
                    donlist(arg, 1);
                } else {
                    addreply(211, "http://www.pureftpd.org");
                }
#endif
            } else if (!strcmp(cmd, "list")) {
                opt_l = 1;
#ifndef MINIMAL
                modern_listings = 0;
#endif
                donlist(arg, 0);
            } else if (!strcmp(cmd, "nlst")) {
                opt_l = 0;
#ifndef MINIMAL                
                modern_listings = 0;
#endif
                donlist(arg, 0);
#ifndef MINIMAL
            } else if (!strcmp(cmd, "mlst")) {
                domlst(*arg != 0 ? arg : ".");
            } else if (!strcmp(cmd, "mlsd")) {
                opt_l = 1;
                modern_listings = 1;
                donlist(arg, 0);
#endif
            } else if (!strcmp(cmd, "abor")) {
                addreply(226, MSG_ABOR_SUCCESS);
#ifndef MINIMAL
            } else if (!strcmp(cmd, "site")) {
                if ((sitearg = arg) != NULL) {
                    while (*sitearg != 0 && !isspace((unsigned char) *sitearg)) {
                        sitearg++;
                    }
                    if (*sitearg != 0) {
                        *sitearg++ = 0;
                    }
                }
                if (!strcasecmp(arg, "idle")) {
                    if (sitearg == NULL || *sitearg == 0) {
                        addreply(501, "SITE IDLE: " MSG_MISSING_ARG);
                    } else {
                        unsigned long int i = 0;

                        i = strtoul(sitearg, &sitearg, 10);
                        if (sitearg && *sitearg)
                            addreply(501, MSG_GARBAGE_FOUND " : %s", sitearg);
                        else if (i > MAX_SITE_IDLE)
                            addreply(501, MSG_VALUE_TOO_LARGE);
                        else {
                            idletime = i;
                            addreply(200, MSG_IDLE_TIME, idletime);
                            idletime_noop = (double) idletime * 2.0;
                        }
                    }
                } else if (!strcasecmp(arg, "help")) {
                    help_site:
                    
                    addreply_noformat(214, MSG_SITE_HELP CRLF
# ifdef WITH_DIRALIASES
                                      " ALIAS" CRLF
# endif
                                      " CHMOD" CRLF " IDLE");
                    addreply(214, "Pure-FTPd - http://pureftpd.org");
                } else if (!strcasecmp(arg, "chmod")) {
                    char *sitearg2;
                    mode_t mode;
                    
                    parsechmod:
                    if (sitearg == NULL || *sitearg == 0) {
                        addreply(501, "SITE CHMOD: " MSG_MISSING_ARG);
                        goto chmod_wayout;
                    }
                    sitearg2 = sitearg;
                    while (*sitearg2 != 0 && !isspace((unsigned char) *sitearg2)) {
                        sitearg2++;
                    }                    
                    while (*sitearg2 != 0 && isspace((unsigned char) *sitearg2)) {
                        sitearg2++;
                    }                    
                    if (*sitearg2 == 0) {
                        addreply(550, "SITE CHMOD: " MSG_NO_FILE_NAME);
                        goto chmod_wayout;
                    }
                    mode = (mode_t) strtoul(sitearg, NULL, 8);
                    if (mode > (mode_t) 07777) {
                        addreply(501, "SITE CHMOD: " MSG_BAD_CHMOD);
                        goto chmod_wayout;
                    }
                    dochmod(sitearg2, mode);
                  chmod_wayout:
                    (void) 0;
# ifdef WITH_DIRALIASES
                } else if (!strcasecmp(arg, "alias")) {
                    if (sitearg == NULL || *sitearg == 0) {
                        print_aliases();
                    } else {
                        register const char *alias;
                        
                        if ((alias = lookup_alias(sitearg)) == NULL) {
                            addreply(214, MSG_ALIASES_ALIAS, sitearg, alias);
                        } else {
                            addreply(502, MSG_ALIASES_UNKNOWN, sitearg);
                        }
                    }
# endif
                } else if (*arg != 0) {
                    addreply(500, "SITE %s " MSG_UNKNOWN_EXTENSION, arg);
                } else {
                    addreply(500, "SITE: " MSG_MISSING_ARG);
                }
#endif
            } else if (!strcmp(cmd, "mdtm")) {
                domdtm(arg);
            } else if (!strcmp(cmd, "size")) {
                dosize(arg);
#ifndef MINIMAL
            } else if (!strcmp(cmd, "chmod")) {
                sitearg = arg;
                goto parsechmod;
#endif
            } else if (!strcmp(cmd, "rnfr")) {
                if (*arg != 0) {
                    dornfr(arg);
                } else {
                    addreply(550, MSG_NO_FILE_NAME);
                }
            } else if (!strcmp(cmd, "rnto")) {
                arg = revealextraspc(arg);
                if (*arg != 0) {
                    dornto(arg);
                } else {
                    addreply(550, MSG_NO_FILE_NAME);
                }
            } else {
                addreply(500, MSG_UNKNOWN_COMMAND);
            }
        }
        noopidle = (time_t) -1;
      wayout:
#ifdef THROTTLING
        if (throttling_delay != 0UL) {
            usleep2(throttling_delay);
        }
#else
        (void) 0;
#endif
    }
}
