
################################################################################
# Copyright (c) QinetiQ Plc 2003
#
# Licensed under the LGPL. For full license details see the LICENSE file.
################################################################################

"""
Initialisation support for hooking the GeoTypes into the psycopg typecast machinery.
"""
import sys, traceback

from _Point import Point
from _LineSeg import LineSeg
from _Path import Path
from _Polygon import Polygon
from _Circle import Circle
from _Box import Box

_class_map = {'point':   {'class': Point, 'oid': 600, 'name': 'POINT'},
              'lseg':    {'class': LineSeg, 'oid': 601, 'name': 'LINESEG'},
              'path':    {'class': Path,  'oid': 602, 'name': 'PATH'},
              'polygon': {'class': Polygon, 'oid': 604, 'name': 'POLYGON'},
              'circle':  {'class': Circle, 'oid': 718, 'name': 'CIRCLE'},
              'box':     {'class': Box, 'oid': 603, 'name': 'BOX'}}

from _OGMultiPolygon import OGMultiPolygon
from _OGPoint import OGPoint
from _OGLineString import OGLineString
from _OGPolygon import OGPolygon
from _OGMultiPoint import OGMultiPoint
from _OGMultiLineString import OGMultiLineString
from _OGGeometryCollection import OGGeometryCollection
from _OGGeoTypeFactory import OGGeoTypeFactory
from _WKTParser import WKTParser
from _WKBParser import WKBParser

class _PostWKTGISClassFactory:
    """
    Private class used as a factory for OpenGID types.
    """
    def __init__(self):
        pass
    
    def __call__(self,s=None):
        """
        A factory method for creating objects of the correct OpenGIS type.
        """
        factory = OGGeoTypeFactory()
        parser = WKTParser(factory)
        parser.parseGeometry(s)

        return factory.getGeometry()

class _PostWKBGISClassFactory:
    """
    Private class used as a factory for OpenGID types.
    """
    def __init__(self):
        pass
    
    def __call__(self,s=None):
        """
        A factory method for creating objects of the correct OpenGIS type.
        """
        factory = OGGeoTypeFactory()
        parser = WKBParser(factory)
        parser.parseGeometry(s)

        return factory.getGeometry()

def _getTypeOid(conn,curs,typename):
    curs.execute("select oid from pg_type where typname='%s'" % (typename,))
    conn.commit()
    return curs.fetchall()[0][0]
    
def initialisePsycopgTypes(psycopg_module, subclass_map={}, connect_string=None, register_opengis_types=None):
    """
    Inform psycopg about the GeoType types.

    This ensures that when the 'bound' variable method of query generation is used
    the GeoTypes are automatically coverted into the right format for use in postgres
    queries.

    It also ensures that any results columns from queries that are of a geometric
    type are returned as instances of the GeoType classes.

    (arg psycopg_module) is the psycopg module itself. This is passes in as a parameter
    to ensure that the GeoTypes package can be used with psycopg is required. If it
    were imported in the module directly it work reduce the flexibility of the package.

    (arg subclass_map) is a dictionary of 'postgres_type':'GeoType class' pairs. It allows
    a caller to subclass the GeoType classes and ensure that the subclasses get hooked into
    the psycopg type machinery rather the plain GeoType classes. See the PsycopgInit_Test.py
    file for an example of its use. Default {}.

    (arg connect_string) is a postgres connection string of the form
    'dbname=schema_test user=postgres. If this is not None an attempt it made to create a
    test table from which to dynamically workout the type OIDS for the geometric types. If this
    is None as list of OID values that where correct when I tested it are used. I am not sure
    how oftern these OIDs change but if you have problems with typecast registration try
    passing a connect_string and see if it fixes it. Default None.

    (arg register_opengis_types) is a flag to control whether the OpenGIS types are registered.
    The the flag is not None the OpenGIS types are registered with Psycopg. If this is not None
    a (arg connect_string) must be provided as the type oids for the OpenGIS types differ
    between databases and the correct oid must be calculated dynamically. Default None.
    
    """

    if subclass_map != {}:
        for override in subclass_map.keys():
            _class_map[override]['class'] = subclass_map[override]

    if connect_string != None:
        conn = psycopg_module.connect(connect_string)

        # Start by working out the oids for the standard Postgres geo types
        curs = conn.cursor()

        for typename in _class_map.keys():
            _class_map[typename]['oid'] = _getTypeOid(conn,curs,typename)
            
        if register_opengis_types:
            # Now calculate the type oid for the OpenGIS Geometry type.
            # This one is different for every database.
            try:
                geometry_type_oid = _getTypeOid(conn,curs,'geometry')
                wkb_type_oid = _getTypeOid(conn,curs,'wkb')
            except:
                type, value, tb = sys.exc_info()[:3]
                error = ("%s , %s \n" % (type, value))
                for bits in traceback.format_exception(type,value,tb):
                    error = error + bits + '\n'
                del tb

                raise RuntimeError, \
                      "Failed to get the type oid for the 'geometry' type from the database:\n\n"\
                      "                   connection_string = '%s' \n\n"\
                      "This is probably because you have not initialised the OpenGIS types\n"\
                      "for this database. Look at http://postgis.refractions.net/docs/x83.html\n"\
                      "for instructions on how to do this.\n\n"\
                      "The actual exception raised was:\n\n"\
                      "%s" % (connect_string, error)
            
            # Register the type factory for the OpenGIS types.
            psycopg_module.register_type(psycopg_module.new_type((geometry_type_oid,), 'Geometry', _PostWKTGISClassFactory()))
            psycopg_module.register_type(psycopg_module.new_type((wkb_type_oid,), 'WKB', _PostWKBGISClassFactory()))


    # Finally, register the standard Postgres GIS types.
    # If no connect_string is given these use a default set of type oids that were
    # correct for the database I tested it on.
    for new_class in _class_map.values():
        psycopg_module.register_type(psycopg_module.new_type((new_class['oid'],), new_class['name'], new_class['class']))


    
    
        
        
    
