
################################################################################
# Copyright (c) QinetiQ Plc 2003
#
# Licensed under the LGPL. For full license details see the LICENSE file.
################################################################################

"""
Implementation of Box.
"""

from _Point import Point, pointFromSequence, pointFromValues

class Box:
    """
    Box provides and implementation for a rectangular area in 2d space.

    It has accessor functions for the upper right hand and lower left hand
    corners of the box, which are Points. It provides a constructor that
    can handle the strings returned from Postgres for the box type and
    support equallity operations with other Box objects.

    NOTE: The accessor methods imply that they must recieve the
    coords for the upper_right and lower_left corners. Passing the wrong
    corners into the setXXX methods will result in a ValueError being raised.

    """
    
    def __init__(self,s=None):
        """
        Constructor. Optional (arg s) is a string as returned by postgres.
        It is of the form '((x.y),(x,y))' where x and y are floating point numbers.

        If (arg s) is None or omitted the Box is initalised to ((0.0,0.0),(0.0,0.0)).

        """
        if s:
            self.fromString(s)
        else:
            self._setUpperRight(Point())
            self._setLowerLeft(Point())
        self._normaliseCornerCoords()
        
    def _normaliseCornerCoords(self):
        #print self._upper_right
        xr = self._upper_right.getX()
        xl = self._lower_left.getX()
        yu = self._upper_right.getY()
        yl = self._lower_left.getY()

        if xl>xr and yl>yu:
            self._upper_right = pointFromValues(xl,yl)
            self._lower_left  = pointFromValues(xr,yu)                        
        elif xl>xr:
            self._upper_right = pointFromValues(xl,yu)
            self._lower_left  = pointFromValues(xr,yl)            
        elif yl>yu:
            self._upper_right = pointFromValues(xr,yl)
            self._lower_left  = pointFromValues(xl,yu)
        #print self
     
        
    def fromString(self,s):
        """
        Initialise the Box from a string.        

        (arg s) should be of the form '((x.y),(x,y))' where x and y are floating
        point numbers.
        
        """        
        seq = eval(s,{},{})
        self._setUpperRight(pointFromSequence(seq[0]))
        self._setLowerLeft(pointFromSequence(seq[1]))
        self._normaliseCornerCoords()
       
    def setUpperRight(self,p):
        """
        Set the upper right corner of the Box.

        (arg p) is a Point
        """
        if hasattr(self,'_lower_left'):
            if p.getX() < self._lower_left.getX() \
               or  p.getY() < self._lower_left .getY() :
                raise ValueError, "coord is not upper right of box."
        self._setUpperRight(p)
            
    def _setUpperRight(self,p):
        "Unckecked, private version"
        self._upper_right = p


    def getUpperRight(self):
        """
        Return a Point that represents the uppoer right corner of the Box.
        """
        self._normaliseCornerCoords()
        return self._upper_right

    def setLowerLeft(self,p):
        """
        Set the lower left corner of the Box.

        (arg p) is a Point
        """
        if  hasattr(self,'_upper_right'):
            if p.getX() > self._upper_right.getX() \
                   or  p.getY() > self._upper_right.getY() :
                raise ValueError, "coord is not lower left of box."
        self._setLowerLeft(p)
            
    def _setLowerLeft(self,p):
        "Unckecked, private version"
        self._lower_left = p
             
        
    def getLowerLeft(self):

        """
        Return a Point that represents the lower left corner of the Box.
        """
        self._normaliseCornerCoords()
        return self._lower_left

    def __str__(self):
        """
        Generate a string representation of the Box that is
        suitable to use in a Postgres query.
        """
        self._normaliseCornerCoords()
        return "'(%s,%s)'" % (self.getUpperRight().__repr__(),
                              self.getLowerLeft().__repr__())
    
    def __repr__(self):
        """
        Generate a represention of the Box as a string
        suitable for 'evaling' as a tuple.
        """
        self._normaliseCornerCoords()
        return "(%s,%s)" % (self.getUpperRight().__repr__(),
                            self.getLowerLeft().__repr__())

    def __eq__(self,other):
        """
        The is a simple equallity operator.
        """
        if (type(self) != type(other)):
            return False
        self._normaliseCornerCoords()
        other._normaliseCornerCoords()
        if self.getUpperRight() == other.getUpperRight() and \
           self.getLowerLeft() == other.getLowerLeft():
            return True
        return False

    def __ne__(self,other):
        return not self.__eq__(other)


# factory methods

def boxFromPoints(upper_right,lower_left):
    """
    Return a Box.

    (arg upper_right) is a Point object.
    (arg lower_left) is a Point object.
    """
    b = Box()
    b._setUpperRight(upper_right)
    b._setLowerLeft(lower_left)
    b._normaliseCornerCoords()
    return b

def boxFromSequence(seq):
    """
    Return a Box.

    (arg seq) is a sequence of the form '((x,y),(x1,y1))' where (x,y) is
    the upper right corner and (x1,y1) is the lower left corner.
    """
    b = Box()
    b._setUpperRight(pointFromSequence(seq[0]))
    b._setLowerLeft(pointFromSequence(seq[1]))
    b._normaliseCornerCoords()
    return b
               
