#include "prefs.h"

prefs::prefs() {
  autoSave = 0;            // Don't autosave settings when we quit
  loginStartup = 0;        // Don't login automatically
  numFetchLines = 100;     // 100 lines (TOP 100 n)
  savePassword = 0;        // Don't save passwords
  showFromName = 0;        // Show email addresses instead
  showHeadersRead = 1;     // Show headers when read
  sizeConvert = 1024;      // Show sizes in kbytes (1,1024,1024*1024 = b,kb,Mb)
  currentBookmark = 0;     // The default bookmark to be opened at startup
  
  scrollDelay = 10;        // Default delay in ms if scrolling out text
  
  showHeader[0] = 1;       // Default headers to display when reading message
  showHeader[1] = 1;
  showHeader[2] = 1;
  showHeader[3] = 1;
  showHeader[4] = 0;
  showHeader[5] = 0;
  showHeader[6] = 1;
  
  for (i=0; i<BSIZE; i++)
    host[i] = port[i] = user[i] = pass[i] = '\0';
}

// Save prepop settings in ~/.prepoprc (prefsFile)
int
prefs::saveSettings() {
  FILE* inFile;
  mode_t um;
  
  um = umask(077);
  inFile = fopen(prefsFile, "w");
  (void) umask(um);
  if (inFile) {
    fprintf(inFile, "autoSave=%d\n", autoSave);
    fprintf(inFile, "loginStartup=%d\n", loginStartup);
    fprintf(inFile, "savePassword=%d\n", savePassword);
    fprintf(inFile, "showFromName=%d\n", showFromName);
    fprintf(inFile, "showHeadersRead=%d\n", showHeadersRead);
    fprintf(inFile, "numFetchLines=%d\n", numFetchLines);
    fprintf(inFile, "sizeConvert=%d\n", sizeConvert);
    fprintf(inFile, "scrollDelay=%d\n", scrollDelay);
    
    fprintf(inFile, "showHeader0=%d\n", showHeader[0]);
    fprintf(inFile, "showHeader1=%d\n", showHeader[1]);
    fprintf(inFile, "showHeader2=%d\n", showHeader[2]);
    fprintf(inFile, "showHeader3=%d\n", showHeader[3]);
    fprintf(inFile, "showHeader4=%d\n", showHeader[4]);
    fprintf(inFile, "showHeader5=%d\n", showHeader[5]);
    fprintf(inFile, "showHeader6=%d\n", showHeader[6]);
    
    fprintf(inFile, "currentBookmark=%d\n\n", currentBookmark);
    
    for (i=0; i<BSIZE; i++) {
      if (!host[i] || !port[i] || !user[i])
	continue;
      
      fprintf(inFile, "bookmark=%d\n", i);
      fprintf(inFile, "%s\n", host[i]);
      fprintf(inFile, "%s\n", port[i]);
      fprintf(inFile, "%s\n", user[i]);
      if (savePassword && pass[i])
	fprintf(inFile, "%s\n\n", pass[i]);
      else
	fprintf(inFile, "*\n\n");
    }
    fclose(inFile);
    return(1);
  }
  else
    return(0);
}

// Load prepop settings from ~/.prepoprc (prefsFile)
int
prefs::loadSettings() {
  struct stat fileStatus;
  FILE* inFile;
  char* ptr;
  int count;
  
  // If file isn't present on disk
  if (access(prefsFile, F_OK) != 0)
    return(0);
  
  // If stat for file can't be accessed
  if (stat(prefsFile, &fileStatus) != 0)
    return(0);
  
  // If the correct permissions for the file can't be set
  if (fileStatus.st_mode != 33152)
    if (chmod(prefsFile, 33152) == -1)
      return(0);
  
  // If file can't be accessed in READ ONLY mode
  if (access(prefsFile, R_OK) != 0)
    return(0);
  
  ptrAlloc->freeMem(host[0]);
  ptrAlloc->freeMem(port[0]);
  ptrAlloc->freeMem(user[0]);
  
  host[0] = ptrAlloc->allocMem(10);
  port[0] = ptrAlloc->allocMem(4);
  user[0] = ptrAlloc->allocMem(5);
  
  strcpy(host[0], "localhost");
  strcpy(port[0], "110");
  strcpy(user[0], "user");
  
  for (count=0; count<2; count++) {
    
    if (count == 0)
      inFile = fopen("/etc/prepoprc", "r");
    else
      inFile = fopen(prefsFile, "r");
    
    if (inFile) {
      while (!feof(inFile)) {
	fgets(buffer, sizeof(buffer), inFile);
	buffer[strlen(buffer)-1] = '\0';
	
	ptr = strstr(buffer, "#");
	if (ptr)
	  *ptr = '\0';
	
	ptr = buffer;
	while (*ptr && isspace(*ptr))
	  ptr++;
	if (*ptr == '\0')
	  continue;
	
	if (strncasecmp(ptr, "autoSave=", 9) == 0)
	  autoSave = atoi(ptr+9);
	else if (strncasecmp(ptr, "loginStartup=", 13) == 0)
	  loginStartup = atoi(ptr+13);
	else if (strncasecmp(ptr, "savePassword=", 13) == 0)
	  savePassword  = atoi(ptr+13);
	else if (strncasecmp(ptr, "showFromName=", 13) == 0)
	  showFromName = atoi(ptr+13);
	else if (strncasecmp(ptr, "showHeadersRead=", 16) == 0)
	  showHeadersRead = atoi(ptr+16);
	else if (strncasecmp(ptr, "numFetchLines=", 14) == 0)
	  numFetchLines = atoi(ptr+14);
	else if (strncasecmp(ptr, "sizeConvert=", 12) == 0)
	  sizeConvert = atoi(ptr+12);
	else if (strncasecmp(ptr, "scrollDelay=", 12) == 0)
	  scrollDelay = atoi(ptr+12);
	
	else if (strncasecmp(ptr, "showHeader0=", 12) == 0)
	  showHeader[0] = atoi(ptr+12);
	else if (strncasecmp(ptr, "showHeader1=", 12) == 0)
	  showHeader[1] = atoi(ptr+12);
	else if (strncasecmp(ptr, "showHeader2=", 12) == 0)
	  showHeader[2] = atoi(ptr+12);
	else if (strncasecmp(ptr, "showHeader3=", 12) == 0)
	  showHeader[3] = atoi(ptr+12);
	else if (strncasecmp(ptr, "showHeader4=", 12) == 0)
	  showHeader[4] = atoi(ptr+12);
	else if (strncasecmp(ptr, "showHeader5=", 12) == 0)
	  showHeader[5] = atoi(ptr+12);
	else if (strncasecmp(ptr, "showHeader6=", 12) == 0)
	  showHeader[6] = atoi(ptr+12);
	
	else if (strncasecmp(ptr, "currentBookmark=", 16) == 0)
	  currentBookmark = atoi(ptr+16);
	else if (strncasecmp(ptr, "bookmark=", 9) == 0) {
	  i = atoi(ptr+9);
	  if ((i < 0) || (i > (BSIZE-1)))
	    continue;
	  
	  fgets(buffer, sizeof(buffer), inFile);
	  buffer[strlen(buffer)-1] = '\0';
	  ptrAlloc->freeMem(host[i]);
	  host[i] = ptrAlloc->allocMem(strlen(buffer)+1);
	  strcpy(host[i], buffer);
	  
	  fgets(buffer, sizeof(buffer), inFile);
	  buffer[strlen(buffer)-1] = '\0';
	  ptrAlloc->freeMem(port[i]);
	  port[i] = ptrAlloc->allocMem(strlen(buffer)+1);
	  strcpy(port[i], buffer);
	  
	  fgets(buffer, sizeof(buffer), inFile);
	  buffer[strlen(buffer)-1] = '\0';
	  ptrAlloc->freeMem(user[i]);
	  user[i] = ptrAlloc->allocMem(strlen(buffer)+1);
	  strcpy(user[i], buffer);
	  
	  fgets(buffer, sizeof(buffer), inFile);
	  buffer[strlen(buffer)-1] = '\0';
	  ptrAlloc->freeMem(pass[i]);
	  if (buffer[0] == '*')
	    continue;
	  pass[i] = ptrAlloc->allocMem(strlen(buffer)+1);
	  strcpy(pass[i], buffer);
	}
      }
      fclose(inFile);
    }
  }
  
  // Set currentBookmark to 0 if out of range
  if ((currentBookmark < 0) || (currentBookmark > (BSIZE-1)))
    currentBookmark = 0;
  
  // Set currentBookmark to 0 if currentBookmark is empty
  if (!host[currentBookmark] || !port[currentBookmark] || !user[currentBookmark])
    currentBookmark = 0;
  
  return(1);
}

// Save bookmark in memory arrays
int
prefs::saveBookmark(int num) {
  if ((num < 0) || (num > (BSIZE-1)) || (num == currentBookmark))
    return(0);
  
  if (!host[currentBookmark] || !port[currentBookmark] || !user[currentBookmark])
    return(0);
  
  ptrAlloc->freeMem(host[num]);
  ptrAlloc->freeMem(port[num]);
  ptrAlloc->freeMem(user[num]);
  ptrAlloc->freeMem(pass[num]);
  
  host[num] = ptrAlloc->allocMem(strlen(host[currentBookmark])+1);
  port[num] = ptrAlloc->allocMem(strlen(port[currentBookmark])+1);
  user[num] = ptrAlloc->allocMem(strlen(user[currentBookmark])+1);
  if (pass[currentBookmark])
    pass[num] = ptrAlloc->allocMem(strlen(pass[currentBookmark])+1);
  
  strcpy(host[num], host[currentBookmark]);
  strcpy(port[num], port[currentBookmark]);
  strcpy(user[num], user[currentBookmark]);
  if (pass[currentBookmark])
    strcpy(pass[num], pass[currentBookmark]);
  
  return(1);
}

// Convert size accordingly to sizeConvert
int
prefs::getSize(int size) {
  return ((int) size/sizeConvert);
}

// Return the sizePrefix
char*
prefs::getPrefix() {
  if (sizeConvert == 1)
    return("bytes");
  else if (sizeConvert == 1024)
    return("kbytes");
  else if (sizeConvert == 1048576)
    return("Mbytes");
  
  return('\0');
}

prefs::~prefs() {
}
