#include <iostream.h>
#include <strstream.h>
#include <stdlib.h>
#include <setjmp.h>
#include <unistd.h>
#include <sys/utsname.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#include "pop.h"
#include "sockets.h"
#include "header.h"
#include "help.h"
#include "display.h"
#include "prefs.h"
#include "alloc.h"
#include "signals.h"
#include "base64.h"
#include "util.h"
#include "config.h"

char* prompt;
char buffer[100];
long int i;
jmp_buf jmpbuffer;

POP *ptrPOP, POPObj;
sockets *ptrSockets, socketsObj;
help *ptrHelp, helpObj;
display *ptrDisplay, displayObj;
prefs *ptrPrefs, prefsObj;
alloc *ptrAlloc, allocObj;
signals *ptrSignals, signalsObj;
base64 *ptrBase64, base64Obj;
header *ptrHeader, headerObj;
util *ptrUtil, utilObj;

// Parse numbers in argumentlist [n,n-n]
int
parseArgs(const char* ptr) {
  int j;
  int start;
  int end;
  
  // Initialize array with zeros
  for (i=0; i<MAXMSG; i++)
    ptrHeader->numArgs[i] = 0;
  
  // Check syntax first
  for (i=0; ptr[i]; i++) {
    if (i==0 && !isdigit(ptr[i]))
      ptrDisplay->showError("Argument must start with a number");
    else if (!isdigit(ptr[strlen(ptr)-1]))
      ptrDisplay->showError("Argument must end with a number");
    else if (!isdigit(ptr[i]) && (ptr[i] != ',') && (ptr[i] != '-'))
      ptrDisplay->showError("Invalid characters");
    else if (((ptr[i] == ',') || (ptr[i] == '-')) &&
	     ((ptr[i+1] == ',') || (ptr[i+1] == '-')))
      ptrDisplay->showError("Invalid syntax");
    else
      continue;
    return(0);
  }
  
  // If we've never checked messages before
  if (!ptrHeader->maxNumber) {
    ptrHeader->maxNumber = MAXMSG;
  }
  
  // Create int array with messages to list/delete
  for (; *ptr; ptr++) {
    if ((atoi(ptr) <= ptrHeader->maxNumber) && (atoi(ptr) > 0)) {
      start = atoi(ptr);
      while (isdigit(*ptr) && *ptr)
	ptr++;
      if ((*ptr == ',') || (*ptr == '\0'))
	ptrHeader->numArgs[start] = 1;
      else {
	ptr++;
	end = atoi(ptr);
	if (end>=start && start>0 && end<=MAXMSG) {
	  for(j=start; j<=end; j++)
	    ptrHeader->numArgs[j] = 1;
	  while (isdigit(*ptr) && *ptr)
	    ptr++;
	}
	else {
	  ptrDisplay->showError("Invalid range of numbers");
	  return(0);
	}
      }
    }
    else {
      ptrDisplay->showError("Number out of range");
      return(0);
    }
    if (*ptr == '\0')
      break;
  }
  
  return(1);
}

// Set the prompt
void
setPrompt() {
  int allocSize = 14;
  
  ptrPOP->host = ptrPrefs->host[ptrPrefs->currentBookmark];
  ptrPOP->port = ptrPrefs->port[ptrPrefs->currentBookmark];
  ptrPOP->user = ptrPrefs->user[ptrPrefs->currentBookmark];
  
  if (ptrPOP->host)
    allocSize += strlen(ptrPOP->host);
  if (ptrPOP->port)
    allocSize += strlen(ptrPOP->port);
  if (ptrPOP->user)
    allocSize += strlen(ptrPOP->user);
  
  ptrAlloc->freeMem(prompt);
  prompt = ptrAlloc->allocMem(allocSize);
  
  strcpy(prompt, "[prepop ");
  strcat(prompt, ptrPOP->user);
  strcat(prompt, "@");
  strcat(prompt, ptrPOP->host);
  strcat(prompt, " ");
  strcat(prompt, ptrPOP->port);
  strcat(prompt, "]> ");  
}

// Load bookmark from memory arrays in prefs class
void
loadBookmark(int num) {
  if ((num >= 0) && (num < (BSIZE-1)) && ptrPrefs->host[num] &&
      ptrPrefs->port[num] && ptrPrefs->user[num])
    ptrPrefs->currentBookmark = num;
  else
    ptrDisplay->showError("One of the host, port or user fields in the bookmark is empty");
  
  setPrompt();
}

// Delete bookmark from memory arrays in prefs class
void
deleteBookmark(int num) {
  if ((num >= 0) && (num < (BSIZE-1)) && (ptrPrefs->currentBookmark != num)) {
    ptrAlloc->freeMem(ptrPrefs->host[num]);
    ptrAlloc->freeMem(ptrPrefs->port[num]);
    ptrAlloc->freeMem(ptrPrefs->user[num]);
    ptrAlloc->freeMem(ptrPrefs->pass[num]);
  }
  else
    ptrDisplay->showError("Out of range or trying to delete current prompt");
}

// Load default settings from ~/.prepoprc file
void
loadSettings() {
  if (!ptrPrefs->loadSettings()) {
    ptrDisplay->showText("\nCouldn't load settings: ");
    ptrDisplay->showText(ptrPrefs->prefsFile);
    ptrDisplay->showText("\n\n");
    ptrDisplay->showText("Note:\n");
    ptrDisplay->showText("You should be the owner of the prefsfile.\n");
    ptrDisplay->showText("Access permissions for prefsfile should be 600.\n\n");
  }
  else
    setPrompt();
}

// Set the host variable
void
setHost(const char* ptrCmd) {
  if (strlen(ptrCmd) < 1)
    return;
  
  ptrAlloc->freeMem(ptrPrefs->host[ptrPrefs->currentBookmark]);
  ptrPrefs->host[ptrPrefs->currentBookmark] = ptrAlloc->allocMem(strlen(ptrCmd)+1);
  strcpy(ptrPrefs->host[ptrPrefs->currentBookmark], ptrCmd);
  ptrAlloc->freeMem(ptrPrefs->pass[ptrPrefs->currentBookmark]);
  
  setPrompt();
}

// Set the port variable
void
setPort(char* ptrCmd) {
  if (strlen(ptrCmd) < 1)
    return;
  
  for (i=0; isdigit(ptrCmd[i]); i++) { }
  ptrCmd[i] = '\0';
  
  if ((atol(ptrCmd) < 65536) && (atol(ptrCmd) > 0)) {
    ptrAlloc->freeMem(ptrPrefs->port[ptrPrefs->currentBookmark]);
    ptrPrefs->port[ptrPrefs->currentBookmark] = ptrAlloc->allocMem(strlen(ptrCmd)+1);
    strcpy(ptrPrefs->port[ptrPrefs->currentBookmark], ptrCmd);
  }
  
  setPrompt();
}

// Set the user variable
void
setUser(const char* ptrCmd) {
  if (strlen(ptrCmd) < 1)
    return;
  
  ptrAlloc->freeMem(ptrPrefs->user[ptrPrefs->currentBookmark]);
  ptrPrefs->user[ptrPrefs->currentBookmark] = ptrAlloc->allocMem(strlen(ptrCmd)+1);
  strcpy(ptrPrefs->user[ptrPrefs->currentBookmark], ptrCmd);
  ptrAlloc->freeMem(ptrPrefs->pass[ptrPrefs->currentBookmark]);
  
  setPrompt();
}

// Set the pass variable
int
setPass() {
  char* passwd;
  
  // Only ask for password if one doesn't exist
  if (ptrPrefs->pass[ptrPrefs->currentBookmark] == NULL) {
    passwd = getpass("Password: ");
    if (strlen(passwd) < 1)
      return(0);
  }
  else {
    ptrPOP->pass = ptrPrefs->pass[ptrPrefs->currentBookmark];
    return(1);
  }
  
  ptrAlloc->freeMem(ptrPrefs->pass[ptrPrefs->currentBookmark]);
  ptrPrefs->pass[ptrPrefs->currentBookmark] = ptrAlloc->allocMem(strlen(passwd)+1);
  strcpy(ptrPrefs->pass[ptrPrefs->currentBookmark], passwd);
  ptrUtil->encryptPassword(ptrPrefs->pass[ptrPrefs->currentBookmark]);
  ptrPOP->pass = ptrPrefs->pass[ptrPrefs->currentBookmark];
  
  return(1);
}

// Setup the environment and default variables
void
initialize() {
  ptrPOP = &POPObj;
  ptrSockets = &socketsObj;
  ptrHelp = &helpObj;
  ptrDisplay = &displayObj;
  ptrPrefs = &prefsObj;
  ptrAlloc = &allocObj;
  ptrSignals = &signalsObj;
  ptrBase64 = &base64Obj;
  ptrHeader = &headerObj;
  ptrUtil = &utilObj;
  
  if (getenv("HOME") != NULL)
    ptrPrefs->prefsFile = ptrAlloc->allocMem(strlen(getenv("HOME"))+11);
  else
    ptrDisplay->showFatalError("Environment variable $HOME must be set!");
  
  strcpy(ptrPrefs->prefsFile, getenv("HOME"));
  strcat(ptrPrefs->prefsFile, "/.prepoprc");
  
  if (!ptrPrefs->loadSettings()) {
    ptrPrefs->host[0] = ptrAlloc->allocMem(10);
    ptrPrefs->port[0] = ptrAlloc->allocMem(4);
    ptrPrefs->user[0] = ptrAlloc->allocMem(5);
    
    strcpy(ptrPrefs->host[0], "localhost");
    strcpy(ptrPrefs->port[0], "110");
    strcpy(ptrPrefs->user[0], "user");
  }
  
  setPrompt();
  
  ptrSignals->ignoreSIGPIPE();
}

// Login to server
void
login() {
  ptrPOP->serverLogout();
  
  if (!setPass())
    return;
  
  if (ptrPOP->serverConnect()) {
    ptrDisplay->showText("Connect: ");
    ptrDisplay->showText(ptrPOP->message+4);
    ptrDisplay->showText("\n");
  }
  else
    return;
  
  ptrUtil->decryptPassword(ptrPrefs->pass[ptrPrefs->currentBookmark]);
  if (ptrPOP->serverLogin()) {
    ptrDisplay->showText(" Server: ");
    ptrDisplay->showText(ptrPOP->message+4);
    ptrDisplay->showText("\n");
    ptrUtil->encryptPassword(ptrPrefs->pass[ptrPrefs->currentBookmark]);
  }
  else {
    ptrAlloc->freeMem(ptrPrefs->pass[ptrPrefs->currentBookmark]);
    ptrDisplay->showError(ptrPOP->message+5);
  }
}

// Logout of server
void
logout() {
  if (ptrPOP->serverLogout()) {
    ptrDisplay->showText("Disconnect: ");
    ptrDisplay->showText(ptrPOP->message+4);
    ptrDisplay->showText("\n");
  }
  else
    ptrDisplay->showError(ptrPOP->message+5);
}

// Execute STAT command and show prettified output
void
status() {
  if (ptrPOP->serverSTAT()){
    ptrDisplay->showText("\n[01;04m");
    ptrDisplay->showText("Num", 15, "left");
    ptrDisplay->showText("Size");
    ptrDisplay->showText("[0m\n\n");
    ptrDisplay->showText(ptrUtil->intToASCII(ptrHeader->maxNumber), 15, "left");
    ptrDisplay->showText(ptrPrefs->getSize(ptrHeader->maxSize));
    ptrDisplay->showText("\n\nSize: ");
    ptrDisplay->showText(ptrPrefs->getPrefix());
    ptrDisplay->showText("\n\n");
  }
  else
    ptrDisplay->showError(ptrPOP->message+5);
}

// Execute *short* LIST command and show prettified output
void
shortList(const char* ptrCmd) {
  if (strlen(ptrCmd) > 0)
    if (!parseArgs(ptrCmd+1))
      return;
  
  ptrSignals->enableSIGINTHandler();
  
  if (setjmp(jmpbuffer) != 0) {
    ptrSignals->resetSIGINTHandler();
    ptrDisplay->showText("[interrupt]\n\n");
    ptrSockets->readToEOF();
    return;
  }
  
  if (!ptrPOP->serverShortLIST()) {
    ptrDisplay->showError(ptrPOP->message+5);
    ptrSignals->resetSIGINTHandler();
    return;
  }
  
  if (ptrHeader->maxNumber == 0) {
    ptrDisplay->showError("No messages on server");
    ptrSignals->resetSIGINTHandler();
    return;
  }
  
  if (strlen(ptrCmd) < 1)
    for (i=0; i<MAXMSG; i++)
      ptrHeader->numArgs[i] = 1;
  
  ptrDisplay->showText("\n[01;04m");
  ptrDisplay->showText("Msg", 15, "left");
  ptrDisplay->showText("Size");
  ptrDisplay->showText("[0m\n\n");
  
  for (i=1; i<=ptrHeader->maxNumber; i++) {
    // Debugging
    //usleep(100000);
    ptrDisplay->showText(ptrUtil->intToASCII(i), 15, "left");
    if (ptrHeader->numArgs[i]) {
      if (ptrHeader->size[i])
	ptrDisplay->showText(ptrPrefs->getSize(ptrHeader->size[i]));
      else
	ptrDisplay->showText("-");
    }
    ptrDisplay->showText("\n");
  }
  ptrDisplay->showText("\nSize: ");
  ptrDisplay->showText(ptrPrefs->getPrefix());
  ptrDisplay->showText("\n\n");
  
  ptrSignals->resetSIGINTHandler();
}

// Execute *long* LIST commands and show prettified output
void
longList(const char* ptrCmd) {
  if (strlen(ptrCmd) > 0)
    if (!parseArgs(ptrCmd+1))
      return;
  
  ptrSignals->enableSIGINTHandler();
  
  if (setjmp(jmpbuffer) != 0) {
    ptrSignals->resetSIGINTHandler();
    ptrDisplay->showText("[interrupt]\n\n");
    ptrSockets->readToEOF();
    return;
  }
  
  if (!ptrPOP->serverShortLIST()) {
    ptrDisplay->showError(ptrPOP->message+5);
    ptrSignals->resetSIGINTHandler();
    return;
  }
  
  if (ptrHeader->maxNumber == 0) {
    ptrDisplay->showError("No Messages on server");
    ptrSignals->resetSIGINTHandler();
    return;
  }
  
  if (strlen(ptrCmd) < 1)
    for (i=0; i<MAXMSG; i++)
      ptrHeader->numArgs[i] = 1;
  
  ptrDisplay->showText("\n[01;04m");
  ptrDisplay->showText("Msg", 8, "left");
  ptrDisplay->showText("Size", 9, "left");
  ptrDisplay->showText("From", 28, "left");
  ptrDisplay->showText("Subject", 38, "left");
  ptrDisplay->showText("Date", 18, "left");
  ptrDisplay->showText("[0m\n\n");
  
  for (i=1; i<=ptrHeader->maxNumber; i++) {
    if (ptrHeader->numArgs[i])
      if (ptrPOP->serverLongLIST(i)) {
	ptrDisplay->showText(ptrUtil->intToASCII(i), 8, "left");
	ptrDisplay->showText(ptrUtil->intToASCII(ptrPrefs->getSize(ptrHeader->size[i])), 9, "left");
	ptrDisplay->showText(ptrHeader->from[i], 28, "left");
	ptrDisplay->showText(ptrHeader->subject[i], 38, "left");
	ptrDisplay->showText(ptrHeader->date[i], 18, "left");
	ptrDisplay->showText("\n");
	
	// Debugging code!
	//cerr << i << ": " << ptrHeader->size[i] << "\n";
	//cerr << i << ": " << ptrHeader->from[i] << "\n";
	//cerr << i << ": " << ptrHeader->subject[i] << "\n";
	//cerr << i << ": " << ptrHeader->date[i] << "\n";
	//cerr << i << ": " << ptrHeader->contType[i] << "\n";
	//cerr << i << ": " << ptrHeader->charSet[i] << "\n";
	//cerr << i << ": " << ptrHeader->contTransEnc[i] << "\n";
	//cerr << i << ": " << ptrHeader->boundary[i] << "\n";
	//cerr << i << ": " << ptrHeader->xSender[i] << "\n";
	//cerr << i << ": " << ptrHeader->xMailer[i] << "\n";
	//cerr << i << ": " << ptrHeader->xPriority[i] << "\n";
      }
      else {
	ptrDisplay->showText(ptrUtil->intToASCII(i), 8, "left");
	ptrDisplay->showText("-", 9, "left");
	ptrDisplay->showText("-", 28, "left");
	ptrDisplay->showText("-", 38, "left");
	ptrDisplay->showText("-", 18, "left");
	ptrDisplay->showText("\n");
      }
  }
  ptrDisplay->showText("\nSize: ");
  ptrDisplay->showText(ptrPrefs->getPrefix());
  ptrDisplay->showText("\n\n");
  
  ptrSignals->resetSIGINTHandler();
}

// Delete specified message with DELE command
void
deleteMsg(const char* ptrCmd) {
  if (!parseArgs(ptrCmd))
    return;
  
  for (i=1; i<MAXMSG; i++)
    if (ptrHeader->numArgs[i]) {
      strcpy(buffer, "DELE ");
      strcat(buffer, ptrUtil->intToASCII(i));
      ptrDisplay->showText("\nServer: ");
      ptrDisplay->showText(i);
      ptrDisplay->showText(" (");
      if (ptrPOP->serverCmd(buffer)) {
	ptrDisplay->showText(ptrPOP->message+4);
	
	ptrHeader->size[i] = 0;
	ptrAlloc->freeMem(ptrHeader->from[i]);
	ptrAlloc->freeMem(ptrHeader->subject[i]);
	ptrAlloc->freeMem(ptrHeader->date[i]);
	ptrAlloc->freeMem(ptrHeader->contType[i]);
	ptrAlloc->freeMem(ptrHeader->charSet[i]);
	ptrAlloc->freeMem(ptrHeader->contTransEnc[i]);
	ptrAlloc->freeMem(ptrHeader->boundary[i]);
	ptrAlloc->freeMem(ptrHeader->xSender[i]);
	ptrAlloc->freeMem(ptrHeader->xMailer[i]);
	ptrAlloc->freeMem(ptrHeader->xPriority[i]);
      }
      else
	ptrDisplay->showText(ptrPOP->message+5);
      ptrDisplay->showText(")");
    }
  ptrDisplay->showText("\n\n");
}

// Read specified message with TOP command
void
readMsg(const char* ptrCmd) {
  ptrSignals->enableSIGINTHandler();
  
  if (setjmp(jmpbuffer) != 0) {
    ptrSignals->resetSIGINTHandler();
    ptrDisplay->showText("\n[interrupt]\n\n");
    ptrSockets->readToEOF();
    return;
  }
  
  if (ptrPOP->serverCmdRead(atoi(ptrCmd))) { 
    if (ptrPrefs->showHeadersRead) {  
      ptrDisplay->showText("\n");
      
      if (ptrPrefs->showHeader[0]) {
	ptrDisplay->showText("   [01;04mFrom:[0m ");
	ptrDisplay->showText(ptrHeader->from[atoi(ptrCmd)]);
	ptrDisplay->showText("\n");
      }
      
      if (ptrPrefs->showHeader[1]) {
	ptrDisplay->showText("[01;04mSubject:[0m ");
	ptrDisplay->showText(ptrHeader->subject[atoi(ptrCmd)]);
	ptrDisplay->showText("\n");
      }
      
      if (ptrPrefs->showHeader[2]) {
	ptrDisplay->showText("   [01;04mDate:[0m ");
	ptrDisplay->showText(ptrHeader->date[atoi(ptrCmd)]);
	ptrDisplay->showText("\n");
      }
      
      if (ptrPrefs->showHeader[3]) {
	ptrDisplay->showText(" [01;04mMailer:[0m ");
	ptrDisplay->showText(ptrHeader->xMailer[atoi(ptrCmd)]);
	ptrDisplay->showText("\n");
      }
      
      if (ptrPrefs->showHeader[4]) {
	ptrDisplay->showText("[01;04mCharset:[0m ");
	ptrDisplay->showText(ptrHeader->charSet[atoi(ptrCmd)]);
	ptrDisplay->showText("\n");
      }
      
      if (ptrPrefs->showHeader[5]) {
	ptrDisplay->showText("   [01;04mPrio:[0m ");
	ptrDisplay->showText(ptrHeader->xPriority[atoi(ptrCmd)]);
	ptrDisplay->showText("\n");
      }
      
      if (ptrPrefs->showHeader[6]) {
	ptrDisplay->showText("   [01;04mSize:[0m ");
	ptrDisplay->showText(ptrPrefs->getSize(ptrHeader->size[atoi(ptrCmd)]));
	ptrDisplay->showText(" ");
	ptrDisplay->showText(ptrPrefs->getPrefix());
	ptrDisplay->showText("\n");
      }
      ptrDisplay->showText("\n");
    }
    
    // If we should scroll out text or just dump it there
    if (ptrPrefs->scrollDelay)
      ptrDisplay->showScroll(ptrPOP->message+4, ptrPrefs->scrollDelay);
    else
      ptrDisplay->showText(ptrPOP->message+4);
    
    ptrDisplay->showText("\n\n");
  }
  else
    ptrDisplay->showError(ptrPOP->message+5);
  
  ptrSignals->resetSIGINTHandler();
}

// Send a POP3 command (RFC 1939, 2449)
void
commandRFC(char* ptrCmd) {
  for (i=0; ptrCmd[i]; i++)
    buffer[i] = toupper(ptrCmd[i]);
  buffer[i] = '\0';
  
  if (ptrPOP->serverCmd(buffer)) {
    ptrDisplay->showText("\n");
    ptrDisplay->showText(ptrPOP->message+4);
    ptrDisplay->showText("\n\n");
  }
  else
    ptrDisplay->showError(ptrPOP->message+5);
}

// Save specified bookmark
void
saveBookmark(int num) {
  if (ptrPrefs->saveBookmark(num)) {
    ptrDisplay->showText("\nBookmark saved in position: ");
    ptrDisplay->showText(num);
    ptrDisplay->showText("\n\n");
  }
  else
    ptrDisplay->showError("Invalid number, bookmark was not saved");
}

// Show program settings
void
showSettings() {
  ptrDisplay->showText("\n   autosave: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->autoSave));
  ptrDisplay->showText("\n loginstart: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->loginStartup));
  ptrDisplay->showText("\n   savepass: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->savePassword));
  ptrDisplay->showText("\n   fromname: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showFromName));
  ptrDisplay->showText("\n    headers: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeadersRead));
  ptrDisplay->showText("\n linesfetch: ");
  ptrDisplay->showText(ptrPrefs->numFetchLines);
  ptrDisplay->showText(" lines");
  ptrDisplay->showText("\nscrolldelay: ");
  ptrDisplay->showText(ptrPrefs->scrollDelay);
  ptrDisplay->showText(" ms");
  ptrDisplay->showText("\n   showsize: ");
  ptrDisplay->showText(ptrPrefs->getPrefix());
  ptrDisplay->showText("\n");
  ptrDisplay->showText("\n   From [0]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[0]));
  ptrDisplay->showText("\nSubject [1]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[1]));
  ptrDisplay->showText("\n   Date [2]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[2]));
  ptrDisplay->showText("\n Mailer [3]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[3]));
  ptrDisplay->showText("\nCharset [4]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[4]));
  ptrDisplay->showText("\n   Prio [5]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[5]));
  ptrDisplay->showText("\n   Size [6]: ");
  ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[6]));
  ptrDisplay->showText("\n\n");
}

// Save settings to prefsfile
void
saveSettings() {
  if (ptrPrefs->saveSettings()) {
    ptrDisplay->showText("\nSettings saved in: ");
  }
  else {
    ptrDisplay->showText("\nUnable to open: ");
  }
  
  ptrDisplay->showText(ptrPrefs->prefsFile);
  ptrDisplay->showText("\n\n");
}

// Enter the settings mode
void
changeSettings() {
  char* command;
  
  ptrDisplay->showText("\nType 'help' to display available commands in settings mode\n\n");
  
  while (true) {
    command = readline("[prepop Settings]> ");
    
    if (strlen(command) > 100) {
      ptrAlloc->freeMem(command);
      continue;
    }
    
    if (*command != '\0')
      add_history(command);
    
    ptrUtil->fixInput(command);
    
    if ((strcmp(command, "help") == 0) || (strcmp(command, "?") == 0))
      ptrHelp->showSettingsHelp();
    else if (strcmp(command, "save") == 0)
      saveSettings();
    else if (strcmp(command, "load") == 0)
      loadSettings();
    else if (strcmp(command, "show") == 0)
      showSettings();
    else if (strcmp(command, "autosave") == 0) {
      ptrPrefs->autoSave = (~ptrPrefs->autoSave) & (1);
      ptrDisplay->showText("autosave: ");
      ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->autoSave));
      ptrDisplay->showText("\n");
    }
    else if (strcmp(command, "loginstart") == 0) {
      ptrPrefs->loginStartup = (~ptrPrefs->loginStartup) & (1);
      ptrDisplay->showText("loginstart: ");
      ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->loginStartup));
      ptrDisplay->showText("\n");
    }
    else if (strcmp(command, "savepass") == 0) {
      ptrPrefs->savePassword = (~ptrPrefs->savePassword) & (1);
      ptrDisplay->showText("savepass: ");
      ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->savePassword));
      ptrDisplay->showText("\n");
    }
    else if (strcmp(command, "fromname") == 0) {
      ptrPrefs->showFromName = (~ptrPrefs->showFromName) & (1);
      ptrDisplay->showText("fromname: ");
      ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showFromName));
      ptrDisplay->showText("\n");
    }
    else if (strcmp(command, "headers") == 0) {
      ptrPrefs->showHeadersRead = (~ptrPrefs->showHeadersRead) & (1);
      ptrDisplay->showText("headers: ");
      ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeadersRead));
      ptrDisplay->showText("\n");
    }
    else if (strncmp(command, "linesfetch ", 11) == 0) {
      if ((atoi(command+11) > 0) && (atoi(command+11) < 65535)) {
	ptrPrefs->numFetchLines = atoi(command+11);
	ptrDisplay->showText("linesfetch: ");
	ptrDisplay->showText(ptrPrefs->numFetchLines);
	ptrDisplay->showText(" lines\n");
      }
    }
    else if (strcmp(command, "showsize") == 0) {
      if (ptrPrefs->sizeConvert == 1)
	ptrPrefs->sizeConvert = 1024;
      else if (ptrPrefs->sizeConvert == 1024)
	ptrPrefs->sizeConvert = 1024*1024;
      else if (ptrPrefs->sizeConvert == 1024*1024)
	ptrPrefs->sizeConvert = 1;
      ptrDisplay->showText("showsize: ");
      ptrDisplay->showText(ptrPrefs->getPrefix());
      ptrDisplay->showText("\n");
    }
    else if (strncmp(command, "scrolldelay ", 12) == 0) {
      if ((atoi(command+12) >= 0) && (atoi(command+12) <= 1000)) {
	ptrPrefs->scrollDelay = atoi(command+12);
	ptrDisplay->showText("scrolldelay: ");
	ptrDisplay->showText(ptrPrefs->scrollDelay);
	ptrDisplay->showText(" ms\n");
      }
    }
    else if (strncmp(command, "header ", 7) == 0) {
      if (atoi(command+7) == 0) {
	ptrPrefs->showHeader[0] = (~ptrPrefs->showHeader[0]) & (1);
	ptrDisplay->showText("From [0]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[0]));
	ptrDisplay->showText("\n");
      }
      if (atoi(command+7) == 1) {
	ptrPrefs->showHeader[1] = (~ptrPrefs->showHeader[1]) & (1);
	ptrDisplay->showText("Subject [1]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[1]));
	ptrDisplay->showText("\n");
      }
      if (atoi(command+7) == 2) {
	ptrPrefs->showHeader[2] = (~ptrPrefs->showHeader[2]) & (1);
	ptrDisplay->showText("Date [2]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[2]));
	ptrDisplay->showText("\n");
      }
      if (atoi(command+7) == 3) {
	ptrPrefs->showHeader[3] = (~ptrPrefs->showHeader[3]) & (1);
	ptrDisplay->showText("Mailer [3]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[3]));
	ptrDisplay->showText("\n");
      }
      if (atoi(command+7) == 4) {
	ptrPrefs->showHeader[4] = (~ptrPrefs->showHeader[4]) & (1);
	ptrDisplay->showText("Charset [4]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[4]));
	ptrDisplay->showText("\n");
      }
      if (atoi(command+7) == 5) {
	ptrPrefs->showHeader[5] = (~ptrPrefs->showHeader[5]) & (1);
	ptrDisplay->showText("Prio [5]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[5]));
	ptrDisplay->showText("\n");
      }
      if (atoi(command+7) == 6) {
	ptrPrefs->showHeader[6] = (~ptrPrefs->showHeader[6]) & (1);
	ptrDisplay->showText("Size [6]: ");
	ptrDisplay->showText(ptrUtil->boolToEnglish(ptrPrefs->showHeader[6]));
	ptrDisplay->showText("\n");
      }
    }
    else if (strcmp(command, "exit") == 0)
      break;
    else if (strlen(command) > 0) {
      ptrDisplay->showText(command);
      ptrDisplay->showText(": no such command\n");
    }
    
    ptrAlloc->freeMem(command);
  }
}

// Show environment settings for prepop
void
showEnv() {
  struct utsname os;
  
  ptrDisplay->showText("\n[01;04mCurrent Bookmark[0m\n");
  ptrDisplay->showText("    host: ");
  ptrDisplay->showText(ptrPrefs->host[ptrPrefs->currentBookmark]);
  ptrDisplay->showText("\n");
  ptrDisplay->showText("    port: ");
  ptrDisplay->showText(ptrPrefs->port[ptrPrefs->currentBookmark]);
  ptrDisplay->showText("\n");
  ptrDisplay->showText("    user: ");
  ptrDisplay->showText(ptrPrefs->user[ptrPrefs->currentBookmark]);
  ptrDisplay->showText("\n");
  ptrDisplay->showText("password: ");
  if (ptrPrefs->pass[ptrPrefs->currentBookmark] != '\0')
    ptrDisplay->showText("****");
  ptrDisplay->showText("\n\n");
  ptrDisplay->showText("[01;04mEnvironment[0m\n");
  
  if (uname(&os) == 0) {
    ptrDisplay->showText("    host: ");
    ptrDisplay->showText(os.sysname);
    ptrDisplay->showText(" ");
    ptrDisplay->showText(os.release);
    ptrDisplay->showText(" (");
    ptrDisplay->showText(os.machine);
    ptrDisplay->showText(")\n");
  }
  
  ptrDisplay->showText("settings: ");
  ptrDisplay->showText(ptrPrefs->prefsFile);
  ptrDisplay->showText("\n");
  ptrDisplay->showText(" columns: ");
  ptrDisplay->showText(getenv("COLUMNS"));
  ptrDisplay->showText("\n");
  ptrDisplay->showText("   lines: ");
  ptrDisplay->showText(getenv("LINES"));
  ptrDisplay->showText("\n\n");
}

// Show bookmarks from memory arrays
void showBookmarks() {
  ptrDisplay->showText("\n[01;04m");
  ptrDisplay->showText("Num", 15, "left");
  ptrDisplay->showText("Host", 35, "left");
  ptrDisplay->showText("Port", 15, "left");
  ptrDisplay->showText("User", 20, "left");
  ptrDisplay->showText("Pass[0m\n\n");
  
  for (i=0; i<BSIZE; i++) {
    if (!ptrPrefs->host[i] || !ptrPrefs->port[i] || !ptrPrefs->user[i])
      continue;
    
    if (i == ptrPrefs->currentBookmark) {
      ptrDisplay->showText(i);
      ptrDisplay->showText("+", 15-strlen(ptrUtil->intToASCII(i)), "left");
    }
    else
      ptrDisplay->showText(ptrUtil->intToASCII(i), 15, "left");
    ptrDisplay->showText(ptrPrefs->host[i], 35, "left");
    ptrDisplay->showText(ptrPrefs->port[i], 15, "left");
    ptrDisplay->showText(ptrPrefs->user[i], 20, "left");
    if (ptrPrefs->pass[i] != '\0')
      ptrDisplay->showText("****");
    ptrDisplay->showText("\n");
  }
  ptrDisplay->showText("\n");
}

// Where the fun starts
int
main(int argc) {
  char* command;
  
  initialize();
  
  /*
  char fire[10];
  login();
  while (true) {
    for (int a=1; a<611; a++) {
      strcpy(fire, ptrUtil->intToASCII(a));
      readMsg(fire);
      cerr << "FIRE: " << fire << "\n";
    }
  }
  */
  
  /*
  // START beta expire program
  time_t lt;
  struct tm *timePtr;
  int year;
  short int month;
  short int day;
  short int hour;
  short int min;
  short int sec;
  
  lt = time('\0');
  timePtr = localtime(&lt);
  year = timePtr->tm_year;
  month = timePtr->tm_mon + 1;
  day = timePtr->tm_mday;
  hour = timePtr->tm_hour;
  min = timePtr->tm_min;
  sec = timePtr->tm_sec;
  
  // Y2K Fix
  if (year < 1000) {
    if (year > 99) {
      year += 1900;
    }
    else {
      if (year < 50) {
        year += 2000;
      }
      else {
        year += 1900;
      }
    }
  }
  
  if ((month < 1) || (month > 2) || (year != 2000))
  {
    cerr << "\nprepop " << VERSION << " has expired, please find a new version at http://www.rundegren.com\n\n";
    exit(1);
  }
  // END beta expire program
  */
  
  if (argc != 1) {
    ptrHelp->showUsage();
    exit(1);
  }
  
  if (ptrPrefs->loginStartup)
    login();
  
  while (true) {
    command = readline(prompt);
    if (strlen(command) > 255) {
      ptrAlloc->freeMem(command);
      ptrDisplay->showError("Input is limited to 255 characters!");
      continue;
    }
    if (*command != '\0')
      add_history(command);
    
    ptrUtil->fixInput(command);
    
    if (strcmp(command, "vars") == 0)
      showEnv();
    else if (strncmp(command, "host ", 5) == 0)
      setHost(command+5);
    else if (strncmp(command, "port ", 5) == 0)
      setPort(command+5);
    else if (strncmp(command, "user ", 5) == 0)
      setUser(command+5);
    else if (strcmp(command, "prefs") == 0)
      changeSettings();
    else if (strcmp(command, "save") == 0)
      saveSettings();
    else if (strcmp(command, "load") == 0)
      loadSettings();
    else if (strcmp(command, "login") == 0)
      login();
    else if (strcmp(command, "logout") == 0)
      logout();
    else if (strcmp(command, "stat") == 0)
      status();
    else if (strncmp(command, "slist", 5) == 0)
      shortList(command+5);
    else if (strncmp(command, "list", 4) == 0)
      longList(command+4);
    else if (strncmp(command, "del ", 4) == 0)
      deleteMsg(command+4);
    else if (strncmp(command, "read ", 5) == 0)
      readMsg(command+5);
    else if (strncmp(command, "cmd ", 4) == 0)
      commandRFC(command+4);
    else if ((strcmp(command, "help") == 0) || (strcmp(command, "?") == 0))
      ptrHelp->showHelp();
    else if (strcmp(command, "about") == 0)
      ptrHelp->aboutBox();
    else if (strcmp(command, "bshow") == 0)
      showBookmarks();
    else if (strncmp(command, "bsave ", 6) == 0)
      saveBookmark(atoi(command+6));
    else if (strncmp(command, "bload ", 6) == 0)
      loadBookmark(atoi(command+6));
    else if (strncmp(command, "bdel ", 5) == 0)
      deleteBookmark(atoi(command+5));
    else if (strcmp(command, "quit") == 0)
      break;
    else if (strlen(command) > 0) {
      ptrDisplay->showText(command);
      ptrDisplay->showText(": no such command\n");
    }
    
    ptrAlloc->freeMem(command);
  }
  
  ptrPOP->serverLogout();
  
  if (ptrPrefs->autoSave)
    saveSettings();
  
  return(0);
}
