/*
** pork_format.c - helper functions for formatting text.
** Copyright (C) 2002-2003 Amber Adams <amber@ojnk.net>
** Copyright (C) 2003 Ryan McCabe <ryan@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
*/

#include <config.h>

#include <unistd.h>
#include <ncurses.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>

#include <pork.h>
#include <pork_missing.h>
#include <pork_util.h>
#include <pork_list.h>
#include <pork_color.h>
#include <pork_buddy.h>
#include <pork_imwindow.h>
#include <pork_buddy_list.h>
#include <pork_acct.h>
#include <pork_cstr.h>
#include <pork_misc.h>
#include <pork_screen.h>
#include <pork_screen_cmd.h>
#include <pork_transfer.h>
#include <pork_format.h>

static int format_status_timestamp(	char opt,
									char *buf,
									size_t len,
									va_list ap __notused)
{
	int ret = 0;
	time_t cur_time;
	struct tm *tm;

	cur_time = time(NULL);
	tm = localtime(&cur_time);

	if (tm == NULL)
		return (-1);

	switch (opt) {
		/* Hours, 24-hour format */
		case 'H':
			ret = snprintf(buf, len, "%02d", tm->tm_hour);
			break;

		/* Hours, 12-hour format */
		case 'h':
		{
			u_int32_t time_12hour;

			if (tm->tm_hour == 0)
				time_12hour = 12;
			else {
				time_12hour = tm->tm_hour;

				if (time_12hour > 12)
					time_12hour -= 12;
			}

			ret = snprintf(buf, len, "%02d", time_12hour);
			break;
		}

		/* Minutes */
		case 'm':
			ret = snprintf(buf, len, "%02d", tm->tm_min);
			break;

		/* Seconds */
		case 's':
			ret = snprintf(buf, len, "%02d", tm->tm_sec);
			break;

		/* AM or PM */
		case 'z':
			ret = snprintf(buf, len, "%s",
				(tm->tm_hour >= 12) ? "pm" : "am");
			break;

		default:
			return (-1);
	}

	if (ret < 0 || (size_t) ret >= len)
		return (-1);

	return (0);
}

static int format_status_activity(	char opt,
									char *buf,
									size_t len,
									va_list ap __notused)
{
	switch (opt) {
		/* Activity display */
		case 'A':
		case 'a':
		{
			dlist_t *cur = screen.window_list;
			size_t n = 0;

			do {
				struct imwindow *imwindow = cur->data;

				if (imwindow->swindow.activity &&
					!imwindow->ignore_activity &&
					!imwindow->swindow.visible)
				{
					int ret = snprintf(&buf[n], len - n, "%u,",
								imwindow->refnum);

					if (ret < 0 || (size_t) ret >= len - n)
						return (-1);

					n += ret;
				}

				cur = cur->next;
			} while (cur != screen.window_list);

			if (n > 0)
				buf[n - 1] = '\0';
			else
				return (1);

			break;
		}

		default:
			return (-1);
	}

	return (0);
}

static int format_status_typing(char opt, char *buf, size_t len, va_list ap) {
	int ret = 0;
	struct imwindow *imwindow = va_arg(ap, struct imwindow *);

	switch (opt) {
		/* Typing status */
		case 'Y':
			if (imwindow->typing) {
				char *str;

				if (imwindow->typing == 1)
					str = opt_get_str(OPT_TEXT_TYPING_PAUSED);
				else
					str = opt_get_str(OPT_TEXT_TYPING);

				if (xstrncpy(buf, str, len) == -1)
					ret = -1;
			} else
				return (1);

			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_status_held(char opt, char *buf, size_t len, va_list ap) {
	struct imwindow *imwindow = va_arg(ap, struct imwindow *);
	int ret = 0;
	int r_code;

	if (imwindow->swindow.held == 0)
		return (1);

	switch (opt) {
		/* Held indicator */
		case 'H':
			r_code = snprintf(buf, len, "%02d", imwindow->swindow.held);
			if (r_code < 0 || (size_t) r_code >= len)
				ret = -1;
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_status_idle(char opt, char *buf, size_t len, va_list ap) {
	struct pork_acct *acct = va_arg(ap, struct pork_acct *);
	int hide_if_zero = va_arg(ap, int);

	switch (opt) {
		/* Idle time */
		case 'I':
		own:
			if (acct->idle_time == 0 && hide_if_zero)
				return (1);

			time_to_str(acct->idle_time, buf, len);
			break;

		/*
		** If the current window is a chat window, $i will display the
		** idle time of the user we're talking to.
		*/
		case 'i':
		{
			struct imwindow *win = cur_window();

			if (win->type == TYPE_PRIVMSG) {
				struct buddy *buddy;

				buddy = buddy_find(acct, win->target);
				if (buddy == NULL || (buddy->idle_time == 0 && hide_if_zero))
					return (1);

				time_to_str(buddy->idle_time, buf, len);
				break;
			}

			goto own;
		}

		default:
			return (-1);
	}

	return (0);
}

static int format_status(char opt, char *buf, size_t len, va_list ap) {
	struct imwindow *imwindow = va_arg(ap, struct imwindow *);
	struct pork_acct *acct = va_arg(ap, struct pork_acct *);
	int ret = 0;

	switch (opt) {
		/* Screen name */
		case 'n':
		case 'N':
			if (xstrncpy(buf, acct->username, len) == -1)
				ret = -1;
			break;

		/* Window name */
		case 'z':
		case 'Z':
			if (xstrncpy(buf, imwindow->name, len) == -1)
				ret = -1;
			break;

		/* Chat room, if inactive */
		case 'c':
		case 'C':
			if (imwindow->type == TYPE_CHAT && imwindow->data == NULL) {
				if (xstrncpy(buf, opt_get_str(OPT_TEXT_NO_ROOM), len) == -1)
					ret = -1;
			}
			break;

		/* Timestamp */
		case 't':
		case 'T':
			ret = fill_format_str(OPT_FORMAT_STATUS_TIMESTAMP, buf, len);
			break;

		/* Activity */
		case 'a':
		case 'A':
			ret = fill_format_str(OPT_FORMAT_STATUS_ACTIVITY, buf, len);
			break;

		/* Typing */
		case 'y':
		case 'Y':
			ret = fill_format_str(OPT_FORMAT_STATUS_TYPING, buf, len, imwindow);
			break;

		/* Held Messages */
		case 'h':
		case 'H':
			ret = fill_format_str(OPT_FORMAT_STATUS_HELD, buf, len, imwindow);
			break;

		/* Idle Time */
		case 'i':
		case 'I':
			ret = fill_format_str(OPT_FORMAT_STATUS_IDLE, buf, len, acct,
					isupper(opt));
			break;

		/* Warn Level */
		case 'w':
		case 'W':
			ret = fill_format_str(OPT_FORMAT_STATUS_WARN, buf, len, acct,
					isupper(opt));
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_status_warn(char opt, char *buf, size_t len, va_list ap) {
	struct pork_acct *acct= va_arg(ap, struct pork_acct *);
	int hide_if_zero = va_arg(ap, int);
	int ret = 0;
	int r_code;

	switch (opt) {
		/* Warn level */
		case 'W':
		own:
			if (acct->warn_level == 0 && hide_if_zero)
				return (1);

			r_code = snprintf(buf, len, "%02d", acct->warn_level);
			if (r_code < 0 || (size_t) r_code >= len)
				ret = -1;
			break;

		case 'w':
		{
			struct imwindow *win = cur_window();

			if (win->type == TYPE_PRIVMSG) {
				struct buddy *buddy;

				buddy = buddy_find(acct, win->target);
				if (buddy == NULL || (buddy->warn_level == 0 && hide_if_zero))
					return (1);

				r_code = snprintf(buf, len, "%02d", buddy->warn_level);
				if (r_code < 0 || (size_t) r_code >= len)
					ret = -1;
				break;
			}

			goto own;
		}

		default:
			return (-1);
	}

	return (ret);
}

static int format_privmsg(char opt, char *buf, size_t len, va_list ap) {
	char *nick = va_arg(ap, char *);
	char *dest = va_arg(ap, char *);
	char *msg = va_arg(ap, char *);
	int ret = 0;

	switch (opt) {
		/* Screen name of sender */
		case 'n':
		case 'N':
			if (xstrncpy(buf, nick, len) == -1)
				ret = -1;
			break;

		/* Screen name of the receiver */
		case 'R':
		case 'r':
			if (xstrncpy(buf, dest, len) == -1)
				ret = -1;
			break;

		/* Message text */
		case 'm':
		case 'M':
			if (xstrncpy(buf, msg, len) == -1)
				ret = -1;
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_chat_msg(char opt, char *buf, size_t len, va_list ap) {
	char *source = va_arg(ap, char *);
	char *dest = va_arg(ap, char *);
	char *msg = va_arg(ap, char *);
	int ret = 0;

	switch (opt) {
		/* Message source */
		case 'N':
		case 'n':
			if (xstrncpy(buf, source, len) == -1)
				ret = -1;
			break;

		/* Message destination */
		case 'C':
		case 'c':
			if (xstrncpy(buf, dest, len) == -1)
				ret = -1;
			break;

		/* Message text */
		case 'm':
		case 'M':
			if (xstrncpy(buf, msg, len) == -1)
				ret = -1;
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_chat_info(char opt, char *buf, size_t len, va_list ap) {
	char *name = va_arg(ap, char *);
	char *action = va_arg(ap, char *);
	char *room = va_arg(ap, char *);
	int ret = 0;

	switch (opt) {
		case 'n':
		case 'N':
			if (xstrncpy(buf, name, len) == -1)
				ret = -1;
			break;

		/* Action (join, part, etc) */
		case 'a':
		case 'A':
			if (xstrncpy(buf, action, len) == -1)
				ret = -1;
			break;

		case 'r':
		case 'R':
			if (xstrncpy(buf, room, len) == -1)
				ret = -1;
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_chat_arginfo(char opt, char *buf, size_t len, va_list ap) {
	char *name = va_arg(ap, char *);
	char *room = va_arg(ap, char *);
	char *altname = va_arg(ap, char *);
	char *msg = va_arg(ap, char *);
	int ret = 0;

	switch (opt) {
		case 'n':
		case 'N':
			if (xstrncpy(buf, name, len) == -1)
				ret = -1;
			break;

		case 'u':
		case 'U':
			if (xstrncpy(buf, altname, len) == -1)
				ret = -1;
			break;

		case 'm':
		case 'M':
		{
			char *chat_msg = msg;
			if (chat_msg == NULL)
				chat_msg = " ";

			if (xstrncpy(buf, chat_msg, len) == -1)
				ret = -1;

			break;
		}

		case 'r':
		case 'R':
			if (xstrncpy(buf, room, len) == -1)
				ret = -1;
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_warning(char opt, char *buf, size_t len, va_list ap) {
	char *you = va_arg(ap, char *);
	char *warner = va_arg(ap, char *);
	u_int16_t warn_level = va_arg(ap, unsigned int);
	int ret = 0;

	switch (opt) {
		case 'n':
		case 'N':
			if (xstrncpy(buf, you, len) == -1)
				ret = -1;
			break;

		case 'u':
		case 'U':
			if (xstrncpy(buf, warner, len) == -1)
				ret = -1;
			break;

		case 'w':
		case 'W':
		{
			int r_code;

			r_code = snprintf(buf, len, "%u", warn_level);
			if (r_code < 0 || (size_t) r_code >= len)
				ret = -1;
			break;
		}

		default:
			return (-1);
	}

	return (ret);
}

static int format_blist_idle(char opt, char *buf, size_t len, va_list ap) {
	struct buddy *buddy = va_arg(ap, struct buddy *);
	int ret = 0;

	switch (opt) {
		case 'I':
			time_to_str(buddy->idle_time, buf, len);
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_blist_warn(char opt, char *buf, size_t len, va_list ap) {
	struct buddy *buddy = va_arg(ap, struct buddy *);
	int ret = 0;

	switch (opt) {
		case 'W':
		{
			int r_code;

			r_code = snprintf(buf, len, "%u", buddy->warn_level);
			if (r_code < 0 || (size_t) r_code >= len)
				ret = -1;
			break;
		}

		default:
			return (-1);
	}

	return (ret);
}

static int format_blist_buddy_label(char opt, char *buf, size_t len, va_list ap)
{
	struct buddy *buddy = va_arg(ap, struct buddy *);
	int ret = 0;

	switch (opt) {
		case 'B':
		case 'b':
		{
			char *status_text;

			if (buddy->status == STATUS_ACTIVE)
				status_text = opt_get_str(OPT_TEXT_BUDDY_ACTIVE);
			else if (buddy->status == STATUS_AWAY)
				status_text = opt_get_str(OPT_TEXT_BUDDY_AWAY);
			else if (buddy->status == STATUS_IDLE)
				status_text = opt_get_str(OPT_TEXT_BUDDY_IDLE);
			else
				status_text = "%p?%x";

			if (xstrncpy(buf, status_text, len) == -1)
				ret = -1;

			break;
		}

		case 'N':
		case 'n':
			if (xstrncpy(buf, buddy->name, len) == -1)
				ret = -1;
			break;

		case 'I':
			ret = fill_format_str(OPT_FORMAT_BLIST_IDLE, buf, len, buddy);
			break;

		case 'i':
			if (buddy->idle_time > 0)
				ret = fill_format_str(OPT_FORMAT_BLIST_IDLE, buf, len, buddy);
			break;

		case 'W':
			ret = fill_format_str(OPT_FORMAT_BLIST_WARN, buf, len, buddy);
			break;

		case 'w':
			if (buddy->warn_level > 0)
				ret = fill_format_str(OPT_FORMAT_BLIST_WARN, buf, len, buddy);
			break;

		default:
			return (-1);
	}

	return (ret);
}

static int format_blist_group_label(char opt, char *buf, size_t len, va_list ap)
{
	struct bgroup *group = va_arg(ap, struct bgroup *);
	dlist_t *node = group->blist_line;
	int ret = 0;

	switch (opt) {
		case 'E':
		{
			char *expand_str;

			if (node != NULL) {
				struct slist_cell *cell = node->data;

				if (cell != NULL && cell->collapsed) {
					expand_str =
						opt_get_str(OPT_TEXT_BLIST_GROUP_COLLAPSED);
				} else {
					expand_str =
						opt_get_str(OPT_TEXT_BLIST_GROUP_EXPANDED);
				}
			} else {
				expand_str =
					opt_get_str(OPT_TEXT_BLIST_GROUP_COLLAPSED);
			}

			if (xstrncpy(buf, expand_str, len) == -1)
				ret = -1;
			break;
		}

		case 'N':
		case 'n':
			if (xstrncpy(buf, group->name, len) == -1)
				ret = -1;
			break;

		case 'T':
		case 't':
		{
			int r_code;

			r_code = snprintf(buf, len, "%u", group->num_members);
			if (r_code < 0 || (size_t) r_code >= len)
				ret = -1;
			break;
		}

		case 'O':
		case 'o':
		{
			int r_code;

			r_code = snprintf(buf, len, "%u", group->num_online);
			if (r_code < 0 || (size_t) r_code >= len)
				ret = -1;
			break;
		}

		default:
			return (-1);
	}

	return (ret);
}

static int format_file_transfer(char opt, char *buf, size_t len, va_list ap) {
	struct file_transfer *xfer = va_arg(ap, struct file_transfer *);
	struct pork_acct *acct = xfer->acct;
	int ret;

	switch (opt) {
		/* File transfer reference number */
		case 'I':
			ret = snprintf(buf, len, "%u", xfer->refnum);
			break;

		/* Actual file name */
		case 'N':
			ret = xstrncpy(buf, xfer->fname_local, len);
			break;

		/* Original (requested) file name */
		case 'n':
			ret = xstrncpy(buf, xfer->fname_orig, len);
			break;

		/* Local IP address */
		case 'L':
			ret = xstrncpy(buf, xfer->laddr_ip, len);
			break;

		/* Local hostname */
		case 'l':
			ret = xstrncpy(buf, transfer_get_local_hostname(xfer), len);
			break;

		/* Remote IP address */
		case 'F':
			ret = xstrncpy(buf, xfer->faddr_ip, len);
			break;

		/* Remote hostname */
		case 'f':
			ret = xstrncpy(buf, transfer_get_remote_hostname(xfer), len);
			break;

		/* Local port */
		case 'P':
			ret = snprintf(buf, len, "%d", xfer->lport);
			break;

		/* Remote port */
		case 'p':
			ret = snprintf(buf, len, "%d", xfer->fport);
			break;

		/* Average transfer rate */
		case 'R':
			ret = snprintf(buf, len, "%.04f", transfer_avg_speed(xfer));
			break;

		/* File size */
		case 'S':
			ret = snprintf(buf, len, "%lld", xfer->file_len);
			break;

		/* Number of bytes sent */
		case 's':
			ret = snprintf(buf, len, "%lld", xfer->bytes_sent);
			break;

		/* Time elapsed since the start of the transfer in seconds */
		case 'T':
			ret = snprintf(buf, len, "%.04f", transfer_time_elapsed(xfer));
			break;

		/* Local user */
		case 'U':
			ret = xstrncpy(buf, acct->username, len);
			break;

		/* Remote user */
		case 'u':
			ret = xstrncpy(buf, xfer->peer_username, len);
			break;

		default:
			return (-1);
	}

	if (ret < 0 || (size_t) ret >= len)
		return (-1);

	return (0);
}

static int (*const format_handler[])(char, char *, size_t, va_list) = {
	format_blist_buddy_label,	/* OPT_FORMAT_BLIST					*/
	format_blist_group_label,	/* OPT_FORMAT_BLIST_GROUP			*/
	format_blist_idle,			/* OPT_FORMAT_BLIST_IDLE			*/
	format_blist_warn,			/* OPT_FORMAT_BLIST_WARN			*/
	format_chat_arginfo,		/* OPT_FORMAT_CHAT_CREATE			*/
	format_chat_arginfo,		/* OPT_FORMAT_CHAT_IGNORE			*/
	format_chat_info,			/* OPT_FORMAT_CHAT_INFO				*/
	format_chat_arginfo,		/* OPT_FORMAT_CHAT_INVITE			*/
	format_chat_arginfo,		/* OPT_FORMAT_CHAT_KICK				*/
	format_chat_msg,			/* OPT_FORMAT_CHAT_RECV				*/
	format_chat_msg,			/* OPT_FORMAT_CHAT_RECV_NOTICE		*/
	format_chat_msg,			/* OPT_FORMAT_CHAT_SEND				*/
	format_chat_msg,			/* OPT_FORMAT_CHAT_SEND_NOTICE		*/
	format_chat_msg,			/* OPT_FORMAT_CHAT_TOPIC			*/
	format_chat_arginfo,		/* OPT_FORMAT_CHAT_UNIGNORE			*/
	format_file_transfer,		/* OPT_FORMAT_FILE_RECV_ACCEPT		*/
	format_file_transfer,		/* OPT_FORMAT_FILE_RECV_ASK			*/
	format_file_transfer,		/* OPT_FORMAT_FILE_RECV_COMPLETE	*/
	format_file_transfer,		/* OPT_FORMAT_FILE_SEND_ACCEPT		*/
	format_file_transfer,		/* OPT_FORMAT_FILE_SEND_ASK			*/
	format_file_transfer,		/* OPT_FORMAT_FILE_SEND_COMPLETE	*/
	format_privmsg,				/* OPT_FORMAT_IM_RECV				*/
	format_privmsg,				/* OPT_FORMAT_IM_RECV_AUTO			*/
	format_privmsg,				/* OPT_FORMAT_IM_SEND				*/
	format_privmsg,				/* OPT_FORMAT_IM_SEND_AUTO			*/
	format_privmsg,				/* OPT_FORMAT_NOTICE_RECV			*/
	format_privmsg,				/* OPT_FORMAT_NOTICE_SEND			*/
	format_status,				/* OPT_FORMAT_STATUS				*/
	format_status_activity,		/* OPT_FORMAT_STATUS_ACTIVITY		*/
	format_status_held,			/* OPT_FORMAT_STATUS_HELD			*/
	format_status_idle,			/* OPT_FORMAT_STATUS_IDLE			*/
	format_status_timestamp,	/* OPT_FORMAT_STATUS_TIMESTAMP		*/
	format_status_typing,		/* OPT_FORMAT_STATUS_TYPING			*/
	format_status_warn,			/* OPT_FORMAT_STATUS_WARN			*/
	format_warning,				/* OPT_FORMAT_WARN					*/
};

/*
** Do all appropriate '$' substitutions in format strings, except for
** $>, which must be handled after everything else.
*/

int fill_format_str(int type, char *buf, size_t len, ...) {
	va_list ap;
	char *format;
	size_t i = 0;
	int (*handler)(char, char *, size_t, va_list);

	format = opt_get_str(type);
	if (format == NULL)
		return (-1);

	handler = format_handler[type - OPT_FORMAT_OFFSET];

	va_start(ap, len);
	--len;

	while (*format != '\0' && i < len) {
		if (*format == FORMAT_VARIABLE) {
			char result[len + 1];
			int ret_code;

			format++;

			result[0] = '\0';
			ret_code = handler(*format, result, sizeof(result), ap);
			if (ret_code == 0) {
				/*
				** The subsitution was successful.
				*/
				int ret;

				ret = xstrncpy(&buf[i], result, len - i);
				if (ret == -1)
					break;

				i += ret;
			} else if (ret_code == 1) {
				/*
				** We should suppress the whole string.
				*/

				*buf = '\0';
				break;
			} else {
				/*
				** They specified an invalid option -- treat
				** their '$' as a literal.
				*/

				if (i > len - 2)
					break;

				buf[i++] = FORMAT_VARIABLE;
				buf[i++] = *format;
			}
		} else
			buf[i++] = *format;

		format++;
	}

	buf[i] = '\0';

	va_end(ap);
	return (0);
}

void format_apply_justification(char *buf, chtype *ch, size_t len) {
	char *p = buf;
	char *left = NULL;
	char *right = NULL;
	size_t len_left;
	chtype fill_char;

	while ((p = strchr(p, '$')) != NULL) {
		if (p[1] == '>') {
			left = buf;

			*p = '\0';
			right = &p[2];
			break;
		}
	}

	len_left = plaintext_to_cstr(buf, ch, len);
	fill_char = ch[len_left - 1];
	chtype_set(fill_char, ' ');

	/*
	** If there's right-justified text, paste it on.
	*/
	if (right != NULL) {
		chtype ch_right[len];
		size_t len_right;
		size_t diff;
		size_t i;
		size_t j;

		plaintext_to_cstr(right, ch_right, len);
		len_right = cstrlen(ch_right);

		/*
		** If the right side won't fit, paste as much
		** as possible, leaving at least one cell of whitespace.
		*/

		if (len_left + len_right >= len - 1) {
			int new_rlen;

			new_rlen = len_right - ((len_left + 1 + len_right) - (len - 1));
			if (new_rlen < 0)
				len_right = 0;
			else
				len_right = new_rlen;
		}

		diff = len_left + (--len - (len_left + len_right));

		i = len_left;
		while (i < diff)
			ch[i++] = fill_char;

		j = 0;
		while (i < len && j < len_right && ch_right[j] != 0)
			ch[i++] = ch_right[j++];

		ch[i] = 0;
	} else if (len_left < len - 1) {
		/*
		** If there's no right-justified text, pad out the
		** string to its maximum length with blank characters (spaces)
		** having the color/highlighting attributes of the last character
		** in the string.
		*/
		size_t i = len_left;

		--len;
		while (i < len)
			ch[i++] = fill_char;

		ch[i] = 0;
	}
}
