/*
	in.pop3gwd, a POP3 proxy
	Copyright (C) 1997 Andrea Borgia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* get_remote_data.c:	read user command, parse and verify it */
/* ----------------------------------------------------------- */

#include "pop3-gw.h"



static int parse_cmd(char input[], char delim, int maxlen, const char user_c[],
	const char quit_c[], const char pos_re[], const char neg_re[], 
	char error[], char cmd[], char username[], char hostname[]) {
	
	int result = GOOD;
	char *next_tok;
	char *delim_pos;
	
	
	/* we must find the separator now, otherwise strtok messes things up */
	delim_pos = strrchr(input, delim);

	/* look for user command, can be USER or QUIT */
	if ((next_tok = strtok(input, " ")) != NULL)
		strncpy(cmd, next_tok, maxlen);
	else {
		result = BAD;
		strncpy(error, "no command in input", maxlen);
		cmd[0] = 0;	/* we terminate the string anyway, better be safe than sorry */
	}

	if (result == GOOD && strcasecmp(cmd, user_c) == 0) {
		/* look for hostname to connect to (after last delimiter, if present) */
		if (delim_pos != NULL) {
			strncpy(hostname, delim_pos+1, MAXHOSTNAMELEN);
			*delim_pos = 0;	/* terminate the string here, so strtok will ignore this part */			
			if (strlen(hostname) != 0) {
				if ((next_tok = strtok(NULL, " ")) != NULL) {
					strncpy(username, next_tok, maxlen);
				}
				else {
					result = BAD;
					strncpy(error, "no username in input", maxlen);
				}
			
			}
			else {
				result = BAD;
				strncpy(error, "no hostname in input", maxlen);
			}
		}
		else {
			result = BAD;
			strncpy(error, "no username/hostname delimiter in input", maxlen);
		}
	}
	
	/* command *must* be either USER or QUIT */
	if (result == GOOD) {
		if (strcasecmp(cmd, user_c) != 0 && strcasecmp(cmd, quit_c) != 0) {
			result = BAD;
			strncpy(error, "command must be either USER or QUIT", maxlen);
		}
	}
	
	if (result != GOOD) {
		cmd[0] = 0;		/* all strings are returned empty */
		username[0] = 0;	/* in case of incorrect input */
		hostname[0] = 0;
	}

	return(result);
}



/*
 * get remote command and parse it; return only if command is valid or disconnected
*/

int get_remote_data(int client_filedes, char username[], char hostname[], 
	int maxlen, int maxwait, char termin[], char delim, const char user_c[],
	const char quit_c[], const char pos_re[], const char neg_re[], 
	int *in, int *out) {
	
	int connected = TRUE;	/* we are here only if connected is TRUE */
	int parse_res;		/* GOOD if a valid command has been entered */
	char input[MAX_IO_LEN], output[MAX_IO_LEN], error[MAX_IO_LEN], cmd[MAX_IO_LEN];
	char server_name[MAXHOSTNAMELEN];
	int count;
	
	
	do {
		if ((count = readline(client_filedes, input, MAX_IO_LEN, maxwait, termin)) != BAD) {
			(*in) += count;
			if ((parse_res = parse_cmd(input, delim, MAX_IO_LEN, user_c, quit_c, pos_re, neg_re, error, cmd, username, hostname)) != GOOD) {
#ifdef DEBUG
				syslog(LOG_PRIO, "%s", error);
#endif
				snprintf(output, MAX_IO_LEN, "%s %s%s", neg_re, error, termin);
				if ((count = writen(client_filedes, output, strlen(output), maxwait)) == strlen(output))
					*out += count;
				else
					connected = FALSE;
			}		
		}
		else
			connected = FALSE;
	} while (connected == TRUE && parse_res == BAD);

#ifdef NOTDEF
	if (parse_res == GOOD)
		syslog(LOG_PRIO, "cmd: %s; username: %s; hostname: %s", cmd, username, hostname);
#endif

	if (parse_res == GOOD && strcasecmp(cmd, quit_c) == 0) {
		/* set server's hostname and setup farewell */
		if (gethostname(server_name, MAXHOSTNAMELEN) != 0)
			strncpy(server_name, "localhost", MAXHOSTNAMELEN);
		snprintf(output, MAX_IO_LEN, "%s %s %s %s%s", pos_re, server_name,
				GREETING, "signing off", termin);
		if  ((count = writen(client_filedes, output, strlen(output), maxwait)) == strlen(output))
			*out += count;
		close(client_filedes);
		connected = FALSE;		
	}
	
	return(connected);
}
