
% common utilities 
:- use_module(library('/org/cs3/pdt/util/pdt_util')).

% deal with annotated terms
:- use_module(library('/org/cs3/pdt/util/pdt_util_aterm')).

% rb-tree based, non-destructive associative data structure
:- use_module(library('/org/cs3/pdt/util/pdt_util_map')).

% deal with subterm_positions terms as returned by read_term
:- use_module(library('/org/cs3/pdt/util/pdt_util_term_position')).

% attach comments to the corresponding subterms
:- use_module(library('/org/cs3/pdt/util/pdt_util_comments')).

% comment_positions(+Comments, -Positions)
%
% extract the character offsets from a list of comments as returned by read_term/*
comment_positions([],[]).
comment_positions([CPos-_|Cs],[Position|Positions]):-
    stream_position_data(char_count,CPos,Position),
    comment_positions(Cs,Positions).


% comments_map(+Comments,-Map)
%
% create a associative datastructure that maps comment positions to comment strings
% Comments is a list of comments as returned by read_term
% Map is a associative map. (See pdt_util_map) 
% The map contains character offsets of comments as keys and comment strings as values.
comments_map(Comments,Map):-
    pdt_map_empty(Map0),
    comments_map(Map0,Comments,Map).

% comments_map(+MapIn, +Comments,-MapOut)
%
% add position --> comment mappings to an existing map.
% Comments is a list of comments as returned by read_term
% Map is a associative map. (See pdt_util_map) 
% The map contains character offsets of comments as keys and comment strings as values.    
comments_map(In,[],In).
comments_map(In,[CPos-Comment|Cs],Out):-
    stream_position_data(char_count,CPos,Position),
    pdt_map_put(In,Position,Comment,Next),
    comments_map(Next,Cs,Out).



/*
i changed the semantic of the embed predicate.
What it does now is attaching the comment positions to the 
corresponding subterms.
*/

% embed_comments(+ATermIn,+CPositions,-ATermOut)
%
% attach comments to the corresponding subterms 
%
% ATermIn is an annotated Term as generated by wrap_term/6
% CPositions is a list of character offsets as generated by comment_positions/2
% ATermOut will be unified with an annotated Term based on ATermIn, but with added
% annotations comments_left and comments_right. Both have as value a list of
% character positions of comments.
embed_comments(ATermIn,CPositions,ATermOut) :-
	pdt_term_annotation(ATermIn,_,Annos),
	pdt_member(file_ref(FileRef),Annos),
	pdt_file_spec(file_ref(FileRef),File),
	open(File,read,Stream),
	b_getval(comments,Map),
	pdt_attach_comments(ATermIn,Map,Stream,CPositions,_,ATermOut),
	close(Stream).

    

		
	

reformat_comment(Comment) :-
	atom_chars(Comment, Chars),
	Chars = [First|_],
	( First = '%' ->
		write(Comment), nl
	;
		append([_,_|Nub], [_,_], Chars),
		format("\n\n/*\n  "), 
		reformat_comment_(Nub),
		format("\n*/\n\n")
	).

reformat_comment_([]).
reformat_comment_([C|Cs]) :-
	( C = '\n' ->
		( Cs \== [] -> format("\n  ") ; true )
	;
		format(C)
	),
	reformat_comment_(Cs).


plpp(File) :-
    pdt_file_ref(File,FileRef),
	open(File, read, In),
	read_term(In, Term, [subterm_positions(Pos),variable_names(Names),comments(Comments)]),
	term_predicate(Term, Pred),
	(	Term==end_of_file
	->	true
	;	% create an annotated term
		wrap_term(Term,Pos,FileRef,0,ATerm0,_),
		
		% generate the comments table into Map and store it.
		% note: The use of b_setval is for the sake of simplicity.
		% In a bigger context, the Map would probably not stored using b_setval/2. 
		% For instance in my framework, i would store it together with other, file-level annotations, 
		% once per processed source file and propagate it as argument while processing the file.
		comments_map(Comments,Map),		
	    b_setval(comments,Map),
	    
	    % extract comment positions and attach them to the subterms.
	    comment_positions(Comments,Positions),
		embed_comments(ATerm0,Positions,ATerm),
		
		% recurse.
		call_cleanup(pp(ATerm, Names, Pred, In), close(In))
		
	).


/*added a clause here to handle annotated terms */
term_predicate(ATerm, Pred) :-
    pdt_aterm(ATerm),
    !,
    pdt_strip_annotation(ATerm,Term,_),
    term_predicate(Term,Pred).
term_predicate(Term, Pred) :-
	functor(Term, Functor, Arity),
	( Functor == (:-) ->
		% rule or declaration
		arg(1, Term, Head),
		functor(Head, F, A),
		( F == module, A =:= 2 ->
			arg(2, Head, PublicList),
			module_ops(PublicList)
		; F == op, A =:= 3 ->
			call(Head)
		; F == use_module, A =:= 1 ->
			call(Head)
		;
			true
		),
		Pred = F/A
	;
		% fact
		Pred = Functor/Arity
	).

module_ops([]).
module_ops([P|Ps]) :-
	( functor(P, op, _) -> call(P) ; true ),
	module_ops(Ps).


/* this case is now caught above */
%pp(end_of_file, _, _, _) :- !.

pp(ATerm, Names, Previous, In) :-
	term_predicate(ATerm, Pred),
	( Pred \== Previous -> nl, nl ; true ),
	bind_names(Names),
	
	% portray_clause unfortunately does not call the portray hook.
	% So i use print and i try to achieve a moreless readable result by
	% defining a portray/1 hook for annotated terms below. 
	% The result should be interpreted as a "proove of concept" :-)
	print(ATerm),
	read_term(In, Term2, [subterm_positions(Pos),variable_names(Names2),comments(Comments)]),
	( Comments \== [] -> nl ; true ),
	(	Term2==end_of_file	
	->	b_getval(comments,Map0),
		comments_map(Map0,Comments,Map),
		b_setval(comments,Map),
		comment_positions(Comments,Positions),
		print_comments(Positions)
	;	% basically the same as above.
	 	pdt_term_annotation(ATerm,_,Annos),
		pdt_member(file_ref(FileRef),Annos),
		pdt_member(last_n(N),Annos),
		% increment subterm counter. For this example this is irrelevant, 
		% but it is important in my context.
		M is N+1, 
		wrap_term(Term2,Pos,FileRef,M,ATerm0,_),
		
		% update comments table
		b_getval(comments,Map0),
		comments_map(Map0,Comments,Map),
		b_setval(comments,Map),
		
		comment_positions(Comments,Positions),
		embed_comments(ATerm0,Positions,ATerm2),
		
		pp(ATerm2, Names2, Pred, In)
	).
	

bind_names([]).
bind_names([Name = Var|T]) :-
	Var = '$VAR'(Name),
	bind_names(T).


% this definitly needs work. But at least it prints comments and subterms in the correct order.
portray(ATerm):-
    pdt_aterm(ATerm),
    pdt_term_annotation(ATerm,SubTerm,Annos),
    (	pdt_member(comments_left(Cs),Annos)
    ->	print_comments(Cs)
    ;	true
    ),    
    print(SubTerm),
    (	pdt_member(comments_right(Cs2),Annos)
    ->	print_comments(Cs2)
    ;	true
    ).

print_comments([]).
print_comments([C|Cs]):-
	b_getval(comments,Map),
	pdt_map_get(Map,C,Comment),
	reformat_comment(Comment),
	print_comments(Cs).
    


