/* $Id: pfstat.c,v 1.1.1.1 2007/01/11 16:01:58 dhartmei Exp $ */

/*
 * Copyright (c) 2002-2006, Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer. 
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

static const char rcsid[] = "$Id: pfstat.c,v 1.1.1.1 2007/01/11 16:01:58 dhartmei Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "data.h"
#include "graph.h"
#include "pf.h"

extern int	 parse_config(const char *, struct matrix **);

struct col {
	unsigned	 nr;
	int		 type;
	char		 arg[128];
	int		 idx;
	int		 diff;
	double		 val;
} cols[512];

unsigned maxcol = 0;
unsigned since = 0;
int debug = 0;

int
add_col(unsigned nr, int type, const char *arg, int idx, int diff)
{
	int i;

	if (type < COL_TYPE_GLOBAL || type > COL_TYPE_QUEUE) {
		fprintf(stderr, "add_col: type %d invalid\n", type);
		return (1);
	}
	for (i = 0; i < maxcol; ++i) {
		if (cols[i].nr == nr) {
			fprintf(stderr, "add_col: %d already defined\n", nr);
			return (1);
		}
	}
	if (maxcol == sizeof(cols) / sizeof(cols[0])) {
		fprintf(stderr, "add_col: limit of %d collects reached\n",
		    maxcol);
		return (1);
	}
	cols[maxcol].nr = nr;
	cols[maxcol].type = type;
	strlcpy(cols[maxcol].arg, arg, sizeof(cols[maxcol].arg));
	cols[maxcol].idx = idx;
	cols[maxcol].diff = diff;
	maxcol++;
	return (0);
}

static void
set_col(int type, const char *arg, int idx, double val)
{
	int i;

	if (type == COL_TYPE_SINCE) {
		since = val;
		return;
	}
	for (i = 0; i < maxcol; ++i) {
		if (cols[i].type != type || cols[i].idx != idx ||
		    (strcmp(arg, "-") && strcmp(cols[i].arg, arg)))
			continue;
		cols[i].val = val;
	}
}

static void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-v] [-c config] [-d data] [-r host[:port]] "
	    "[-p] [-q] [-t days[:days]]\n", __progname);
	exit(1);
}

int
main(int argc, char *argv[])
{
	const char *configfn = "/etc/pfstat.conf";
	const char *datafn = "/var/db/pfstat.db";
	const char *addr = NULL;
	const char *serv = "9999";
	int ch, query = 0, draw = 0, trunc = 0, i;
	int days[2] = { 31, 365 };
	struct matrix *matrices = NULL, *m;
	struct graph *g;

	while ((ch = getopt(argc, argv, "c:d:pqr:t:v")) != -1) {
		switch (ch) {
		case 'c':
			configfn = optarg;
			break;
		case 'd':
			datafn = optarg;
			break;
		case 'p':
			draw = 1;
			break;
		case 'q':
			query = 1;
			break;
		case 'r': {
			char *o, *p;

			o = strdup(optarg);
			if (!o) {
				fprintf(stderr, "main: strdup: %s\n",
				    strerror(errno));
				return (1);
			}
			p = strchr(o, ':');
			if (p != NULL) {
				*p = 0;
				serv = p + 1;
			}
			addr = o;
			break;
		}
		case 't': {
			char *o, *p;

			o = strdup(optarg);
			if (!o) {
				fprintf(stderr, "main: strdup: %s\n",
				    strerror(errno));
				return (1);
			}
			p = strchr(o, ':');
			if (p != NULL) {
				*p = 0;
				days[1] = atoi(p + 1);
			}
			days[0] = atoi(o);
			if (days[0] <= 0 || days[1] <= 0)
				usage();
			trunc = 1;
			break;
		}
		case 'v':
			debug++;
			break;
		default:
			usage();
		}
	}
	if (argc != optind)
		usage();
	if (!query && !draw && !trunc)
		usage();

	if (parse_config(configfn, &matrices))
		return (1);

	if (data_open(datafn))
		return (1);

	if (trunc) {

		if (debug)
			printf("truncating database\n");
		if (data_truncate(days[0], days[1])) {
			fprintf(stderr, "main: data_truncate() failed\n");
			return (1);
		}

	}

	if (query) {

		if (debug)
			printf("querying values\n");
		if (addr != NULL) {
			int fd = -1;
			struct addrinfo hints, *res0 = NULL, *res;
			int gai_err;
			FILE *f;
			char h[256], s[256], arg[256];
			int type, idx;
			double val;

			if (debug)
				printf("querying remote %s:%s\n", addr, serv);
			memset(&hints, 0, sizeof(hints));
			hints.ai_family = PF_UNSPEC;
			hints.ai_socktype = SOCK_STREAM;
			hints.ai_protocol = IPPROTO_TCP;
			if ((gai_err = getaddrinfo(addr, serv, &hints,
			    &res0))) {
				fprintf(stderr, "main: getaddrinfo: %s:%s: "
				    "%s\n", addr, serv, gai_strerror(gai_err));
				return (1);
			}
			for (res = res0; res; res = res->ai_next) {
				if ((gai_err = getnameinfo(res->ai_addr,
				    res->ai_addrlen, h, sizeof(h), s, sizeof(s),
				    NI_NUMERICHOST|NI_NUMERICSERV))) {
					fprintf(stderr, "main: getnameinfo: "
					    "%s\n", gai_strerror(gai_err));
					continue;
				}
				if (debug > 0)
					printf("trying %s:%s...\n", h, s);
				if ((fd = socket(res->ai_family,
				    res->ai_socktype, res->ai_protocol)) < 0) {
					fprintf(stderr, "main: socket: %s\n",
					    strerror(errno));
					continue;
				}
				if (connect(fd, res->ai_addr,
				    res->ai_addrlen)) {
					fprintf(stderr, "main: connect: %s:%s: "
					    "%s\n", h, s, strerror(errno));
					close(fd);
					fd = -1;
					continue;
				}
				if (debug > 0)
					printf("connected to %s:%s\n", h, s);
				break;
			}
			if (fd < 0)
				return (1);
			freeaddrinfo(res0);
			if ((f = fdopen(fd, "r")) == NULL) {
				fprintf(stderr, "main: fdopen: %s\n",
				    strerror(errno));
				return (1);
			}
			while (fgets(s, sizeof(s), f) != NULL) {
				if (sscanf(s, "%d %s %d %lf", &type, arg,
				    &idx, &val) != 4)
					fprintf(stderr, "main: sscanf: %s", s);
				else
					set_col(type, arg, idx, val);
			}
			fclose(f);
		} else {
			int fd;

			if (debug > 0)
				printf("querying local /dev/pf\n");
			if ((fd = open("/dev/pf", O_RDONLY)) < 0) {
				fprintf(stderr, "main: open: /dev/pf: %s\n",
				    strerror(errno));
				return (1);
			}
			if (pf_query(fd, set_col))
				return (1);
			close(fd);
		}

		if (debug)
			printf("storing values in database\n");
		for (i = 0; i < maxcol; ++i)
			if (data_put_value(since, time(NULL), cols[i].nr,
			    cols[i].val, cols[i].diff)) {
				fprintf(stderr, "main: data_put_value() "
				    "failed\n");
				return (1);
			}

	}

	if (draw) {

		if (debug)
			printf("generating images\n");
		for (m = matrices; m != NULL; m = m->next)
			for (i = 0; i < 2; ++i)
				for (g = m->graphs[i]; g != NULL; g = g->next) {
					if (debug)
						printf("fetching values for "
						    "unit %u from database\n",
						    g->desc_nr);
					if (data_get_values(g->desc_nr, m->beg,
					    m->end, g->type, m->w0, g->data)) {
						fprintf(stderr, "main: "
						    "data_get_values() "
						    "failed\n");
						return (1);
					}
				}
		if (debug)
			printf("drawing and writing images\n");
		if (graph_generate_images(matrices)) {
			fprintf(stderr, "main: graph_generate_images() "
			    "failed\n");
			return (1);
		}

	}

	data_close();
	return (0);
}
