
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 


#include "pflogx.h"
#include "CmdLineParser.h"
#include "EntriesFilter.h"
#include "LogEntryFormat.h"
#include "LogParser.h"
#include "XmlFileWriter.h"

#ifdef WITH_EXPAT
#include "XmlFileParser.h"
#include "LogEntryParser.h"
#endif

#include <iostream>
#include <stdlib.h>


/* Constructor */
cPfLogx::cPfLogx(int argc, char** argv) : m_argc(argc), m_argv(argv), m_options(), m_vectLogEntries()
{
}


/* Main */
int cPfLogx::Main()
{
	int nRet=EXIT_FAILURE;

	// Parse command line
	enum eParseCmdLineRet eParseCmdLine=ParseCmdLine();
	
	if (eParseCmdLine == PARSECMDLINE_CONTINUE)
	{
		// Parse XML output file (if "-m" command line switch is defined)
		if (ParseXmlFile())
		{
			// Parse pflog input file
			if (ParseLogFile())
			{
				// Write XML output file
				if (WriteXmlFile())
				{
					nRet=EXIT_SUCCESS;
				}
			}
		}
	}
	else if (eParseCmdLine == PARSECMDLINE_EXIT_SUCCESS)
		nRet=EXIT_SUCCESS;
	else if (eParseCmdLine == PARSECMDLINE_EXIT_FAILURE)
		nRet=EXIT_FAILURE;
	
	return nRet;
}


/* Parse command line */
cPfLogx::eParseCmdLineRet cPfLogx::ParseCmdLine()
{
	enum eParseCmdLineRet eRet=PARSECMDLINE_CONTINUE;

	// Parse command line
	cCmdLineParser cmdLineParser(m_argc, m_argv);
	switch(cmdLineParser.Parse())
	{
		case cCmdLineParser::PARSE_NO_ERROR:
		{
			// Store options
			m_options=cmdLineParser.GetOptions();
			break;
		}
		
		case cCmdLineParser::PARSE_HELP_DISPLAY:
		{
			DisplayUsage();
			eRet=PARSECMDLINE_EXIT_SUCCESS;
			break;
		}
	
		case cCmdLineParser::PARSE_VERSION_DISPLAY:
		{
			DisplayVersion();
			eRet=PARSECMDLINE_EXIT_SUCCESS;
			break;
		}

		case cCmdLineParser::PARSE_ERROR:
		{
			cerr<<"Option error: "<<cmdLineParser.GetErrorString()<<"."<<endl;
			eRet=PARSECMDLINE_EXIT_FAILURE;
			break;
		}
	}

	return eRet;
}


/* Parse pflog file */
bool cPfLogx::ParseLogFile()
{
	bool bRet=true;

	cLogParser logParser;
	vector<pair<int, enum cLogParser::eParsePacketError> > vectLogErrors;

	// Parse pflog input file
	if (logParser.ParseLog(m_options.GetInputFile(), m_vectLogEntries, vectLogErrors))
	{
		// Display erroneous packets
		if (m_options.GetDisplayErrorsPackets())
		{
			for(vector<pair<int, enum cLogParser::eParsePacketError> >::const_iterator it=vectLogErrors.begin(); it!=vectLogErrors.end(); it++)
			{
				int packet_num=(*it).first;
				const string packet_error=logParser.GetParsePacketErrorMessage((*it).second);
			
				cerr<<"Error in packet "<<packet_num<<": "<<packet_error<<"."<<endl;
			}
		}
	}
	else
	{
		bRet=false;
		cerr<<"Libpcap error: "<<logParser.GetPcapErrorMessage()<<"."<<endl;
	}

	return bRet;
}


/* Parse XML file */
bool cPfLogx::ParseXmlFile()
{
	bool bRet=true;

#ifdef WITH_EXPAT

	if (m_options.GetMergeEvents())
	{
		// Entry parser
		cLogEntryParser entryParser;
	
		// Parse XML file
		cXmlFileParser xmlParser(m_options.GetOutputFile(), entryParser);
		enum cXmlFileParser::eParseRet parseRet=xmlParser.Parse();

		if ((parseRet == cXmlFileParser::PARSE_NO_ERROR) || (parseRet == cXmlFileParser::PARSE_INPUTFILE_OPEN_ERROR))
		{
			// It's not an error if we could not open the output file, file may not exist
			
			// Store entries
			m_vectLogEntries=xmlParser.GetLogEntries();  		
		}
		else
		{
			bRet=false;
			cerr<<"XML file parsing error: "<<xmlParser.GetErrorString()<<"."<<endl;
		}
	}

#endif

	return bRet;
}


/* Write XML output file */
bool cPfLogx::WriteXmlFile() const
{
	bool bRet=true;

	// Filter entries
	cEntriesFilter entriesFilter(m_vectLogEntries, m_options); 
	entriesFilter.ApplyFilters();
		
	// Entry format
	cLogEntryFormat entryFormat(m_options);

	// Write output XML file
	cXmlFileWriter xmlWriter(m_options.GetOutputFile(), entryFormat, entriesFilter.GetEntries());
		
	if (!xmlWriter.Write())
	{
		bRet=false;
		cerr<<"XML file writing error: "<<xmlWriter.GetErrorString()<<"."<<endl;
	}

	return bRet;
}


/* Display program usage */
void cPfLogx::DisplayUsage() const
{
	cout<<"pflogx - Export packet filter logs to XML"<<endl;
	cout<<"Copyright (c) 2005, Arnaud KLEIN "<<endl;
	cout<<endl;
	cout<<"Usage:"<<endl;
	cout<<"    "<<m_argv[0]<<" [OPTIONS]"<<endl;
	cout<<endl;
	cout<<"Options:"<<endl;
	cout<<"    -i <input_file>: Packet filter logfile (Default: standard input)"<<endl;
	cout<<"    -o <output_file>: Output XML file (Default: standard output)"<<endl;
#ifdef WITH_EXPAT
	cout<<"    -m: Merge new events with already existing events in output XML file"<<endl;
#endif
	cout<<"    -a <filter>: Define filter on action (Default: no filter)"<<endl;
	cout<<"    -d <filter>: Define filter on direction (Default: no filter)"<<endl;
	cout<<"    -p <filter>: Define filter on protocol (Default: no filter)"<<endl;
	cout<<"    -n <filter>: Define filter on interface (Default: no filter)"<<endl;
	cout<<"    -t <h|s>: Specify output format for date (Default: \"h\")"<<endl;
	cout<<"    -r: Sort events in reverse chronological order"<<endl;
	cout<<"    -e: Show erroneous packets"<<endl;
	cout<<"    -h: Display program usage"<<endl;
	cout<<"    -v: Display program version"<<endl;
}


/* Display program version */
void cPfLogx::DisplayVersion() const
{
	cout<<"pflogx - Export packet filter logs to XML"<<endl;
	cout<<"Copyright (c) 2005, Arnaud KLEIN "<<endl;
	cout<<endl;
	cout<<"Version "<<PFLOGX_VERSION<<endl;
}


/* main */
int main(int argc, char** argv)
{
	cCmdLineParser::Initialize();
	
	cPfLogx p(argc, argv);
	return p.Main();
}
